/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License version 3 as published by the Free Software Foundation;
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib.h>
#include <gio/gio.h>
#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>
#include <gconf/gconf-client.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>

#include "launcher-util.h"

#define GCONF_ICON_KEY "icon_name"
#define ICON_SIZE 24
#define DEFAULT_ICON_NAME "distributor-logo"
#define XDG_OPEN "xdg-open"

typedef struct 
{
  GtkWidget    *image;
  GtkWidget    *applet;
  
} GoHomeApp;

/* D&D stuff */
static const GtkTargetEntry drop_types[] = 
{
	{ "STRING", 0, 0 }
};
static const gint n_drop_types = G_N_ELEMENTS (drop_types);

/* Forwards */
static void cw_panel_background_changed (PanelApplet               *applet,
                                         PanelAppletBackgroundType  type,
				                                 GdkColor                  *colour,
				                                 GdkPixmap                 *pixmap,
                                         gpointer                   user_data);
static void display_about_dialog (BonoboUIComponent *component, 
                                  gpointer           user_data, 
                                  const gchar       *verb);

static const char Context_menu_xml [] =
   "<popup name=\"button3\">\n"
  "   <menuitem name=\"About Item\" "
   "             verb=\"MenuAbout\" "
   "           _label=\"_About\"\n"
   "          pixtype=\"stock\" "
   "          pixname=\"gtk-about\"/>\n"
   "</popup>\n";

static const BonoboUIVerb _menu_verbs [] = {
        BONOBO_UI_VERB ("MenuAbout", display_about_dialog),
        BONOBO_UI_VERB_END
}; 

static const gchar *close_window_authors [] = {
	"Neil J. Patel <neil.patel@canonical.com>",
	NULL
};

static gchar *
get_applet_icon_name (PanelApplet *applet)
{
  gchar *icon_name;

  icon_name = panel_applet_gconf_get_string (applet, GCONF_ICON_KEY, NULL);
  if (icon_name == NULL || icon_name[0] == 0)
    icon_name = g_strdup (DEFAULT_ICON_NAME);

  return icon_name;
}


static void
set_icon (GoHomeApp *app)
{
  gchar *icon_name;

  icon_name = get_applet_icon_name (PANEL_APPLET (app->applet));
  gtk_image_set_from_icon_name (GTK_IMAGE (app->image), icon_name, ICON_SIZE);
  g_free (icon_name);
}


/* GConf callback */
static void
icon_changed_cb (GConfClient *client,
                 guint        connection_id,
                 GConfEntry  *entry,
                 gpointer     data)
{
  GoHomeApp *app;

  app = (GoHomeApp*)data;

  set_icon (app);
}

static void
on_drag_data_recieved (GtkWidget        *widget, 
                       GdkDragContext   *context,
				               gint              x,      
                       gint              y, 
                       GtkSelectionData *selection_data,
                       guint             target_type, 
                       guint             time,
                       GoHomeApp        *app)
{
  gchar   *_sdata = NULL;
  gboolean dnd_success = FALSE;
  gboolean delete_selection_data = FALSE;
    
  /* Deal with what we are given from source */
  if((selection_data != NULL) && (selection_data-> length >= 0))
  {
    /* Check that we got the format we can use */
    _sdata = (gchar*)selection_data-> data;
    g_debug ("Received %s\n", _sdata);
    dnd_success = TRUE;

  } 
  
  _sdata = g_strdup (_sdata);
  _sdata = g_strstrip (_sdata);
  
  if (g_str_has_prefix (_sdata, "file:///") ||
      g_str_has_prefix (_sdata, "http") ||
      g_str_has_prefix (_sdata, "ftp"))
  {
    gint argc;
    gchar **argv;
    gchar *args = g_strdup_printf ("netbook-launcher --add-favorite %s", _sdata);
    
    if (g_shell_parse_argv (args, &argc, &argv, NULL))
    {
      g_spawn_async (NULL,
                     argv,
                     NULL,
                     G_SPAWN_SEARCH_PATH,
                     NULL,
                     NULL,
                     NULL,
                     NULL);
      g_strfreev (argv);
    }
    g_free (args);
    
  }
  else
  {
    dnd_success = FALSE;
  }
  
  gtk_drag_finish (context, dnd_success, delete_selection_data, time);
}


/* Event callbacks */
static gboolean
on_button_press (GtkWidget      *eb, 
                 GdkEventButton *event,
                 GoHomeApp *app)
{
  g_return_val_if_fail (app, FALSE);

  if (event->button != 1)
    return FALSE;
  
  int argc;
  gchar **argv;
  
  if (g_shell_parse_argv ("netbook-launcher --show-favorites", &argc, &argv, NULL))
    {
      g_spawn_async (NULL,
                     argv,
                     NULL,
                     G_SPAWN_SEARCH_PATH,
                     NULL,
                     NULL,
                     NULL,
                     NULL);
      g_strfreev (argv);
    }

  return TRUE;
}

static gboolean
cw_applet_fill (PanelApplet *applet, 
                const gchar *iid, 
                gpointer     data)
{
  GoHomeApp *app;
  GtkWidget *eb;
  AtkObject *atk;
  gchar *key;

  if (strcmp (iid, "OAFIID:GNOME_GoHome") != 0)
		return FALSE;
  	
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  app = g_slice_new0 (GoHomeApp);

  app->applet = GTK_WIDGET (applet);
  panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);
  gtk_container_set_border_width (GTK_CONTAINER (applet), 0);
  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), 
                               _("Click here to hide all windows and show the home screen"));

  panel_applet_add_preferences (applet, "/schemas/apps/go-home-applet/prefs", NULL);

  key = panel_applet_gconf_get_full_key (applet, GCONF_ICON_KEY);
  gconf_client_notify_add (gconf_client_get_default (), key, icon_changed_cb,
                           app, NULL, NULL);
  g_free (key);

  eb = gtk_event_box_new ();
  g_object_set (eb, 
                "above-child", TRUE, 
                "visible-window", FALSE, 
                "border-width", 0, 
                NULL);
	gtk_container_add (GTK_CONTAINER (applet), eb);

  /* D&D */
	gtk_widget_add_events (GTK_WIDGET (eb),GDK_ALL_EVENTS_MASK);
	gtk_drag_dest_set (GTK_WIDGET (eb),
                           GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
                           drop_types, n_drop_types,
                           GDK_ACTION_COPY);
	gtk_drag_dest_add_uri_targets (GTK_WIDGET (eb));
	gtk_drag_dest_add_text_targets (GTK_WIDGET (eb));

  g_signal_connect (eb, "drag-data-received",
			              G_CALLBACK(on_drag_data_recieved), app);

  /* The actual icon */
  app->image = gtk_image_new ();
  set_icon (app);
  gtk_container_add (GTK_CONTAINER (eb), app->image);

	gtk_widget_show_all (GTK_WIDGET (applet));
		
  atk = gtk_widget_get_accessible (app->image);
  atk_object_set_name (atk, _("Go Home"));
  atk_object_set_description (atk, _("Go home, show the home window."));
  atk_object_set_role (atk, ATK_ROLE_PUSH_BUTTON);
  
  /* Signals */
	g_signal_connect (applet, "change-background",
			             G_CALLBACK (cw_panel_background_changed), NULL);
  g_signal_connect (eb, "button-release-event",
                    G_CALLBACK (on_button_press), app);
  g_signal_connect (applet, "button-release-event",
                    G_CALLBACK (on_button_press), app);
 
  panel_applet_setup_menu (PANEL_APPLET (applet),
                         	Context_menu_xml,
                         	_menu_verbs,
                         	applet);
	
	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_GoHome_Factory",
                             PANEL_TYPE_APPLET,
                             "Show Desktop",
                             "0",
                             cw_applet_fill,
                             NULL);

static void 
cw_panel_background_changed (PanelApplet               *applet,
                             PanelAppletBackgroundType  type,
                             GdkColor                  *colour,
                             GdkPixmap                 *pixmap,
                             gpointer                   user_data)
{
  GtkRcStyle *rc_style;
  GtkStyle *style;

  /* reset style */
  gtk_widget_set_style (GTK_WIDGET (applet), NULL);
  rc_style = gtk_rc_style_new ();
  gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
  gtk_rc_style_unref (rc_style);

  switch (type) 
  {
    case PANEL_NO_BACKGROUND:
      break;
    case PANEL_COLOR_BACKGROUND:
      gtk_widget_modify_bg (GTK_WIDGET (applet), GTK_STATE_NORMAL, colour);
      break;
    
    case PANEL_PIXMAP_BACKGROUND:
      style = gtk_style_copy (GTK_WIDGET (applet)->style);
      if (style->bg_pixmap[GTK_STATE_NORMAL])
        g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
      style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);
      gtk_widget_set_style (GTK_WIDGET (applet), style);
      g_object_unref (style);
      break;
  }
}

static void
display_about_dialog (BonoboUIComponent *component, 
                      gpointer           user_data, 
                      const gchar       *verb)
{
  GtkWidget *panel_about_dialog;
  gchar *icon_name;
  PanelApplet *applet = (PanelApplet*)user_data;
	
  panel_about_dialog = gtk_about_dialog_new ();

  icon_name = get_applet_icon_name (applet);
  g_object_set (panel_about_dialog,
                "name", _("Go Home"),
                "comments", _("Go Home"),
                "version", "0.1",
                "authors", close_window_authors,
                "logo-icon-name", icon_name,
                "icon-name", icon_name,
                "copyright", "Copyright \xc2\xa9 2008 Canonical Ltd",
                NULL);
  g_free (icon_name);

  gtk_widget_show (panel_about_dialog);

  g_signal_connect (panel_about_dialog, "response",
                    G_CALLBACK (gtk_widget_destroy), NULL);
	

  gtk_window_present (GTK_WINDOW (panel_about_dialog));
}
