/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef DESKTOP_H
#define DESKTOP_H

/*
 * a desktop; a collection width * height workspaces.  workspaces exist to provide the
 * user with independant focus lists for seperate screens of a large virtual desktop;
 * which is a _lot_ nicer if you use click-focus modes: the pointer people probably
 * wont notice.
 *
 * when switching either desktops or just workspaces on the same desktop, a client
 * with flags.sticky set to true will "stick" to the glass.  this works by not moving
 * it/unmaping it, and we maintain it's status in terms of focus after adding it to
 * the other focuslist.  so if it is focused it will stay focused, and if not, it will
 * not become focused, even if there are no windows on the destination workspace.
 *
 * workspaces work by moving windows in the opposite direction that the viewport
 * is moved; the viewport _must_ be moved in screen-sized icrements (otherwise
 * maintaining independant focus lists would be really really odd). if a window is
 * visible across two of the screen-viewport boundries, it will be added to the newly
 * visible workspace's focus list, and removed from the old one.  every screen has
 * at least one desktop, which can be any size, as long as it has at least
 * one workspace (1x1).
 */
struct desktop {
	int		num;		/* unique (on each screen) desktop number */

	int		width, height;	/* dimensions */
	int		viewx, viewy;	/* current viewport location */
	workspace_t	**workspaces;	/* grid of workspaces on this desktop */
	workspace_t	*current_space;	/* in-use workspace */

	/*
	 * stacking lists; this is an array of tailq heads for
	 * each client stacking layer.  a count of the total
	 * number of windows on this desktop (in all stacking
	 * layers) is maintained in window_count.
	 */
	TAILQ_HEAD(stacklayer, client)	stacking_list[STACKLAYER_COUNT];
	int		window_count;

	TAILQ_ENTRY(desktop) d_list;
};

void desktop_shutdown();
int desktop_create(screen_t *screen, int width, int height, int count);
void desktop_remove(screen_t *screen);
void desktop_add_client(client_t *client);
void desktop_rm_client(client_t *client);
void desktop_switch(screen_t *screen, int num);

#endif
