/*
 * Copyright (c) 2011 Paulo Zanoni
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <cstring>
#include <iostream>
#include <set>

extern "C" {
#include <sys/types.h>
#include <dirent.h>
}

#include "XDGAutostart.h"
#include "XDGUtils.h"

static bool strEndsWith(const char *s, const char *substr)
{
    size_t i;
    size_t s_len = strlen(s);
    size_t substr_len = strlen(substr);

    if (substr_len > s_len)
	return false;

    for(i = s_len - substr_len; i < s_len; i++)
	if (s[i] != substr[i - (s_len - substr_len)])
	    return false;

    return true;
}


void XDGAutostart::run(const char *environment)
{
    std::list<std::string> autostartDirs;
    std::list<std::string>::iterator it;

    std::set<std::string> autostartFiles;
    DIR *autostartDir;
    struct dirent *dirEntry;

    autostartDirs = XDGBasedir::findConfigFile("autostart");

    for(it = autostartDirs.begin(); it != autostartDirs.end(); ++it) {

	autostartDir = opendir(it->c_str());
	if (!autostartDir) {
	    std::cerr << "Error opening " << *it << " directory\n";
	    return;
	}

	while ((dirEntry = readdir(autostartDir))) {
	    if (strEndsWith(dirEntry->d_name, ".desktop")) {
		XDGUtils::log(XDGUtils::Verbose, "File found: %s\n",
			      dirEntry->d_name);

		if (autostartFiles.count(dirEntry->d_name) == 0) {
		    std::string fullFileName = *it + std::string("/") +
			std::string(dirEntry->d_name);
		    runDesktopFile(fullFileName.c_str(), environment);
		    autostartFiles.insert(dirEntry->d_name);
		} else {
		    std::cout << "not running: repeated\n";
		}
	    }
	}

	if (closedir(autostartDir) != 0)
	    std::cerr << "Error: failed to close dir " << it->c_str() << "\n";
    }
}

int XDGAutostart::runDesktopFile(const char *fileName,
				 const char *environment)
{
    XDGDesktopEntry entry(fileName);
    if (entry.showInEnvironment(environment))
	return entry.run("");

    return 0;
}
