# dbplugs/todo.py
#
#

""" provides todo related commands """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import strtotime, striptime, getwho, today, \
handle_exception
from gozerbot.commands import cmnds
from gozerbot.examples import examples
from gozerbot.users import users
from gozerbot.aliases import aliases
from gozerbot.plughelp import plughelp
from gozerbot.db import db
from gozerplugs.plugs.alarm import alarms
import time

plughelp.add('todo', 'manage todo lists .. by user or by channel')

class Todo(object):

    """ database todo interface """

    def size(self):
        result = db.execute(""" SELECT COUNT(*) FROM todo """)
        return result[0][0]
        
    def get(self, name):
        """ get todo list of <name> """
        name = name.lower()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s ORDER BY \
priority DESC, indx ASC """, name)
        return result

    def getid(self, idnr):
        """ get todo data of <idnr> """
        result = db.execute(""" SELECT * FROM todo WHERE indx = %s """, idnr)
        return result

    def setprio(self, todonr, prio):
        """ set priority of todonr """
        result = db.execute(""" UPDATE todo SET priority = %s WHERE indx \
= %s """, (prio, todonr))
        return result

    def getprio(self, todonr):
        """ get priority of todonr """
        result = db.execute(""" SELECT name, priority FROM todo WHERE \
indx = %s """, todonr)
        return result

    def getwarnsec(self, todonr):
        """ get priority of todonr """
        result = db.execute(""" SELECT warnsec FROM todo WHERE \
indx = %s """, todonr)
        return result

    def settime(self, todonr, ttime):
        """ set time of todonr """
        result = db.execute(""" UPDATE todo SET time = %s WHERE indx \
= %s """, (ttime, todonr))
        return result

    def add(self, name, txt, ttime, alarmnr=None):
        """ add a todo """
        name = name.lower()
        txt = txt.strip()
        if ttime:
            if not alarmnr:
                result = db.execute(""" INSERT INTO todo(name, time, \
descr) VALUES (%s, %s, %s) """, (name, ttime, txt))
            else:
                result = db.execute(""" INSERT INTO todo(name, time, \
descr, warnsec) VALUES (%s, %s, %s, %s) """, (name, ttime, txt, 0-alarmnr))

        else:
            result = db.execute(""" INSERT INTO todo(name, descr) \
VALUES (%s, %s) """, (name, txt))
        return result

    def delete(self, name, indexnr):
        """ delete todo item """
        name = name.lower()
        try:
            warnsec = self.getwarnsec(indexnr)[0][0]
            if warnsec:
                alarmnr = 0 - warnsec
                if alarmnr > 0:
                    alarms.delete(alarmnr)
        except (IndexError, TypeError):
            pass
        result = db.execute(""" DELETE FROM todo WHERE name = %s AND \
indx = %s """, (name, indexnr))
        return result

    def toolate(self, name):
        """ show if there are any time related todoos that are too late """
        name = name.lower()
        now = time.time()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s AND \
time < %s """, (name, now))
        return result

    def withintime(self, name, time1, time2):
        """ show todo list within time frame """
        name = name.lower()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s AND \
time > %s AND time < %s """, (name, time1, time2))
        return result

todo = Todo()

def size():
    return todo.size()

def handle_todo(bot, ievent):
    """ todo [<item>] .. show todo list or set todo item """
    if len(ievent.args) > 0:
        handle_todo2(bot, ievent)
        return
    name = users.getname(ievent.userhost)
    saytodo(bot, ievent, name)

def handle_todo2(bot, ievent):
    """ set todo item """
    if not ievent.rest:
        ievent.missing("<what>")
        return
    else:
        what = ievent.rest
    name = users.getname(ievent.userhost)
    ttime = strtotime(what)
    res = 0
    try:
        if not ttime  == None:
            ievent.reply('time detected ' + time.ctime(ttime))
            what = striptime(what)
            alarmnr = alarms.add(bot.name, ievent.nick, ttime, what)
            res = todo.add(name, what, ttime, alarmnr)
        else:
            res = todo.add(name, what, None)
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if res:
        ievent.reply('todo added')
    else:
        ievent.reply('add failed')

cmnds.add('todo', handle_todo, 'USER')
examples.add('todo', 'todo [<item>] .. show todo items or add a todo \
item', '1) todo 2) todo program the bot')

def handle_todoshow(bot, ievent):
    """ todo-show <nr> .. show todo item data """
    try:
        todonr = int(ievent.args[0])
    except (IndexError, ValueError):
        ievent.missing("<nr>")
        return
    name = users.getname(ievent.userhost)
    result = todo.get(name)
    nrs = []
    if not result:
        ievent.reply('nothing todo ;]')
        return
    for i in result:
        nrs.append(i[0])
    if todonr not in nrs:
        ievent.reply('%s is not on your todo list' % todonr)
        return
    result = todo.getid(todonr)
    if result:
        res = list(result[0])
        res[2] = time.ctime(res[2])
        ievent.reply("indx: %s name: %s time: %s duration: %s \
warnsec: %s descr: %s priority: %s" % tuple(res))
    else:
        ievent.reply("can't fetch data for %s" % todonr)

cmnds.add('todo-show', handle_todoshow, 'USER')
examples.add('todo-show', 'todo-show <nr> .. show data of todo item', \
'todo-show 10')

def handle_todosettime(bot, ievent):
    """ todo-settime [<channel|name] <nr> <prio> .. set todo item time"""
    ttime = strtotime(ievent.rest)
    if not ttime:
        ievent.reply("can't detect time")
        return
    what = striptime(ievent.rest).split()
    try:
        (name, todonr) = what
        todonr = int(todonr)
    except ValueError:
        try:
            todonr = int(what[0])
            name = users.getname(ievent.userhost)
        except (ValueError, IndexError):
            ievent.missing("[<channel>] <todnr> <timestring>")
            return
    name = name.lower()
    result = todo.getid(todonr)
    if not result:
        ievent.reply("can't find todo %s for %s" % (todonr, name))
        return
    if result[0][1] != name:
        ievent.reply('%s is not in %s todolist' % (todonr, name))
        return
    ok = 0
    try:
        ok = todo.settime(todonr, ttime)
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if ok:
        ievent.reply('time of todo %s set to %s' % (todonr, time.ctime(ttime)))
    else:
        ievent.reply('set time failed')

cmnds.add('todo-settime', handle_todosettime, 'USER')
examples.add('todo-settime', 'todo-settime [<channel|name>] <nr> <timestring> \
.. set todo time', '1) todo-settime #dunkbots 2 22:00 2) todo-settime 2 22:00')

def handle_setpriority(bot, ievent):
    """ todo-setprio [<channel|name] <nr> <prio> .. set todo item priority"""
    try:
        (name, todonr, prio) = ievent.args
        todonr = int(todonr)
        prio = int(prio)
    except ValueError:
        try:
            todonr = int(ievent.args[0])
            prio = int(ievent.args[1])
            name = users.getname(ievent.userhost)
        except (ValueError, IndexError):
            ievent.missing("[<channel>] <todnr> <prio>")
            return
    name = name.lower()
    result = todo.getid(todonr)
    if not result:
        ievent.reply("can't find todo %s for %s" % (todonr, name))
        return
    if result[0][1] != name:
        ievent.reply('%s is not in %s todolist' % (todonr, name))
        return
    ok = 0
    try:
        ok = todo.setprio(todonr, prio)
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if ok:
        ievent.reply('priority set')
    else:
        ievent.reply('set priority failed')

cmnds.add('todo-setprio', handle_setpriority, 'USER')
examples.add('todo-setprio', 'todo-setprio [<channel|name>] <nr> <prio> .. \
set todo priority', '1) todo-setprio #dunkbots 2 5 2) todo-setprio 2 10')
aliases.data['setprio'] = 'todo-setprio'

def handle_getpriority(bot, ievent):
    """ todo-prio [<channel|name>] <nr> .. get priority of todo item"""
    try:
        (name, itemnr) = ievent.args
        itemnr = int(itemnr)
    except ValueError:
        try:
            itemnr = int(ievent.args[0])
            name = users.getname(ievent.userhost)
        except (IndexError, ValueError):
            ievent.missing('[<channel|name>] <itemnr>')
            return
    name = name.lower()
    prio = todo.getprio(itemnr)
    if prio: 
        if prio[0][0] != name:
            ievent.reply("todonr %s is not yours" % itemnr)
            return
        ievent.reply('priority is %s' % prio[0][1])
    else:
        ievent.reply('no priority found for todoitem %s' % itemnr)

cmnds.add('todo-prio', handle_getpriority, 'USER')
examples.add('todo-prio', 'todo-prio [<channel|name>] <nr> .. get todo \
priority', '1) todo-prio #dunkbots 5 2) todo-prio 3')
aliases.data['prio'] = 'todo-prio'

def handle_tododone(bot, ievent):
    """ todo-done <listofnrs> .. remove todo items """
    if len(ievent.args) == 0:
        ievent.missing('<list of nrs>')
        return
    try:
        nrs = []
        for i in ievent.args:
            nrs.append(int(i))
    except ValueError:
        ievent.reply('%s is not an integer' % i)
        return
    name = users.getname(ievent.userhost)
    if not name:
        ievent.reply("can't find name of %s" % ievent.userhost)
        return
    teller = 0
    failed = []
    try:
        for i in nrs:
            result = todo.delete(name, i)
            if not result:
                failed.append(str(i))
            else:
                teller += 1
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if failed:
        ievent.reply('failed to delete %s' % " ".join(failed))
    if teller == 1:
        ievent.reply('1 item deleted')
    elif teller == 0:
        ievent.reply('no items deleted')
    else:
        ievent.reply('%s items deleted' % teller)

cmnds.add('todo-done', handle_tododone, 'USER')
examples.add('todo-done', 'todo-done <listofnrs> .. remove items from todo \
list', '1) todo-done 1 2) todo-done 3 5 8')
aliases.data['done'] = 'todo-done'

def handle_settodo(bot, ievent):
    """ todo-set <nick> <txt> .. add to todo of another user """
    try:
        who = ievent.args[0]
        what = ' '.join(ievent.args[1:])
    except IndexError:
        ievent.missing('<nick> <txt>')
        return
    if not what:
        ievent.missing('<nick> <txt>')
        return
    userhost = getwho(bot, who)
    if not userhost:
        ievent.reply("can't find userhost for %s" % who)
        return
    whoname = users.getname(userhost)
    if not whoname:
        ievent.reply("can't find user for %s" % userhost)
        return
    name = users.getname(ievent.userhost)
    if not users.permitted(userhost, name, 'todo'):
        ievent.reply("%s doesn't permit todo sharing for %s " % \
(whoname, name))
        return
    what = "%s: %s" % (ievent.nick, what)
    ttime = strtotime(what)
    result = 0
    try:
        if not ttime  == None:
            ievent.reply('time detected ' + time.ctime(ttime))
            what = striptime(what)
            alarmnr = alarms.add(bot.name, who, ttime, what)
            result = todo.add(whoname, what, ttime, alarmnr)
        else:
            result = todo.add(whoname, what, None)
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if result:
        ievent.reply('todo added')
    else:
        ievent.reply('add failed')

cmnds.add('todo-set', handle_settodo, 'USER')
examples.add('todo-set', 'todo-set <nick> <txt> .. set todo item of \
<nick>', 'todo-set dunker bot proggen')

def handle_gettodo(bot, ievent):
    """ todo-get <nick> .. get todo of other user """
    try:
        who = ievent.args[0]
    except IndexError:
        ievent.missing('<nick>')
        return
    userhost = getwho(bot, who)
    if not userhost:
        ievent.reply("can't find userhost for %s" % who)
        return
    whoname = users.getname(userhost)
    if not whoname:
        ievent.reply("can't find user for %s" % userhost)
        return
    name = users.getname(ievent.userhost)
    if not users.permitted(userhost, name, 'todo'):
        ievent.reply("%s doesn't permit todo sharing for %s " % \
(who, name))
        return
    result = todo.get(whoname)
    if result:
        res = []
        for i in result:
            res.append("%s) %s" % (i[0], i[5]))
        ievent.reply(res)
    else:
        ievent.reply('%s has nothing todo' % who)

cmnds.add('todo-get', handle_gettodo, ['USER', 'WEB'])
examples.add('todo-get', 'todo-get <nick> .. get the todo list of \
<nick>', 'todo-get dunker')

def handle_chantodo(bot, ievent):
    """ todo-chan [<txt>] .. show todoos or set todo item for channel"""
    if ievent.rest:
        handle_chantodo2(bot, ievent)
        return
    saytodo(bot, ievent, ievent.channel)

def handle_chantodo2(bot, ievent):
    """ set todo item for channel"""
    what = ievent.rest
    ttime = strtotime(what)
    ok = 0
    try:
        if not ttime  == None:
            ievent.reply('time detected ' + time.ctime(ttime))
            result = '(%s) ' % ievent.nick + striptime(what)
            alarmnr = alarms.add(bot.name, ievent.channel, ttime, result)
            ok = todo.add(ievent.channel, result, ttime, alarmnr)
        else:
            result = '(%s) ' % ievent.nick + what
            ok = todo.add(ievent.channel, result, None)
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if ok:
        ievent.reply('todo item added')
    else:
        ievent.reply('add failed')

cmnds.add('todo-chan', handle_chantodo, 'USER')
examples.add('todo-chan', 'todo-chan [<txt>] .. add channel todo or show \
channel todoos', '1) todo-chan 2) todo-chan fix bla')
aliases.data['chantodo'] = 'todo-chan'

def handle_chandone(bot, ievent):
    """ todo-chandone <listofnrs> .. remove channel todo item """
    if not ievent.rest:
        ievent.missing('<listofnrs>')
        return
    data = ievent.rest.split()
    try:
        nrs = []
        for i in data:
            nrs.append(int(i))
    except ValueError:
        ievent.reply('%s is not an integer' % i)
        return
    teller = 0
    failed = []
    try:
        for i in nrs:
            result = todo.delete(ievent.channel, i)
            if not result:
                failed.append(str(i))
            else:
                teller += 1
    except Exception, ex:
        handle_exception()
        ievent.reply('ERROR: %s' % str(ex))
        return
    if failed:
        ievent.reply('failed to delete %s' % ' '.join(failed))
    if teller == 1:
        ievent.reply('1 item deleted')
    elif teller == 0:
        ievent.reply('no items deleted')
    else:
        ievent.reply('%s items deleted' % teller)

cmnds.add('todo-chandone', handle_chandone, 'USER')
examples.add('todo-chandone', 'todo-chandone <listofnrs> .. remove item \
from channel todo list', '1) todo-chandone 2 2) todo-chandone 1 4 7')
aliases.data['chandone'] = 'todo-chandone'

def handle_todotime(bot, ievent):
    """ todo-time .. show time related todoos """
    name = users.getname(ievent.userhost)
    if not name:
        ievent.reply("can't find name of %s" % ievent.userhost)
        return
    saytodotime(bot, ievent, name)

cmnds.add('todo-time', handle_todotime, 'USER')
examples.add('todo-time', 'show todo items with time fields', 'todo-time')
aliases.data['tt'] = 'todo-time'

def handle_todoweek(bot, ievent):
    """ todo-week .. show time related todo items for this week """
    name = users.getname(ievent.userhost)
    if not name:
        ievent.reply("can't find name of %s" % ievent.userhost)
        return
    saytodotime2(bot, ievent, name, today(), today()+7*24*60*60)

cmnds.add('todo-week', handle_todoweek, 'USER')
examples.add('todo-week', 'todo items for this week', 'todo-week')

def handle_today(bot, ievent):
    """ todo-today .. show time related todo items for today """
    name = users.getname(ievent.userhost)
    if not name:
        ievent.reply("can't find name of %s" % ievent.userhost)
        return
    saytodotime2(bot, ievent, name, today(), today()+24*60*60)

cmnds.add('todo-today', handle_today, 'USER')
examples.add('todo-today', 'todo items for today', 'todo-today')
aliases.data['today'] = 'todo-today'

def handle_tomorrow(bot, ievent):
    """ todo-tomorrow .. show time related todo items for tomorrow """
    name = users.getname(ievent.userhost)
    if ievent.rest:
        what = ievent.rest
        ttime = strtotime(what)
        if ttime != None:
            if ttime < today() or ttime > today() + 24*60*60:
                ievent.reply("can't set time on %s" % \
time.ctime(ttime + 24*60*60))
                return
            ttime += 24*60*60
            ievent.reply('time detected ' + time.ctime(ttime))
            what = striptime(what)
        else:
            ttime = today() + 42*60*60
        ok = 0
        try:
            ok = todo.add(name, what, ttime)
        except Exception, ex:
            handle_exception()
            ievent.reply('ERROR: %s' % str(ex))
            return
        if ok:
            ievent.reply('todo added')
        else:
            ievent.reply('add failed')
        return
    saytodotime2(bot, ievent, name, today()+24*60*60, today()+2*24*60*60)

cmnds.add('todo-tomorrow', handle_tomorrow, 'USER')
examples.add('todo-tomorrow', 'todo items for tomorrow', 'todo-tomorrow')
aliases.data['tomorrow'] = 'todo-tomorrow'

def saytodo(bot, ievent, name):
    """ output todo items of <name> """
    res = []
    result = todo.get(name)
    if not result:
        ievent.reply('nothing todo ;]')
        return
    now = time.time()
    for i in result:
        txt = ""
        if i[2] and i[2] < now:
            txt = "%s) " % i[0]
            txt += 'TOO LATE: '
            ttime = time.ctime(i[2])
            txt += "%s %s" % (ttime, i[5])
        else:
            txt = "%d) %s" % (i[0], i[5])
        if i[6]:
            txt += " [%+d]" % i[6]
        res.append(txt)                    
    ievent.reply(res, nritems=True)

def saytodotime(bot, ievent, name):
    """ say time related todo items of <name> """
    result = todo.get(name)
    if not result:
        ievent.reply('nothing todo ;]')
        return
    res = []
    now = time.time()
    for i in result:
        txt = "%s) " % i[0]
        if not i[2]:
            continue
        if i[2] < now:
            txt += 'TOO LATE: '
        ttime = time.ctime(i[2])
        txt += "%s %s" % (ttime, i[5])
        if i[6]:
            txt += " [%+d]" % i[6]
        res.append(txt)
    if res:
        ievent.reply("todolist of %s: " % ievent.nick, res, nritems=True)
    else:
        ievent.reply('nothing todo ;]')

def saytodotime2(bot, ievent, name, time1, time2):
    """ output todo items within time frame """
    result = todo.withintime(name, time1, time2)
    if not result:
        ievent.reply('nothing todo ;]')
        return
    now = time.time()
    res = []
    for i in result:
        txt = "%s) " % i[0]
        if i[2] < now:
            txt += 'TOO LATE: '
        txt += "%s %s" % (time.ctime(i[2]), i[5])
        if i[6]:
            txt += " [%+d]" % i[6]
        res.append(txt)
    ievent.reply("timed todolist of %s: " % ievent.nick, res, nritems=True)
