/*
Copyright (C) 2000-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "header.h"

static int multi=0;
static int newprivvar=1;
extern int initnode;
extern int optstrict;
extern int linecount;
extern int safecoeff;

int newdecl(int flag, int t, int initval, int *v)
{
  char s[33];
  sprintf(s,"%d",newprivvar++);
  *v=newnode(Fentry,newentry(strdup(s)),-1);
  return pushvar(*v,flag,t,initval);
}
int newcall(const char *s, int y)
{
  return newnode(Ffunction,newentry(s),y);
}

int stacktofun(int fun, int *stack, int nb)
{
  int n,i;
  if (nb==-1) return -1;
  for(n=stack[0],i=1;i<nb;n=newnode(fun,n,stack[i++]));
  return n;
}

int stacktoargs(int *stack, int nb)
{
  return stacktofun(Flistarg,stack,nb);
}

int stacktoseq(int *stack, int nb)
{
  return stacktofun(Fseq,stack,nb);
}

int newseq(int x, int y)
{
  int seq;
  if (tree[x].f==Fnoarg || tree[y].f==Fnoarg)
  {
    if (tree[x].f==Fnoarg && tree[y].f==Fnoarg)
      return GNIL;
    else if (tree[x].f==Fnoarg)
      return y;
    else
      return x;
  }
  seq=newnode(Fseq,x,y);
  return seq;
}

int addlistleft(int n, int l)
{
  int i,b;
  if (l==GNOARG)
    return n;
  if (tree[l].f!=Flistarg)
    return newnode(Flistarg,n,l);
  for(i=l;tree[tree[i].x].f==Flistarg;i=tree[i].x);
  b=newnode(Flistarg,n,tree[i].x);
  tree[i].x=b;
  return l;
}

int addseqleft(int n, int seq)
{
  int i;
  int bseq;
  if(seq==-1)
    return n;
  if (tree[seq].f!=Fseq)
    return newnode(Fseq,n,seq);
  for(i=seq;tree[tree[i].x].f==Fseq;i=tree[i].x);
  bseq=newnode(Fseq,n,tree[i].x);
  tree[i].x=bseq;
  return seq;
}
int addseqright(int seq, int n)
{
  if(seq==-1)
    return n;
  return newnode(Fseq,seq,n);
}

static int insertreturn(int y)
{
  if (isfunc(y,"return"))
    return y;
  return newcall("return",y);
}

int geninsertreturn(int seq)
{
  int bseq;
  if (seq==GNOARG) return seq;
  if (tree[seq].f==Fseq)
  {
    bseq=insertreturn(tree[seq].y);
    bseq=newnode(Fseq,tree[seq].x,bseq);
  }
  else
    bseq=insertreturn(seq);
  return bseq;
}

int geninsertvar(int seq, int ret)
{
  int bseq;
  if (seq==-1)
    return -1;
  if (ret>=0)
  {
    if (tree[seq].f==Fseq)
    {
      bseq=newnode(Fassign,ret,tree[seq].y);
      bseq=newnode(Fseq,tree[seq].x,bseq);
    }
    else
      bseq=newnode(Fassign,ret,seq);
  }
  else
    bseq=seq;
  return bseq;
}
int geninsertvarop(int seq, int ret, OPerator op)
{
  int bseq;
  if (seq==-1)
    return -1;
  if (ret>=0)
  {
    if (tree[seq].f==Fseq)
    {
      bseq=newopcall(op,ret,tree[seq].y);
      bseq=newnode(Fseq,tree[seq].x,bseq);
    }
    else
      bseq=newopcall(op,ret, seq);
  }
  else
    bseq=seq;
  return bseq;
}

int newcoeff(int n, int x, int y)
{
  int arg[3];
  int nb=y<0?2:3;
  arg[0]=n; arg[1]=x; arg[2]=y;
  return newnode(Ffunction,OPcoeff,stacktoargs(arg,nb));
}

int newcoeffsafe(int n, int x, int y)
{
  int arg[3];
  int nb=y<0?2:3;
  arg[0]=n; arg[1]=x; arg[2]=y;
  if (!safecoeff)
    return newnode(Ffunction,OPcoeff,stacktoargs(arg,nb));
  else
    return newcall("_safecoeff",stacktoargs(arg,nb));
}

void makeblock(int bl, int n, int aseq, int ret, int savx)
{
  /*create a block*/
  tree[n].f=Fblock;
  tree[n].x=bl;
  block[bl].ret=ret;
  tree[n].y=aseq;
  copyctx(savx,block+bl);
  restorectx(savx);
}
void makeblocks(int bl1, int bl2, int n, int bseq, int aseq, int ret, int savx)
{
  int y;
  tree[n].f=Fblock;
  tree[n].x=bl1;
  block[tree[n].x].ret=ret;
  y=addseqright(bseq,newnode(Fblock,bl2,aseq));
  tree[n].y=y;
  copyctx(savx,block+bl2);
  restorectx(savx);
}
void makeblocks3(int bl1, int bl2, int bl3, int n, int bseq, int aseq, int ret, int sav0, int savx)
{
  int y;
  tree[n].f=Fblock;
  tree[n].x=bl1;
  block[tree[n].x].ret=ret;
  y=addseqright(bseq,newnode(Fblock,bl2,newnode(Fblock,bl3,aseq)));
  tree[n].y=y;
  copyctx(savx,block+bl3);
  restorectx(savx);
  copyctx(sav0,block+bl2);
  restorectx(sav0);
}

void affectval(int vn, int val, int *seq)
{
  ctxvar *v = ctxstack+vn;
  if (val<0) die(-1,"Internal error: affectval");
  int valn=newnode(Fassign,newleaf(v->node),val);
  *seq=addseqright(*seq,valn);
}

int genblockdeclaration(int args, int n, int flag, int type, int *seq)
{
  int stack[STACKSZ];
  int i;
  enum {local,global,function} decl;
  int mint=(type>=0)?type:Ggen;
  int nb=listtostack(args,Flistarg,stack,STACKSZ,"function declaration",n);
  if (nb==1 && stack[0]==GNOARG)
    nb--;
  decl=(flag&(1<<Carg))?function:(flag&(1<<Cglobal))?global:local;
  for(i=0;i<nb;i++)
  {
    int var,tv;
    int val=-1;
    switch(tree[stack[i]].f)
    {
    case Ftag:
      var=tree[stack[i]].x;
      tv=tree[stack[i]].y;
      if (tree[var].f!=Fentry)
        die(stack[i],"Incorrect declaration");
      if (decl!=function && autogc && ctype[tv]==Vgen)
        /*Make sure GEN objects are gerepilable.*/
        val = newnode(Ftag, newnode(Ftag, newsmall(0), Ggen), tv);
      if (decl==global)
      {
        int f=fillvar(var,flag,tv,-1);
        if (autogc && ctype[tv]==Vgen)
          affectval(f,val,seq);
      }
      else
        pushvar(var,flag,tv,val);
      break;
    case Fentry:
      switch(decl)
      {
      case local:
        if (type<0 || autogc)
          /*Make sure (implicitly GEN) objects are gerepilable.*/
          val=newsmall(0);
        pushvar(stack[i],flag,mint,val);
        break;
      case global:
        if (type<0)
        {
          val=newcall("_const_quote",newstringnode(entryname(stack[i]),-1));
          affectval(fillvar(stack[i],flag,mint,-1),val,seq);
        }
        else
        {
          int f = fillvar(stack[i],flag,mint,-1);
          /*Make sure (implicitly GEN) objects are gerepilable.*/
          if (autogc)
            affectval(f,  newsmall(0), seq);
        }
        break;
      case function:
        if (!optstrict)
          val=newsmall(0);
        pushvar(stack[i],flag,mint,val);
        break;
      }
      break;
    case Fassign:
      switch (decl)
      {
      case global:
        genequal(stack[i],"declaration",&var,&val,&tv);
        if (tv==Gnotype)
          tv=mint;
        affectval(fillvar(var,flag,tv,-1),val,seq);
        genblock(val,n);
        break;
      case local:
        {
          int x = tree[stack[i]].x;
          if (tree[x].f==Fvec)
          {
            int vars[STACKSZ];
            int j, ret;
            int nb=listtostack(tree[x].x,Fmatrixelts,vars,STACKSZ," declaration",n);
            val = tree[stack[i]].y;
            newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&ret);
            *seq = addseqright(*seq, geninsertvar(val,ret));
            for (j=0;j<nb;j++)
            {
              checkisvar(vars[j],"my",&var,&tv);
              if (tv==Gnotype)
                tv=mint;
              affectval(pushvar(var,flag,tv,-1),newcoeff(newleaf(ret),newsmall(j+1),-1),seq);
            }
          } else
          {
            genequal(stack[i],"declaration",&var,&val,&tv);
            if (tv==Gnotype)
              tv=mint;
            if (tree[val].f==Fsmall)
              pushvar(var,flag,tv,val);
            else
              affectval(pushvar(var,flag,tv,-1),val,seq);
          }
          genblock(val,n);
        }
        break;
      case function:
        {
          int vn;
          ctxvar *v;
          genequal(stack[i],"declaration",&var,&val,&tv);
          if (tv==Gnotype)
            tv=mint;
          vn = pushvar(var,flag,tv,val);/* can change ctxstack */
          if (tv!=Gsmall || tree[val].f!=Fsmall)
          {
            v = ctxstack+vn;
            if (descfindrules1(v->node,FC_default_check))
              v->flag|=(1<<Cdefmarker);
          }
        }
        break;
      }
      break;
    case Fvararg:
      if (decl==function && i==nb-1)
        pushvar(tree[stack[i]].x,flag|(1<<Cvararg),Gvec,val);
      else
        die(n,"invalid variadic");
      break;
    default:
      die(n,"Incorrect node %s in function declaration",
          funcname(tree[stack[i]].f));
    }
  }
  return nb;
}

int genblockcheckargs(int sav1, int seq, userfunc *ufunc)
{
  int i;
  gpfunc *gp=lfunc+currfunc;
  seq=addseqleft(newcall("_copyarg",-1),seq);
  for(i=ufunc->narg-1;i>=0;i--)
  {
    ctxvar *v=ctxstack+sav1+i;
    gpdescarg *rule;
    int check=-1;
    if (!(v->flag&(1<<Carg))) continue;
    rule=descfindrules1(v->node, FC_badtype);
    if (rule && rule->args[0].type==vartype(*v))
    {
      check=newcall("_badtype",newleaf(v->node));
      if (rule->type!=Gvoid)
      {
        int name=newstringnode(gp->gpname,-1);
        int args = descfindrules1(name,findfuncdesc("_err_type")) ? name :
                     newnode(Flistarg,name,v->node);
        check=newcall("if", newnode(Flistarg,check, newcall("_err_type", args)));
      }
    }
    if (v->initval>=0 && (v->flag&(1<<Cdefmarker)))
    {
      int n=newnode(Flistarg,
          newcall("_default_check",newleaf(v->node)),
          newnode(Fassign,newleaf(v->node),v->initval));
      check=newcall("if", (check>=0)?newnode(Flistarg,n,check):n);
      v->initval=-1;
    }
    if (check>=0)
      seq=addseqleft(check,seq);
  }
  return seq;
}

#if 0
/*The code for
  global(globalvars)
  f(args)=local(localvars);code;...
is */
{
  ulong ltop=avma;
  GEN p1;
  {
    GEN args;
    GEN localvars;
    code;
    p1=...;
  }
  gerepileall(ltop,2,&p1,&globalsvar);
  return p1;
}
#endif
void genblockdeffunc(int n)
{
  int funcid=tree[n].x;
  int seq=tree[n].y;
  const char *name=entryname(funcid);
  int args=tree[funcid].y;
  int bl1,gltop=-1,level=preclevel;
  int savcf,savnpv;
  int sav1,nodebl1;
  userfunc *ufunc;
  /*save global var*/
  savcf=currfunc;
  savnpv=newprivvar;
  /*reset private var counter*/
  newprivvar=1;
  /*create function*/
  currfunc=findfunction(entryname(funcid));
  if (currfunc<0)
    die(n,"internal error: unknown function %s in genblockdeffunc",name);
  if (lfunc[currfunc].spec!=GPuser)
  {
    if (lfunc[currfunc].spec==GPinstalled)
      die(n,"Cannot redefine install'ed function %s",name);
    else
      die(n,"internal error: not a user function %s in genblockdeffunc",name);
  }
  ufunc=lfunc[currfunc].user;
  /*create external block*/
  bl1=newblock();
  sav1=s_ctx.n;
  ufunc->bctx=sav1;
  ufunc->bl=bl1;
  if (autogc)
  {
    /*declare ltop*/
    gltop=newnode(Fentry,newentry("ltop"),-1);
    pushvar(gltop,0,Gpari_sp,newcall("_avma",-1));
  }
  /*declare function arguments*/
  ufunc->sarg=s_ctx.n-sav1;
  ufunc->narg=genblockdeclaration(args,n,(1<<Cuser)|(1<<Carg)|(1<<Ccompo),-1,NULL);
  /*Add return at the end*/
  seq=geninsertreturn(seq);
  seq=genblockcheckargs(sav1+ufunc->sarg,seq,ufunc);
  /*generate block*/
  genblock(seq,-1);
  if (tree[funcid].x!=initnode)
  {
    /*FIXME: For now, clear Cglobal flag to avoid initialization
      problem with implicit fetch_user_var().*/
    int i;
    for (i=sav1;i<s_ctx.n;i++)
    {
      if (ctxstack[i].flag&(1<<Cfunction))
        ctxstack[i].flag&=~(1<<Cfunction);
    }
  }
  copyctx(sav1,block+bl1);
  restorectx(sav1);
  nodebl1=newnode(Fblock,bl1,seq);
  tree[n].y=nodebl1;
  /*restore globalvar*/
  currfunc=savcf;
  newprivvar=savnpv;
  preclevel=level;
}

int newvectoridx(int var, int ind)
{
  return newcoeff(var, newsmall(ind), -1);
}

int newmatidx(int var, int x, int y)
{
  return newcoeff(var, newsmall(x), newsmall(y));
}

void genblockmatrixl(int n, int x, int y)
{
  int arg[6];
  int var,bsup,seq,ret;
  int vx,vy,vsup;
  int savx,bl1,bl2;
  int aseq,bseq=-1;
  /*we have x[y,] x is a matrix, y a small*/
  newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vx);
  newdecl((1<<Cconst),Gsmall,-1,&vy);
  newdecl((1<<Cconst),Glg,-1,&vsup);
  newdecl(0,Gvec,-1,&ret);
  newdecl(0,Gsmall,-1,&var);
  bsup=newcall("length",newnode(Ftag,newleaf(vx),Gvec));
  bseq=addseqright(bseq,geninsertvar(x,vx));
  bseq=addseqright(bseq,geninsertvar(y,vy));
  bseq=addseqright(bseq,geninsertvar(bsup,vsup));
  genblock(bseq,-1);
  savx=s_ctx.n;
  bl1=newblock();
  bl2=newblock();
  seq=newcoeff(newleaf(vx),newleaf(vy),newleaf(var));
  aseq=newgetg(ret,newleaf(vsup), "t_VEC");
  arg[0]=newnode(Fassign,var,newsmall(1));
  arg[1]=newopcall(OPle,newleaf(var),vsup);
  arg[2]=newcoeff(newleaf(ret),newleaf(var),-1);
  arg[2]=geninsertvar(seq,arg[2]);
  genblock(arg[2],-1);
  arg[3]=newopcall(OPpp,newleaf(var),-1);
  aseq=addseqright(aseq,newcall("for",stacktoargs(arg,4)));
  makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
}
void genblockaffmatrixl(int n, int x, int y, int z)
{
  int arg[6];
  int var,bsup,seq;
  int vy,vz,vsup;
  int savx,bl1,bl2;
  int aseq,bseq=-1;
  /*we have x[y,] x is a matrix, y a small, z is a vector*/
  newdecl((1<<Cconst),Gsmall,-1,&vy);
  newdecl((1<<Cconst),Glg,-1,&vsup);
  newdecl(0,Gsmall,-1,&var);
  bsup=newcall("length",newnode(Ftag,newleaf(x),Gvec));
  if (tree[z].f==Fentry)
  {
    genblock(z,n);
    vz=z;
  }
  else
  {
    newdecl(0,Gvec,-1,&vz);
    bseq=addseqright(bseq,geninsertvar(z,vz));
  }
  bseq=addseqright(bseq,geninsertvar(y,vy));
  bseq=addseqright(bseq,geninsertvar(bsup,vsup));
  genblock(bseq,-1);
  savx=s_ctx.n;
  bl1=newblock();
  bl2=newblock();
  seq=newcoeff(newleaf(x),newleaf(vy),newleaf(var));
  arg[0]=newnode(Fassign,var,newsmall(1));
  arg[1]=newopcall(OPle,newleaf(var),vsup);
  arg[2]=newcoeff(newleaf(vz),newleaf(var),-1);
  arg[2]=geninsertvar(arg[2],seq);
  genblock(arg[2],-1);
  arg[3]=newopcall(OPpp,newleaf(var),-1);
  aseq=newcall("for",stacktoargs(arg,4));
  makeblocks(bl1,bl2,n,bseq,aseq,vz,savx);
}
void genblockvector(int n, const char *typ)
{
  int arg[STACKSZ];
  int x=tree[n].x;
  int i,nb,ret;
  int aseq;
  nb=listtostack(x,Fmatrixelts,arg+1,STACKSZ-1,"Vector too long.",n);
  newdecl(0,Gvec,-1,&ret);
  arg[0]=newgetg(ret, newsmall(nb), typ);
  for(i=1;i<=nb;i++)
    arg[i]=geninsertvar(arg[i],newvectoridx(newleaf(ret),i));
  for(i=1;i<=nb;i++)
    genblock(arg[i],-1);
  aseq=newcall("_makevec",stacktoargs(arg,nb+1));
  makeblock(newblock(),n,aseq,ret,s_ctx.n);
}
void genblockindex(int n)
{
  int ret,z;
  int aseq;
  if (n==GNOARG || tree[n].f==Fsmall)
    return;
  newdecl((1<<Cconst),Gsmall,-1,&ret);
  z=newleaf(n);
  tree[z]=tree[n];
  aseq=geninsertvar(z,ret);
  makeblock(newblock(),n,aseq,ret,s_ctx.n);
}
void genblockmatrix(int n)
{
  int line[STACKSZ];
  int arg[STACKSZ];
  int x=tree[n].x,xx;
  int i,j,k,nb=1,nbline,nbcol=-1,ret;
  int aseq;
  nbline=listtostack(x,Fmatrixlines,line,STACKSZ,"[...;...;...]",n);
  for(xx=line[0],nbcol=1;tree[xx].f==Fmatrixelts;xx=tree[xx].x,nbcol++);
  nb+=nbcol;
  for(i=0;i<nbline;i++)
  {
    int k;
    k=listtostack(line[i],Fmatrixelts,arg+nb,STACKSZ-nb,"[...;...;...]",n);
    if (k!=nbcol) die(n,"Matrix must be rectangular");
    nb+=k;
  }
  newdecl(0,Gvec,-1,&ret);
  arg[0]=newgetg(ret,newsmall(nbcol),"t_MAT");
  for(i=1;i<=nbcol;i++)
    arg[i]=newgetg(newcoeff(newleaf(ret),newsmall(i),-1),
                     newsmall(nbline),"t_COL");
  for(j=1,k=1;i<nb;i++)
  {
    arg[i]=geninsertvar(arg[i],newmatidx(newleaf(ret),k,j));
    if (j==nbcol)
    {
      j=1;
      k++;
    }
    else
      j++;
  }
  for(i=0;i<nb;i++)
    genblock(arg[i],-1);
  aseq=newcall("_makevec",stacktoargs(arg,nb));
  makeblock(newblock(),n,aseq,ret,s_ctx.n);
}

void
genblockvecaff(int n)
{
  int vars[STACKSZ];
  int x=tree[n].x, y=tree[n].y;
  int aseq, ret, i;
  int nv=listtostack(tree[x].x,Fmatrixelts,vars,STACKSZ,"[,...,]=",n);
  genblock(y,n);
  newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&ret);
  aseq = geninsertvar(y,ret);
  for(i=0; i<nv; i++)
  {
    genblock(vars[i],n);
    vars[i] = newnode(Fassign,vars[i],newcoeff(newleaf(ret),newsmall(i+1),-1));
  }
  aseq = addseqright(aseq,stacktoseq(vars,nv));
  makeblock(newblock(),n,aseq,ret,s_ctx.n);
}

void genblockproto(int n, gpfunc *gp)
{
  int arg[STACKSZ];
  const char *p=gp->proto.code ? gp->proto.code : "&DG"; /* 2.3 compat */
  char c;
  PPproto mod;
  int nb=genlistargs(n,arg,0,STACKSZ);
  int i=0;
  while((mod=parseproto(&p,&c))!=PPend)
  {
    if (i<nb && tree[arg[i]].f!=Fnoarg && (mod==PPdefault || mod==PPdefaultmulti))
      mod=PPstd;
    switch(mod)
    {
    case PPstd:
      if (i>=nb)
        die(n,"too few arguments");
      switch(c)
      {
      case 'r':
        if (tree[arg[i]].f==Fentry)
          tree[arg[i]].f=Fconst;
        else if (tree[arg[i]].f!=Fconst)
          die(n,"argument must be a raw string");
        break;
      case '&':
        if (tree[arg[i]].f!=Frefarg)
          die(arg[i],"missing &");
        arg[i]=tree[arg[i]].x;
      case '*': /*Fall through */
        {
          int a = detag(arg[i]);
          if (tree[a].f==Fmatcoeff)
          {
            int l=getlvaluerr(a);
            genblock(l,n);
            ctxstack[getvarerr(l)].flag|=1<<Ccompo;
          }
          if (c=='*') multi++;
          genblock(arg[i],n);
          if (c=='*') multi--;
          break;
        }
      default:
        genblock(arg[i],n);
      }
      break;
    case PPstar:
      genblock(tree[n].y,n);
      i=nb-1;
      break;
    case PPauto: /* Skip */
      i--;
      break;
    default:
      break;
    }
    i++;
  }
}

/*
n: node
p: caller node

  Rule of the game:
  genblock must never be used twice on the same node.
  Using p=-1 is allowed but this assumes the value of n is to be discarded.

  genblock must not change p.
*/
int newgetgvec(const char *t)
{
  return newtag(newcall("_cgetg",
                        newnode(Flistarg, newsmall(0), newstringnode(t,-1))),
                "vec", -1);
}

int genclosedlist(int nf, int n)
{
  int i;
  gpfunc *gp=lfunc+nf;
  context *fc=block+gp->user->bl;
  for (i=0; i<fc->s.n; i++)
  {
    ctxvar *v=fc->c+i;
    if (v->flag&(1<<Cclosed))
      n = newnode(Flistarg,n,newleaf(v->node));
  }
  return n;
}

static int has_mand_arg(const char *code)
{
  PPproto mod;
  char c;
  while ((mod=parseproto(&code,&c))!=PPend)
    if(mod==PPstd) return 1;
  return 0;
}

void genblock(int n, int p)
{
  int x,y;
  if (n<0)
    return;
  linecount=tree[n].lineno;
  x=tree[n].x;
  y=tree[n].y;
  switch(tree[n].f)
  {
  case Fseq:
      genblock(x, n);
      genblock(y, n);
    break;
  case Fmat:
      if (x==-1)
      {
        int z=newgetgvec("t_MAT");
        tree[n]=tree[z];
      }
      else
        genblockmatrix(n);
    break;
  case Fvec:
      if (x==-1)
      {
        int z=newgetgvec("t_VEC");
        tree[n]=tree[z];
      }
      else
        genblockvector(n, "t_VEC");
    break;
  case Fassign:
    x=detag(x);
    if (tree[x].f==Fvec)
      genblockvecaff(n);
    else
    {
      if (tree[x].f==Fmatcoeff)
      {
        int l=getlvaluerr(x);
        int m=matindex_type(tree[x].y);
        genblock(l,n);
        ctxstack[getvarerr(l)].flag|=1<<Ccompo;
        if (m==MAT_line)
        {
          int xx=tree[x].x;
          int xy=tree[tree[tree[x].y].x].x;
          genblockaffmatrixl(n,xx,xy,y);
          break;
        }
      }
      genblock(x, n);
      genblock(y, n);
    }
    break;
  case Fmatrix:
    genblock(x,n);
    genblock(y,n);
    if (multi)
    {
      genblockindex(tree[x].x);
      genblockindex(tree[x].y);
      if (y>=0)
      {
        genblockindex(tree[y].x);
        genblockindex(tree[y].y);
      }
    }
    break;
  case Fmatcoeff:
    {
      int nn;
      int m=matindex_type(y);
      int yx=tree[y].x, yy=tree[y].y;
      genblock(x,n);
      genblock(y,n);
      switch(m)
      {
      case VEC_std:
        nn=newcoeffsafe(x,tree[yx].x,-1);
        break;
      case MAT_std:
        nn=newcoeffsafe(x,tree[yx].x,tree[yy].x);
        break;
      case MAT_line:
        tree[n].y=yx;
        if (FC_matrixrow>=0)
          nn=newcall("_[_,]",newnode(Flistarg,x,tree[yx].x));
        else
        {
          genblockmatrixl(n,x,tree[yx].x);
          nn=n;
        }
        break;
      case MAT_column:
        nn=newcoeff(x,tree[yy].x,-1);
        break;
      case MAT_range:
        {
          int arg[5];
          arg[0] = x;
          arg[1] = tree[yx].x;
          arg[2] = tree[yx].y;
          if(yy<0)
            nn=newnode(Ffunction,newentry("_[_.._]"),stacktoargs(arg,3));
          else
          {
            arg[3] = tree[yy].x;
            arg[4] = tree[yy].y;
            nn=newnode(Ffunction,newentry("_[_.._,_.._]"),stacktoargs(arg,5));
          }
        }
        break;
      default:
        die(-1,"Unknown matindex");
      }
      tree[n]=tree[nn];
      break;
    }
  case Fcall:
    {
      int seq = newcall("_(_)",addlistleft(x,y));
      if (FC_call<0) die(n,"closures not available in this version");
      tree[n]=tree[seq];
      genblock(n,p);
    }
    break;
  case Fentry:
    {
      int nf, nv = getvar(n);
      if (nv>=0) break;
      nf = findfunction(entryname(n));
      if (nf<0) /*Undeclared variable*/
      {
        gpfunc *gp = lfunc+currfunc;
        userfunc *uf = gp->user;
        if (uf->flag&(1<<UFclosure))
        {
          /* We assume the closure contains the variable */
          pushvar(n,(1<<Cuser)|(1<<Cfunction)|(1<<Carg)|(1<<Ccompo)|(1<<Cclosed),Ggen,-1);
          uf->narg++;
        }
        else
        {/*The variable has not been declared.
          * We declare it as a function-local variable initialized
          * to 'var. The global flag is set to avoid being block-local.
          * It is discarded in genblockdeffunc. This can be changed.
          */
          int iv=newquotenode(entryname(n),-1);
          pushvar(n,(1<<Cuser)|(1<<Cfunction)|(1<<Cglobal)|(1<<Cundeclared),Ggen,iv);
          if (warn)
            warning(n,"variable undeclared");
        }
        break;
      } else
      {
        /* This is a closure */
        gpfunc *gp = lfunc+nf;
        const char * code = gp->proto.code;
        GPspec spec = gp->spec;
        if (spec==GPuser && gp->user->flag&(1<<UFclosure))
        {
          int z=newcall("_closure",genclosedlist(nf,newstringnode(lfunc[nf].gpname,-1)));
          tree[n]=tree[z];
          genblock(n, p);
          break;
        }
        else if (spec <= 0 && (spec!=GPpari || (code && has_mand_arg(code))))
        {
          int z=newcall("_closure",newstringnode(lfunc[nf].gpname,-1));
          tree[n]=tree[z];
          genblock(n, p);
          break;
        }
      }
      /*else it is a function call*/
      tree[n].f=Ffunction;
    }
  case Ffunction:/*fall through*/
    if (x==OPtrans && tree[y].f==Fvec)
    {
      if (tree[y].x==-1)
      {
        int z=newgetgvec("t_COL");
        tree[n]=tree[z];
      }
      else
      {
        tree[n]=tree[y];
        genblockvector(n,"t_COL");
      }
    }
    else
    {
      int nf=findfunction(value[x].val.str);
      gpfunc *gp = lfunc+currfunc;
      userfunc *uf = gp->user;
      int nv = getvar(n);
      if (nf<0 && uf->flag&(1<<UFclosure) && nv==-1)
      {
        pushvar(n,(1<<Cuser)|(1<<Cfunction)|(1<<Carg)|(1<<Ccompo)|(1<<Cclosed),Ggen,-1);
        uf->narg++;
        genblock(y,n);
      }
      else if (nf<0 && nv>=0)
      {
        int seq = newnode(Fcall,newnode(Fentry,x,-1),y);
        tree[n]=tree[seq];
        genblock(n,p);
      }
      else if (nf>=0 && lfunc[nf].iter)
        genblockfunciter(n,lfunc+nf);
      else if (nf>=0 && lfunc[nf].spec>0)
        genblockfuncspec(n,p,lfunc+nf);
      else if (nf>=0 && lfunc[nf].proto.code)
        genblockproto(n,lfunc+nf);
      else if (nf>=0 && x>=OPss && x<=OPme) /* For compatibility with PARI 2.3 */
        genblockproto(n,lfunc+nf);
      else
        genblock(y,n);
    }
    break;
  case Fdeffunc:
    if (currfunc!=-1)
      die(n,"Internal error : nested function");
    genblockdeffunc(n);
    break;
  case Fconst:
  case Fsmall:
  case Fnoarg:
    break;
  case Ftag:
    genblock(x,n);
    break;
  case Fblock:
    die(p,"Internal error : looping in genblock");
  default:
    if (tree[n].f>=FneedENTRY)
    {
      die(p,"Internal error : unknown func %s in genblock",funcname(tree[n].f));
    }
    else
    {
      genblock(x, n);
      genblock(y, n);
    }
  }
}

