/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002, 2003, 2004 
   Gerber van der Graaf <gerber_graaf@users.sourceforge.net>

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
 * Dac callbacks
 * $Id: dac.c,v 1.5 2005/02/26 09:43:30 gerber Exp $
 */


#ifndef DISABLE_DAC

#include <pthread.h>
#include "gpiv_gui.h"
#include "utils.h"
#include "console.h"
#include "display.h"
#include "dac.h"
/* #include "dac_cam.h" */
#include <libdc1394/dc1394_control.h>

/* #include <sys/types.h> */
/* #include <sys/stat.h> */
/* #include <fcntl.h> */


/*
 * BUGFIX: Put somwhere to update on regular basis:
        gtk_label_set_text(GTK_LABEL(gpiv->dac->label_temp), 
                           "Temp");
*/

static int init, start, stop, error;
static struct tm *dac_time;
static GDate *date;


typedef struct _ImgFrame ImgFrame;
struct _ImgFrame {
    unsigned char **A;
    unsigned char **B;
/*     gint count; */
};

pthread_mutex_t mutex_rec = PTHREAD_MUTEX_INITIALIZER;

void
thread_rec_frame(void *img_frame);


static void
thread_rec_frame_serie(ImgFrame img_frame[GPIV_NIMG_MAX]
                );

static void
rec_frame(ImgFrame *img_frame
          );

static void
add_buffer(GpivConsole *gpiv,
           gchar *fname_base,
           gint ibuf, 
           unsigned char **frameA, 
           unsigned char **frameB
           );

static void
renew_buffer(GpivConsole *gpiv,
             gchar *fname_base,
             gint ibuf, 
             unsigned char **frameA, 
             unsigned char **frameB
             );

static void 
recreate_img(Display * disp,
             unsigned char **frameA, 
             unsigned char **frameB
             );


static int
open_dac_img(GpivCamVar *cam_var,
             unsigned char **frameA,
             unsigned char **frameB,
             Image * img,
             gboolean alloc_mem
             )
/*-----------------------------------------------------------------------------
 * Opens an image obtained from the camera (Counterpart of open_img from file)
 */
{
    gint i, j;
    gchar *month;
/*
 * parameter initializing of image
 */
    img->image_par.depth_logic = FALSE;
    img->image_par.ncolumns_logic = FALSE;
    img->image_par.nrows_logic = FALSE;
    img->image_par.x_corr_logic = FALSE;
    img->image_par.s_scale_logic = FALSE;
    img->image_par.t_scale_logic = FALSE;
    img->image_par.z_off_x_logic = FALSE;
    img->image_par.z_off_y_logic = FALSE;

    img->image_par.title_logic = FALSE;
    img->image_par.creation_date_logic = FALSE;
    img->image_par.location_logic = FALSE;
    img->image_par.author_logic = FALSE;
    img->image_par.software_logic = FALSE;
    img->image_par.source_logic = FALSE;
    img->image_par.usertext_logic = FALSE;
    img->image_par.warning_logic = FALSE;
    img->image_par.disclaimer_logic = FALSE;
    img->image_par.comment_logic = FALSE;

    img->image_par.ncolumns = cam_var->capture[0].frame_width;
    img->image_par.nrows = cam_var->capture[0].frame_height;
    img->image_par.x_corr = gpiv_par.x_corr;
    img->image_par.ncolumns_logic = TRUE;
    img->image_par.nrows_logic = TRUE;
    img->image_par.x_corr_logic = TRUE;

    gpiv_img_cp_parameters (image_par, &img->image_par, FALSE, FALSE);
    g_message("open_dac_img:: 1 creation_date = %s", 
              img->image_par.creation_date);
/* 
 * Apply time and date to creation_date (buffer) parameter
 */
    month = month_name(g_date_month(date));
    g_snprintf(img->image_par.creation_date,
               GPIV_MAX_CHARS,
               "%d %s %d %d:%d:%d",
               g_date_day(date), 
               month, 
               g_date_year(date),
               dac_time->tm_hour,
               dac_time->tm_min, 
               dac_time->tm_sec);
    g_free(month);
    g_message("open_dac_img:: 2 creation_date = %s", img->image_par.creation_date);


/*     if(alloc_mem) { */
    img->img1 = gpiv_alloc_img(img->image_par);
/*     } */

    for (i = 0; i < img->image_par.nrows ; i++) {
        for (j = 0; j < img->image_par.ncolumns; j++) {
/*
 * format 640x480 = 1:1.3
 */
            img->img1[i][j/*  + i/3 */] = (guint16) frameA[i][j];
        }
    }



    if (gpiv_par.x_corr) {
 /*        if(alloc_mem) { */
        img->img2 = gpiv_alloc_img(img->image_par);
/*         } */
        for (i = 0; i < img->image_par.nrows ; i++) {
            for (j = 0; j < img->image_par.ncolumns; j++) {
/*
 * format 640x480 = 1:1.3
 */
                img->img2[i][j/*  + i/3 */] = (guint16) frameB[i][j];
            }
        }
    } else {
        img->img2 = img->img1;
    }

 
    img->exist_img = TRUE;
    return 0;
}


static void
load_dac_buffer(GpivConsole *gpiv,
                GpivCamVar *cam_var,
                gboolean second_image,
                char *fname,
                int ibuf,
                unsigned char **frameA,
                unsigned char **frameB,
                enum ClistPut clist_put
                )
/*-----------------------------------------------------------------------------
 * create display and its (popup) menu, load image, piv data and puts 
 * buffername into clist
 */
{
    gint return_val = 0;
    gchar *clist_buf_txt[MAX_BUFS][2], cl_int[4];


    display[ibuf] = create_display(fname, ibuf, gpiv);
    display_act = display[ibuf];
    gtk_widget_show(display_act->mwin);

/*     zoom_display(display_act, display_act->zoom_index); */



    g_snprintf(display_act->fname_base, GPIV_MAX_CHARS, "%s", fname);
    if((return_val = open_dac_img(cam_var, frameA, frameB, 
                                  &display_act->img, FALSE))
       != 0) {
        gtk_object_destroy(GTK_OBJECT(display[ibuf]->mwin));
        if (ibuf > 0) {
            ibuf = nbufs - 1;;
            display_act = display[ibuf];
        } else {
            display_act = NULL;
        }
        nbufs--;
        return;
    }

/* 
 * Pop-up menu after loading data
 */
    display_act->display_menu = create_display_menu(display_act);
    gtk_widget_show(display_act->display_menu);
    gtk_signal_connect_object(GTK_OBJECT(display_act->mwin), 
                              "button_press_event",
                              GTK_SIGNAL_FUNC (on_my_popup_handler), 
                              GTK_OBJECT(display_act->display_menu));


    zoom_display(display_act, display_act->zoom_index);

/*
 * Displaying
 */
    create_background(display_act);
    create_img(display_act);
    if (display_act->display_img1) {
        show_img1(display_act);
    } else {
        hide_img1(display_act);
    }
    
    if (image_par.x_corr) {
        if (display_act->display_img2) {
            show_img2(display_act);
        } else {
            hide_img2(display_act);
        }
    }


/*     display_act->img.img_mean = image_mean(display_act->img.img1, */
/*                                            gpiv_par.img_width,  */
/*                                            gpiv_par.img_height); */


    if (gpiv_par.display_intregs == 1) {
        g_message("load_dac_buffer:: create_all_intregs");
        create_all_intregs(display_act);
    }

/*
 * Add buffer to list
 */
    g_snprintf(cl_int, 4, "%d", ibuf);
    clist_buf_txt[ibuf][0] = (gchar *) cl_int;
    clist_buf_txt[ibuf][1] = fname;
    if (clist_put == PREPEND) {
        gtk_clist_prepend(GTK_CLIST(gpiv->clist_buf), 
                          clist_buf_txt[ibuf]);
    } else if (clist_put == INSERT) {
        gtk_clist_insert(GTK_CLIST(gpiv->clist_buf), ibuf, 
                         clist_buf_txt[ibuf]);
    } else if (clist_put == APPEND) {
        gtk_clist_append(GTK_CLIST(gpiv->clist_buf), 
                         clist_buf_txt[ibuf]);
    } else {
        error_gpiv("non-existent CLIST enumerate");
    }
    
   gtk_clist_set_row_data(GTK_CLIST(gpiv->clist_buf), ibuf, 
                           display_act);
 

/*
 * Copy general image parameters to the buffer parameters 
 * and display in imgh tabulator
 */
/*    gpiv_img_cp_parameters(image_par, display_act->img.image_par); */

    gtk_widget_set_size_request (display_act->canvas, 
                                 (gint) (display_act->zoom_factor * 
                                         gpiv_par.img_width), 
                                 (gint) (display_act->zoom_factor * 
                                         gpiv_par.img_height));
    
}



static void
rec_and_display(GpivConsole *gpiv
/*                 , ImgFrame *img_frame */
                )
/*-----------------------------------------------------------------------------
 */
{
    gint ibuf = 0;
    ImgFrame img_frame[GPIV_NIMG_MAX];

    gint return_val = 0;
    pthread_t thread_rec;
    pthread_attr_t attr;

    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);


    if (dac_par.ttime.mode == GPIV_TIMER_MODE__DURATION) {
/*
 * Closing existing buffer windows and cleaning up Clist
 */
        for (ibuf = 0; ibuf < nbufs; ibuf++) {
            if (display[ibuf] != NULL) {
                close_buffer(gpiv, display[ibuf], &ibuf);
            }
        }
        gtk_clist_clear(GTK_CLIST(gpiv->clist_buf));
        nbufs = 0;
            
/*
 * allocating image mem
 */
        for (ibuf = 0; ibuf < dac_par.ttime.cycles; ibuf++) {
            img_frame[ibuf].A = 
                gpiv_ucmatrix(cam_var.capture[0].frame_height, 
                              cam_var.capture[0].frame_width);
            if (gpiv_par.x_corr) {
                img_frame[ibuf].B = gpiv_ucmatrix(gpiv_par.img_height, 
                                                  gpiv_par.img_width);
            }
        }

/*
 * recording
 */

#define THREADING
/* 
 * use threading
 */
/*         if (verbose)  */
/*             g_message("rec_and_display:: Creating thread_rec from %d", pthread_self ()); */
#ifdef THREADING
        if ((return_val = pthread_create(&thread_rec, 
                                         &attr,
                                         (void *) &thread_rec_frame_serie, 
                                         (void *) &img_frame)) == TRUE) {
            pthread_exit(NULL);
            g_error("return code from pthread_create() is %d\n", return_val);
        }
        pthread_join(thread_rec, NULL);
        g_message ("rec_and_display:: waited on thread_rec. Done");

/*
 * use no threading
 */
#else
        for (ibuf = 0; ibuf < dac_par.ttime.cycles; ibuf++) {
            if (cancel_process) break;
            rec_frame(&img_frame[ibuf]);

            progress_value = (gfloat) (ibuf +  1.0) / 
                (gfloat) dac_par.ttime.cycles;
            g_snprintf(progress_string, GPIV_MAX_CHARS, 
                       "recording image #%d", ibuf);
            while (g_main_iteration(FALSE));
            gtk_progress_set_value(gnome_appbar_get_progress
                                   (GNOME_APPBAR(gpiv->appbar)), 
                                   progress_value * 100.0);
            gnome_appbar_push(GNOME_APPBAR(gpiv->appbar), 
                              progress_string);
    }

#endif /* THREADING */
#undef THREADING
/*
 * displaying and freeing image mem
 */
        for (ibuf = 0; ibuf < dac_par.ttime.cycles; ibuf++) {
            add_buffer(gpiv, gpiv_var.fn_last[0], ibuf, img_frame[ibuf].A, 
                       img_frame[ibuf].B);
            gpiv_free_ucmatrix(img_frame[ibuf].A);
            if (gpiv_par.x_corr) {
                gpiv_free_ucmatrix(img_frame[ibuf].B);
            }
        }

/*
 * timing is indefinite periodic
 * display only and continuously images in buffer 0
 */
    } else {
        ibuf = 0;
        gtk_clist_select_row(GTK_CLIST(gpiv->clist_buf), ibuf, 0); 
        g_message("rec_and_display:: INDEFINITE");

/* BUGFIX: for Gtk2 */
/*                 gtk_progress_bar_set_pulse_step (GNOME_APPBAR(gpiv->appbar), */
/*                                                  0.1); */

        img_frame[ibuf].A = 
            gpiv_ucmatrix(cam_var.capture[0].frame_height, 
                          cam_var.capture[0].frame_width);
        if (gpiv_par.x_corr) {
            img_frame[ibuf].B = gpiv_ucmatrix(gpiv_par.img_height, 
                                              gpiv_par.img_width);
        }

        while (!cancel_process) {
            rec_frame(&img_frame[ibuf]);
            while (g_main_iteration(FALSE));
            renew_buffer(gpiv, gpiv_var.fn_last[0], ibuf, img_frame[ibuf].A, 
                         img_frame[ibuf].B);
/* BUGFIX: for Gtk2 */
/*                     gtk_progress_bar_pulse ((GNOME_APPBAR(gpiv->appbar)) */


        }
            
        gpiv_free_ucmatrix(img_frame[ibuf].A);
        if (gpiv_par.x_corr) {
            gpiv_free_ucmatrix(img_frame[ibuf].B);
        }
        
    }

   pthread_attr_destroy(&attr);
}





/*
 * Global functions
 */

/* static */ void
exec_trigger_start (void
            )
/*-----------------------------------------------------------------------------
 */
{
    gchar *err_msg = NULL;
    int on = 1, param_ok;

    g_message("exec_trigger_start: 0");
    if (!gpiv_dac_openrtfs(&init, &start, &stop, &error)) {
        err_msg = _("Fail in fifo open");
        warning_gpiv(err_msg);
        return;
    }


    g_message("exec_trigger_start: 1");
    gpiv_dac_test_parameter(&dac_par);

/*
 * write the timing details to /dev/rtf/1
 */
    g_message("exec_trigger_start: 2");
    if((write(init, &dac_par.ttime, sizeof(GpivTrigTime))) < 0)  {
        err_msg = _("Fail in setting camera and Laser timing");
        warning_gpiv(err_msg);
        return;
    }

    g_message("exec_trigger_start: 3");
    if((read(error, &param_ok, sizeof(int))) < 0) {
        err_msg = _("Fail in receipt of confirmation");
        warning_gpiv(err_msg);
        return;
    }
        
    g_message("exec_trigger_start: 4");
    if (param_ok != 1) {
            warning_gpiv("Invalid parameters entered \n");
        
    } else { 
        if (verbose) 
            g_message (_("Parameters: \n\
cam_acq_period: %lld \n\
laser_trig_pw:  %lld \n\
time2laser:     %lld \n\
dt:             %lld \n\
mode:           %d \n\
cycles:         %d \n\
increment:      %d\n"),
                       dac_par.ttime.cam_acq_period, 
                       dac_par.ttime.laser_trig_pw,
                       dac_par.ttime.time2laser, 
                       dac_par.ttime.dt, 
                       dac_par.ttime.mode, 
                       dac_par.ttime.cycles, 
                       (int) dac_par.ttime.increment);
        
        g_message("exec_trigger_start: 5");
        if((write(start, &on, sizeof(int))) < 0) {
            err_msg = _("Fail in starting camera and Laser timing");
            warning_gpiv(err_msg);
            return;
        }
    }
    
    g_message("exec_trigger_start: 6");
/*     gpiv_par.process_dac = TRUE; */
}



void
exec_trigger_stop (void
           )
/*-----------------------------------------------------------------------------
 */
{
    gchar *err_msg = NULL;
    int off = 0;

    if((write(stop, &off, sizeof(int))) < 0)  {
        err_msg = _("Fail stopping camera and Laser timing");
        warning_gpiv(err_msg);
        return;
    }

/*     gpiv_par.process_dac = FALSE; */
}



void 
exec_rec_start(GpivConsole * gpiv
               )
/*-----------------------------------------------------------------------------
 */
{
/*
 * dc1394 stuff
 */
/*     quadlet_t value; */

   cancel_process = FALSE;

/*
 * report camera's feature set
 * in: on_menu_camera_select
 */
/*         dc1394_print_feature_set(&cam_var.feature_set[0]); */

/*    g_message("exec_rec_start:: mode[%d] = %s", */
/*              cam_var.misc_info[0].mode, */
/*              format0_desc[cam_var.misc_info[0].mode - MODE_FORMAT0_MIN]); */
/*    g_message("exec_rec_start:: format[%d] = %s", */
/*              cam_var.misc_info[0].format,  */
/*              format_desc[cam_var.misc_info[0].format - FORMAT_MIN]); */

   g_warning("exec_rec_start:: fps = %d", cam_var.misc_info[0].framerate/*  - FRAMERATE_MIN */);
    if (dc1394_setup_capture(cam_var.camera[0].handle, 
                             cam_var.camera[0].id, 
                             cam_var.misc_info[0].iso_channel, 
                             cam_var.misc_info[0].format, 
                             cam_var.misc_info[0].mode, 
                             cam_var.maxspeed,

                             cam_var.misc_info[0].framerate, 

/*                              FRAMERATE_7_5,   */

/* FRAMERATE_3_75, */

                             &cam_var.capture[0])
        != DC1394_SUCCESS) {
        dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
        raw1394_destroy_handle(cam_var.handle);
        warning_gpiv(_("Unable to setup camera-\n\
check line %d of %s to make sure\n\
that the video mode,framerate and format are\n\
supported by your camera"),
                    __LINE__,__FILE__);
        return;
    }


/*
 * have the camera start sending us data
 */
    if (dc1394_start_iso_transmission(cam_var.handle, cam_var.capture[0].node) 
        != DC1394_SUCCESS) {
        dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
        raw1394_destroy_handle(cam_var.handle);
        warning_gpiv(_("Unable to start camera iso transmission"));
        return;
    }

/*
 * let sending data for a while before grabbing and storing in order to stabilize
 * data transfer
 */
/*     sleep(1); */
    usleep(1000);


/*
 * set trigger mode and use triggering or just free run
 */
    
    if (gpiv_par.trigger_cam) {
            g_warning("exec_rec_start:: trigger_cam");
/*             if (dc1394_set_trigger_mode(cam_var.handle,  */
/*                                         cam_var.capture[0].node,  */
/*                                         TRIGGER_MODE_0) != DC1394_SUCCESS) */
/*                 { */
/*                     warning_gpiv(_("unable to set camera trigger mode")); */
/*                 } */
/*             exec_trigger_start(); */

            rec_and_display(gpiv);

        /*             exec_trigger_stop(); */
    } else {
        g_warning("exec_rec_start:: !trigger_cam");
            rec_and_display(gpiv);
    }


/*     pthread_exit(NULL); */

/*
 * Stop data transmission
 */
        g_warning("exec_rec_start:: Stop data transmission and release camera");
    if (dc1394_stop_iso_transmission(cam_var.handle, cam_var.capture[0].node)
        != DC1394_SUCCESS
/*         && dc1394_release_camera(cam_var.handle, &cam_var.capture[0])   */
/*         != DC1394_SUCCESS  */
        ) {
        warning_gpiv(_("couldn't stop the camera?"));
    }
/* in: exec_stop_rec    dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);     */
/*in: exec_stop_rec:    raw1394_destroy_handle(cam_var.handle); */

}



void
thread_rec_frame(void *img_frame)
/*-----------------------------------------------------------------------------
 */
{
    ImgFrame *my_frame = NULL;
    unsigned char **frameA = my_frame->A;
    unsigned char **frameB = my_frame->B;

    g_warning("thread_rec_frame:: 0");
    my_frame = (ImgFrame *) img_frame;


/*     unsigned char **frameA = gpiv_ucmatrix(cam_var.capture[0].frame_height,  */
/*                           cam_var.capture[0].frame_width); */


g_message("thread_rec_frame:: 0.1");
/*
 * capture one frame
 */
    if (dc1394_single_capture(cam_var.handle, &cam_var.capture[0]) 
        != DC1394_SUCCESS) {
        dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
        raw1394_destroy_handle(cam_var.handle);
        warning_gpiv(_("unable to capture a frame"));
/*         return(); */
    } else {
g_message("thread_rec_frame:: 1");


/*
 * Copy buffer content to tmp buffer array
 * block data for thread
 */

        pthread_mutex_lock(&mutex_rec);
        memcpy( frameA[0], (const char *) cam_var.capture[0].capture_buffer, 
                sizeof(char) * cam_var.capture[0].frame_width * 
                cam_var.capture[0].frame_height);
        pthread_mutex_unlock(&mutex_rec);
g_message("thread_rec_frame:: 2");
    }    

/*
 * Repeat if cross correlation; x_corr = TRUE
 */
    if (gpiv_par.x_corr) {
        if (dc1394_single_capture(cam_var.handle, &cam_var.capture[0]) 
            != DC1394_SUCCESS) {
            dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
            raw1394_destroy_handle(cam_var.handle);
            warning_gpiv(_("unable to capture a frame"));
/*             return(); */
        } else {
            
            pthread_mutex_lock(&mutex_rec);
            memcpy( frameB[0], (const char *) cam_var.capture[0].capture_buffer, 
                    sizeof(char) * cam_var.capture[0].frame_width * 
                    cam_var.capture[0].frame_height);
            pthread_mutex_unlock(&mutex_rec);
        }
    }
    
    
/*     gpiv_free_ucmatrix(frameA); */

g_warning("thread_rec_frame:: 3/3");
    pthread_exit(0);
}




static void
thread_rec_frame_serie(ImgFrame img_frame[GPIV_NIMG_MAX]
          )
/*-----------------------------------------------------------------------------
 * capture a series of images
 */
{
    gint ibuf;

/* g_message("rec_frame_series:: 0 with id = %d", (int) pthread_self ()); */
    for (ibuf = 0; ibuf < dac_par.ttime.cycles; ibuf++) {
        if (cancel_process) break;
        rec_frame(&img_frame[ibuf]);
    }
/* g_message("rec_frame_series:: 1/1"); */
    pthread_exit(NULL);
}


static void
rec_frame(ImgFrame *img_frame
          )
/*-----------------------------------------------------------------------------
/*
 * capture a single image frame or frame pair
 */
{

/* g_message("rec_frame:: 0"); */

    if (dc1394_single_capture(cam_var.handle, &cam_var.capture[0]) 
        != DC1394_SUCCESS) {
        dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
        raw1394_destroy_handle(cam_var.handle);
        g_warning("unable to capture a frame");
        return;
    }

/*
 * Copy buffer content to tmp buffer array
 */
    memcpy( img_frame->A[0], 
            (const char *) cam_var.capture[0].capture_buffer, 
            sizeof(char) * cam_var.capture[0].frame_width * 
            cam_var.capture[0].frame_height);
    
    if (gpiv_par.x_corr) {
/*
 * capture second frame
 */
        if (dc1394_single_capture(cam_var.handle, &cam_var.capture[0]) 
            != DC1394_SUCCESS) {
            dc1394_release_camera(cam_var.handle, &cam_var.capture[0]);
            raw1394_destroy_handle(cam_var.handle);
            g_warning("unable to capture a frame");
            return;
        }

        memcpy( img_frame->B[0], 
                (const char *) cam_var.capture[0].capture_buffer, 
                sizeof(char) * cam_var.capture[0].frame_width * 
                cam_var.capture[0].frame_height);   
    }
    
/* g_message("rec_frame:: 3/3"); */
}



static void
add_buffer(GpivConsole *gpiv,
           gchar *fname_base,
           gint ibuf, 
           unsigned char **frameA, 
           unsigned char **frameB
           )
/*-----------------------------------------------------------------------------
 */
{
    gchar cbuf[4], cdate[GPIV_MAX_CHARS], ctime[GPIV_MAX_CHARS];
/*     GTimeVal *my_gtime = NULL; */
    time_t ltime_t;
/*
 * Use numbers or timing to concanate after buffer names
 */


/* BUGFIX GTimeVal might be used for more accurate timing */
/*     my_gtime = g_malloc(sizeof(GTimeVal)); */
/*     g_get_current_time(my_gtime); */

    date = g_date_new ();
    g_date_set_time(date, time(NULL));
    ltime_t = time(&ltime_t);
    dac_time = localtime(&ltime_t);
    g_snprintf(ctime, sizeof(gint) * 3 + 2, 
               "%d.%d.%d_", 
               dac_time->tm_hour, 
               dac_time->tm_min, 
               dac_time->tm_sec);


    g_snprintf(cdate, sizeof(gint) * 3 + 2, 
               "%d.%d.%d_", 
               g_date_day(date), 
               g_date_month(date), 
               g_date_year(date));

    g_snprintf(cbuf, 4, "%d", ibuf);

    if (gpiv_var.fname_date
        && gpiv_var.fname_time) {
        fname_base = g_strdup(g_strconcat(gpiv_var.fn_last[0],
                                          (gchar *) cdate, 
                                          (gchar *) ctime, 
                                          (gchar *) cbuf, 
                                          NULL));
    } else if (gpiv_var.fname_date) {
        fname_base = g_strdup(g_strconcat(gpiv_var.fn_last[0],
                                          (gchar *) cdate, 
                                          (gchar *) cbuf, 
                                          NULL));
    } else if (gpiv_var.fname_time) {
        fname_base = g_strdup(g_strconcat(gpiv_var.fn_last[0],
                                          (gchar *) ctime, 
                                          (gchar *) cbuf, 
                                          NULL));
    } else {
        fname_base = g_strdup(g_strconcat(gpiv_var.fn_last[0],
                                          (gchar *) cbuf, 
                                          NULL));   
    }

    load_dac_buffer(gpiv, &cam_var, FALSE, fname_base, ibuf, frameA, frameB, 
                    INSERT);
    nbufs++;

/*     g_free(my_gtime);  */
    g_free(fname_base); 
    g_date_free(date); 
    return;
}



static void
renew_buffer(GpivConsole *gpiv,
             gchar *fname_base,
             gint ibuf, 
             unsigned char **frameA, 
             unsigned char **frameB
             )
/*-----------------------------------------------------------------------------
 * updates image of buffer
 */
{
/*
 * Open buffer 0 if not exist
 */

    if (display[ibuf] == NULL) {
        add_buffer(gpiv, fname_base, ibuf, frameA, frameB);
    } else {
        display_act = display[ibuf];

        recreate_img(display_act, frameA, frameB);
        if (display_act->display_img1) {
            show_img1(display_act);
        } else {
            hide_img1(display_act);
        }
        
        if (image_par.x_corr) {
            if (display_act->display_img2) {
                show_img2(display_act);
            } else {
                hide_img2(display_act);
            }
        }
        
    }


}



static void 
recreate_img(Display * disp,
             unsigned char **frameA,
             unsigned char **frameB
             )
/* ----------------------------------------------------------------------------
 * Re-creates image in gnome canvas; without memallocating of rgbbuf_img1 and
 * rgbbuf_img2
 * row stride; each row is a 4-byte buffer array
 */
{

    GpivImagePar image_par = disp->img.image_par;
    guchar *pos1 = NULL;
/*     guchar *pos2 = NULL; */
    gint i, j;
    GdkPixbuf *pixbuf1 = NULL;
/*     GdkPixbuf *pixbuf2 = NULL; */
    guint16 fact = 1;
    gint depth = 8;

    assert (disp != NULL);
    assert (disp->img.img1 != NULL);
    assert (disp->img.rgbbuf_img1 != NULL);
    if(image_par.x_corr) {
        assert (disp->img.img2 != NULL);
        assert (disp->img.rgbbuf_img2 != NULL);
    }
    assert (disp->img.exist_img);
    

    fact = fact << (image_par.depth - depth);
    disp->img.rgb_img_width = gpiv_par.img_width * 3;
    while ((disp->img.rgb_img_width) % 4 != 0) {
        disp->img.rgb_img_width++;
    } 
    
    pixbuf1 = gdk_pixbuf_new_from_data(disp->img.rgbbuf_img1,
                                       GDK_COLORSPACE_RGB,
                                       FALSE,          /* gboolean has_alpha */
                                       depth,          /* image_par.depth */
                                       gpiv_par.img_width, 
                                       gpiv_par.img_height,
                                       disp->img.rgb_img_width, /* rowstride */ 
                                       NULL, 
                                       NULL);
    


    pos1 = disp->img.rgbbuf_img1;
   for (i = 0; i < gpiv_par.img_height; i++) {
	for (j = 0; j < gpiv_par.img_width; j++) {
	    *pos1++ =  (guchar) (frameA[i][j] / fact);
	    *pos1++ =  (guchar) (frameA[i][j] / fact);
	    *pos1++ =  (guchar) (frameA[i][j] / fact);
	}
    }


    gnome_canvas_item_set(GNOME_CANVAS_ITEM (disp->img.gci_img1),
                          "pixbuf", pixbuf1,
                          NULL);
/*     gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (disp->img.gci_img1)); */
/*     gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (disp->img.gci_img1)); */

/* gtk_widget_pop_visual(); */
/*   gtk_widget_pop_colormap(); */


    gdk_pixbuf_unref (pixbuf1);
    
    
    
/*     if(image_par.x_corr) { */
/*         pos2 = NULL; */
        
/*         disp->img.rgbbuf_img2 = g_malloc(disp->img.rgb_img_width * 3 * */
/*                                          gpiv_par.img_height); */
        
/*         pixbuf2 = gdk_pixbuf_new_from_data(disp->img.rgbbuf_img2, */
/*                                            GDK_COLORSPACE_RGB, */
/*                                            FALSE,  */
/*                                            depth,  */
/*                                            gpiv_par.img_width,  */
/*                                            gpiv_par.img_height, */
/*                                            disp->img.rgb_img_width,  */
/*                                            NULL,  */
/*                                            NULL); */
        
/*         if (disp->img.gci_img2 != NULL) { */
/*             destroy_img(disp); */
/*         } */

/*         disp->img.gci_img2 =  */
/*             gnome_canvas_item_new( gnome_canvas_root( GNOME_CANVAS */
/*                                                       (disp->canvas)), */
/*                                    gnome_canvas_pixbuf_get_type (), */
/*                                    "pixbuf", pixbuf2, */
/*                                    NULL); */
        
        
/*         pos2 = disp->img.rgbbuf_img2; */
/*         for (i = 0; i < gpiv_par.img_height; i++) { */
/*             for (j = 0; j < gpiv_par.img_width; j++) { */
/*                 *pos2++ = (guchar) (frameB[i][j] / fact); */
/*                 *pos2++ = (guchar) (frameB[i][j] / fact); */
/*                 *pos2++ = (guchar) (frameB[i][j] / fact); */
/*             } */
/*         } */
        
/*         gdk_pixbuf_unref (pixbuf2); */
        
/*     } else { */
        disp->img.gci_img2 = disp->img.gci_img2;
/*     } */
    
}



void 
exec_rec_stop(void
               )
/*-----------------------------------------------------------------------------
 */
{
    cancel_process = TRUE;
/*     if (dc1394_stop_iso_transmission(cam_var.handle, cam_var.capture[0].node) != DC1394_SUCCESS */
/*         != DC1394_SUCCESS)  */
/*         { */
/*             warning_gpiv(_("couldn't stop the camera?")); */
/*         } */
    
/*    pthread_exit(NULL); */
/*     dc1394_release_camera(cam_var.handle, &cam_var.capture[0]); */
/*     raw1394_destroy_handle(cam_var.handle); */
}



/*
 * Callback functions
 */

void
on_entry_dac_fname(GtkSpinButton *widget, 
		      GtkWidget *entry
                      )
/*-----------------------------------------------------------------------------
 */
{
    gint i;
    Dac * dac = gtk_object_get_data(GTK_OBJECT(widget), "dac");

    push_list_lastfnames(gtk_entry_get_text(GTK_ENTRY(dac->entry_fname)));
    dac->combo_fname_items = NULL;
    for (i = 0; i < gpiv_var.number_fnames_last; i++) {
        dac->combo_fname_items = g_list_append (dac->combo_fname_items, 
                                                gpiv_var.fn_last[i]);
    }
    gtk_combo_set_popdown_strings (GTK_COMBO (dac->combo_fname), 
                                   dac->combo_fname_items);
    g_list_free (dac->combo_fname_items);

/*
 * not used:
 *     dac_par.fname_logic = TRUE;
 */
}


void
on_checkbutton_fname_date_enter(GtkWidget *widget, 
                                 gpointer data
                                 )
/* ----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Extends the name with current date");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_checkbutton_fname_date(GtkWidget *widget, 
                          gpointer data
                          )
/* ----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_var.fname_date = TRUE;
    } else {
	gpiv_var.fname_date = FALSE;
    }
    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    gnome_config_set_bool("fname_date", gpiv_var.fname_date);
    gnome_config_pop_prefix();
    gnome_config_sync();
}



void
on_checkbutton_fname_time_enter(GtkWidget *widget, 
                                 gpointer data
                                 )
/* ----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Extends the name with current time");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_checkbutton_fname_time(GtkWidget *widget, 
                          gpointer data
                          )
/* ----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_var.fname_time = TRUE;
    } else {
	gpiv_var.fname_time = FALSE;
    }
    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    gnome_config_set_bool("fname_time", gpiv_var.fname_time);
    gnome_config_pop_prefix();
    gnome_config_sync();
}



void
on_radiobutton_dac_mouse_1_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("indefinite periodic recording");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse_2_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("with pre-defined number of images");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse_3_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("one shot with interrupd");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse_4_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("periodic with interrupt");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse_5_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("with incremental separation time dt");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse_6_enter(GtkWidget * widget, 
                                  GtkWidget * entry
                                  )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("double exposure");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_dac_mouse(GtkWidget * widget, 
                          GtkWidget * entry
                          )
/*-----------------------------------------------------------------------------
 */
{
    gint mouse_select = atoi(gtk_object_get_data(GTK_OBJECT(widget),
					"mouse_select"));
    dac_par.ttime.mode = mouse_select;
/*     g_warning("on_radiobutton_dac_mouse:: ttime.mode = %d",  */
/*               dac_par.ttime.mode); */

    if (mouse_select == GPIV_TIMER_MODE__PERIODIC) {}
    if (mouse_select == GPIV_TIMER_MODE__DURATION) {}
    if (mouse_select == GPIV_TIMER_MODE__ONE_SHOT_IRQ) {}
    if (mouse_select == GPIV_TIMER_MODE__TRIGGER_IRQ) {}
    if (mouse_select == GPIV_TIMER_MODE__INCREMENT) {}
    if (mouse_select == GPIV_TIMER_MODE__DOUBLE) {}

}



void
on_spinbutton_dac_trigger_dt(GtkSpinButton * widget, 
                            GtkWidget * entry
                            )
/*-----------------------------------------------------------------------------
 */
{
    gfloat val = gtk_spin_button_get_value_as_float(widget);
    dac_par.ttime.dt =  (RTIME)  (GPIV_MILI2NANO * val);
}



void
on_spinbutton_dac_trigger_incrdt(GtkSpinButton * widget, 
                            GtkWidget * entry
                            )
/*-----------------------------------------------------------------------------
 */
{
    gfloat val = gtk_spin_button_get_value_as_float(widget);
    dac_par.ttime.increment = (RTIME) (GPIV_MILI2NANO * val);
}



void
on_spinbutton_dac_trigger_cap(GtkSpinButton * widget, 
                            GtkWidget * entry
                            )
/*-----------------------------------------------------------------------------
 */
{
    gfloat val = gtk_spin_button_get_value_as_float(widget);
    dac_par.ttime.cam_acq_period  =  (RTIME) (GPIV_MILI2NANO * val);
}



void
on_spinbutton_dac_trigger_nf(GtkSpinButton * widget, 
                            GtkWidget * entry
                            )
/*-----------------------------------------------------------------------------
 */
{
    dac_par.ttime.cycles = gtk_spin_button_get_value_as_int(widget);
}



void 
on_button_dac_triggerstart_enter(GtkWidget *widget, 
                    gpointer data
                    )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Uploads and Starts timings to kernel module");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_button_dac_triggerstart(GtkWidget * widget, 
              gpointer data
              )
/* ----------------------------------------------------------------------------
 */
{
/*     gchar *msg =  */
/*         warning_gpiv(msg); */
    cancel_process = FALSE;
    exec_trigger_start();
}



void 
on_button_dac_triggerstop_enter(GtkWidget *widget, 
                    gpointer data
                    )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Stops timings");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_button_dac_triggerstop(GtkWidget * widget, 
              gpointer data
              )
/* ----------------------------------------------------------------------------
 */
{
/*     gchar *msg =  */
/*         warning_gpiv(msg); */
    cancel_process = TRUE;
    exec_trigger_stop();
}



void
on_menu_camera_select(GtkWidget *widget, 
                          gpointer data)
/* ----------------------------------------------------------------------------
 */
{
}



void
on_menu_format(GtkWidget *widget, 
                          gpointer data)
/* ----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint mode = (int) data;
    gint format = 0;
    gint was_iso_on;
/*         change_mode_and_format(format, FORMAT_VGA_NONCOMPRESSED); */
/*   IsoFlowCheck(state); */

       g_message("on_menu_format:: ");
    if (!dc1394_get_iso_status(cam_var.camera[0].handle, cam_var.camera[0].id, 
                               &cam_var.misc_info[0].is_iso_on)) {
        g_warning("Cannot get ISO status");
    } else {
        if (cam_var.misc_info[0].is_iso_on > 0) {
            cancel_process = TRUE;
        }
    }
    was_iso_on = cam_var.misc_info[0].is_iso_on;
/*        g_message("on_menu_format:: is_iso_on = %d", was_iso_on); */


    if (mode >= MODE_FORMAT0_MIN && mode <= MODE_FORMAT0_MAX) {
        format = FORMAT_VGA_NONCOMPRESSED;

/*         g_message("on_menu_format:: mode[%d] = %s",  */
/*                   mode, format0_desc[mode - MODE_FORMAT0_MIN]); */
        
/*         g_message(":n_menu_format: format[%d] = %s", */
/*                   format, */
/*                   format_desc[format - FORMAT_MIN]); */
    } else if  (mode >= MODE_FORMAT1_MIN && mode <= MODE_FORMAT1_MAX) {
        format = FORMAT_SVGA_NONCOMPRESSED_1;
    } else if  (mode >= MODE_FORMAT2_MIN && mode <= MODE_FORMAT2_MAX) {
        format = FORMAT_SVGA_NONCOMPRESSED_2;
    } else if  (mode >= MODE_FORMAT6_MIN && mode <= MODE_FORMAT6_MAX) {
        format = FORMAT_STILL_IMAGE;
    } else if  (mode >= MODE_FORMAT7_MIN && mode <= MODE_FORMAT7_MAX) {
        format = FORMAT_SCALABLE_IMAGE_SIZE;
/*     } else if  (mode >= COLOR_FORMAT7_MIN && mode <= COLOR_FORMAT7_MAX) { */
/*         format = FORMAT_SCALABLE_IMAGE_SIZE; */
    } else {
        warning_gpiv(_("on_menu_format: non valid format"));
    }

    if (!dc1394_set_video_format(cam_var.camera[0].handle, 
                                 cam_var.camera[0].id,
                                 format)) {
        g_warning("on_menu_format:: Could not set video format");
    } else {
        cam_var.misc_info[0].format = format;
    }

    if (!dc1394_set_video_mode(cam_var.camera[0].handle,
                               cam_var.camera[0].id,
                               mode)) {
        g_warning("on_menu_format:: Could not set video format");
    } else {
        cam_var.misc_info[0].mode = mode;
    }



    if (was_iso_on > 0) {
        exec_rec_start(gpiv);
    }

/*   BuildFpsMenu(); */


/*   UpdateTriggerFrame(); */

/*   IsoFlowResume(state); */


/*     MODE_320x240_YUV422, */
/*     MODE_640x480_YUV411, */
/*     MODE_640x480_YUV422, */
/*     MODE_640x480_RGB, */
/*     MODE_640x480_MONO, */
/*     MODE_640x480_MONO16 */
}



void
on_menu_fps(GtkWidget *widget, 
            gpointer data)
/* ----------------------------------------------------------------------------
 */
{
    cam_var.misc_info[0].framerate = (int) data/*  - FRAMERATE_MIN */;
/* cam_var.misc_info[0].framerat */
    g_warning("on_menu_fps:: fps = %d", cam_var.misc_info[0].framerate);
}


/*
 * BUGFIX: include in trigger callbacks?
 */
void
on_trigger_polarity_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (!dc1394_set_trigger_polarity(cam_var.camera[0].handle,
                                   cam_var.camera[0].id,
                                   togglebutton->active)) {
      message_gpiv(_("Cannot set trigger polarity"));
  } else {
      cam_var.feature_set[0].feature[FEATURE_TRIGGER-FEATURE_MIN].
          trigger_polarity = (int) togglebutton->active;
  }
}



void
on_trigger_external_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (!dc1394_feature_on_off(cam_var.camera[0].handle, 
                             cam_var.camera[0].id, 
                             FEATURE_TRIGGER, 
                             togglebutton->active)) {
      message_gpiv(_("Cannot set external trigger source"));
  } else {
      cam_var.feature_set[0].feature[FEATURE_TRIGGER - FEATURE_MIN].is_on =
          togglebutton->active;
  }
/*   UpdateTriggerFrame(); */
}



void
on_trigger_mode_activate(GtkWidget *widget, 
                         gpointer user_data)
/* ----------------------------------------------------------------------------
 */
{
  if (!dc1394_set_trigger_mode(cam_var.camera[0].handle, cam_var.camera[0].id,
                               (int) user_data)) {
      message_gpiv(_("Cannot set trigger mode"));
  } else {
      cam_var.feature_set[0].feature[FEATURE_TRIGGER-FEATURE_MIN].
          trigger_mode = (int) user_data;
  }

/*   UpdateTriggerFrame(); */
}



/* void */
/* on_trigger_value_changed               (GtkAdjustment    *adj, */
/*                                         gpointer         user_data) */
/* { */
/*   if (!dc1394_set_feature_value(cam_var.camera[0].handle, cam_var.camera[0].id, */
/*                                 FEATURE_TRIGGER, adj->value)) */
/*     message_gpiv(_("Cannot set external trigger count")); */
/*   else */
/*     cam_var.feature_set[0].feature[FEATURE_TRIGGER-FEATURE_MIN].value = */
/*        adj->value; */
/* } */


void
on_checkbutton_camera_trigger_enter(GtkWidget *widget, 
                                    gpointer data)
/* ----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Enables triggering of the camera");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_checkbutton_camera_trigger(GtkWidget *widget, 
                              gpointer data)
/* ----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.trigger_cam = TRUE;
    } else {
	gpiv_par.trigger_cam = FALSE;
    }
}



void
on_man_auto_menu(GtkWidget *widget, 
                 gpointer data)
/* ----------------------------------------------------------------------------
 */
{
    GtkWidget *scale = gtk_object_get_data(GTK_OBJECT(widget), "scale");
    int feature = (int) data;
    enum VariableType {
        MAN,
        AUTO    
    } var_type;
    
    var_type = atoi(gtk_object_get_data(GTK_OBJECT(widget), "var_type"));
    if (!dc1394_auto_on_off(cam_var.camera[0].handle, 
                            cam_var.camera[0].id,
                            feature, var_type)) {
        message_gpiv(_("on_man_auto_menu: Cannnot set auto / manual"));
    }
    
    if (var_type == MAN) {
        gtk_widget_set_sensitive (GTK_WIDGET(scale), TRUE);
    } else if (var_type == AUTO) {
        gtk_widget_set_sensitive (GTK_WIDGET(scale), FALSE);
    }

}



void
on_scale_changed(GtkAdjustment *adj, 
         gpointer user_data)
/* ----------------------------------------------------------------------------
 */
{
    switch((int)user_data) {
    case FEATURE_TEMPERATURE:
        if (!dc1394_set_temperature(cam_var.camera[0].handle,
                                    cam_var.camera[0].id,
                                    adj->value))
            message_gpiv(_("on_scale_changed: Cannot set temperature"));
        else
            cam_var.feature_set[0].feature[FEATURE_TEMPERATURE-FEATURE_MIN].
                value = adj->value;
        break;

    default:
/*         g_message("on_scale_changed:: feature = %d value = %f",  */
/*                   (gint) user_data, adj->value); */
        if (!dc1394_set_feature_value(cam_var.camera[0].handle,
                                      cam_var.camera[0].id,
                                      (gint) user_data,
                                      adj->value)) {
            message_gpiv(_("on_scale_changed: Cannot set feature"));
        } else {
            cam_var.feature_set[0].feature[(gint) user_data - FEATURE_MIN].
                value = adj->value;
        }
        break;
    }

}



void 
on_button_dac_recstart_enter(GtkWidget *widget, 
                             gpointer data
                             )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Records PIV images");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_button_dac_recstart(GtkWidget * widget, 
                       gpointer data
                       )
/* ----------------------------------------------------------------------------
 * The actual calculation of particle image displacements
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");

    cancel_process = FALSE;
/*in exec_rec_start:     exec_trigger_start(); */
    exec_rec_start(gpiv);

}



void 
on_button_dac_recstop_enter(GtkWidget *widget, 
                    gpointer data
                    )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Stops recording PIV images");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_button_dac_recstop(GtkWidget * widget, 
              gpointer data
              )
/* ----------------------------------------------------------------------------
 * The actual calculation of particle image displacements
 */
{
    cancel_process = TRUE;
    exec_trigger_stop();
    exec_rec_stop();
}



#endif /* DISABLE_DAC  */
