/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
     engine.c: Satellite tracking routines stolen from Predict           
               http://www.qsl.net/kd2bd/predict.html                     

  Copyright (C)  2001-2005  Alexandru Csete & John A. Magliacane.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>,
            John A. Magliacane

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gconf/gconf-client.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "defaults.h"
#include "satdata.h"
#include "timeout.h"
#include "util.h"
#include "satlog.h"
#include "sataos.h"
#include "engine.h"
#include "qth.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

extern qth_struc qth;           /* qth.c */
extern sat_t *sat_table[];      /* satdata.c */
extern GConfClient *client;     /* main.c */

gint engine_timer_id = -1;

guchar val[256];
gdouble c[4][3]; /* needed by (Pre)Calc */


/* function protoypes */
static gint engine_timer_cb (gpointer);
static void PreCalc (guint i, gdouble dnum);
static void Calc (guint, gdouble);




gint
engine_init ()
{
	/* Initialize tracking engine. Returns:
	   0: Everything seems all right
	   1: default QTH file not found
	   2: hash table not initialized (FATAL)
	   3: tle file not found
	*/
	GnomeVFSResult result;
	GnomeVFSDirectoryFilter *vfsfilter;
	GnomeVFSFileInfo *fileinfo;
	GList *list,*node;
	guint x,i;
	FILE *tle;
	gchar *tlefn,*buff,*dir,*vfsuri;
	gchar name[80],line1[80],line2[80];
	sat_t *sat;

	if (qth_init ())
		return 1;

	if (satdata_init())
		return 2;

	/* Set up translation table for computing TLE checksums (stolen from predict) */
	for (x = 0; x <= 255; val[x] = 0, x++);
	for (x = '0'; x <= '9'; val[x] = x-'0', x++);
	val['-'] = 1;
     
	/* See if user has a specific TLE directory */
	buff = g_strconcat (QTH_ROOT_PATH, "/engine/tledir", NULL);
	if (gconf_client_get_string (client, buff, NULL))
		dir = g_strdup (gconf_client_get_string (client, buff, NULL));
	else
		dir = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
				   G_DIR_SEPARATOR_S, "tle", NULL);
	g_free (buff);

	buff = g_strconcat (__FUNCTION__, ": ", _("Reading directory"), " ", dir, ":", NULL);
	satlog_log (SAT_LOG_INFO, buff);
	g_free (buff);

	/* Load directory list */
	/* convert to VFS URI */
	vfsuri = gnome_vfs_get_uri_from_local_path (dir);

	/* create a VFS filter and load the directory */
	vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						    GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						    "*.tle");
	result = gnome_vfs_directory_list_load (&list, vfsuri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						vfsfilter);
	/* free some memory */
	g_free (vfsuri);
	gnome_vfs_directory_filter_destroy (vfsfilter);

	if (result == GNOME_VFS_OK) {
		/* load each file */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;   /* fileinfo->name contains file name */
			tlefn = g_strconcat (dir, G_DIR_SEPARATOR_S, fileinfo->name, NULL);

			/* Open TLE file */
			tle = fopen (tlefn, "r");
			if (tle) {
				i = 0;
				/* read in the elements */
				while (!feof(tle) && (satdata_count_sat () < SAT_MAX)) {
					if (fgets (name,  75, tle) && fgets (line1, 75, tle) &&
					    fgets (line2, 75, tle) && KepCheck (line1, line2)) {
						sat = g_malloc (sizeof (sat_t));
						/* remove trailing spaces from name */
						x = strlen (name);
						while (name[x]==32 || name[x]==0 || name[x]==10 || name[x]==13) {
							name[x] = 0;
							x--;
						}
						sat->name = g_strdup (name);
						sat->catnum  = (guint32) g_strtod (SubString (line1,2,6), NULL);
						sat->class   = line1[7];
						sat->lyear   = (g_strtod (SubString (line1,9,10), NULL) < 50) ?
							(2000 + (guint16) g_strtod (SubString (line1,9,10), NULL)) :
							(1900 + (guint16) g_strtod (SubString (line1,9,10), NULL));
						sat->lnum    = (guint) g_strtod (SubString (line1,11,13), NULL);
						sat->piece   = g_strdup (SubString (line1,14,16));
						sat->epyear  = (guint16) g_strtod (SubString (line1,18,19), NULL);
						sat->epday   = g_strtod (SubString (line1,20,31), NULL);
						sat->ephem   = line1[62];
						sat->elnum   = (guint32) g_strtod (SubString (line1,64,67), NULL);
						sat->incl    = g_strtod (SubString (line2,8,15), NULL);
						sat->raan    = g_strtod (SubString (line2,17,24), NULL);
						sat->eccn    = 1e-7*g_strtod (SubString (line2,26,32), NULL);
						sat->argper  = g_strtod (SubString (line2,34,41), NULL);
						sat->meanan  = g_strtod (SubString (line2,43,50), NULL);
						sat->meanmo  = g_strtod (SubString (line2,52,62), NULL);
						sat->drag    = g_strtod (SubString (line1,33,42), NULL);
						sat->decac   = 1e-5*g_strtod (SubString (line1,44,51), NULL);
						sat->bstar   = 1e-5*g_strtod (SubString (line1,53,60), NULL);
						sat->eporb   = (gulong) g_strtod (SubString (line2,63,67), NULL);
						sat->file = g_strdup (fileinfo->name);

						/* add satellite to data table */
						satdata_add_sat (sat);
						i++;
					}
				}
				/* log entry */
				buff = g_strdup_printf ("%s: %s %d %s %s", __FUNCTION__, _("Found"), i,
							_("satellites in"), fileinfo->name);
				satlog_log (SAT_LOG_INFO, buff);
				g_free (buff);
			}
			else {
				buff = g_strconcat (__FUNCTION__, ": ", _("Could not open file"), " ", tlefn, NULL);
				satlog_log (SAT_LOG_CRITICAL, buff);
				g_free (buff);
			}
			g_free (tlefn);
			fclose (tle);
		}
	}
	g_free (dir);

	/* do the precalc stuff */
	PreCalcAll (CurrentDaynum());
	CalcAll ();

	return 0;
}



void engine_close ()
{
	if (engine_timer_id != -1)
		engine_stop ();
	satdata_close ();
	qth_close ();
}


gint
engine_reload ()
{
	/* This function should be called when fatal changes occur to the
	   satellite data or the QTH settings on disk. It will reload
	   everything.
	*/
 	gchar *buff;
 	gint retcode = 0;

 	buff = g_strconcat (__FUNCTION__, ": ", _("Starting reload sequence"), NULL);
 	satlog_log (SAT_LOG_INFO, buff);
 	g_free (buff);

	/* Stop Sun and Moon timers */
	sun_timer_stop ();
	moon_timer_stop ();

	/* Close and reinitialize the engine */
 	engine_close ();
 	retcode = engine_init ();
 	if (!retcode) {
 		engine_start ();
 		satlog_log (SAT_LOG_INFO, _("Retcode: 0 (everything all right)"));

		/* Restart the Sun and moon timers */
		sun_timer_start ();
		moon_timer_start ();
 	}
 	else {
 		buff = g_strdup_printf (_("%s: Received code: %d. SOMETHING IS WRONG!"),
 					__FUNCTION__, retcode);
 		satlog_log (SAT_LOG_CRITICAL, buff);
 		g_free (buff);
 		satlog_log (SAT_LOG_CRITICAL, _("This is where you should panic! "\
 						"From this point on, the program will "\
 						"probably not function properly."));
 	}

	return 0;
}


void
engine_update ()
{
	/* This is a "soft" version of the engine_reload function.
	   It will not delete the whole satellite data structure, 
	   instead it will silently update the elements.
	   This function is safe to use while various modules are
	   running.
	*/
	GnomeVFSResult result;
	GnomeVFSDirectoryFilter *vfsfilter;
	GnomeVFSFileInfo *fileinfo;
	GList *list,*node;
	guint x,i;
	FILE *tle;
	gchar *tlefn,*buff,*dir,*vfsuri;
	gchar name[80],line1[80],line2[80];
	sat_t *sat;


	buff = g_strconcat (__FUNCTION__, _("Starting update sequence:"), NULL);
	satlog_log (SAT_LOG_INFO, buff);
	g_free (buff);

	/* Set up translation table for computing TLE checksums (stolen from predict) */
	for (x = 0; x <= 255; val[x] = 0, x++);
	for (x = '0'; x <= '9'; val[x] = x-'0', x++);
	val['-'] = 1;
     
	/* See if user has a specific TLE directory */
	buff = g_strconcat (QTH_ROOT_PATH, "/engine/tledir", NULL);
	if (gconf_client_get_string (client, buff, NULL))
		dir = g_strdup (gconf_client_get_string (client, buff, NULL));
	else
		dir = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
				   G_DIR_SEPARATOR_S, "tle", NULL);
	g_free (buff);

	buff = g_strconcat (__FUNCTION__, ": ", _("Reading directory"), " ", dir, ":", NULL);
	satlog_log (SAT_LOG_INFO, buff);
	g_free (buff);

	/* Load directory list */
	/* convert to VFS URI */
	vfsuri = gnome_vfs_get_uri_from_local_path (dir);

	/* create a VFS filter and load the directory */
	vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						    GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						    "*.tle");
	result = gnome_vfs_directory_list_load (&list, vfsuri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						vfsfilter);
	/* free some memory */
	g_free (vfsuri);
	gnome_vfs_directory_filter_destroy (vfsfilter);

	if (result == GNOME_VFS_OK) {
		/* load each file */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;   /* fileinfo->name contains file name */
			tlefn = g_strconcat (dir, G_DIR_SEPARATOR_S, fileinfo->name, NULL);

			/* Open TLE file */
			tle = fopen (tlefn, "r");
			if (tle) {
				i = 0;
				/* read in the elements */
				while (!feof(tle) && (satdata_count_sat () < SAT_MAX)) {
					if (fgets (name,  75, tle) && fgets (line1, 75, tle) &&
					    fgets (line2, 75, tle) && KepCheck (line1, line2)) {
						sat = g_malloc (sizeof (sat_t));
						/* remove trailing spaces from name */
						x = strlen (name);
						while (name[x]==32 || name[x]==0 || name[x]==10 || name[x]==13) {
							name[x] = 0;
							x--;
						}
						sat->name = g_strdup (name);
						sat->catnum  = (guint32) g_strtod (SubString (line1,2,6), NULL);
						sat->class   = line1[7];
						sat->lyear   = (g_strtod (SubString (line1,9,10), NULL) < 50) ?
							(2000 + (guint16) g_strtod (SubString (line1,9,10), NULL)) :
							(1900 + (guint16) g_strtod (SubString (line1,9,10), NULL));
						sat->lnum    = (guint) g_strtod (SubString (line1,11,13), NULL);
						sat->piece   = g_strdup (SubString (line1,14,16));
						sat->epyear  = (guint16) g_strtod (SubString (line1,18,19), NULL);
						sat->epday   = g_strtod (SubString (line1,20,31), NULL);
						sat->ephem   = line1[62];
						sat->elnum   = (guint32) g_strtod (SubString (line1,64,67), NULL);
						sat->incl    = g_strtod (SubString (line2,8,15), NULL);
						sat->raan    = g_strtod (SubString (line2,17,24), NULL);
						sat->eccn    = 1e-7*g_strtod (SubString (line2,26,32), NULL);
						sat->argper  = g_strtod (SubString (line2,34,41), NULL);
						sat->meanan  = g_strtod (SubString (line2,43,50), NULL);
						sat->meanmo  = g_strtod (SubString (line2,52,62), NULL);
						sat->drag    = g_strtod (SubString (line1,33,42), NULL);
						sat->decac   = 1e-5*g_strtod (SubString (line1,44,51), NULL);
						sat->bstar   = 1e-5*g_strtod (SubString (line1,53,60), NULL);
						sat->eporb   = (gulong) g_strtod (SubString (line2,63,67), NULL);
						sat->file = g_strdup (fileinfo->name);

						/* add satellite to data table */
						satdata_add_sat (sat);
						i++;
					}
				}
				/* log entry */
				buff = g_strdup_printf ("%s: %s %d %s %s", __FUNCTION__, _("Found"), i,
							_("satellites in"), fileinfo->name);
				satlog_log (SAT_LOG_INFO, buff);
				g_free (buff);
			}
			else {
				buff = g_strconcat (__FUNCTION__, ": ", _("Could not open file"), " ", tlefn, NULL);
				satlog_log (SAT_LOG_CRITICAL, buff);
				g_free (buff);
			}
			g_free (tlefn);
			fclose (tle);
		}
	}
	g_free (dir);

	/* do the precalc stuff */
	PreCalcAll (CurrentDaynum());
	CalcAll ();
}




void engine_start ()
{
	guint32 tval;
	gchar *buff;

	if (engine_timer_id != -1) {
		buff = g_strconcat (__FUNCTION__, ": ", 
				    _("Tryed to start already running timer (action ignored)"),
				    NULL);
		satlog_log (SAT_LOG_CRITICAL, buff);
		g_free (buff);
	}
	else {
		/* get user defined or default timeout value */
		tval = (gconf_client_get_int (client, ENGINE_TIMER_PATH, NULL) == 0) ?
			ENGINE_DEF_TIMER : gconf_client_get_int (client, ENGINE_TIMER_PATH, NULL);
		engine_timer_id = gtk_timeout_add (tval, 
						   engine_timer_cb, 
						   NULL );
		buff = g_strdup_printf (_("%s: Starting engine (%d msec)"), __FUNCTION__, tval);
		satlog_log (SAT_LOG_INFO, buff);
		g_free (buff);
	}

}


void engine_stop ()
{
	gchar *buff;

	if ( engine_timer_id != -1 ) {
		/* we have a real timeout running */
		gtk_timeout_remove (engine_timer_id);
		engine_timer_id = -1;
		buff = g_strconcat (__FUNCTION__, ": ", _("Engine stopped"), NULL);
		satlog_log (SAT_LOG_INFO, buff);
		g_free (buff);
	}
	else {
		buff = g_strconcat (__FUNCTION__, ": ",
				    _("Tryed to stop not running timer (action ignored)"),
				    NULL);
		satlog_log (SAT_LOG_CRITICAL, buff);
		g_free (buff);
	}
}


static gint
engine_timer_cb (gpointer data)
{
	CalcAll();
	return TRUE;
}


void
PreCalcAll (gdouble dnum)
{
	/* This function calls PreCalc for all satellites */
	guint i,cnt;

	cnt = satdata_count_sat();
	for ( i=0; i<cnt; i++)
		PreCalc (i, dnum);
}


static void
PreCalc (guint i, gdouble dnum)
{
	/* This function performs preliminary calculations
	   prior to tracking or prediction.  Shamelessly
	   stolen from Predict (but modified as necessary).
	*/
 
	if ( ( i >= 0 ) && ( i < SAT_MAX ) ) {

		sat_table[i]->daynum = dnum;

		sat_table[i]->epoch = DayNum (1,0,sat_table[i]->epyear) + sat_table[i]->epday;
		sat_table[i]->age = sat_table[i]->daynum-sat_table[i]->epoch;
		sat_table[i]->yr = (gfloat) sat_table[i]->epyear;

		/* Do the Y2K thing... */

		if ( sat_table[i]->yr <= 50.0)
			sat_table[i]->yr += 100.0;

		sat_table[i]->t1 = sat_table[i]->yr - 1.0;
		sat_table[i]->df = 366.0+floor(365.25*(sat_table[i]->t1-80.0)) - 
			floor(sat_table[i]->t1/100.0)+floor(sat_table[i]->t1/400.0+0.75);
		sat_table[i]->t1 = (sat_table[i]->df+29218.5)/36525.0;
		sat_table[i]->t1 = 6.6460656+sat_table[i]->t1*(2400.051262+sat_table[i]->t1*2.581e-5);
		sat_table[i]->se = sat_table[i]->t1/24.0-sat_table[i]->yr;
		sat_table[i]->n0 = sat_table[i]->meanmo+(sat_table[i]->age*sat_table[i]->drag);
		sat_table[i]->sma = 331.25*pow((1440.0/sat_table[i]->n0),(2.0/3.0));
		sat_table[i]->e2 = 1.0-(sat_table[i]->eccn*sat_table[i]->eccn); 
		sat_table[i]->e1 = sqrt(sat_table[i]->e2);
		/**** BIG TIME BUG??? ****/
		sat_table[i]->k2 = 9.95*(exp(log(R0/sat_table[i]->sma)*3.5))/sat_table[i]->e2*sat_table[i]->e2;
		/*************************/
		sat_table[i]->s1 = sin(sat_table[i]->incl*DEG2RAD); 
		sat_table[i]->c1 = cos(sat_table[i]->incl*DEG2RAD);
		sat_table[i]->l8 = qth.lat*DEG2RAD; 
		sat_table[i]->s9 = sin(sat_table[i]->l8); 
		sat_table[i]->c9 = cos(sat_table[i]->l8);
		sat_table[i]->s8 = sin(-qth.lon*DEG2RAD); 
		sat_table[i]->c8 = cos(qth.lon*DEG2RAD);
		sat_table[i]->r9 = R0*(1.0+(FF/2.0)*(cos(2.0*sat_table[i]->l8)-1.0))+(qth.alt/1000.0);
		sat_table[i]->l8 = atan((1.0-FF)*(1.0-FF)*sat_table[i]->s9/sat_table[i]->c9);
		sat_table[i]->z9 = sat_table[i]->r9*sin(sat_table[i]->l8);
		sat_table[i]->x9 = sat_table[i]->r9*cos(sat_table[i]->l8)*sat_table[i]->c8;
		sat_table[i]->y9 = sat_table[i]->r9*cos(sat_table[i]->l8)*sat_table[i]->s8;
		sat_table[i]->apogee = sat_table[i]->sma*(1.0+sat_table[i]->eccn)-R0;
		sat_table[i]->perigee = sat_table[i]->sma*(1.0-sat_table[i]->eccn)-R0;
		sat_table[i]->hasaos = AosHappens(sat_table[i]) ? TRUE : FALSE;
		sat_table[i]->geostat = Geostationary(sat_table[i]) ? TRUE : FALSE;
		sat_table[i]->decayed = Decayed(sat_table[i], 0.0) ? TRUE : FALSE;
		sat_table[i]->status = sat_table[i]->hasaos ? SAT_STATUS_NONE : SAT_STATUS_NOAOS;
		if ( Decayed(sat_table[i], 0.0) )
		     sat_table[i]->status = SAT_STATUS_DECAYED;
		else {
			if ( Geostationary(sat_table[i]) )
				sat_table[i]->status = SAT_STATUS_GEOSTAT;
		}
	}
}


void
CalcAll ()
{
	/* This function calls Calc for all satellites */
	guint i,cnt;
	gdouble daynum;

	daynum = CurrentDaynum ();
	cnt = satdata_count_sat ();
	for ( i=0; i<cnt; i++)
		Calc( i, daynum );
}


gdouble vis_el,vlh,ve,vn,vsi,vco,vha;  /* for visibility */


static void
Calc (guint i , gdouble dnum)
{
	/* This is the stuff we need to do repetitively.
	   Stolen from Predict - modified as necessary.
	*/


	if ( ( i >= 0 ) && ( i < SAT_MAX ) ) {
		sat_table[i]->age = dnum-sat_table[i]->epoch;
		sat_table[i]->o = DEG2RAD*(sat_table[i]->raan-(sat_table[i]->age)*
					   sat_table[i]->k2*sat_table[i]->c1);
		sat_table[i]->s0 = sin(sat_table[i]->o); 
		sat_table[i]->c0 = cos(sat_table[i]->o);
		sat_table[i]->w = DEG2RAD*(sat_table[i]->argper+(sat_table[i]->age)*sat_table[i]->k2*
					   (2.5*sat_table[i]->c1*sat_table[i]->c1-0.5));
		sat_table[i]->s2 = sin(sat_table[i]->w); 
		sat_table[i]->c2 = cos(sat_table[i]->w);
		c[1][1] = sat_table[i]->c2*sat_table[i]->c0-sat_table[i]->s2*sat_table[i]->s0*sat_table[i]->c1;
		c[1][2] =-sat_table[i]->s2*sat_table[i]->c0-sat_table[i]->c2*sat_table[i]->s0*sat_table[i]->c1;
		c[2][1] = sat_table[i]->c2*sat_table[i]->s0+sat_table[i]->s2*sat_table[i]->c0*sat_table[i]->c1;
		c[2][2] =-sat_table[i]->s2*sat_table[i]->s0+sat_table[i]->c2*sat_table[i]->c0*sat_table[i]->c1;
		c[3][1] = sat_table[i]->s2*sat_table[i]->s1;
		c[3][2] = sat_table[i]->c2*sat_table[i]->s1;
		sat_table[i]->q0 = (sat_table[i]->meanan/360.0)+sat_table[i]->eporb;
		sat_table[i]->phase = sat_table[i]->n0*sat_table[i]->age+sat_table[i]->q0; 
		sat_table[i]->orbit = (glong)floor(sat_table[i]->phase);
		sat_table[i]->phase = sat_table[i]->phase-floor(sat_table[i]->phase);
		sat_table[i]->m = sat_table[i]->phase*TP;
		sat_table[i]->e = sat_table[i]->m+sat_table[i]->eccn*
			(sin(sat_table[i]->m)+0.5*sat_table[i]->eccn*sin(sat_table[i]->m*2.0));
		do   /* Kepler's Equation */
		{
			sat_table[i]->s3 = sin(sat_table[i]->e); 
			sat_table[i]->c3 = cos(sat_table[i]->e); 
			sat_table[i]->r3 = 1.0-sat_table[i]->eccn*sat_table[i]->c3;
			sat_table[i]->m1 = sat_table[i]->e-sat_table[i]->eccn*sat_table[i]->s3; 
			sat_table[i]->m5 = sat_table[i]->m1 - sat_table[i]->m;
			sat_table[i]->e  = sat_table[i]->e - sat_table[i]->m5/sat_table[i]->r3;
		} while ( fabs(sat_table[i]->m5) >= 1.0e-6 );
		sat_table[i]->x0 = sat_table[i]->sma*(sat_table[i]->c3-sat_table[i]->eccn); 
		sat_table[i]->yzero = sat_table[i]->sma*sat_table[i]->e1*sat_table[i]->s3;
		sat_table[i]->r = sat_table[i]->sma*sat_table[i]->r3; 
		sat_table[i]->x1 = sat_table[i]->x0*c[1][1]+sat_table[i]->yzero*c[1][2];
		sat_table[i]->yone = sat_table[i]->x0*c[2][1]+sat_table[i]->yzero*c[2][2];
		sat_table[i]->z1 = sat_table[i]->x0*c[3][1]+sat_table[i]->yzero*c[3][2];
		sat_table[i]->g7 = (dnum-sat_table[i]->df)*1.0027379093+sat_table[i]->se;
		sat_table[i]->g7 = TP*(sat_table[i]->g7-floor(sat_table[i]->g7));
		sat_table[i]->s7 = -sin(sat_table[i]->g7); 
		sat_table[i]->c7 = cos(sat_table[i]->g7);
		sat_table[i]->x = sat_table[i]->x1*sat_table[i]->c7-sat_table[i]->yone*sat_table[i]->s7; 
		sat_table[i]->y = sat_table[i]->x1*sat_table[i]->s7+sat_table[i]->yone*sat_table[i]->c7;
		sat_table[i]->z = sat_table[i]->z1; 
		sat_table[i]->x5 = sat_table[i]->x-sat_table[i]->x9; 
		sat_table[i]->y5 = sat_table[i]->y-sat_table[i]->y9; 
		sat_table[i]->z5 = sat_table[i]->z-sat_table[i]->z9;
		sat_table[i]->range2 = sat_table[i]->x5*sat_table[i]->x5+sat_table[i]->y5 *
			sat_table[i]->y5+sat_table[i]->z5*sat_table[i]->z5; 
		sat_table[i]->z8 = sat_table[i]->x5*sat_table[i]->c8*sat_table[i]->c9+sat_table[i]->y5 *
			sat_table[i]->s8*sat_table[i]->c9+sat_table[i]->z5*sat_table[i]->s9;
		sat_table[i]->x8 =-sat_table[i]->x5*sat_table[i]->c8*sat_table[i]->s9-sat_table[i]->y5 * 
			sat_table[i]->s8*sat_table[i]->s9+sat_table[i]->z5*sat_table[i]->c9;
		sat_table[i]->y8 = sat_table[i]->y5*sat_table[i]->c8-sat_table[i]->x5*sat_table[i]->s8; 
		sat_table[i]->alt = sat_table[i]->r-R0;
		sat_table[i]->el = atan(sat_table[i]->z8/sqrt(sat_table[i]->range2-sat_table[i]->z8 *
							      sat_table[i]->z8))/DEG2RAD;
		sat_table[i]->az = atan(sat_table[i]->y8/sat_table[i]->x8)/DEG2RAD;

		if ( sat_table[i]->x8 < 0.0 )
			sat_table[i]->az += 180.0;

		if ( sat_table[i]->az < 0.0 )
			sat_table[i]->az += 360.0;

		/*ma256=(int)256.0*phase;*/
		sat_table[i]->range = sqrt(sat_table[i]->range2); 
		sat_table[i]->vel = 3.6*sqrt(3.98652e+14*((2.0/(sat_table[i]->r*1000.0)) -
							  1.0/(sat_table[i]->sma*1000.0)));
		sat_table[i]->fp = 12756.33*acos(R0/sat_table[i]->r);
		sat_table[i]->lat = atan(sat_table[i]->z/sqrt(sat_table[i]->r*sat_table[i]->r -
							      sat_table[i]->z*sat_table[i]->z))/DEG2RAD;
		sat_table[i]->lon =-atan(sat_table[i]->y/sat_table[i]->x)/DEG2RAD;

		if ( sat_table[i]->x < 0.0)
			sat_table[i]->lon += 180.0;
		/* 0..360 */
		if ( sat_table[i]->lon < 0.0 )
			sat_table[i]->lon += 360.0;
		/* change to -180..180 */
		if ( sat_table[i]->lon > 180.0 )
			sat_table[i]->lon -= 360.0;

		sat_table[i]->range = sqrt(sat_table[i]->range2);

		/* Update status */
		switch (sat_table[i]->status) {
		case SAT_STATUS_NONE:
			if (sat_table[i]->el > 0.0) {
				sat_table[i]->status = SAT_STATUS_CAMEUP;
				sat_table[i]->los = aos_find_los( *sat_table[i],
								  CurrentDaynum()+0.005 );
				sat_table[i]->aos = aos_find_aos( *sat_table[i],
								  sat_table[i]->los+0.015 );
			}
			if (sat_table[i]->el < 0.0) {
				sat_table[i]->status = SAT_STATUS_WENTDOWN;
				sat_table[i]->aos = aos_find_aos( *sat_table[i],
								  CurrentDaynum()+0.005 );
				sat_table[i]->los = aos_find_los( *sat_table[i],
								  sat_table[i]->aos+0.005 );
			}
			break;
		case SAT_STATUS_BHOR:
			if (sat_table[i]->el > 0.0) {
				sat_table[i]->los = aos_find_los( *sat_table[i],
								  CurrentDaynum()+0.005);
				sat_table[i]->aos = aos_find_aos( *sat_table[i],
								  sat_table[i]->los+0.015);
				sat_table[i]->status = SAT_STATUS_CAMEUP;
			}
			break;
		case SAT_STATUS_AHOR:
			if (sat_table[i]->el < 0.0) {
				sat_table[i]->aos = aos_find_aos( *sat_table[i],
								  CurrentDaynum()+0.005 );
				sat_table[i]->los = aos_find_los( *sat_table[i],
								  sat_table[i]->aos+0.005 );
				sat_table[i]->status = SAT_STATUS_WENTDOWN;
			}
			break;
		default:
			break;
		}

	}
}


gboolean AosHappens (sat_t *sat)
{
	/* This function returns TRUE if the satellite pointed to by
	   "sat" can ever rise above the horizon of the ground station.
	   Stolen from Predict.
	*/

	gdouble lin, sma, apogee;

	lin=sat->incl;

	if (lin >= 90.0)
		lin=180.0-lin;

	sma=331.25*exp(log(1440.0/sat->meanmo)*(2.0/3.0));
	apogee=sma*(1.0+sat->eccn)-R0;

	if ((acos(R0/(apogee+R0))+(lin*DEG2RAD)) > fabs(qth.lat*DEG2RAD))
		return TRUE;
	else
		return FALSE;
}


gboolean Decayed (sat_t *sat, gdouble time)
{
	/* This function returns TRUE if it appears that the
	   satellite pointed to by "sat" has decayed at the
	   time of 'time'.  If 'time' is 0.0, then
	   the current date/time is used.
	*/

	gdouble satepoch;

	if (time==0.0)
		time=CurrentDaynum();

	satepoch=DayNum(1,0,sat->epyear)+sat->epday;

	if (satepoch+((16.666666-sat->meanmo)/(10.0*fabs(sat->drag))) < time)
		return TRUE;
	else
		return FALSE;
}


gboolean Geostationary (sat_t *sat)
{
	/* This function returns TRUE if the satellite pointed
	   to by "sat" appears to be in a geostationary orbit.
	   Stolen from Predict.
	*/

	if (fabs(sat->meanmo-1.0027)<0.0002) 

		return TRUE;
	else
		return FALSE;
}
