/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  satconfig-locations.c: Configuration wizard for Gnome Predict: Locations.
                         This module is responsible for the widgets and
			 callbacks related to ground station locations.


  Copyright (C)  2002-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <stdio.h>
#include "satloc-tree.h"
#include "satconfig.h"
#include "satconfig-locations.h"
#include "qth.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


#if defined (HAVE_HAMLIB)
#  include <hamlib/rotator.h>


static void satconfig_locations_latlon_cb         (GtkAdjustment *adj, gpointer data);
static void satconfig_locations_latlon_toggled_cb (GtkToggleButton *button, gpointer locbut);
static void satconfig_locations_qra_changed_cb    (GtkWidget *qra, gpointer data);

#endif

/* initialized in satconfig.c */
extern GdkColor logobg,mainbg;
extern gboolean CONFIGERROR;
extern gboolean CONFIGOK;

/* initialized in main.c */
extern GtkWidget *app;
extern GConfClient *client;

static GtkWidget *qth_name,*qth_desc;                  /* name and descr. entries */
static GtkWidget *qth_loc,*latsp,*lonsp,*altsp;        /* location, lat, lon and alt entries */
static GtkWidget *wxsta;                               /* nearest weather station */
static GtkWidget *qra_loc;                             /* QRA locator */

GtkWidget *radio1,*radio2;
extern GtkWidget *druid;    /* in satconfig.c */

/* This structure is used as buffer when the user
   selects a row in the locations tree.
*/
static struct {
	gchar *location;
	gdouble lat;
	gdouble lon;
	gint alt;
	gchar *wx;
} selection;


/* default QTH, if already configured */
static gint defqth = -1;


/* private function prototypes */
static void     satconfig_locations_locbut_cb                   (GtkWidget *button,
								 gpointer data);

static void     satconfig_locations_loctree_select_cb           (GtkCTree *ctree,
								 GList    *node,
								 gint      column,
								 gpointer  dialog);

static void     satconfig_locations_loctree_unselect_cb         (GtkCTree *ctree,
								 GList    *node,
								 gint     column,
								 gpointer   dialog);

static void     satconfig_locations_wxbut_cb                    (GtkWidget *button, 
								 gpointer   data);

static gboolean satconfig_locations_read_predict_qth            (void);
static gboolean satconfig_locations_read_gnome_predict_qth      (void);
static gboolean satconfig_locations_read_gnome_predict_0_4_qth  (void);

static void     satconfig_locations_prepare_cb                  (GnomeDruidPage *druidpage,
								 GtkWidget      *druid,
								 gpointer        data);

static void     satconfig_locations_name_cb                     (GtkEntry *entry, gpointer data);


GtkWidget *
satconfig_locations_open (void)
{
	/* This function creates and returns the QTH config
	   page. It also checks, whether the user already has
	   a QTH configured (either from earlier versions of
	   Gnome Predict or from Predict.
	*/
	GtkWidget *qthpage,*table;
	GtkWidget *label;
	GtkWidget *locbut,*wxbut;
	GtkTooltips *tips;
	GtkObject *adj1,*adj2,*adj3;
	GdkImlibImage *logo;
	gchar *fname;

	/* create a new table */
	table = gtk_table_new (10, 4, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);

	/* Name / Call */
	label = gtk_label_new (_("Name/Call:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 0, 1);
	qth_name = gtk_entry_new ();
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_name,
			      _("Enter the name of your ground station or your callsign, "\
				"if you happen to be a radio amateur. This field is mandatory."),
			      _("This name will be used on various occasions like prints or saved images."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_name, 2, 4, 0, 1);

	/* connect the changed signal */
	gtk_signal_connect (GTK_OBJECT (qth_name), "changed",
			    GTK_SIGNAL_FUNC (satconfig_locations_name_cb), NULL);
	
	/* Description */
	label = gtk_label_new (_("Description:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 1, 2);
	qth_desc = gtk_entry_new ();
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_desc,
			      _("You can enter a description of your ground station here, if you wish."),
			      _("You can enter a description of your ground station here, if you wish."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_desc, 2, 4, 1, 2);

	/* separator */
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 4, 2, 3);
	
	/* Latitude */
	label = gtk_label_new (_("Latitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 3, 4);
	adj1 = gtk_adjustment_new (0.0, -90, 90, 0.1, 1.0, 1);
	latsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj1), 0.1, 4);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (latsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (latsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, latsp,
			      _("The latitude of your ground station in decimal degrees North (South = negative)."),
			      _("The latitude of your ground station in decimal degrees North (South = negative)."));
	gtk_table_attach (GTK_TABLE (table), latsp, 2, 3, 3, 4, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* Longitude */
	label = gtk_label_new (_("Longitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 4, 5);
	adj2 = gtk_adjustment_new (0.0, -180.0, 180.0, 0.1, 1.0, 1);
	lonsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj2), 0.1, 4);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (lonsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (lonsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, lonsp,
			      _("The longitude of your ground station in decimal degrees West (East = negative)."),
			      _("The longitude of your ground station in decimal degrees West (East = negative)."));
	gtk_table_attach (GTK_TABLE (table), lonsp, 2, 3, 4, 5, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 6, 6, 7);

	/* Location */
	label = gtk_label_new (_("Location:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 7, 8);
	qth_loc = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (qth_loc), 100);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_loc,
			      _("The location of your ground station (City, Country)."),
			      _("The location of your ground station (City, Country)."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_loc, 2, 3, 7, 8);

	/* button */
	locbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
				      _("Select Location"));
	gtk_signal_connect (GTK_OBJECT (locbut), "clicked",
			    GTK_SIGNAL_FUNC (satconfig_locations_locbut_cb), NULL);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, locbut,
			      _("Click to select a location from a list of more than 2500 "\
				"predefined locations around the world."),
			      _("Click to select a location from a list of more than 2500 "\
				"predefined locations around the world."));
	gtk_table_attach_defaults (GTK_TABLE (table), locbut, 3, 4, 7, 8);

	/* altitude */
	label = gtk_label_new (_("Altitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 8, 9);
	adj3 = gtk_adjustment_new (0, -1000, 9000, 5, 100, 1);
	altsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj3), 1, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (altsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (altsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, altsp,
			      _("The altitude above sea level in meters."),
			      _("The altitude above sea level in meters."));
	gtk_table_attach (GTK_TABLE (table), altsp, 2, 3, 8, 9, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* weather station */
	label = gtk_label_new (_("Weather:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 9, 10);
	wxsta = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (wxsta), 4);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, wxsta,
			      _("Nearest weather station. This is a four letter combination "\
				"and is used to obtain weather information from the internet."),
			      _("Nearest weather station. This is a four letter combination "\
				"and is used to obtain weather information from the internet."));
	gtk_table_attach_defaults (GTK_TABLE (table), wxsta, 2, 3, 9, 10);

	/* button */
	wxbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
				     _("Select Station"));
	gtk_signal_connect (GTK_OBJECT (wxbut), "clicked",
			    GTK_SIGNAL_FUNC (satconfig_locations_wxbut_cb), NULL);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, wxbut,
			      _("Click to select a weather station from a list of more than 2500 "\
				"predefined stations around the world. You will probably find one "\
				"near your location."),
			      _("Click to select a weather station from a list of more than 2500 "\
				"predefined stations around the world. You will probably find one "\
				"near your location."));
	gtk_table_attach_defaults (GTK_TABLE (table), wxbut, 3, 4, 9, 10);

#if defined (HAVE_HAMLIB)
	/* QRA locator */
	label = gtk_label_new (_("Locator:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 5, 6);
//	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 5, 6, GTK_SHRINK, GTK_SHRINK, 0, 0);
	qra_loc = gtk_entry_new ();
	gtk_signal_connect (GTK_OBJECT (qra_loc), "changed",
			    GTK_SIGNAL_FUNC (satconfig_locations_qra_changed_cb),
			    NULL);
	gtk_entry_set_max_length (GTK_ENTRY (qra_loc), 6);
	gtk_widget_set_sensitive (qra_loc, FALSE);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qra_loc,
			      _("You can enter your Mainhead grid locator here."),
			      _("You can enter your Mainhead grid locator here."));
	gtk_table_attach (GTK_TABLE (table), qra_loc, 2, 3, 5, 6, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* connect changed signals to calc functions */
	gtk_signal_connect (adj1, "value-changed",
			    GTK_SIGNAL_FUNC (satconfig_locations_latlon_cb),
			    NULL);
	gtk_signal_connect (adj2, "value-changed",
			    GTK_SIGNAL_FUNC (satconfig_locations_latlon_cb),
			    NULL);

	/* Add radio buttons */
	radio1 = gtk_radio_button_new (NULL);
	gtk_signal_connect (GTK_OBJECT (radio1), "toggled",
			    GTK_SIGNAL_FUNC (satconfig_locations_latlon_toggled_cb),
			    locbut);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, radio1,
			      _("Select this option if you wish to specify your location "\
				"using geographical coordinates"),
			      _("Select this option if you wish to specify your location "\
				"using geographical coordinates"));
	radio2 = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON (radio1));
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, radio2,
			      _("Select this option if you wish to specify your location "\
				"using the Mainhead locator system"),
			      _("Select this option if you wish to specify your location "\
				"using the Mainhead locator system"));
	gtk_table_attach (GTK_TABLE (table), radio1, 0, 1, 3, 4, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_table_attach (GTK_TABLE (table), radio2, 0, 1, 5, 6, GTK_SHRINK, GTK_SHRINK, 0, 0);
#endif

	/* Check for existing configuration */
	if (!satconfig_locations_read_gnome_predict_0_4_qth ()) {
		if (!satconfig_locations_read_gnome_predict_qth ()) {
			satconfig_locations_read_predict_qth ();
		}
	}

	/* create the page */
	qthpage = gnome_druid_page_standard_new ();
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (qthpage),
					     _("Default Location:"));
	gnome_druid_page_standard_set_bg_color (GNOME_DRUID_PAGE_STANDARD (qthpage), &logobg);
	gnome_druid_page_standard_set_title_color (GNOME_DRUID_PAGE_STANDARD (qthpage), &mainbg);
	gnome_druid_page_standard_set_logo_bg_color (GNOME_DRUID_PAGE_STANDARD (qthpage), &logobg);

	/* Connect the "prepare" signal so that we can disable the next button. It will be enabled
	   if we have a reasonable value in the QTH name field.
	*/
	gtk_signal_connect (GTK_OBJECT (qthpage), "prepare",
			    GTK_SIGNAL_FUNC (satconfig_locations_prepare_cb), NULL);

	/* add logo */
	fname = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S,
			     "icons", G_DIR_SEPARATOR_S, "gpredict-home.png",
			     NULL);
	logo = gdk_imlib_load_image (fname);
	g_free (fname);
	gnome_druid_page_standard_set_logo (GNOME_DRUID_PAGE_STANDARD (qthpage), logo);
	gdk_imlib_destroy_image (logo);

	gtk_container_set_border_width (GTK_CONTAINER (GNOME_DRUID_PAGE_STANDARD (qthpage)->vbox), 5);

	/* Add contents to page */
	gtk_box_pack_start (GTK_BOX ((GNOME_DRUID_PAGE_STANDARD (qthpage))->vbox),
			    table, FALSE, FALSE, 10);

	return qthpage;
}



void
satconfig_locations_finish (void)
{
	/* This function should be called by the configuration
	   druid when the user clicks on the FINISH button.
	*/
	gchar *key;
	gboolean ok = TRUE;

	/* Check whether we have a default QTH, otherwise make default 0.
	   One may argue, that if we already did have a QTH configuration,
	   we should not rewrite it. Then again, the user might have made some
	   modifications.
	*/
	if (defqth == -1) {
		defqth = 0;
	}

	/* QTH Name */
	key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, defqth);
	if (gtk_entry_get_text (GTK_ENTRY (qth_name)) != NULL)
		ok = ok && gconf_client_set_string (client, key, 
						    gtk_entry_get_text (GTK_ENTRY (qth_name)),
						    NULL);
	/* FIXME: As of 0.4.99.8 this field can't be empty! */
	else
		ok = ok && gconf_client_set_string (client, key,
						    _("Some Station"),
						    NULL);
	g_free (key);

	/* QTH Description */
	key = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, defqth);
	if (gtk_entry_get_text (GTK_ENTRY (qth_desc)) != NULL)
		ok = ok && gconf_client_set_string (client, key,
						    gtk_entry_get_text (GTK_ENTRY (qth_desc)),
						    NULL);
	else
		ok = ok && gconf_client_set_string (client, key,
						    _("Some Description"),
						    NULL);
	g_free (key);

	/* QTH Location */
	key = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, defqth);
	if (gtk_entry_get_text (GTK_ENTRY (qth_loc)) != NULL)
		ok = ok && gconf_client_set_string (client, key,
						    gtk_entry_get_text (GTK_ENTRY (qth_loc)),
						    NULL);
	else
		ok = ok && gconf_client_set_string (client, key,
						    _("Some Location"),
						    NULL);
	g_free (key);

	/* Weather Station */
	if (gtk_entry_get_text (GTK_ENTRY (wxsta)) != NULL) {
		key = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, defqth);
		ok = ok && gconf_client_set_string (client, key,
						    gtk_entry_get_text (GTK_ENTRY (wxsta)),
						    NULL);
		g_free (key);
	}

	/* QRA Locator */
	if (gtk_entry_get_text (GTK_ENTRY (qra_loc)) != NULL) {
		key = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, defqth);
		ok = ok && gconf_client_set_string (client, key,
						    gtk_entry_get_text (GTK_ENTRY (qra_loc)),
						    NULL);
		g_free (key);
	}

	/* QTH latitude */
	key = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, defqth);
	ok = ok && gconf_client_set_float (client, key,
					   gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (latsp)),
					   NULL);
	g_free (key);

	/* QTH Longitude */
	key = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, defqth);
	ok = ok && gconf_client_set_float (client, key,
					   gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (lonsp)),
					   NULL);
	g_free (key);

	/* QTH Altitude */
	key = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, defqth);
	ok = ok && gconf_client_set_int (client, key,
					 gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (altsp)),
					 NULL);
	g_free (key);

	gconf_client_suggest_sync (client, NULL);

	if (selection.location)
		g_free (selection.location);
	if (selection.wx)
		g_free (selection.wx);

	if (!ok) {
		CONFIGOK = FALSE;
		CONFIGERROR = TRUE;
	}
}


void
satconfig_locations_cancel (void)
{
	/* This function should be called by the configuration
	   druid, when the user clicks on the CANCEL button.
	*/
	if (selection.location)
		g_free (selection.location);
	if (selection.wx)
		g_free (selection.wx);
}


static void
satconfig_locations_locbut_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   "Select" location button.
	*/
	GtkWidget *tree,*dialog,*swin;
	gint yesno;

	/* create the tree and hide WX station column */
	tree = satloc_tree_create ();
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_WX, FALSE);

	/* put tree into scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), tree);
	gtk_widget_show_all (swin);

	/* create a dialog window */
	dialog = gnome_dialog_new (_("Select a location"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gtk_widget_set (dialog, "width", 400, "height", 300, NULL);
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);

	/* OK is disabled until a selection is made */
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);

	/* conect signal for row selection */
	gtk_signal_connect (GTK_OBJECT (tree), "tree-select-row",
			    GTK_SIGNAL_FUNC (satconfig_locations_loctree_select_cb),
			    dialog);
	gtk_signal_connect (GTK_OBJECT (tree), "tree-unselect-row",
			    GTK_SIGNAL_FUNC (satconfig_locations_loctree_unselect_cb),
			    dialog);

	/* Fianlly, run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		gtk_entry_set_text (GTK_ENTRY (qth_loc), selection.location);
		gtk_entry_set_text (GTK_ENTRY (wxsta), selection.wx);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), selection.lat);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), selection.lon);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), selection.alt);
	}
}


static void
satconfig_locations_loctree_select_cb      (GtkCTree *ctree,
					    GList *node,
					    gint column,
					    gpointer dialog)
{
	/* This function is called when the user selects a row in
	   the locations tree. It updates the current selection
	   variables and enables the OK button in the dialog.
	*/
	GtkCTreeNode *par;
	gchar *buff1,*buff2;

	g_return_if_fail (dialog != NULL);

	if (selection.location)
		g_free (selection.location);
	if (selection.wx)
		g_free (selection.wx);

	/* Get parent node containing the country */
	par = GTK_CTREE_NODE (GTK_CTREE_ROW (GTK_CTREE_NODE (node))->parent);
	gtk_ctree_node_get_pixtext (ctree, par, 
				    SATLOC_TREE_COL_LOC,
				    &buff1,
				    0,
				    NULL, NULL);
	/* location */
	gtk_ctree_node_get_pixtext (ctree, 
				    GTK_CTREE_NODE (node),
				    SATLOC_TREE_COL_LOC,
				    &buff2,
				    0,
				    NULL,
				    NULL);
	
	selection.location = g_strdup_printf ("%s, %s", buff2, buff1);

	/* Latitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_LAT, &buff2);
	selection.lat = g_strtod (buff2, NULL);

	/* Longitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_LON, &buff2);
	selection.lon = g_strtod (buff2, NULL);

	/* Altitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_ALT, &buff2);
	selection.alt = (gint) g_strtod (buff2, NULL);

	/* weather station */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_WX, &buff2);
	selection.wx = g_strdup (buff2);

	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, TRUE);
}


static void
satconfig_locations_loctree_unselect_cb    (GtkCTree *ctree,
					    GList *node,
					    gint column,
					    gpointer dialog)
{
	/* This function is called when the user unselects a row in
	   the locations tree. It disables the OK button in the dialog.
	*/

	g_return_if_fail (dialog);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
}


static void
satconfig_locations_wxbut_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   "Select" nearest weather station button.
	*/
	GtkWidget *tree,*dialog,*swin;
	gint yesno;

	/* create the tree and hide unused columns */
	tree = satloc_tree_create ();
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_LAT, FALSE);
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_LON, FALSE);
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_ALT, FALSE);

	/* put tree into scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), tree);
	gtk_widget_show_all (swin);

	/* create a dialog window */
	dialog = gnome_dialog_new (_("Select a weather station"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gtk_widget_set (dialog, "width", 400, "height", 300, NULL);
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);

	/* OK is disabled until a selection is made */
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);

	/* recycle functions from "select location" dialog */
	gtk_signal_connect (GTK_OBJECT (tree), "tree-select-row",
			    GTK_SIGNAL_FUNC (satconfig_locations_loctree_select_cb),
			    dialog);
	gtk_signal_connect (GTK_OBJECT (tree), "tree-unselect-row",
			    GTK_SIGNAL_FUNC (satconfig_locations_loctree_unselect_cb),
			    dialog);

	/* Fianlly, run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		gtk_entry_set_text (GTK_ENTRY (wxsta), selection.wx);
	}
}



static gboolean
satconfig_locations_read_predict_qth       (void)
{
	/* This function checks whether the user has a
	   ~/.predict/predict.qth. If the file exists, it
	   reads the data into the widgets and returns
	   TRUE. Otherwise it returns FALSE.
	*/
	gchar *path;

	path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
			    ".predict", G_DIR_SEPARATOR_S,
			    "predict.qth", NULL);
	if (g_file_test (path, G_FILE_TEST_ISFILE)) {
		FILE *qthfile;

		qthfile = fopen (path, "r");
		if (qthfile) {
			gchar *buff;
			gdouble latlon;
			gint alt;

			/* Call sign */
			buff = g_malloc (80);
			fgets (buff, 80, qthfile);
			g_strdelimit (buff, "\n", '\0');
			gtk_entry_set_text (GTK_ENTRY (qth_name), buff);
			g_free (buff);

			/* latitude */
			fscanf (qthfile, "%lf", &latlon);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), latlon);

			/* longitude */
			fscanf (qthfile, "%lf", &latlon);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), latlon);

			/* altitude */
			fscanf (qthfile, "%d", &alt);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), alt);

			fclose (qthfile);
		}

		g_free (path);

		return TRUE;
	}

	g_free (path);

	return FALSE;
}


static gboolean
satconfig_locations_read_gnome_predict_qth (void)
{
	/* This function checks whether the user has a
	   ~/.gpredict/default.qth. If the file exists, it
	   reads the data into the widgets and returns
	   TRUE. Otherwise it returns FALSE.
	*/
	gchar *path;

	path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
			    ".gpredict", G_DIR_SEPARATOR_S,
			    "default.qth", NULL);
	if (g_file_test (path, G_FILE_TEST_ISFILE)) {
		FILE *qthfile;

		qthfile = fopen (path, "r");
		if (qthfile) {
			gchar *buff;
			gdouble latlon;
			gint alt;

			/* Call sign */
			buff = g_malloc (80);
			fgets (buff, 80, qthfile);
			g_strdelimit (buff, "\n", '\0');
			gtk_entry_set_text (GTK_ENTRY (qth_name), buff);
			g_free (buff);

			/* Description */
			buff = g_malloc (80);
			fgets (buff, 80, qthfile);
			g_strdelimit (buff, "\n", '\0');
			gtk_entry_set_text (GTK_ENTRY (qth_desc), buff);
			g_free (buff);

			/* Location */
			buff = g_malloc (80);
			fgets (buff, 80, qthfile);
			g_strdelimit (buff, "\n", '\0');
			gtk_entry_set_text (GTK_ENTRY (qth_loc), buff);
			g_free (buff);

			/* latitude */
			fscanf (qthfile, "%lf", &latlon);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), latlon);

			/* longitude */
			fscanf (qthfile, "%lf", &latlon);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), latlon);

			/* altitude */
			fscanf (qthfile, "%d", &alt);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), alt);

			fclose (qthfile);
		}

		g_free (path);

		return TRUE;
	}

	g_free (path);

	return FALSE;
}

static gboolean
satconfig_locations_read_gnome_predict_0_4_qth (void)
{
	/* Check for QTH configuration (post 0.4.0 versions) */
	gchar *key;
	gchar *buff;

	if (gconf_client_dir_exists (client, QTH_ROOT_PATH, NULL)) {

		/* get default */
		key = g_strdup_printf ("%s/default", QTH_ROOT_PATH);
		defqth = gconf_client_get_int (client, key, NULL);
		g_free (key);

		/* get name */
		key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, defqth);
		buff = g_strdup (gconf_client_get_string (client, key, NULL));
		if (buff) {
			gtk_entry_set_text (GTK_ENTRY (qth_name), buff);
		}
		g_free (buff);
		g_free (key);

		/* get description */
		key = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, defqth);
		buff = g_strdup (gconf_client_get_string (client, key, NULL));
		if (buff) {
			gtk_entry_set_text (GTK_ENTRY (qth_desc), buff);
		}
		g_free (buff);
		g_free (key);

		/* get location */
		key = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, defqth);
		buff = g_strdup (gconf_client_get_string (client, key, NULL));
		if (buff) {
			gtk_entry_set_text (GTK_ENTRY (qth_loc), buff);
		}
		g_free (buff);
		g_free (key);

		/* get weather station */
		key = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, defqth);
		buff = g_strdup (gconf_client_get_string (client, key, NULL));
		if (buff) {
			gtk_entry_set_text (GTK_ENTRY (wxsta), buff);
		}
		g_free (buff);
		g_free (key);

		/* get latitude */
		key = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, defqth);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), gconf_client_get_float (client, key, NULL));
		g_free (key);

		/* get longitude */
		key = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, defqth);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), gconf_client_get_float (client, key, NULL));
		g_free (key);

		/* get altitude */
		key = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, defqth);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), gconf_client_get_int (client, key, NULL));
		g_free (key);

#if defined (HAVE_HAMLIB)
		/* get QRA locator */
		key = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, defqth);
		buff = g_strdup (gconf_client_get_string (client, key, NULL));
		if (buff) {
			gtk_entry_set_text (GTK_ENTRY (qra_loc), buff);
		}
		g_free (buff);
		g_free (key);
#endif

		return TRUE;
	}

	return FALSE;
}



/**
 * \brief Handle prepare signals for the QTH config page.
 * \param druidpage The GnomeDruidPage that received the signal.
 * \param druid     The GnomeDruid that emitted the signal.
 * \param data      User data.
 *
 * This function is called when the satconfig-locations druid page receives the
 * prepare signal from the man druid. This means that the druid is abouet to show
 * this page. The idea with this function is to diable the Next button until a
 * reasonable value has been entered to the Name/Call field.
 *
 * \internal
 */
static void
satconfig_locations_prepare_cb             (GnomeDruidPage *druidpage,
					    GtkWidget      *druid,
					    gpointer        data)
{
	gchar *text;

	/* get text from qth_name widget */
	text = gtk_entry_get_text (GTK_ENTRY (qth_name));

	/* If there is some text, enable all buttons */
	if ((text != NULL) && g_strcasecmp (text, "")) {
		gnome_druid_set_buttons_sensitive (GNOME_DRUID (druid), TRUE, TRUE, TRUE);
	}
	/* otherwise disable the next button */
	else {
		gnome_druid_set_buttons_sensitive (GNOME_DRUID (druid), TRUE, FALSE, TRUE);
	}
}



/**
 * \brief Handle changed signals for the QTH name entry widget.
 * \param entry The GtkEntry that received the signal.
 * \param data  User data.
 * 
 * This function is called when the user modifies the text in the QTH name entry. The
 * purpose with this function is to enable or disable the Next button, depending on whether
 * the name entry contains some text or not.
 *
 * \internal
 */
static void
satconfig_locations_name_cb                (GtkEntry *entry, gpointer data)
{
	gchar *text;

	text = gtk_entry_get_text (entry);
	if ((text != NULL) && g_strcasecmp (text, "")) {
		gnome_druid_set_buttons_sensitive (GNOME_DRUID (druid), TRUE, TRUE, TRUE);
	}
	else {
		gnome_druid_set_buttons_sensitive (GNOME_DRUID (druid), TRUE, FALSE, TRUE);
	}
}



/* The rest is only included if hamlib 1.2.0 or later is used */
#if defined (HAVE_HAMLIB)

static void
satconfig_locations_latlon_cb              (GtkAdjustment *adj, gpointer data)
{
	/* This function is called when either the latitude or the longitude spin
	   button is changed. It updates the locator entry.
	*/
	gchar locator[2*QRA_PAIR_COUNT+1];

	/* Check whether user is typing into lat/lon entries or the change comes from
	   automatic conversion (this can be checked by reading the status of the
	   longitude-latitude radio button).
	*/
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio1))) {
		/* Try to convert input */
		if (longlat2locator (-gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (lonsp)),
				     gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (latsp)),
				     locator, QRA_PAIR_COUNT) == RIG_OK) {
			
			/* Conversion is OK, update locator field */
			gtk_entry_set_text (GTK_ENTRY (qra_loc), g_strdup (locator));
		}
		else {
			/* Clear locator field and disable "Next" button
			   because user is typing some junk.
			   This could be avoided by using spinbuttons! */
			gtk_entry_set_text (GTK_ENTRY (qra_loc), "");
		}
	}
}


static void
satconfig_locations_latlon_toggled_cb (GtkToggleButton *button, gpointer locbut)
{
	/* This function is called when the LAT/LON radio button
	   is toggled. It checks whether the button is pressed in
	   or not, and performs the corresponding actions.
	*/

	if (gtk_toggle_button_get_active (button)) {
		/* User selected LAT/LON */
		gtk_widget_set_sensitive (qra_loc, FALSE);
		gtk_widget_set_sensitive (latsp, TRUE);
		gtk_widget_set_sensitive (lonsp, TRUE);
		gtk_widget_set_sensitive (GTK_WIDGET (locbut), TRUE);
	}
	else {
		/* User selected QRA */
		gtk_widget_set_sensitive (qra_loc, TRUE);
		gtk_widget_set_sensitive (latsp, FALSE);
		gtk_widget_set_sensitive (lonsp, FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (locbut), FALSE);
	}

}

static void
satconfig_locations_qra_changed_cb    (GtkWidget *qra, gpointer data)
{
	/* This function is called when the text in the Locator
	   entry is changed.
	*/

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio2))) {
		gdouble lat,lon;

		if (!locator2longlat (&lon, &lat, gtk_entry_get_text (GTK_ENTRY (qra)))) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), lat);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), -lon);
		}
	}
}

#endif


