/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  satlog.c: Logger terminal for Gnome Predict.

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <time.h>
#include "defaults.h"
#include "satlog.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

extern GtkWidget *app;
extern GConfClient *client;

gchar times[22];
guint satloglevel = SAT_LOG_ERROR | SAT_LOG_CRITICAL | SAT_LOG_WARNING;

GtkWidget *msgwin = NULL;    /* message window */
GtkWidget *msgtext = NULL;   /* message box */


/* some function prototypes */
void satlog_hide_func    (GtkWidget *, gpointer);
void satlog_clear_func   (GtkWidget *, gpointer);
void satlog_filter_func  (GtkWidget *, gpointer);
void satlog_set_loglevel (GtkWidget *, gpointer);
gchar *now (void);


void satlog_init ()
{
	/* This function initializes the logger and
	   creates the message window.
	*/
	GtkWidget *msgvbar,*msghbox;      /* gtk_text related */
	GtkWidget *hbut,*cbut,*fbut;      /* hide and clear buttons */
	GtkWidget *vbox,*bbox;

	/* get user defined loglevel */
	satloglevel |= gconf_client_get_int (client, SATLOG_LOGLEVEL_PATH, NULL);

	/* text display */
	msghbox = gtk_hbox_new (FALSE, 0);
	msgtext = gtk_text_new (NULL, NULL);
	gtk_text_set_editable (GTK_TEXT (msgtext), FALSE);
	gtk_box_pack_start (GTK_BOX (msghbox), msgtext, TRUE, TRUE, 0 );

	msgvbar = gtk_vscrollbar_new (GTK_TEXT (msgtext)->vadj );

	gtk_box_pack_end (GTK_BOX (msghbox), msgvbar, FALSE, FALSE, 0 );

	/* message window */
	msgwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (msgwin), _("Gnome Predict Messages"));
	gtk_window_set_policy (GTK_WINDOW (msgwin), FALSE, TRUE, TRUE );

	hbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_CLOSE),
				    _("Hide") );
	gtk_signal_connect (GTK_OBJECT (hbut), "clicked",
			    GTK_SIGNAL_FUNC (satlog_hide_func ),
			    NULL );

	cbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_CLEAR),
				    _("Clear") );
	gtk_signal_connect (GTK_OBJECT( cbut ), "clicked",
			    GTK_SIGNAL_FUNC (satlog_clear_func),
			    NULL );

	fbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_PROPERTIES),
				    _("Filter"));
	gtk_signal_connect (GTK_OBJECT (fbut), "clicked",
			    GTK_SIGNAL_FUNC (satlog_filter_func),
			    NULL);

	/* Create buttons and pack them into a button box */
	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 0 );
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox),
				   GTK_BUTTONBOX_START);
	gtk_box_pack_start (GTK_BOX (bbox), GTK_WIDGET (hbut), TRUE, TRUE, 0 );
	gtk_box_pack_start (GTK_BOX (bbox), GTK_WIDGET (cbut), TRUE, TRUE, 0 );
	gtk_box_pack_start (GTK_BOX (bbox), GTK_WIDGET (fbut), TRUE, TRUE, 0 );

	/* main vertical box */
	vbox = gtk_vbox_new (FALSE , 0 );
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (msghbox), TRUE, TRUE, 0 );
	gtk_box_pack_end (GTK_BOX (vbox), GTK_WIDGET (bbox), FALSE, FALSE, 0 );

	gtk_container_add (GTK_CONTAINER (msgwin), GTK_WIDGET (vbox) );

	/* We have to watch out for delete or destroy signal from sloppy users */
	gtk_signal_connect (GTK_OBJECT (msgwin), "delete_event",
			    GTK_SIGNAL_FUNC (satlog_hide_func ), NULL);
/*	gtk_signal_connect (GTK_OBJECT (msgwin), "destroy",
	GTK_SIGNAL_FUNC (msgwin_delete), NULL );*/

	gtk_widget_set_usize (GTK_WIDGET (msgwin), 500, 150 );
}


void satlog_show ()
{
	if (msgwin)
		gtk_widget_show_all (GTK_WIDGET (msgwin));
	else
		satlog_init();
	gconf_client_set_bool (client, SATLOG_VISIBLE_PATH, TRUE, NULL );
}

void satlog_hide ()
{
	if (msgwin) {
		gtk_widget_hide_all (GTK_WIDGET (msgwin));
		gconf_client_set_bool (client, SATLOG_VISIBLE_PATH, FALSE, NULL );
	}
}

void satlog_hide_func (GtkWidget *widget, gpointer p)
{
	/* This function is caaled when the user clicks on
	   the "hide" button.
	*/
	if (msgwin) {
		gtk_widget_hide_all( GTK_WIDGET (msgwin) );
		gconf_client_set_bool (client, SATLOG_VISIBLE_PATH, FALSE, NULL );
	}
}


void satlog_clear_func (GtkWidget *widget, gpointer p)
{
	/* This function is caaled when the user clicks on
	   the "clear" button.
	*/
	if (msgwin)
		gtk_editable_delete_text( GTK_EDITABLE(msgtext), 0, -1 );
}


void satlog_filter_func( GtkWidget *widget, gpointer p )
{
	/*This function is called when the user
	  clicks on the  the filter button.
	*/
	GtkWidget *dialog,*chk;

	/* create the dialog window */
	dialog = gnome_dialog_new (_("SatLog Filter"), GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (app));
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);
	
	chk = gtk_check_button_new_with_label (_("Errors"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_ERROR);
	gtk_signal_connect (GTK_OBJECT (chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_ERROR));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	chk = gtk_check_button_new_with_label (_("Critical Warnings"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_CRITICAL);
	gtk_signal_connect (GTK_OBJECT (chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_CRITICAL));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	chk = gtk_check_button_new_with_label (_("Warnings"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_WARNING);
	gtk_signal_connect (GTK_OBJECT(chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_WARNING));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	chk = gtk_check_button_new_with_label (_("Messages"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_MESSAGE);
	gtk_signal_connect (GTK_OBJECT (chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_MESSAGE));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	chk = gtk_check_button_new_with_label(_("Informations"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_INFO);
	gtk_signal_connect (GTK_OBJECT (chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_INFO));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	chk = gtk_check_button_new_with_label (_("Debug Infos"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chk),
				      satloglevel & SAT_LOG_DEBUG);
	gtk_signal_connect (GTK_OBJECT (chk), "toggled",
			    GTK_SIGNAL_FUNC (satlog_set_loglevel),
			    GINT_TO_POINTER (SAT_LOG_DEBUG));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    chk, TRUE, TRUE, 0 );
	
	gtk_widget_show_all (dialog);
}

void satlog_set_loglevel (GtkWidget *widget, gpointer level)
{
	/* This function is called when the user changes
	   the log level.
	*/
	gint lev;

	lev = GPOINTER_TO_INT (level);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		satloglevel = satloglevel | lev;
	else
		satloglevel = satloglevel & ~lev;

	/* save new value */
	gconf_client_set_int (client, SATLOG_LOGLEVEL_PATH, satloglevel, NULL);
	gconf_client_suggest_sync (client, NULL);
}


gchar *now ()
{
	/* Utility function witch returns the current time
	   as a formatted string. It should be replaced
	   by the new functions that exist.
	*/
	time_t *mytp;

	mytp = (time_t *) g_malloc (sizeof (time_t));
	*mytp = time (NULL);
	strftime (times, 22, "[%Y/%m/%d %H:%M:%S]", localtime (mytp));
	g_free (mytp);

	return times;
}

void satlog_log (sat_log_level_t level, const gchar *message)
{
	/* This is the main API function to call */
	
	gchar *tmp;

	/* do we need to log this message */
	if ( (satloglevel & level) && msgwin ) {
		switch (level) {
		case SAT_LOG_NONE:
			break;
		case SAT_LOG_ERROR:
			tmp = g_strconcat (now (), _(" ERROR: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		case SAT_LOG_CRITICAL:
			tmp = g_strconcat (now (), _(" CRITICAL: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		case SAT_LOG_WARNING:
			tmp = g_strconcat (now (), _(" WARNING: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		case SAT_LOG_MESSAGE:
			tmp = g_strconcat (now (), _(" MESSAGE: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		case SAT_LOG_INFO:
			tmp = g_strconcat (now (), _(" INFO: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		case SAT_LOG_DEBUG:
			tmp = g_strconcat (now (), _(" DEBUG: "), message, "\n", NULL);
			gtk_text_insert (GTK_TEXT (msgtext), NULL,
					 &msgtext->style->black, NULL,
					 tmp, -1);
			g_free (tmp);
			break;
		}	
	}
}
