/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satmap-druid.c: Tools and widgets to set up a new satmap.

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>


  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gconf/gconf-client.h>
#include "satlog.h"
#include "satmap.h"
#include "satdata.h"
#include "satmap-druid.h"
#include "sat-ui-utils.h"
#include "sattree.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

extern GConfClient *client; /* main.c */

/* Private function prototypes */
static gint satmap_druid_list_compare_func   (GtkCList *, gconstpointer, gconstpointer);
static void satmap_druid_select_all_cb       (GtkWidget *, gpointer);
static void satmap_druid_unselect_all_cb     (GtkWidget *, gpointer);
static void satmap_druid_add_selected_cb     (GtkWidget *, gpointer);
static void satmap_druid_delete_selected_cb  (GtkWidget *, gpointer);
static gint satmap_druid_uint_compare_func   (gconstpointer, gconstpointer);
static void satmap_druid_name_changed_cb     (GtkEditable *, gpointer);
static void satmap_druid_cancel_cb           (GtkWidget *, gpointer);
static void satmap_druid_ok_cb               (GtkWidget *, GtkCList *);
static void satmap_druid_update_clist_title  (GtkCList *, const gchar *);
static void satmap_map_selector_cb           (GtkButton *button, gpointer defmap);
static void satmap_druid_read_data_file      (GtkCList *, const gchar *);


extern GtkWidget *app;  /* in main.c */
static gchar *name = NULL;
static gchar *mapfile = NULL;

static gchar *title2[1] = { N_("Shown Satellites") };

static GtkWidget *dp = NULL; /* pointer to dialog */

gchar *satmap_druid_run (const gchar *module)
{
	/* This function is called when satmap_open is invoked
	   with a NULL argument. This function pops up a dialog
	   window, where the user can configure a new satmap module,
	   and saves the configuration in a file wich name is specified
	   by the user.
	   The function returns the configuration file path or NULL
	   if the user cancels the configuration process.
	*/
	GtkWidget *list1,*list2,*swin1,*swin2,*table;
	GtkWidget *selal1,*desal1,*addsel,*selal2,*desal2,*delsel;
	GtkWidget *entry,*dialog,*help,*preview,*mapbut;
	guint satnum;
	gchar *iconfile,*configpath;
	/* sattree stuff */
	gchar **loc,*filter,*buff;

	/* check if we have any satellites at all */
	satnum = satdata_count_sat ();
	if (!satnum)
		return NULL;

	/* Module Name Widget with history */
	entry = gtk_entry_new ();
	sat_ui_utils_set_tooltip (entry, _("Enter the name of the new satellite map"));
	gtk_entry_set_max_length (GTK_ENTRY (entry), 50);
	/* "changed" signal is connected later when the dialog is ready */
	if (module != NULL) {
		gtk_entry_set_text (GTK_ENTRY (entry), module);
		gtk_widget_set_sensitive (entry, FALSE);
		if (name)
			g_free (name);
		name = g_strdup (module);  /* otherwise the "OK-callback" will think there is no module name */
	}
	
	/* help button */
	help = gnome_stock_button (GNOME_STOCK_BUTTON_HELP);
	gtk_widget_set_sensitive (help, FALSE);

	/* Create map preview */
	if (module != NULL) {
		/* create a true config path */
		configpath = g_strconcat ("=", g_get_home_dir (), G_DIR_SEPARATOR_S,
					  ".gpredict", G_DIR_SEPARATOR_S,
					  module, ".satmap=", SATMAP_SAT_MAP_PATH, NULL);
		mapfile = gnome_config_get_string (configpath);
	}
	else {
		mapfile = gconf_client_get_string (client, SATMAP_DEFMAP_PATH, NULL);
	}

	if (!mapfile) {
		/* get the hardcoded default */
		mapfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps",
				       G_DIR_SEPARATOR_S, "earth_800.jpg", NULL);
	}
	preview = gnome_pixmap_new_from_file_at_size (mapfile, PREVIEW_SIZE_X, PREVIEW_SIZE_Y);
	/**** FIXME: g_free (mapfile) ??? */

	mapbut = gtk_button_new_with_label (_("Select..."));
	sat_ui_utils_set_tooltip (mapbut, _("Select another map"));
	gtk_signal_connect (GTK_OBJECT (mapbut), "clicked",
			    GTK_SIGNAL_FUNC (satmap_map_selector_cb), preview);

	/* create and fill the tree */
	buff = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
			    G_DIR_SEPARATOR_S, "tle", NULL);
	loc = g_strsplit (buff, "<!SEP>", 0);
	filter = g_strdup ("*.tle");

	list1 = sattree_load_uris_recursive (loc, NULL, filter, SATTREE_MODE_SAT);
	g_strfreev (loc);
	g_free (buff);
	g_free (filter);

	/* Expand the tree */
	gtk_ctree_expand_to_depth (GTK_CTREE (list1), NULL, 2);

	/* sort the tree */
	gtk_clist_set_compare_func (GTK_CLIST (list1), satmap_druid_list_compare_func );
	gtk_clist_set_sort_column (GTK_CLIST (list1), 0);
	gtk_clist_set_sort_type (GTK_CLIST (list1), GTK_SORT_ASCENDING);
	gtk_clist_sort (GTK_CLIST (list1));

	/* put the tree in a scrolled window */
	swin1 = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin1),
					GTK_POLICY_NEVER,
					GTK_POLICY_ALWAYS);
	gtk_container_add (GTK_CONTAINER (swin1), list1);

	/* the satmap list */
	list2 = gtk_clist_new_with_titles (1, title2);
	gtk_clist_column_titles_passive (GTK_CLIST (list2));
	gtk_clist_set_column_width (GTK_CLIST (list2), 0,
				    gtk_clist_optimal_column_width (GTK_CLIST (list2), 0));
	gtk_clist_set_selection_mode (GTK_CLIST (list2), GTK_SELECTION_EXTENDED);

	/* read in the satellites if module != NULL */
	if (module)
		satmap_druid_read_data_file (GTK_CLIST (list2), module);
	else
		/* attach information that this is a new module we are creating */
		gtk_object_set_data (GTK_OBJECT (list2), "newmod", GUINT_TO_POINTER (1));

	/* update title with number of satellites */
	satmap_druid_update_clist_title (GTK_CLIST (list2), title2[0]);

	/* set up and enable sorting */
	gtk_clist_set_compare_func (GTK_CLIST (list2), satmap_druid_list_compare_func );
        gtk_clist_set_sort_column (GTK_CLIST (list2), 0);
	gtk_clist_set_sort_type (GTK_CLIST (list2), GTK_SORT_ASCENDING);
	gtk_clist_set_auto_sort (GTK_CLIST (list2), TRUE); /* Way Cool! */

	/* put the list in a scrolled window */
	swin2 = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin2),
					GTK_POLICY_NEVER,
					GTK_POLICY_ALWAYS);
	gtk_container_add (GTK_CONTAINER (swin2), list2);

	/* buttons */
	selal1 = gtk_button_new_with_label (_("Select All"));
	sat_ui_utils_set_tooltip (selal1, _("Select all satellites in the list"));
	gtk_signal_connect (GTK_OBJECT (selal1), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_select_all_cb),
			    list1);

	desal1 = gtk_button_new_with_label (_("Unselect All"));
	sat_ui_utils_set_tooltip (desal1, _("Unselect all satellites in the list"));
	gtk_signal_connect (GTK_OBJECT (desal1), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_unselect_all_cb),
			    list1);

	addsel = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_FORWARD),
				      _("Add"));
	sat_ui_utils_set_tooltip (addsel, _("Add selected satellites to the new map"));
	/* We need to pass both lists to the "Add Selected" callback
	   so we attach the lists to the button */
	gtk_object_set_data (GTK_OBJECT (addsel), "list1", list1);
	gtk_object_set_data (GTK_OBJECT (addsel), "list2", list2);
	gtk_signal_connect (GTK_OBJECT (addsel), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_add_selected_cb),
			    NULL);

	selal2 = gtk_button_new_with_label (_("Select All"));
	sat_ui_utils_set_tooltip (selal2, _("Select all satellites in the list"));
	gtk_signal_connect (GTK_OBJECT (selal2), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_select_all_cb),
			    list2);

	desal2 = gtk_button_new_with_label (_("Unselect All"));
	sat_ui_utils_set_tooltip (desal2, _("Unselect all satellites in the list"));
	gtk_signal_connect (GTK_OBJECT (desal2), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_unselect_all_cb),
			    list2);

	delsel = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_TRASH),
				      _("Remove"));
	sat_ui_utils_set_tooltip (delsel, _("Remove selected satellites from the list"));
	gtk_signal_connect (GTK_OBJECT (delsel), "clicked",
			    GTK_SIGNAL_FUNC (satmap_druid_delete_selected_cb),
			    list2);

	/* create table that contains widgets */
	table = gtk_table_new (9, 6, TRUE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 10);
	gtk_table_set_col_spacings (GTK_TABLE (table), 10);
	gtk_table_attach_defaults (GTK_TABLE (table),
				   gtk_label_new (_("Module Name:")),
				   0, 1, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table),
				   entry,
				   1, 4, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table),
				   gtk_label_new (_("(required)")),
				   4, 5, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), help, 5, 6, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table),
				   gtk_label_new (_("Background Map:")),
				   0, 1, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), preview, 1, 3, 1, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), mapbut, 0, 1, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), swin1, 0, 3, 3, 8);
	gtk_table_attach_defaults (GTK_TABLE (table), swin2, 3, 6, 3, 8);
	gtk_table_attach_defaults (GTK_TABLE (table), selal1, 0, 1, 8, 9);
	gtk_table_attach_defaults (GTK_TABLE (table), desal1, 1, 2, 8, 9);
	gtk_table_attach_defaults (GTK_TABLE (table), addsel, 2, 3, 8, 9);
	gtk_table_attach_defaults (GTK_TABLE (table), selal2, 3, 4, 8, 9);
	gtk_table_attach_defaults (GTK_TABLE (table), desal2, 4, 5, 8, 9);
	gtk_table_attach_defaults (GTK_TABLE (table), delsel, 5, 6, 8, 9);

	gtk_widget_show_all (table);

	/* create the dialog */
	dialog = gnome_dialog_new (_("Gnome Predict Map Configuration Utility"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL );
	gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (app));

	/* store global pointer */
	dp = dialog;

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table,
			    TRUE, TRUE , 0);

	if (module == NULL) {
		/* diable "OK" button while module-name is NULL */
		gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
	}

	/* connect the "module-name-changed" signal */
	gtk_signal_connect (GTK_OBJECT (entry), "changed",
			    GTK_SIGNAL_FUNC (satmap_druid_name_changed_cb),
			    dialog);

	/* connect dialog button signals */
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
				     GTK_SIGNAL_FUNC (satmap_druid_ok_cb), list2);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
				     GTK_SIGNAL_FUNC (satmap_druid_cancel_cb), NULL);
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);

	/* window icon */
	iconfile = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				G_DIR_SEPARATOR_S, "map-druid.png", NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog), iconfile);
	g_free (iconfile);
	
	gnome_dialog_run (GNOME_DIALOG (dialog));

	if (name) {
		iconfile = g_strconcat ("=", g_get_home_dir (), G_DIR_SEPARATOR_S,
					".gpredict", G_DIR_SEPARATOR_S, name,
					".satmap=", SATMAP_SAT_MAP_PATH, NULL);
		gnome_config_set_string (iconfile, mapfile);
		gnome_config_sync ();
		g_free (iconfile);
	}

	return name;
}



static gint
satmap_druid_list_compare_func (GtkCList *clist, gconstpointer s1, gconstpointer s2)
{
	/* This function is used as sorting function for the
	   list widgets.
	   s1 and s2 are pointers to GtkClistRow structures.
	*/
        GtkCListRow *row1 = (GtkCListRow *) s1;
        GtkCListRow *row2 = (GtkCListRow *) s2;

	return g_strcasecmp (GTK_CELL_TEXT (row1->cell[0])->text,
			     GTK_CELL_TEXT (row2->cell[0])->text);
}


static void
satmap_druid_select_all_cb (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on one
	   of the "Select All" buttons. The clist widget to which
	   the button belongs to is passed as the user data "clist".
	*/

	gtk_clist_select_all (GTK_CLIST (clist));
}

static void
satmap_druid_unselect_all_cb (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on one
	   of the "Unselect All" buttons. The clist widget to which
	   the button belongs to is passed as the user data "clist".
	*/

	gtk_clist_unselect_all (GTK_CLIST (clist));
}


static void
satmap_druid_add_selected_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the "Add Selected"
	   button. The selected rows from list1 are stolen out of the clist's
	   private data structure and the row contents are added to list2 if it
	   doesn't contain it already.

	   The clist->selection structure member is a pointer to a GList where
	   each node is the row number of a selected row.
	*/

	GtkCTree *ctree = (GtkCTree *) gtk_object_get_data (GTK_OBJECT (button), "list1");
	GtkCList *list2 = (GtkCList *) gtk_object_get_data (GTK_OBJECT (button), "list2");
	GList *selection = GTK_CLIST (ctree)->selection;
	GtkCTreeNode *node;
	guint catnum,row2;
	gint test;
	gchar *name,*tmp;

/*	num = g_list_length (selection);  */

	while (selection) {
		/* Get the selected CTree node */
		node = GTK_CTREE_NODE (selection->data);
		catnum = GPOINTER_TO_UINT (gtk_ctree_node_get_row_data (ctree, node));

		/* We need to fetch both text and pixmap */
		if (gtk_ctree_node_get_pixtext (ctree, node, 0, &name, 0, NULL, NULL)) {
			test = gtk_clist_find_row_from_data (list2, GUINT_TO_POINTER (catnum));
			if (test == -1) {
				row2 = gtk_clist_append (list2, &name);
				gtk_clist_set_row_data (list2, row2, GUINT_TO_POINTER (catnum));
			}
			else row2 = test;
			/* make a log entry */
			tmp = g_strdup_printf (_("%s: %s is in list2 at pos %d"),
					       __FUNCTION__, name, row2);
			satlog_log (SAT_LOG_DEBUG, tmp);
			g_free (tmp);
		}

		selection = selection->next;
	}

	/* update title with number of satellites */
	satmap_druid_update_clist_title (list2, title2[0]);		

}


static void
satmap_druid_delete_selected_cb (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the "Delete Selected"
	   button.  The selected rows from list2 are stolen out of the clist's
	   private data structure and the row is deleted from the clist.

	   The clist->selection structure member is a pointer to a GList where
	   each node is the row number of a selected row.

	   This function is a bit tricker than the previous because the
	   clist->selection list is changing as we remove rows from the clist.
	*/

	GtkCList *list2 = (GtkCList *) clist;
	GList *selection;
	guint *list,len,i;

	/* copy the selection list and sort it */
	selection = g_list_alloc ();
	selection = g_list_copy (list2->selection);
	g_list_sort (selection, satmap_druid_uint_compare_func);
	selection = g_list_first (selection);

	/* copy the selection list to a regular array */
	len = g_list_length (selection);
	list = g_malloc (len*sizeof (guint));
	for (i=0; i<len; i++) {
		list[i] = GPOINTER_TO_UINT (selection->data);
		selection = selection->next;
	}
	selection = g_list_first (selection);
	g_list_free (selection);
	
	/* remove selected rows */
	for (i=0; i<len; i++) {
		gtk_clist_remove (list2, list[i]);
	}

	g_free (list);

	/* update title with number of satellites */
	satmap_druid_update_clist_title (list2, title2[0]);
}


static gint
satmap_druid_uint_compare_func (gconstpointer p1, gconstpointer p2)
{
	/* This function is used to sort a GList with
	   unsigned integer entries.
	*/
	guint a = GPOINTER_TO_UINT (p1);
	guint b = GPOINTER_TO_UINT (p2);

	return (a==b) ? 0 : ((a>b) ? -1 : 1);
}


static void
satmap_druid_name_changed_cb (GtkEditable *entry, gpointer data)
{
	/* This function is called when the filename text in the
	   file selector widget is changed. If the entered/selected
	   name already exists, the contents will be laoded into
	   list2.
	*/
	GnomeDialog *dialog = GNOME_DIALOG (data);
	gchar *msg;

	if (name)
		g_free (name);
	name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
	msg = g_strconcat (__FUNCTION__, _(": modname: "), name, NULL);
	satlog_log (SAT_LOG_DEBUG, msg);
	g_free (msg);

	if ((name) && (g_strcasecmp (name, "")))
		gnome_dialog_set_sensitive (dialog, 0, TRUE);
	else
		gnome_dialog_set_sensitive (dialog, 0, FALSE);

	/* FIXME: Check whether there is a module with the specified name */
}


static void
satmap_druid_cancel_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   "Cancel" button.
	*/

	name = NULL;
}


static void
satmap_druid_ok_cb (GtkWidget *widget, GtkCList *list2)
{
	/* This function is called when the user clicks on the
	   "OK" button.

	*/
	gchar *configpath,*satname,**argv;
	gchar *buff1 = NULL;
	gchar *buff2 = NULL;
	guint argc,catnum,i;
	gchar *dummy = "ZZZZZZ";  /* DANGEREOUS! List is sorted! */

	if (name) {
		/* check whether we already have a list with that name */
		configpath = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
					  ".gpredict", G_DIR_SEPARATOR_S,
					  name, ".satmap", NULL);
		if (g_file_test (configpath, G_FILE_TEST_ISFILE) &&
		    GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (list2), "newmod"))) {
			gnome_app_error (GNOME_APP (app),
					 _("There is already a map with that name!"));

			/* clear file name */
			g_free (name);
			name = NULL;

			/* clear config path */
			g_free (configpath);
			return;
		}
		g_free (configpath);

		/* create vector to save */
		argc = gtk_clist_append (list2, &dummy);
		if (!argc) {

			/* no satellites in list */
			buff2 = g_strconcat (__FUNCTION__, _(": No satellites selected!"), NULL);
			satlog_log (SAT_LOG_WARNING, buff2);
			g_free (buff2);
		
		}
		else {
			gtk_clist_remove (list2, argc);
			for (i=0; i<argc; i++) {
				catnum = GPOINTER_TO_UINT (gtk_clist_get_row_data (list2, i));
				gtk_clist_get_text (list2, i, 0, &satname);
				if (buff1) {
					buff2 = g_strdup (buff1);
					g_free (buff1);
					buff1 = g_strconcat (buff2, ";", satname, NULL);
					g_free (buff2);
				}
				else {
					buff1 = g_strdup (satname);
				}
			}
			argv = g_strsplit (buff1, ";", argc);
			
			/* create a gnome-config path string */
			configpath = g_strconcat ("=", g_get_home_dir (), G_DIR_SEPARATOR_S,
						  ".gpredict", G_DIR_SEPARATOR_S, name,
						  ".satmap=", SATMAP_SAT_LIST_PATH, NULL);
			satlog_log (SAT_LOG_DEBUG, _("SATMAP-DRUID: created new config path:"));
			satlog_log (SAT_LOG_DEBUG, configpath);
			/* store vector */
			gnome_config_set_vector (configpath, argc, argv);
			gnome_config_sync ();
			/* gnome_config_set_string (configpath, buff1); */
			g_free (configpath);
			g_free (buff1);
			g_strfreev (argv);
		}
	}
	else {
		GtkWidget *err;

		err = gnome_warning_dialog (_("You did not enter a name!\n"\
					      "The map has not been created!"));
		gtk_widget_show_all (err);
	}
}


static void
satmap_druid_update_clist_title (GtkCList *maplist, const gchar *text)
{
	/* This function counts the number of rows in clist and updates
	   the column title of the list to text (number).
	*/
	guint count;
	gchar *title, *dummy[1] = { "ZZZZZZZ" }; /* The list is auto-sorted! */

	count = gtk_clist_append (maplist, dummy);
	gtk_clist_remove (maplist, count);
	title = g_strdup_printf ("%s (%d)", text, count);
	gtk_clist_set_column_title (maplist, 0, title);
	g_free (title);

	if (count > SATMAP_TOO_MANY_SATS) {
		/* User wants too many satellites on the map. Isuue a warning!
		   Yes, this could be done much simpler, but the simple solutions
		   would not allow the warning dialog to have the mouse/keyboard
		   focus, until the druid dialog is closed. Go figure...
		*/
		GtkWidget *dialog,*pixmap,*label,*hbox;
		gchar *message;

		message = g_strdup_printf (_("You have selected too many (%d)\n"
					     "satellites for this map. You should\n"
					     "not have more than %d satellites\n"
					     "on the same map!"), count,
					   SATMAP_TOO_MANY_SATS);
		label = gtk_label_new (message);
		gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_FILL);
		g_free (message);

		/* pixmap */
		message = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				       G_DIR_SEPARATOR_S, "crash.png", NULL);
		pixmap = gnome_pixmap_new_from_file (message);
		g_free (message);

		/* horibox */
		hbox = gtk_hbox_new (FALSE, 20);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), pixmap);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
		gtk_widget_show_all (hbox);


		dialog = gnome_dialog_new (_("Warning"), GNOME_STOCK_BUTTON_OK, NULL);
		gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (dp));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    hbox, TRUE, TRUE, 0);
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	       
	}
}



static void
satmap_map_selector_cb (GtkButton *button, gpointer defmap)
{
	/* This function is called when the user clicks on the default map
	   selector button. It pops up a dialog showing a list of icons in
	   the standard gpredict map directory and in $HOME/.gpredict/maps/
	*/
	GtkWidget *dialog,*selector;
	gchar *dp1,*dp2;
	gint yesno;

	/* create directory entries where maps may be located */
	dp1 = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps", NULL);
	dp2 = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
			   G_DIR_SEPARATOR_S, "maps", NULL);

	/* Create the selector */
	selector = gnome_icon_selection_new ();
	gnome_icon_selection_add_directory (GNOME_ICON_SELECTION (selector), dp1);
	/* The user might not have installed extra maps */
	if (g_file_test (dp2, G_FILE_TEST_ISDIR)) {
		gnome_icon_selection_add_directory (GNOME_ICON_SELECTION (selector), dp2);
	}
	gnome_icon_selection_show_icons (GNOME_ICON_SELECTION (selector));

	/* Create dialog box */
	dialog = gnome_dialog_new (_("Select a new map"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	/* we need to keep the dialog so that we can access the file name of the
	   selection after the dialog is closed.
	*/
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), selector, TRUE, TRUE, 0);
	gtk_widget_show_all (dialog);

	/* run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		if (gnome_icon_selection_get_icon (GNOME_ICON_SELECTION (selector), TRUE)) {
			/* we have a selection */
			g_free (mapfile);
			mapfile = g_strdup (gnome_icon_selection_get_icon (GNOME_ICON_SELECTION (selector), TRUE));

			/* update the preview widget */
			gnome_pixmap_load_file_at_size (GNOME_PIXMAP (defmap), mapfile,
							PREVIEW_SIZE_X, PREVIEW_SIZE_Y);
		}
	}
	
	/* we are done with the widgets in the dialog. close it */
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), FALSE);
	gnome_dialog_close (GNOME_DIALOG (dialog));

	g_free (dp1);
	g_free (dp2);
}


static void
satmap_druid_read_data_file     (GtkCList *clist, const gchar *module)
{
	/* This function read the contents of module.satlist into
	   the CList. It attaches the catalog number of the satellite
	   to each row.
	*/
	gchar *configpath,*buff,**argv,**text;
	gint argc,i,index;
	sat_t *sat;

	g_return_if_fail (clist != NULL);
	g_return_if_fail (module != NULL);

	configpath = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
				  ".gpredict", G_DIR_SEPARATOR_S,
				  module, ".satmap", NULL);
	if (!g_file_test (configpath, G_FILE_TEST_ISFILE)) {
		/* log entry */
		buff = g_strconcat (__FUNCTION__, ": ", _("Could not find data file: "),
				    configpath, NULL);
		satlog_log (SAT_LOG_CRITICAL, buff);
		g_free (buff);

		/* Error dialog */
		buff = g_strdup_printf (_("Could not find file %s"), configpath);
		gnome_app_error (GNOME_APP (app), buff);
		g_free (buff);
		g_free (configpath);

		/* bail out */
		return;
	}
	g_free (configpath);

	/* create a true config path */
	configpath = g_strconcat ("=", g_get_home_dir (), G_DIR_SEPARATOR_S,
				  ".gpredict", G_DIR_SEPARATOR_S,
				  module, ".satmap=", SATMAP_SAT_LIST_PATH, NULL);
	/* get satellites */
	gnome_config_get_vector (configpath, &argc, &argv);
	g_free (configpath);

	/* Empty list? */
	if (!g_strcasecmp (argv[0], "")) {
		return;
	}

	for (i=0; i<argc; i++) {
		/* Get index of satellite and add to list if the
		   satellite really exists in our database
		*/
		index = satdata_get_sat_index (0, argv[i]);
		if (index != -1) {
			sat = satdata_get_sat (index);
			text = g_strsplit (argv[i], "<!SEP>", 1);
			index = gtk_clist_append (clist, text);
			gtk_clist_set_row_data (clist, index, GUINT_TO_POINTER (sat->catnum));
			g_strfreev (text);
		}
		else {
			buff = g_strdup_printf (_("%s: Satellite %s not in database!"),
						__FUNCTION__, argv[i]);
			satlog_log (SAT_LOG_WARNING, buff);
			g_free (buff);
		}
	}
}
