/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satprint.c: Printing functions.

  Copyright (C)  2001-2005  Ran Hadary & Alexandru Csete.

  Authors:   Rand Hadary <hadary@users.sourceforge.net>
             Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include <math.h>
#include "satdata.h"
#include "satlog.h"
#include "qth.h"
#include "util.h"
#include "satprint.h"



#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#include <libgnomeprint/gnome-printer.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-meta.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-print-pixbuf.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gnome-printer-dialog.h>


gdouble paper_width, paper_height;
gint pos_y, gHighlight;

extern qth_struc qth; /* qth.c */

/* Private Functions prototype */
static void satp_calc_paper_size  (gchar *);
static void satp_highlight_line   (GnomePrintContext *, gint, gint, gint);
static void satp_page_header      (GnomePrintContext *);
static void satp_page_footer      (GnomePrintContext *, guint, guint);
static void satp_print_qth        (GnomePrintContext *);
static void satp_sat_data_print   (GnomePrintContext *, sat_t *);
static gint satp_aosline_print    (GnomePrintContext *, gchar *);
static gint satp_sat_textline     (GnomePrintContext *, gchar **);
static gint satp_satline_print    (GnomePrintContext *, guint);
static void satp_satlist_header   (GnomePrintContext *);
static void satp_satlist_print    (GnomePrintContext *);
static void satp_aosdet_header    (GnomePrintContext *, sat_t *, guint);
static gint satp_aosdetline_print (GnomePrintContext * ,gchar *);


/* Calculate the paper size */
static void satp_calc_paper_size (gchar *PaperSize) 
{
	const GnomePaper *gp;

	/* Claculate the paper size */
	gp = gnome_paper_with_name (PaperSize);
	paper_height = gnome_paper_psheight (gp);
	paper_width = gnome_paper_pswidth (gp);

}


static void 
satp_highlight_line (GnomePrintContext *pc, gint pos_lh , gint pos_rh, gint height)
{

	/* Set the background color to grey95                             */
	/* Draw a box from top right-hand to button left-hand             */
	/* and fill with the desired color.                               */

	gnome_print_gsave (pc);
	gnome_print_setrgbcolor (pc, 0.952, 0.952, 0.952);
	gnome_print_moveto (pc, pos_lh, pos_y+height-1);
	gnome_print_lineto (pc, pos_rh, pos_y+height-1);
	gnome_print_lineto (pc, pos_rh, pos_y-1);
	gnome_print_lineto (pc, pos_lh, pos_y-1);
	gnome_print_closepath (pc);
	gnome_print_fill (pc);
	gnome_print_grestore (pc);
}


static void
satp_page_header (GnomePrintContext *pc) 
{

	/* Standard page header for Gnome-Predict                        */
	/* Include the icon on the left hand side                        */
	/* Include package name and number from the header files         */

	GnomeFont* font;
	gdouble text_width;
	gchar* buf;
	gchar* path;
	GdkPixbuf *pb = NULL;

	/* Load application icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
			    G_DIR_SEPARATOR_S, "gpredict-icon.png", NULL);
	pb = gdk_pixbuf_new_from_file (path);
	if (pb == NULL)
		satlog_log (SAT_LOG_CRITICAL, _("SATPRINT: Could not load icon"));
	else {
		gnome_print_gsave (pc);
		gnome_print_translate (pc, 60, (paper_height - 60));
		gnome_print_scale (pc, gdk_pixbuf_get_width(pb), gdk_pixbuf_get_height (pb));
		gnome_print_moveto (pc, 0, 0);
		gnome_print_pixbuf (pc, pb);
		gnome_print_grestore (pc);
	}
	gdk_pixbuf_unref (pb);
	g_free (path);

	/* Set the font closest to Times New Roman */
	font = gnome_font_new_closest ("Times", GNOME_FONT_BOLD, 1, 24);
	if (font == NULL ) {
		satlog_log (SAT_LOG_CRITICAL, _("SATPRINT: Font TIMES not found"));
		return; 
	}

	/* Print package name and version */
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);
	pos_y -= 30;
	buf = g_strconcat (_("Gnome Predict"), "  ", VERSION, NULL); 
	text_width = gnome_font_get_width_string (font, buf);
	gnome_print_moveto (pc, ((paper_width - text_width)/2), pos_y);
	gnome_print_show (pc, buf);
	g_free (buf);
	gnome_print_grestore (pc);
}


static void
satp_page_footer (GnomePrintContext *pc, guint pg, guint pgs)
{
	/* Print page footer                                 */
	/* Print the "Page # of ##" at the buttom right side */
	GnomeFont* font;
	gchar* buf;

	gnome_print_gsave (pc);
	font = gnome_font_new_closest ("Times", GNOME_FONT_BOLD, 1, 10);
	gnome_print_setfont (pc, font);
	buf = g_strdup_printf (_("Page %2.0i of %2.0i"), pg, pgs);

	gnome_print_moveto (pc, 40, 40);
	gnome_print_lineto (pc, paper_width - 95, 40);
	gnome_print_stroke (pc);
	gnome_print_moveto (pc, 450, 25);
	gnome_print_show (pc, buf);
	g_free (buf);
	gtk_object_unref (GTK_OBJECT (font));
	gnome_print_grestore (pc);            

}


static void
satp_print_qth (GnomePrintContext *pc)
{
	GnomeFont* font;
	gchar* buf;

	font = gnome_font_new ("Courier", 14);
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);

	/* First line - includes the name and the description        */
	buf = g_strdup_printf (_("Station: %s (%s)"), qth.name, qth.desc);
	pos_y -= 22;
	gnome_print_moveto (pc, 40, pos_y);
	gnome_print_show (pc, buf);
	g_free (buf);

	/* Second line - the location as registered                  */
	/**** FIXME: string not translated (bug #522419) */
	buf = g_strdup_printf (_("Location: %s %.4f%c %.4f%c  Alt: %dM  "),
			      qth.loc, fabs (qth.lat), (qth.lat < 0) ? 'S' : 'N',
			      fabs (qth.lon), (qth.lon < 0) ? 'E' : 'W', qth.alt );
	pos_y -= 14;
	gnome_print_moveto (pc, 40, pos_y);
	gnome_print_show (pc, buf);
	g_free (buf);

	/* Third line shows the effctive date of printing            */
	buf = g_strdup_printf (_("Effective: %s"), dnum2lstr (CurrentDaynum ()));
	pos_y -= 14;
	gnome_print_moveto (pc, 40, pos_y);
	gnome_print_show (pc, buf);
	g_free (buf);

	/* Draw a break line                                         */
	pos_y -= 10;
	gnome_print_moveto (pc, 40, pos_y);
	gnome_print_lineto (pc, paper_width - 95, pos_y);
	gnome_print_stroke (pc);
	gnome_print_grestore (pc);

}


static void
satp_sat_data_print (GnomePrintContext *pc, sat_t *sat)
{
	/* Print a single SAT's AOS/LOS data header                      */
	/* Print the satellites name and the AOS/LOS header              */

	GnomeFont *font;
	gchar *textbuf;

	font = gnome_font_new_closest ("Times", GNOME_FONT_BOLD, 1, 14);
  
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);

	pos_y -= 20;
	textbuf = g_strdup_printf (_("Upcomming Passes for %s (%d)"), sat->name, sat->catnum);
	gnome_print_moveto (pc, 150, pos_y);
	gnome_print_show (pc, textbuf);
	g_free (textbuf);

	pos_y -= 18;
	gnome_print_moveto (pc, 120, pos_y);
	gnome_print_show (pc, _("AOS"));
	gnome_print_moveto (pc, 320, pos_y);
	gnome_print_show (pc, _("LOS"));

	gnome_print_grestore (pc);

	gtk_object_unref (GTK_OBJECT (font));
}


static gint
satp_aosline_print (GnomePrintContext *pc,gchar *linebuf)
{
	/* Print the AOS/LOS data                                     */
	/* The function recieves an entry from the global list        */
	/* splits into two elemets and print                          */
	gchar **vbuff;

	vbuff = g_strsplit (linebuf,";",2);
       
	if (gHighlight == 1) {
		satp_highlight_line (pc, 40, 500, 12);
		gHighlight = 0;
	} else { gHighlight = 1;}

	gnome_print_gsave (pc);
       
	/* 1st data element - the AOS */
	gnome_print_moveto (pc, 60, pos_y);
	gnome_print_show (pc, vbuff[0]);

	/* 2nd data element - the LOS */
	gnome_print_moveto (pc, 260, pos_y);
	gnome_print_show (pc, vbuff[1]);

	g_strfreev (vbuff);
	gnome_print_grestore (pc); 

	return 0;
}


static gint
satp_sat_textline (GnomePrintContext *pc, gchar **textbuf)
{

	/* Print a single line of Satellite data                            */

	if (gHighlight == 1) {
		satp_highlight_line (pc, 30, 550, 12);
		gHighlight = 0;
	} else { gHighlight = 1;}

	gnome_print_gsave (pc);

	/* 1st data element - the satellite name                            */
	gnome_print_moveto (pc, 30, pos_y);
	gnome_print_show (pc, textbuf[0]);

	/* 2nd data element - the Azimuth to the satellite                  */
	gnome_print_moveto (pc, 130, pos_y);
	gnome_print_show (pc, textbuf[1]);

	/* 3rd data element - the elevation                                 */
	gnome_print_moveto (pc, 200, pos_y);
	gnome_print_show (pc, textbuf[2]);

	/* 4th data element - the AOS or LOS time, _A_OS or _L_OS           */
	gnome_print_moveto (pc, 270, pos_y);
	gnome_print_show (pc, textbuf[3]);

	/* The satellite height is the 5th element                          */
	gnome_print_moveto (pc, 420, pos_y);
	gnome_print_show (pc, textbuf[4]);

	gnome_print_grestore (pc);

	return 0;
}


static gint
satp_satline_print (GnomePrintContext *pc, guint index )
{

	/* Print all list from the SATLIST on display                       */
	/* Prepare the data for the actual line printing function           */
 
	gchar *buf,**text,*aos_los;
	sat_t *sat = NULL;

	sat = satdata_get_sat (index);

	/**** FIXME: val returned by g_strdup_printf should be freed! */
	aos_los = (sat->el > 0.1 ) ? g_strdup_printf ("L %s", dnum2str (sat->los)) 
		: g_strdup_printf ("A %s", dnum2str (sat->aos));
	switch (sat->status) {
	case SAT_STATUS_DECAYED:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;-- DECAYED --;%6.0f"),
				       sat->name, sat->az, sat->el, sat->alt);
		text = g_strsplit (buf, ";", 5);
		g_free (buf);
		break;
	case SAT_STATUS_GEOSTAT:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;GEOSTATIONARY;%6.0f"),
				       sat->name, sat->az, sat->el, sat->alt);
		text = g_strsplit (buf, ";", 5);
		g_free (buf);
		break;
	case SAT_STATUS_NOAOS:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;--- NEVER ---;%6.0f"),
				       sat->name, sat->az, sat->el, sat->alt);
		text = g_strsplit (buf, ";", 5);
		g_free (buf);
		break;
	default:
		buf = g_strdup_printf ("%s;%7.2f;%7.2f;%s ;%6.0f",
				       sat->name, sat->az, sat->el, aos_los, sat->alt);
		text = g_strsplit (buf, ";", 5);
		g_free (buf);
	}
	pos_y -= 12;
	satp_sat_textline (pc, text);
	g_strfreev (text);
	g_free (aos_los);

	return 0;
}


static void
satp_satlist_header (GnomePrintContext *pc)
{
	/* Header line for sat list                                          */
	GnomeFont *font;
	gchar *buf, **text;

	font = gnome_font_new_closest ("Times", GNOME_FONT_BOLD, 1, 13);
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);
	buf = g_strdup_printf (_("Satellite; Az;  El;  Next AOS/LOS;  Alt"));
	text = g_strsplit (buf, ";", 5);
	pos_y -= 13;
	satp_sat_textline (pc, text);
	gnome_print_grestore (pc);

	gtk_object_unref (GTK_OBJECT (font));
	g_free (buf);
	g_strfreev (text);
}


static void
satp_satlist_print (GnomePrintContext *pc)
{
	/* Print the list of prediction for all the displayed list           */
	guint num_of_sats;
	guint satindex, startindex, endindex;
	GnomeFont* font;
	guint page, pages;

	gnome_print_beginpage (pc, _("Gnome-Predict Sat List"));
	/*        Here will come the page components                         */
	/*         o Header (the usual bla bla....)                          */
	/*         o QTH Information                                         */
	/*         o List Header                                             */
	/*         o SAT List                                                */
	/*         o at the end of every page the copyright and date/time    */

	num_of_sats = satdata_count_sat ();
	font = gnome_font_new ("Courier", 12);

	pages = num_of_sats / PG_DATA_LINES;
	pages++;
	startindex = 0;
	endindex = (num_of_sats > PG_DATA_LINES) ? PG_DATA_LINES : num_of_sats;
    
	for (page = 1; page <= pages; page++) {
		gHighlight = 0;
		pos_y = paper_height;
		satp_page_header (pc);
		satp_print_qth (pc);
		satp_satlist_header (pc);
		gnome_print_setfont (pc, font);
		for (satindex = startindex; satindex < endindex ; satindex++) {
			satp_satline_print (pc, satindex);
		}
		satp_page_footer (pc, page, pages);
		gnome_print_showpage (pc);
		startindex = endindex;
		endindex += PG_DATA_LINES;
		if (endindex > num_of_sats) {
			endindex = num_of_sats;
		}
	}
    
	gnome_print_context_close (pc);
	return;
}


static void satp_aosdet_header (GnomePrintContext *pc, sat_t *sat, guint orbit)
{
	/* Header line for sat list */
	GnomeFont *font;
	gchar *textbuf;

	font = gnome_font_new_closest ("Times", GNOME_FONT_BOLD, 1, 13);
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);

	pos_y -= 15;
	textbuf = g_strdup_printf (_("Upcoming Pass for %s (%d),  orbit %d"), sat->name, sat->catnum, orbit);
	gnome_print_moveto (pc, 140, pos_y);
	gnome_print_show (pc, textbuf);
	g_free (textbuf);

	pos_y -= 20;
	gnome_print_moveto (pc, 80, pos_y);
	gnome_print_show (pc, _("Time"));
  
	gnome_print_moveto (pc, 210, pos_y);
	gnome_print_show (pc, _("Az"));

	gnome_print_moveto (pc, 270, pos_y);
	gnome_print_show (pc, _("El"));


	gnome_print_moveto (pc, 315, pos_y);
	gnome_print_show (pc, _("Lat"));

	gnome_print_moveto (pc, 390, pos_y);
	gnome_print_show (pc, _("Lon"));
  
	gnome_print_moveto (pc, 465, pos_y);
	gnome_print_show (pc, _("Alt"));

	gnome_print_grestore (pc);

	gtk_object_unref (GTK_OBJECT (font));
	return;
}


static gint
satp_aosdetline_print (GnomePrintContext *pc, gchar *aosdetline)
{
	gchar **textbuf;

	if (gHighlight == 1) {
		satp_highlight_line (pc, 60, 500, 12);
		gHighlight = 0;
	} else { gHighlight = 1;}

	textbuf = g_strsplit (aosdetline, ";", 9);
	gnome_print_gsave (pc);

	/* 1st data element - the time                             */
	gnome_print_moveto (pc, 40, pos_y);
	gnome_print_show (pc, textbuf[0]);

	/* 2nd data element - the Azimuth to the satellite              */
	gnome_print_moveto (pc, 190, pos_y);
	gnome_print_show (pc, textbuf[1]);

	/* 3rd data element - the elevation                             */
	gnome_print_moveto (pc, 240, pos_y);
	gnome_print_show (pc, textbuf[2]);

	/* 4th data element - the Latitude of the Satellite             */
	gnome_print_moveto (pc, 290, pos_y);
	gnome_print_show (pc, textbuf[3]);

	/* 5th data element - the Longitude of the Satellite            */
	gnome_print_moveto (pc, 370, pos_y);
	gnome_print_show (pc, textbuf[4]);
  
	/* The satellite height is the 6th element                      */
	gnome_print_moveto (pc, 450, pos_y);
	gnome_print_show (pc, textbuf[5]);

	gnome_print_grestore (pc);
	g_strfreev (textbuf);

	return 0;
}


/* Callback function activated when user presses the print button */
void
sat_print_current (void)
{
	GnomePrinter *printer;
	GnomePrintContext *pc;
	gchar* UsePaper = _("A4");

	printer = gnome_printer_dialog_new_modal ();
	
	if (!printer)
		return;

	satp_calc_paper_size (UsePaper);
	pos_y = paper_height;        /* upper print margin               */ 

	pc = gnome_print_context_new_with_paper_size (printer, UsePaper);
	satp_satlist_print (pc);

	gtk_object_unref (GTK_OBJECT (printer));
	gtk_object_unref (GTK_OBJECT (pc));
}


/* Callback function when user prints the AOS/LOS prediction for a satellite */
void 
sat_print_sat_aos (sat_t *sat, gchar **aoslist, guint size)
{
	/* Print the detailed AOS List                                   */
	GnomePrinter *printer;
	GnomePrintContext *pc;
	GnomeFont *font;
	gchar* UsePaper = _("A4");
	gint index;

	printer = gnome_printer_dialog_new_modal ();
	
	if (!printer)
		return;

	satp_calc_paper_size (UsePaper);
	pos_y = paper_height - 20;         /* upper print margin */ 

	pc = gnome_print_context_new_with_paper_size (printer, UsePaper);
	gnome_print_beginpage (pc, _("Gnome-Predict Sat Prediction"));
	satp_page_header (pc);
	satp_print_qth (pc);
	satp_sat_data_print (pc, sat);

	/* Just initialize the Highlight controller */ 
	gHighlight = 0;

	gnome_print_gsave (pc);
	font = gnome_font_new ("Courier", 12);
	gnome_print_setfont (pc, font);
	for (index = 0; (index < (int) size) && (pos_y >= 30); index++) {
		pos_y -= 12;
		satp_aosline_print (pc, aoslist[index]);
	}
	gnome_print_grestore (pc);
	satp_page_footer (pc, 1, 1);
	gnome_print_showpage (pc);
	gnome_print_context_close (pc);
        
	gtk_object_unref (GTK_OBJECT (printer));
	gtk_object_unref (GTK_OBJECT (pc));
} 


void
sat_print_aos_det (sat_t *sat, gchar **aosdet, guint size, guint orbit)
{
	/* Detailed AOS prediction */

	GnomePrinter *printer;
	GnomePrintContext *pc;
	GnomeFont *font;
	gchar* UsePaper = _("A4");
	guint lindex, sindex, eindex, page, pages;
	gchar* buf;

	printer = gnome_printer_dialog_new_modal ();
	
	if (!printer)
		return;

	satp_calc_paper_size (UsePaper);

	font = gnome_font_new ("Courier", 12);
	pc = gnome_print_context_new_with_paper_size (printer, UsePaper);

	/* get number of datalines and calculate required pages */
	pages = size / PG_DATA_LINES;
	pages++;
	sindex = 1;
	eindex = (size > PG_DATA_LINES) ? PG_DATA_LINES : size -1;

	buf = g_strdup_printf (_("Size: %d, Pages %d, eindex %d"), size, pages, eindex);
	satlog_log (SAT_LOG_DEBUG, buf);
	g_free (buf);

	for (page = 1; page <= pages; page++) {
		pos_y = paper_height - 20;
		/* this will add page numbers as well */
		gnome_print_beginpage (pc, _("Gnome-Predict AOS Details"));
		satp_page_header (pc);
		satp_print_qth (pc);
		satp_aosdet_header (pc, sat, orbit);
		gnome_print_setfont (pc, font);
		gHighlight = 0;
		for (lindex = sindex; lindex <= eindex ; lindex++) {
			pos_y -= 12;
			satp_aosdetline_print (pc, aosdet[lindex]);
		}
		satp_page_footer (pc, page, pages);
		gnome_print_showpage (pc);
		sindex = eindex;
		eindex += PG_DATA_LINES;
		if (eindex > (size - 1)) {
			eindex = (size - 1);
		}
	}
    
	gnome_print_context_close (pc);
        
	gtk_object_unref (GTK_OBJECT (printer));
	gtk_object_unref (GTK_OBJECT (pc));
}
