/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  moon.c: Widgets and functions to display the position of the moon. 

  Copyright (C)  2001-2005  Alexandru Csete & John A. Magliacane.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>,
            John A. Magliacane.

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <math.h>
#include "defaults.h"
#include "satlog.h"
#include "util.h"
#include "qth.h"
#include "moon.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif


#define MOON_X_PAD 3
#define MOON_Y_PAD 0


extern qth_struc qth;

static GtkWidget *moonframe,*moonbox,*moonaz,*moonel;
static GtkWidget *riseinfo,*risedata;
static gfloat moon_az,moon_el;
static gchar *azs,*els;

typedef enum {
	MOON_STATE_NONE = 0,
	MOON_STATE_UP,
	MOON_STATE_DOWN
} moon_state_t;

/* used to indicate whether sun is up or not */
static moon_state_t state = MOON_STATE_NONE;

static void FindMoon    (gdouble);
static void GetRiseTime (gdouble *);
static void GetSetTime  (gdouble *);


GtkWidget *moon_create ()
{
	GtkWidget *misc;

	azs = g_malloc (7);
	els = g_malloc (7);

	moonaz = gtk_label_new ("000.00");
	moonel = gtk_label_new ("+00.00");

	/* the table to contain the labels */
	moonbox = gtk_table_new (3, 2, TRUE);
	misc = gtk_label_new (_("Az:"));
	gtk_misc_set_alignment (GTK_MISC (misc), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox),
			  misc, 0, 1, 0, 1,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD);

	misc = gtk_label_new (_("El:"));
	gtk_misc_set_alignment (GTK_MISC (misc), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox),
			  misc, 0, 1, 1, 2,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD);

	gtk_misc_set_alignment (GTK_MISC (moonaz), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox),
			  moonaz, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD );
	gtk_misc_set_alignment (GTK_MISC (moonel), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox),
			  moonel, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD );

	/* rise/set */
	riseinfo = gtk_label_new ("----:");
	gtk_misc_set_alignment (GTK_MISC (riseinfo), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox), riseinfo,
			  0, 1, 2, 3, 
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD );

	risedata = gtk_label_new ("--:--");
	gtk_misc_set_alignment (GTK_MISC (risedata), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (moonbox), risedata,
			  1, 2, 2, 3, 
			  GTK_FILL | GTK_EXPAND,
			  GTK_FILL | GTK_EXPAND,
			  MOON_X_PAD, MOON_Y_PAD );

	/* add a nice frame */
	moonframe = gtk_frame_new (_(" Moon "));
	gtk_frame_set_label_align (GTK_FRAME (moonframe), 0.5, 0.5 );
	gtk_frame_set_shadow_type (GTK_FRAME (moonframe),
				   GTK_SHADOW_ETCHED_OUT );
	gtk_container_add (GTK_CONTAINER (moonframe),
			   GTK_WIDGET (moonbox) );
	return moonframe;
}

void moon_update ()
{
	gdouble  daynum;
	gchar   *tstr;

	daynum = CurrentDaynum ();

	/* calculate coordinates */
	FindMoon (daynum);

	/* update widgets */
	g_snprintf (azs, 7, "%6.2f", moon_az);
	g_snprintf (els, 7, (moon_el > 0) ? "+%5.2f" : "%6.2f", moon_el);
	gtk_label_set_text (GTK_LABEL (moonaz), azs);
	gtk_label_set_text (GTK_LABEL (moonel), els);

	/* check whether state has changed */
	switch (state) {

		/* no state; first call */
	case MOON_STATE_NONE:
		if (moon_el >= 0.0) {
			state = MOON_STATE_UP;
			GetSetTime (&daynum);
			tstr = dnum2fstr (daynum, "%H:%M");
			gtk_label_set_text (GTK_LABEL (riseinfo), _("Set:"));
			gtk_label_set_text (GTK_LABEL (risedata), tstr);
/* 			g_free (tstr); */
		}
		else {
			state = MOON_STATE_DOWN;
			GetRiseTime (&daynum);
			tstr = dnum2fstr (daynum, "%H:%M");
			gtk_label_set_text (GTK_LABEL (riseinfo), _("Rise:"));
			gtk_label_set_text (GTK_LABEL (risedata), tstr);
/* 			g_free (tstr); */
		}
		break;

		/* moon is up; if elevation is < 0 then we should
		   change the state to MOON_STATE_DOWN and calculate
		   next rise time.
		*/
	case MOON_STATE_UP:
		if (moon_el < 0.0) {
			state = MOON_STATE_DOWN;
			GetRiseTime (&daynum);
			tstr = dnum2fstr (daynum, "%H:%M");
			gtk_label_set_text (GTK_LABEL (riseinfo), _("Rise:"));
			gtk_label_set_text (GTK_LABEL (risedata), tstr);
/* 			g_free (tstr); */
		}
		break;

		/* moon is down; if elevation is >= 0 then we should
		   change the state to MOON_STATE_UP and calculate
		   next set time.
		*/
	case MOON_STATE_DOWN:
		if (moon_el >= 0.0) {
			state = MOON_STATE_UP;
			GetSetTime (&daynum);
			tstr = dnum2fstr (daynum, "%H:%M");
			gtk_label_set_text (GTK_LABEL (riseinfo), _("Set:"));
			gtk_label_set_text (GTK_LABEL (risedata), tstr);
/* 			g_free (tstr); */
		}
		break;

	default:
		break;
	}
}


static void
FindMoon (gdouble daynum)
{
/* This function calculates the position of the moon
   at a given time daynum. The code was shamelessly
   stolen from John Magliacane's excellent satellite
   tracking program Predict www.qsl.net/kd2bd/predict.html
*/
	gdouble t1, t2, t3, ew, en, e, n, si, co, ra, gh,
		lh, gm, de, se, gs, dc, el, az, mm, yr, ro;

	gm=24.0*(daynum-floor(daynum));
	ew=FixAngle(1.134193+daynum*0.229971506);
	mm=FixAngle(1.319238+daynum*0.228027135);
	t1=FixAngle(6.217512+daynum*0.01720196977);
	t2=2.0*FixAngle(2.550677+daynum*0.212768711);
	t3=FixAngle(4.7652214+daynum*0.230895723);
	ew=ew+0.01148*sin(t2)+0.10976*sin(mm);
	ew=ew-0.022235*sin(mm-t2)-0.003246*sin(t1);
	ew=ew+0.003735*sin(2.0*mm)-0.0019897*sin(2.0*t3);
	ew=ew-0.0010297*sin(2.0*mm-t2)-0.0009948*sin(mm+t1-t2);
	en=t3+0.0115070*sin(t2)+0.10873924*sin(mm);
	en=en-0.0222006*sin(mm-t2);
	en=0.0897797*sin(en)-0.002548*sin(t3-t2);

	si=C1*sin(en)+S1*cos(en)*sin(ew);
	co=sqrt(1.0-(si*si));
	dc=atan2(si,co);
	si=sin(ew)*C1-tan(en)*S1;
	co=cos(ew);
	ra=atan2(si,co);

	if (ra<0.0)
		ra+=TP;

	t1=(gdouble)(glong)(daynum-39410.0);
	t2=floor((t1+32044.75)/36524.25);
	t2+=t1-t2/4.0+1486.0;
	yr=2084.0+floor((t2-122.1)/365.25);

	t1=yr-1.0;
	de=floor(365.25*(t1-1980.0))-floor(t1/100.0)+floor(t1/400.0)+381.0;
	t1=(de+29218.5)/36525.0;
	t1=6.6460656+t1*(2400.051262+t1*2.581e-5);
	se=t1-24.0*(yr-1900.0);
	t1=(se+0.0657098*(daynum-de)+gm*1.002738)/24.0;
	gs=24.0*(t1-floor(t1));

	t1=gs/24.0-ra/TP;
	gh=TP*(t1-floor(t1));

	n=qth.lat*DEG2RAD;    /* North latitude of tracking station */
	e=-qth.lon*DEG2RAD;  /* East longitude of tracking station */

	lh=gh+e;

	si=cos(lh)*cos(dc)*cos(n)+sin(dc)*sin(n);
	co=sqrt(1.0-(si*si));
	el=atan2(si,co);

	si=-sin(lh)*cos(dc)*cos(n);
	co=sin(dc)-sin(n)*sin(el);
	az=atan2(si,co);

	if (az<0.0)
		az+=TP;

	ro=0.996986/(1.0+0.0549*cos(mm+0.10976*sin(mm)));
	el=el-0.0166*cos(el)/ro;

	moon_az=az/DEG2RAD;
	moon_el=el/DEG2RAD;
}


/* Starting from daynum, find the time when the elevation
   will become >= 0. Note that there is a conflict in using
   the global variables moon_az and moon_el, but it shouldn't
   give any trouble since FindSun is always called again by
   the the sun_update function before using the variables.
*/
static void
GetRiseTime (gdouble *daynum)
{
	for ( ; moon_el < 0.00; *daynum += 3.0e-4) {
		FindMoon (*daynum);
	}
}


/* Starting from daynum, find the time when the elevation
   will become < 0. Note that there is a conflict in using
   the global variables moon_az and moon_el, but it shouldn't
   give any trouble since FindSun is always called again by
   the the sun_update function before using the variables.
*/
static void
GetSetTime  (gdouble *daynum)
{
	for ( ; moon_el >= 0.0; *daynum += 3.0e-4) {
		FindMoon (*daynum);
	}
}
