/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/*
    Gpredict: Real-time satellite tracking and orbit prediction program

    Copyright (C)  2001-2006  Alexandru Csete, OZ9AEC.

    Authors: Alexandru Csete <csete@users.sourceforge.net>

    Comments, questions and bugreports should be submitted via
    http://sourceforge.net/projects/groundstation/
    More details can be found at the project home page:

            http://groundstation.sourceforge.net/
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, visit http://www.fsf.org/
*/
/** \defgroup menu The menubar
 *  \ingroup gui
 *
 */
/** \defgroup menuif Interface
 *  \ingroup menu
 *
 */
/** \defgroup menupriv Private
 *  \ingroup menu
 */
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "sat-pref.h"
#include "mod-cfg.h"
#include "mod-mgr.h"
#include "sat-log.h"
#include "about.h"
#include "gtk-sat-module.h"
#include "gpredict-help.h"
#include "menubar.h"
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

extern GtkWidget *app;


/* private function prototypes */
static void   menubar_new_mod_cb   (GtkWidget *widget, gpointer data);
static void   menubar_open_mod_cb  (GtkWidget *widget, gpointer data);
static void   menubar_app_exit_cb  (GtkWidget *widget, gpointer data);
static void   menubar_freq_edit_cb (GtkWidget *widget, gpointer data);
static void   menubar_pref_cb      (GtkWidget *widget, gpointer data);
static void   menubar_tle_cb       (GtkWidget *widget, gpointer data);
static void   menubar_list_cb      (GtkWidget *widget, gpointer data);
static void   menubar_plot_cb      (GtkWidget *widget, gpointer data);
static void   menubar_glance_cb    (GtkWidget *widget, gpointer data);
static void   menubar_window_cb    (GtkWidget *widget, gpointer data);
static void   menubar_predict_cb   (GtkWidget *widget, gpointer data);
static void   menubar_lab_cb       (GtkWidget *widget, gpointer data);
static void   menubar_getting_started_cb (GtkWidget *widget, gpointer data);
static void   menubar_help_cb      (GtkWidget *widget, gpointer data);
static void   menubar_about_cb     (GtkWidget *widget, gpointer data);
static gchar *select_module        (void);


/** \brief Regular menu items.
 *  \ingroup menupriv
 */
static GtkActionEntry entries[] = {
	{ "FileMenu", NULL, N_("_File") },
	{ "EditMenu", NULL, N_("_Edit") },
	{ "ToolsMenu", NULL, N_("_Tools") },
	{ "HelpMenu", NULL, N_("_Help") },

	/* File menu */
	{ "New", GTK_STOCK_NEW, N_("_New module"), "<control>N",
	  N_("Create a new module"), G_CALLBACK (menubar_new_mod_cb) },
	{ "Open", GTK_STOCK_OPEN, N_("_Open module"), "<control>O",
	  N_("Open an existing module"), G_CALLBACK (menubar_open_mod_cb) },
	{ "Exit", GTK_STOCK_QUIT, N_("E_xit"), "<control>Q",
	  N_("Exit the program"), G_CALLBACK (menubar_app_exit_cb) },

	/* Edit menu */
	{ "Freq", NULL, N_("_Transponders"), NULL,
	  N_("Edit satellite transponder frequencies"),
	  G_CALLBACK (menubar_freq_edit_cb)},
	{ "Pref", GTK_STOCK_PREFERENCES, N_("_Preferences"), NULL,
	  N_("Edit user preferences"), G_CALLBACK (menubar_pref_cb)},

	/* Tools menu */
	{ "Tle", GTK_STOCK_REFRESH, N_("Update TLE"), "<control>T",
	  N_("Update Keplerian elements"), G_CALLBACK (menubar_tle_cb)},
	{ "Glance", GTK_STOCK_PASTE, N_("Sky at a glance"), "<control>G",
	  N_("Show quick overview of the sky now and in the near future"),
	  G_CALLBACK (menubar_glance_cb)},
	{ "Window", NULL, N_("Window"), "<control>W",
	  N_("Predict windows between two observers"), G_CALLBACK (menubar_window_cb)},
	{ "Predict", GTK_STOCK_DND_MULTIPLE, N_("Advanced Predict"), NULL,
	  N_("Opean advanced pass predictor"), G_CALLBACK (menubar_predict_cb)},

	/* Help menu */
	{ "GettingStarted", GTK_STOCK_EXECUTE, N_("Getting Started"), NULL,
	  N_("Show online user manual, Getting Started Section"),
	  G_CALLBACK (menubar_getting_started_cb)},
	{ "Manual", GTK_STOCK_HELP, N_("User Manual"), NULL,
	  N_("Show online user manual"), G_CALLBACK (menubar_help_cb)},
	{ "About", GTK_STOCK_ABOUT, N_("_About"), NULL,
	  N_("Show about dialog"), G_CALLBACK (menubar_about_cb) },
};


/** \brief Menubar UI description.
 *  \ingroup menupriv
 */
static const char *menu_desc =
"<ui>"
"   <menubar name='GpredictMenu'>"
"      <menu action='FileMenu'>"
"         <menuitem action='New'/>"
"         <menuitem action='Open'/>"
"         <separator/>"
"         <menuitem action='Exit'/>"
"      </menu>"
"      <menu action='EditMenu'>"
/* "         <menuitem action='Freq'/>" */
"         <menuitem action='Pref'/>"
"      </menu>"
/* "      <menu action='ToolsMenu'>" */
/* "         <menuitem action='Tle'/>" */
/* "         <separator/>" */
/* "         <menuitem action='Glance'/>" */
/* "         <menuitem action='Predict'/>" */
/* "         <menuitem action='Window'/>" */
/* "         <separator/>" */
/* "         <menuitem action='Lab'/>" */
/* "      </menu>" */
"      <menu action='HelpMenu'>"
/* "         <menuitem action='GettingStarted'/>" */
/* "         <menuitem action='Manual'/>" */
"         <separator/>"
"         <menuitem action='About'/>"
"      </menu>"
"   </menubar>"
"</ui>";


/** \brief Create menubar.
 *  \param window The main application window.
 *  \return The menubar widget.
 *
 * This function creates and initializes the main menubar for gpredict.
 * It should be called from the main gui_create function.
 */
GtkWidget *
menubar_create (GtkWidget *window)
{
	GtkWidget      *menubar;
	GtkActionGroup *actgrp;
	GtkUIManager   *uimgr;
	GtkAccelGroup  *accgrp;
	GError         *error = NULL;

	/* create action group */
	actgrp = gtk_action_group_new ("MenuActions");
	gtk_action_group_add_actions (actgrp, entries, G_N_ELEMENTS (entries), NULL);

	/* create UI manager */
	uimgr = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (uimgr, actgrp, 0);

	/* accelerator group */
	accgrp = gtk_ui_manager_get_accel_group (uimgr);
	gtk_window_add_accel_group (GTK_WINDOW (window), accgrp);

	/* try to create UI from XML*/
	if (!gtk_ui_manager_add_ui_from_string (uimgr, menu_desc, -1, &error)) {
		g_print (_("Failed to build menubar: %s"), error->message);
		g_error_free (error);

		return NULL;
	}

	/* now, finally, get the menubar */
	menubar = gtk_ui_manager_get_widget (uimgr, "/GpredictMenu");

	return menubar;
}


/** \brief Create new module.
 *
 * This function first executes the mod-cfg editor. If the editor returns
 * the name of an existing .mod file it will create the corresponding module
 * and send it to the module manager.
 */
static void
menubar_new_mod_cb  (GtkWidget *widget, gpointer data)
{
	gchar *modnam = NULL;
	gchar *modfile;
	GtkWidget *module = NULL;
	GtkWidget *parent;



	sat_log_log (SAT_LOG_LEVEL_DEBUG,
		     _("%s: Starting new module configurator..."),
		     __FUNCTION__);

	modnam = mod_cfg_new ();

	if (modnam) {
		sat_log_log (SAT_LOG_LEVEL_DEBUG,
			    _("%s: New module name is %s."),
			    __FUNCTION__, modnam);

		modfile = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
				       ".gpredict2", G_DIR_SEPARATOR_S,
				       "modules", G_DIR_SEPARATOR_S,
				       modnam,
				       ".mod", NULL);

		/* create new module */
		module = gtk_sat_module_new (modfile);
		GTK_SAT_MODULE (module)->state = GTK_SAT_MOD_STATE_DOCKED;

		if (module == NULL) {

			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (GTK_WINDOW (app),
							 GTK_DIALOG_MODAL |
							 GTK_DIALOG_DESTROY_WITH_PARENT,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_OK,
							 _("Could not open %s. Please examine "\
							   "the log messages for details."),
							 modnam);

			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}
		else {
			mod_mgr_add_module (module, TRUE);

			/* get size allocation from parent and set some reasonable
			   initial GtkPaned positions
			*/
			parent = gtk_widget_get_parent (module);
			GTK_SAT_MODULE (module)->hpanedpos = parent->allocation.width / 2;
			GTK_SAT_MODULE (module)->vpanedpos = parent->allocation.height / 2;
			gtk_sat_module_fix_size (module);
		}

		g_free (modnam);
		g_free (modfile);
	}
	else {
		sat_log_log (SAT_LOG_LEVEL_DEBUG,
			    _("%s: New module config cancelled."),
			    __FUNCTION__);
	}
}


static void
menubar_open_mod_cb  (GtkWidget *widget, gpointer data)
{
	gchar *modnam = NULL;
	gchar *modfile;
	GtkWidget *module = NULL;
	GtkWidget *parent;


	sat_log_log (SAT_LOG_LEVEL_DEBUG,
		     _("%s: Open existing module..."),
		     __FUNCTION__);
	
	modnam = select_module ();

	if (modnam) {
		sat_log_log (SAT_LOG_LEVEL_DEBUG,
			    _("%s: Open module %s."),
			    __FUNCTION__, modnam);

		modfile = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
				       ".gpredict2", G_DIR_SEPARATOR_S,
				       "modules", G_DIR_SEPARATOR_S,
				       modnam,
				       ".mod", NULL);

		/* create new module */
		module = gtk_sat_module_new (modfile);
		GTK_SAT_MODULE (module)->state = GTK_SAT_MOD_STATE_DOCKED;

		if (module == NULL) {
			/* mod manager could not create the module */
			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (GTK_WINDOW (app),
							 GTK_DIALOG_MODAL |
							 GTK_DIALOG_DESTROY_WITH_PARENT,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_OK,
							 _("Could not open %s. Please examine "\
							   "the log messages for details."),
							 modnam);

			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}
		else {
			mod_mgr_add_module (module, TRUE);

			/* get size allocation from parent and set some reasonable
			   initial GtkPaned positions
			*/
			parent = gtk_widget_get_parent (module);
			GTK_SAT_MODULE (module)->hpanedpos = parent->allocation.width / 2;
			GTK_SAT_MODULE (module)->vpanedpos = parent->allocation.height / 2;
			gtk_sat_module_fix_size (module);
		}

		g_free (modnam);
		g_free (modfile);
	}
	else {
		sat_log_log (SAT_LOG_LEVEL_DEBUG,
			    _("%s: Open module cancelled."),
			    __FUNCTION__);
	}
}


static void
menubar_app_exit_cb  (GtkWidget *widget, gpointer data)
{

	gtk_widget_destroy (app);
}


static void
menubar_freq_edit_cb (GtkWidget *widget, gpointer data)
{
}


static void
menubar_pref_cb      (GtkWidget *widget, gpointer data)
{
	sat_pref_run ();
}


static void
menubar_tle_cb       (GtkWidget *widget, gpointer data)
{
}


static void
menubar_list_cb (GtkWidget *widget, gpointer data)
{
}


static void
menubar_plot_cb (GtkWidget *widget, gpointer data)
{
}


static void
menubar_glance_cb (GtkWidget *widget, gpointer data)
{
}


static void
menubar_window_cb (GtkWidget *widget, gpointer data)
{
}


static void
menubar_predict_cb   (GtkWidget *widget, gpointer data)
{
}

static void
menubar_lab_cb   (GtkWidget *widget, gpointer data)
{
}

static void
menubar_getting_started_cb (GtkWidget *widget, gpointer data)
{
	gpredict_help_launch (GPREDICT_HELP_GETTING_STARTED);
}

static void
menubar_help_cb (GtkWidget *widget, gpointer data)
{
	gpredict_help_launch (GPREDICT_HELP_USER_MANUAL);
}


static void
menubar_about_cb (GtkWidget *widget, gpointer data)
{
	about_dialog_create ();
}



/** \brief Select an existing module.
 *
 * This function creates a dialog with a list of existing modules
 * from /homedir/.gpredict2/modules/ and lets the user select one
 * of them. The function will return the name of the selected module
 * without the .mod suffix.
 */
static gchar *
select_module        ()
{
	GtkWidget         *dialog;     /* the dialog window */
	GtkWidget         *modlist;    /* the treeview widget */
	GtkListStore      *liststore;  /* the list store data structure */
	GtkCellRenderer   *renderer;
	GtkTreeViewColumn *column;
	GtkTreeIter        item;       /* new item added to the list store */
	GtkTreeSelection  *selection;
	GtkTreeModel      *selmod;
	GDir              *dir = NULL;   /* directory handle */
	GError            *error = NULL; /* error flag and info */
	gchar             *dirname;      /* directory name */
	const gchar       *filename;     /* file name */
	gchar            **buffv;
	guint              count = 0;

	/* create and fill data model */
	liststore = gtk_list_store_new (1, G_TYPE_STRING);

	/* scan for .qth files in the user config directory and
	   add the contents of each .qth file to the list store
	*/
	dirname = g_strconcat (g_get_home_dir (),
			       G_DIR_SEPARATOR_S, ".gpredict2",
			       G_DIR_SEPARATOR_S, "modules",
			       NULL);
	dir = g_dir_open (dirname, 0, &error);

	if (dir) {

		sat_log_log (SAT_LOG_LEVEL_DEBUG,
			     _("%s:%s: Scanning directory %s for modules."),
			     __FILE__, __FUNCTION__, dirname);

		while ((filename = g_dir_read_name (dir))) {

			if (g_strrstr (filename, ".mod")) {

				/* strip extension and add to list */
				buffv = g_strsplit (filename, ".mod", 0);

				gtk_list_store_append (liststore, &item);
				gtk_list_store_set (liststore, &item,
						    0, buffv[0],
						    -1);

				g_strfreev (buffv);

				count++;
			}
		}
	}
	else {
		sat_log_log (SAT_LOG_LEVEL_ERROR,
			     _("%s:%d: Failed to open module dir %s (%s)"),
			     __FILE__, __LINE__, dirname, error->message);
		g_clear_error (&error);
	}

	g_free (dirname);
	g_dir_close (dir);

	if (count < 1) {
		/* tell user that there are no modules, try "New" instead */
		dialog = gtk_message_dialog_new (GTK_WINDOW (app),
						 GTK_DIALOG_MODAL |
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_INFO,
						 GTK_BUTTONS_OK,
						 _("You do not have any modules "\
						   "set up yet. Please use File->New "\
						   "in order to create a module."));

		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		return NULL;
	}

	/* create tree view */
	modlist = gtk_tree_view_new ();
	gtk_tree_view_set_model (GTK_TREE_VIEW (modlist), GTK_TREE_MODEL (liststore));
	g_object_unref (liststore);

	/*** FIXME: Add g_stat info? */

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Module"), renderer,
							   "text", 0,
							   NULL);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (modlist), column, -1);

	gtk_widget_show (modlist);

	/* create dialog */
	dialog = gtk_dialog_new_with_buttons (_("Select a module"),
					      GTK_WINDOW (app),
					      GTK_DIALOG_MODAL |
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_STOCK_CANCEL,
					      GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK,
					      GTK_RESPONSE_OK,
					      NULL);

	gtk_window_set_default_size (GTK_WINDOW (dialog), -1, 200);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), modlist);

	switch (gtk_dialog_run (GTK_DIALOG (dialog))) {

		/* user pressed OK */
	case GTK_RESPONSE_OK:

		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (modlist));

		if (gtk_tree_selection_get_selected (selection, &selmod, &item)) {
			gtk_tree_model_get (selmod, &item,
					    0, &dirname,
					    -1);

			sat_log_log (SAT_LOG_LEVEL_DEBUG,
				     _("%s:%s: Selected module is: %s"),
				     __FILE__, __FUNCTION__, dirname);
		}
		else {
			sat_log_log (SAT_LOG_LEVEL_BUG,
				     _("%s:%s: No selection is list of modules."),
				     __FILE__, __FUNCTION__);

			dirname = NULL;
		}

		break;

		/* everything else is regarded as CANCEL */
	default:
		dirname = NULL;
		break;
	}

	gtk_widget_destroy (dialog);
	

	return dirname;
}

