#include "checkmateViewer.h"
#include "gpsshogi/gui/util.h"
#include "gpsshogi/gui/board.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/checkmate/dfpnRecord.h"
#include "osl/checkmate/proofTreeDepthDfpn.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/record/csa.h"
#include "osl/record/ki2.h"

#include <qtextcodec.h>
#include <qmessagebox.h>
#include <qtooltip.h>
#include <qlayout.h>
#if QT_VERSION >= 0x040000
# include <Q3PopupMenu>
# include <Q3Action>
#else
# include <qpopupmenu.h>
# include <qaction.h>
# define Q3PopupMenu QPopupMenu
# define Q3Action QAction
#endif
#include <sstream>

static const bool show_side_nodes = false;

using gpsshogi::gui::Util;

namespace
{
  template <class Parent>
  void addItemAttack(Parent *parent,
		     const osl::checkmate::DfpnRecord& record,
		     const osl::NumEffectState& state,
		     const osl::checkmate::DfpnTable& table)
  {
    assert(parent);
    if (! record.proof_disproof.isFinal())
      return;
    const osl::Move best_move = record.best_move;
    if (best_move.isNormal())
    {
      osl::checkmate::ProofTreeDepthDfpn depth_analyzer(table);
      osl::HashKey key(state);
      int solution_depth = -1;
      if (record.proof_disproof.isCheckmateSuccess())
	solution_depth = depth_analyzer.depth(key, state, true);
      osl::checkmate::DfpnRecord child
	= table.probe(key.newHashWithMove(best_move),
		      osl::PieceStand(osl::WHITE,state).nextStand(osl::WHITE,best_move));
      new CheckMoveItem(parent, best_move, solution_depth, record, child, true);
    }
#if 0
    if (! show_side_nodes)
      return;
    for (osl::checkmate::CheckMoveList::const_iterator i = record->moves.begin();
	 i != record->moves.end(); i++)
    {
      if (best_move && i->move == best_move->move)
	continue;
      osl::checkmate::ProofTreeDepthDfpn depth_analyzer(table);
      int solution_depth = -1;
      if (i->record && i->record->proofDisproof().isCheckmateSuccess())
	solution_depth = depth_analyzer.depth(i->record, false) + 1;
      new CheckMoveItem(parent, i->move, solution_depth, record, true);
    }
#endif
  }

  template <class Parent>
  void addItemDefense(Parent *parent,
		      const osl::checkmate::DfpnRecord& record,
		      const osl::NumEffectState& state,
		      const osl::checkmate::DfpnTable& table)
  {
    assert(parent);
    if (! record.proof_disproof.isFinal())
      return;
    osl::MoveVector moves;
    osl::LegalMoves::generate(state, moves);
    const osl::HashKey key(state);
    for (osl::MoveVector::const_iterator p=moves.begin();
	 p!=moves.end(); ++p) {
      osl::HashKey new_key = key.newHashWithMove(*p);

      int solution_depth = -1;
      osl::checkmate::DfpnRecord child
	= table.probe(osl::HashKey(state).newHashWithMove(*p),
		      osl::PieceStand(osl::WHITE,state).nextStand(osl::WHITE,*p));
      if (child.proof_disproof.isCheckmateSuccess())
      {
	osl::checkmate::ProofTreeDepthDfpn depth_analyzer(table);
	osl::NumEffectState new_state(state);
	new_state.makeMove(*p);
	solution_depth = depth_analyzer.depth(osl::HashKey(new_state), new_state, true)+1;
      }
      new CheckMoveItem(parent, *p, solution_depth, record, child, false);
    }
  }
}

class CheckMoveItem : public MoveTreeItem
{
public:
  CheckMoveItem(CheckMoveViewer *parent, osl::Move move, int moves, 
		const osl::checkmate::DfpnRecord& pr,
		const osl::checkmate::DfpnRecord& mr, bool attack) :
    MoveTreeItem(parent, move, move.isPass() ? "PASS" : Util::moveToString(move)),
    numMoves(moves), parent_record(pr), record(mr), is_attack(attack), tree(parent), parent_item(0) {
    setRecord();
  }
  CheckMoveItem(CheckMoveItem *parent, osl::Move move, int moves,
		const osl::checkmate::DfpnRecord& pr, 
		const osl::checkmate::DfpnRecord& mr, bool attack) :
    MoveTreeItem(parent, move, move.isPass() ? "PASS" : Util::moveToString(move)),
    numMoves(moves), parent_record(pr), record(mr), is_attack(attack), tree(parent->tree), parent_item(parent) {
    setRecord();
  }
  bool isBestMove() const {
    return numMoves >= 0;
  }
  int getNumMoves() const {
    return numMoves;
  }
  int compare(Q3ListViewItem *i, int col, bool ascending) const {
    CheckMoveItem *item = (CheckMoveItem *)i;
    if (numMoves < item->getNumMoves())
      return 1;
    else if (numMoves > item->getNumMoves())
      return -1;

    return MoveTreeItem::compare(i, col, ascending);
  }
  void setOpen(bool);
  void paintCell(QPainter *p, const QColorGroup &cg,
		 int column, int width, int align);

  const osl::checkmate::DfpnRecord getRecord() const { return record; }
  bool isAttack() const { return is_attack; }

  static const QString proofString(const osl::checkmate::DfpnRecord record)
  {
    const int proof = record.proof_disproof.proof();
    if (proof > osl::ProofDisproof::PROOF_LIMIT)
      return QString("INF");
    return QString::number(proof);
  }
  static const QString disproofString(const osl::checkmate::DfpnRecord record)
  {
    const int disproof = record.proof_disproof.disproof();
    if (disproof > osl::ProofDisproof::DISPROOF_LIMIT)
      return QString("INF");
    return QString::number(disproof);
  }
private:
  void setRecord()
  {
    setExpandable(true);
    if (show_side_nodes) {
      setText(1, proofString(record));
      setText(2, disproofString(record));
    }
  }
public:
  void getMovesToCurrent(osl::MoveVector& moves) 
  {
    CheckMoveItem *i = this;
    while (i) {
      moves.push_back(i->getMove());
      i = i->parent_item;
    }
    std::reverse(moves.begin(), moves.end());
  }
private:
  int numMoves;
  const osl::checkmate::DfpnRecord parent_record, record;
  bool is_attack;
  CheckMoveViewer *tree;
  CheckMoveItem *parent_item;
};

void CheckMoveItem::setOpen(bool open)
{
  if (open && childCount() == 0)
  {
    osl::MoveVector history;
    getMovesToCurrent(history);
    osl::NumEffectState state(tree->initialState());
    for (size_t i=0; i<history.size(); ++i)
      state.makeMove(history[i]);

    if (depth() >= 256)
    {
      setExpandable(false);
      return;
    }
#if 0
    // FIXME: sometimes, this makes some nodes disabled even after
    // setUpdatesEnabled(true) called.
    listView()->setUpdatesEnabled(false);
#endif
    if (is_attack)
      addItemDefense(this, record, state, tree->getTable());
    else
      addItemAttack(this, record, state, tree->getTable());
#if 0
    listView()->setUpdatesEnabled(true);
#endif
  }
  Q3ListViewItem::setOpen(open);
}

void CheckMoveItem::paintCell(QPainter *p, const QColorGroup &cg,
			      int column, int width, int align)
{
  if (column == 0 && numMoves < 0)
  {
    QColorGroup color_group(cg);
    QColor side_color("gray");
    color_group.setColor(QColorGroup::Text, side_color);
    MoveTreeItem::paintCell(p, color_group, column, width, align);
  }
  else
    MoveTreeItem::paintCell(p, cg, column, width, align);
}


#if QT_VERSION < 0x040000
class MoveCountTip : public QToolTip
{
public:
  MoveCountTip(QWidget *parent, Q3ListView *w) : QToolTip(parent), listView(w) {
  }
protected:
  virtual void maybeTip(const QPoint &pos);
private:
  Q3ListView *listView;
};

void MoveCountTip::maybeTip(const QPoint &pos)
{
  CheckMoveItem *item = (CheckMoveItem *)listView->itemAt(pos);
  if (item)
  {
    QRect r = listView->itemRect(item);
    QString s;
    s.sprintf("%d %s", item->getNumMoves(), "手");
    tip(r, s);
  }
}
#endif

CheckMoveViewer::CheckMoveViewer(QWidget *parent, const char *name)
  : MoveTree(parent, name)
{
  setRootIsDecorated(true);
  addColumn("Move");
  if (show_side_nodes) {
    addColumn("proof");
    addColumn("disproof");
    setColumnAlignment(1, Qt::AlignRight);
    setColumnAlignment(2, Qt::AlignRight);
  }
#if QT_VERSION < 0x040000
  tip = new MoveCountTip(viewport(), this);
#endif
}

CheckMoveViewer::~CheckMoveViewer()
{
#if QT_VERSION < 0x040000
  delete tip;
#endif
}

bool CheckMoveViewer::analyze(const osl::state::NumEffectState &sstate,
			      int limit, bool change_turn)
{
  int table_size = 20000000;
  check_state result;
  bool search_again = true;

  while (search_again)
  {
    result = analyze(sstate, table_size, limit, change_turn);
    search_again = false;

    if (result == UNKNOWN)
    {
      QString nodeCount;
      nodeCount.sprintf("%s %ld\n", "ノード数",
			(long int)searcher->totalNodeCount());
      limit *= 2;
      if (table_size < limit)
	table_size *= 2;
      int ret = QMessageBox::question(this, "Checkmate Search",
				      nodeCount +
				      QString("Unknown, Search again with limit %1 size %2 ?").arg(limit).arg(table_size),
				      QMessageBox::Yes,
				      QMessageBox::No | QMessageBox::Default);
      if (ret == QMessageBox::Yes)
      {
	search_again = true;
      }
    }
    else
    {
      QString message;
      if (result == CHECKMATE)
	message = "Checkmate";
      else if (result == NO_CHECKMATE)
	message = "No Checkmate";
      else if (result == NO_CHECKMATE_LOOP)
	message = "No Checkmate (loop)";

      QString nodeCount;
      nodeCount.sprintf("\n%s %ld", "ノード数",
			(long int)searcher->totalNodeCount());
      message.append(nodeCount);
      QMessageBox::information(this, "Checkmate State", message);
    }
  }
  return result == CHECKMATE;
}

CheckMoveViewer::check_state CheckMoveViewer::analyze(const osl::state::NumEffectState &sstate,
						      int table_size,
						      int limit, bool change_turn)
{
  clear();
  searcher.reset(new searcher_t(table_size));

  osl::Move checkmateMove;
  osl::state::NumEffectState state(sstate);
  const bool king_in_check = state.inCheck();
  if (change_turn && (!king_in_check))
    state.changeTurn();
  const osl::PathEncoding path(state.turn());
  const bool is_defense = king_in_check && change_turn;
  const bool win 
    = (is_defense
       ? searcher->isLosingState(limit, state, osl::HashKey(state), path)
       : searcher->isWinningState(limit, state, osl::HashKey(state), path, checkmateMove));

  this->table
    = &(searcher->table(is_defense ? alt(state.turn()) : state.turn()));
  initial_state = sstate;
  const osl::checkmate::DfpnRecord record = table->probe(osl::HashKey(state), osl::PieceStand(osl::WHITE,state));
  if (win || show_side_nodes)
  {
    if (!change_turn && !king_in_check)
      addItemAttack(this, record, state, *table);
    else if (king_in_check)
    {
      addItemDefense(this, record, state, *table);
    }
    else
    {
      assert(change_turn && !king_in_check);
      CheckMoveItem *item = new CheckMoveItem(this,
					      osl::Move::PASS(osl::alt(path.turn())), 
					      0, osl::checkmate::DfpnRecord(), record, false);
      addItemAttack(item, record, state, *table);
    }
  }

  check_state result = UNKNOWN;
  if (record.proof_disproof.isCheckmateSuccess())
  {
    result = CHECKMATE;
  }
  else if (record.proof_disproof.isCheckmateFail())
  {
    result = NO_CHECKMATE;
  }
  
  return result;
}

void CheckMoveViewer::showContextMenu(Q3PopupMenu *contextMenu)
{
  contextMenu->insertSeparator();
  Q3Action *showRecordAction = new Q3Action("Dump Record Data", 0,
					    contextMenu);
  showRecordAction->addTo(contextMenu);
  connect(showRecordAction, SIGNAL(activated()),
	  this, SLOT(showRecord()));

  Q3Action *showMovesAction = new Q3Action("Show moves to this node", 0,
					    contextMenu);
  showMovesAction->addTo(contextMenu);
  connect(showMovesAction, SIGNAL(activated()),
	  this, SLOT(showMoves()));

  Q3Action *showMovesInKanjiAction = new Q3Action("Show moves to this node in Kanji", 0,
					    contextMenu);
  showMovesInKanjiAction->addTo(contextMenu);
  connect(showMovesInKanjiAction, SIGNAL(activated()),
	  this, SLOT(showMovesInKanji()));
}

void CheckMoveViewer::showRecord(const osl::checkmate::DfpnRecord& /*record*/)
{
#if 0 
  if (! record)
    return;
  // not yet implemented
  std::ostringstream oss(std::ostringstream::out);
  record->dump(oss, 1);
  const std::string &record_string = oss.str();
  QMessageBox::information(this, "QuiescenceRecord Dump",
			   QString(record_string.c_str()), QMessageBox::Ok);
#endif
}

void CheckMoveViewer::showRecord()
{
  CheckMoveItem *item = dynamic_cast<CheckMoveItem*>(selectedItem());
  if (!item)
    return;
  showRecord(item->getRecord());
}

void CheckMoveViewer::showMoves()
{
  CheckMoveItem *item = dynamic_cast<CheckMoveItem*>(selectedItem());
  if (!item)
    return;

  osl::MoveVector moves;
  item->getMovesToCurrent(moves);
  QString moves_text;
  for (size_t i = 0; i < moves.size(); ++i)
  {
    moves_text.append(osl::record::csa::show(moves[i]).c_str()).append("\n");
  }
  
  QMessageBox::information(this, "Moves",
			   moves_text, QMessageBox::Ok);
}

void CheckMoveViewer::showMovesInKanji()
{
  CheckMoveItem *item = dynamic_cast<CheckMoveItem*>(selectedItem());
  if (!item)
    return;

  osl::MoveVector moves;
  item->getMovesToCurrent(moves);
  osl::NumEffectState state(initial_state);
  QString moves_text;
  QTextCodec *codec = QTextCodec::codecForName("EUC-JP");
  for (size_t i = 0; i < moves.size(); ++i)
  {
    std::string euc_move = i 
      ? osl::record::ki2::show(moves[i], state, moves[i-1])
      : osl::record::ki2::show(moves[i], state);
    moves_text.append(codec->toUnicode(euc_move.c_str(), euc_move.length()))
      .append("\n");
    state.makeMove(moves[i]);
  }
  
  QMessageBox::information(this, "Moves",
			   moves_text, QMessageBox::Ok);
}

CheckmateViewer::CheckmateViewer(QWidget *parent, const char *name)
  : BoardAndListTabChild(parent, name)
{
  moveTree = new CheckMoveViewer(this);

  QHBoxLayout *mainLayout = new QHBoxLayout(this);
  mainLayout->addWidget(board);
  mainLayout->addWidget(moveTree);

  init();
}

bool CheckmateViewer::analyze(const osl::state::NumEffectState &state,
			      int node_limit, bool change_turn)
{
  initialState = state;
  board->setState(state);
  return ((CheckMoveViewer *)moveTree)->analyze(state, node_limit,
						change_turn);
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
