#include "evalDebug.h"

#include <QAbstractTableModel>
#include <QTableView>
#include <QTreeView>
#include <QPushButton>
#include <QVBoxLayout>
#include <QTableWidget>
#include <QPushButton>
#include "osl/state/numEffectState.h"
#include "osl/eval/progressEval.h"
#include "osl/eval/endgame/attackDefense.h"
#include "osl/eval/endgame/kingPieceValues.h"
#include "osl/eval/ml/openMidEndingEval.h"

using namespace osl::eval::ml;

class EvalDebugModel : public QAbstractTableModel
{
public:
  EvalDebugModel(const osl::state::NumEffectState &s,
		 QObject *parent = 0);
  int rowCount(const QModelIndex &parent = QModelIndex()) const;
  int columnCount(const QModelIndex &parent = QModelIndex()) const;
  QVariant data(const QModelIndex &index, int role) const;
  QVariant headerData(int section, Qt::Orientation orientation,
		      int role = Qt::DisplayRole) const;
  void setStatus(const osl::state::SimpleState &state);
  osl::eval::ProgressDebugInfo debugInfo() const { return debug_info; }
  const osl::state::NumEffectState &getState() const { return state; }
private:
  QStringList headers;
  osl::eval::ProgressDebugInfo debug_info;
  osl::state::NumEffectState state;
};

EvalDebugModel::EvalDebugModel(
  const osl::state::NumEffectState &s,
  QObject *parent) : QAbstractTableModel(parent), state(s)
{
  osl::eval::ProgressEval eval(state);
  debug_info = eval.debugInfo(state);
  headers << "Eval" << "Opening" << "Endgame"
	  << "Progress bonus" << "Progress Independent"
	  << "Progress Dependent" << "Minor Pieces";
}

int EvalDebugModel::rowCount(const QModelIndex &) const
{
  return 1;
}

int EvalDebugModel::columnCount(const QModelIndex &) const
{
  return headers.size();
}

QVariant EvalDebugModel::data(const QModelIndex &index, int role) const
{
  if (!index.isValid() || index.row() > rowCount() ||
      index.column() >  columnCount())
    return QVariant();

  if (role == Qt::DisplayRole)
  {
    switch (index.column())
    {
    case 0:
      return debug_info.eval;
    case 1:
      return QString("%1 (%2)").arg(debug_info.opening * (16 - debug_info.progress)).arg(debug_info.opening);
    case 2:
      return QString("%1 (%2)").arg(debug_info.endgame * debug_info.progress).arg(debug_info.endgame);
    case 3:
      return debug_info.progress_bonus;
    case 4:
      return debug_info.progress_independent_bonus;
    case 5:
      return debug_info.progress_dependent_bonus;
    case 6:
      return debug_info.minor_piece_bonus;
    default:
      return QVariant();
    }
  }
  else
  {
    return QVariant();
  }
}

QVariant EvalDebugModel::headerData(int section, Qt::Orientation orientation,
				    int role) const
{
  if (role != Qt::DisplayRole || section > headers.size())
    return QVariant();

  if (orientation == Qt::Horizontal)
    return headers[section];
  else
    return QString("%1").arg(section);
}

void EvalDebugModel::setStatus(const osl::state::SimpleState &s_state)
{
  state = osl::state::NumEffectState(s_state);
  osl::eval::ProgressEval eval(state);
  debug_info = eval.debugInfo(state);
  emit dataChanged(createIndex(0, 0), createIndex(0, headers.size()));
}



class EvalDebugDetailAttackDefensePanel : public QWidget
{
public:
  EvalDebugDetailAttackDefensePanel(QWidget *parent = 0, const char *name = 0)
    : QWidget(parent, name)
  {
    QVBoxLayout *vlayout = new QVBoxLayout(this);
    attack = new QPushButton("&Attack", this);
    defense = new QPushButton("&Defense", this);
    attack_defense = new QPushButton("Attack Defens&e", this);
    vlayout->addWidget(attack);
    vlayout->addWidget(defense);
    vlayout->addWidget(attack_defense);
    vlayout->addStretch(1);
  }
  QPushButton *attack, *defense, *attack_defense;
  osl::state::SimpleState state;
};

void EvalDebugDetail::clear()
{
  table->clear();
  table->setRowCount(0);
  table->setColumnCount(0);
}


EvalDebugDetail::EvalDebugDetail(QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  table = new QTableWidget(this);
  QHBoxLayout *layout = new QHBoxLayout(this);
  panel = new EvalDebugDetailAttackDefensePanel(this);
  connect(panel->attack, SIGNAL(clicked()), this, SLOT(showAttack()));
  connect(panel->defense, SIGNAL(clicked()), this, SLOT(showDefense()));
  connect(panel->attack_defense, SIGNAL(clicked()),
	  this, SLOT(showAttackDefense()));
  layout->addWidget(table);
  layout->addWidget(panel);
  panel->hide();
}

void EvalDebugDetail::showEndgameBonus(
  const osl::state::SimpleState &state,
  const osl::container::PieceValues &values)
{
  table->clear();
  table->setRowCount(9);
  table->setColumnCount(9);
  for (size_t i = 0; i < values.size(); ++i)
  {
    const osl::Piece piece = state.pieceOf (i);
    if (piece.isOnBoard())
    {
      table->setItem(piece.square().y() - 1, 9 - piece.square().x(),
		      new QTableWidgetItem(tr("%1").arg(values[i])));
    }
  }
  panel->show();
  panel->state = state;
  table->resizeColumnsToContents();
}

void EvalDebugDetail::showAttackDefenseBonus(
  const osl::state::SimpleState &state)
{
  osl::container::PieceValues values;
  osl::eval::endgame::AttackDefense::setValues(state, values);
  showEndgameBonus(state, values);
}

void EvalDebugDetail::showAttackBonus(
  const osl::state::SimpleState &state)
{
  osl::container::PieceValues values;
  osl::eval::endgame::KingPieceValues<osl::eval::endgame::AttackKing>::setValues(state, values);
  showEndgameBonus(state, values);
}

void EvalDebugDetail::showDefenseBonus(
  const osl::state::SimpleState &state)
{
  osl::container::PieceValues values;
  osl::eval::endgame::KingPieceValues<osl::eval::endgame::DefenseKing>::setValues(state, values);
  showEndgameBonus(state, values);
}

void EvalDebugDetail::showAttackDefense()
{
  showAttackDefenseBonus(panel->state);
}
void EvalDebugDetail::showAttack()
{
  showAttackBonus(panel->state);
}
void EvalDebugDetail::showDefense()
{
  showDefenseBonus(panel->state);
}

void EvalDebugDetail::showMinorPiecesBonus(const EvalDebugModel &model)
{
  table->clear();
  table->setRowCount(1);
  table->setColumnCount(4);

  QStringList column_header;
  column_header << "Pawn" << "Lance" << "Knight" << "Gold";
  table->setHorizontalHeaderLabels(column_header);
  const osl::eval::ProgressDebugInfo debug_info = model.debugInfo();
  table->setItem(0, 0,
		  new QTableWidgetItem(tr("%1").arg(debug_info.minor_piece_bonus_info.pawn_bonus)));
  table->setItem(0, 1,
		  new QTableWidgetItem(tr("%1").arg(debug_info.minor_piece_bonus_info.lance_bonus)));
  table->setItem(0, 2,
		  new QTableWidgetItem(tr("%1").arg(debug_info.minor_piece_bonus_info.knight_bonus)));
  table->setItem(0, 3,
		  new QTableWidgetItem(tr("%1").arg(debug_info.minor_piece_bonus_info.gold_bonus)));
  panel->hide();
  table->resizeColumnsToContents();
}

void EvalDebugDetail::showProgressBonus(const EvalDebugModel &model)
{
  table->clear();
  table->setRowCount(2);
  table->setColumnCount(4);

  QStringList row_header, column_header;
  row_header << "WHITE" << "BLACK";
  column_header << "Danger" << "Defense" << "" << "";
  table->setHorizontalHeaderLabels(column_header);
  table->setVerticalHeaderLabels(row_header);

  const osl::eval::ProgressDebugInfo debug_info = model.debugInfo();
  QTableWidgetItem *white_danger =
    new QTableWidgetItem(tr("%1").arg(debug_info.white_danger));
  QTableWidgetItem *white_defense =
    new QTableWidgetItem(tr("%1").arg(debug_info.white_defense));
  QTableWidgetItem *black_danger =
    new QTableWidgetItem(tr("%1").arg(debug_info.black_danger));
  QTableWidgetItem *black_defense =
    new QTableWidgetItem(tr("%1").arg(debug_info.black_defense));
  QTableWidgetItem *bonus_formula =
    new QTableWidgetItem(tr("%1 - %2 / 2 - %3 + %4 / 2").
			 arg(debug_info.white_danger).
			 arg(debug_info.white_defense).
			 arg(debug_info.black_danger).
			 arg(debug_info.black_defense));

  QTableWidgetItem *bonus =
    new QTableWidgetItem(tr("%1").
			 arg(debug_info.white_danger - 
			     debug_info.white_defense / 2
			     - debug_info.black_danger
			     + debug_info.black_defense / 2));

  table->setItem(0, 0, white_danger);
  table->setItem(0, 1, white_defense);
  table->setItem(1, 0, black_danger);
  table->setItem(1, 1, black_defense);
  table->setItem(0, 3, bonus_formula);
  table->setItem(1, 3, bonus);
  panel->hide();
  table->resizeColumnsToContents();
}

void EvalDebugDetail::showProgressIndependentBonus(const EvalDebugModel &model)
{
  table->clear();
  table->setRowCount(1);
  table->setColumnCount(6);

  QStringList column_header;
  column_header << "Mobility" << "Two Rooks" << "Knight CHeck"
		<< "Rook Rank" << "King Enter" << "Middle King";
  table->setHorizontalHeaderLabels(column_header);
  const osl::eval::ProgressDebugInfo debug_info = model.debugInfo();
  table->setItem(0, 0,
		  new QTableWidgetItem(tr("%1").arg(debug_info.mobility_bonus)));
  table->setItem(0, 1,
		  new QTableWidgetItem(tr("%1").arg(debug_info.two_rook_bonus)));
  table->setItem(0, 2,
		  new QTableWidgetItem(tr("%1").arg(debug_info.knight_check_bonus)));
  table->setItem(0, 3,
		  new QTableWidgetItem(tr("%1").arg(debug_info.rook_rank_bonus)));
  table->setItem(0, 4,
		  new QTableWidgetItem(tr("%1").arg(debug_info.enter_king_bonus)));
  table->setItem(0, 5,
		  new QTableWidgetItem(tr("%1").arg(debug_info.middle_king_bonus)));
  panel->hide();
  table->resizeColumnsToContents();
}

void EvalDebugDetail::showProgressDependentBonus(const EvalDebugModel &model)
{
  table->clear();
  table->setRowCount(1);
  table->setColumnCount(2);

  QStringList column_header;
  column_header << "King8 Attack" << "Pin";
  table->setHorizontalHeaderLabels(column_header);
  const osl::eval::ProgressDebugInfo debug_info = model.debugInfo();
  table->setItem(0, 0,
		  new QTableWidgetItem(tr("%1").arg(debug_info.king8_attack_bonus)));
  table->setItem(0, 1,
		  new QTableWidgetItem(tr("%1").arg(debug_info.pin_bonus)));
  panel->hide();
  table->resizeColumnsToContents();
}

EvalDebug::EvalDebug(
  const osl::state::NumEffectState &s,
  QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  model = new EvalDebugModel(s, this);
  QVBoxLayout *layout = new QVBoxLayout(this);
  QTableView *view = new QTableView(this);
  detail = new EvalDebugDetail(this);
  connect(view, SIGNAL(clicked(const QModelIndex &)),
	  this, SLOT(itemClicked(const QModelIndex &)));
  view->setModel(model);
  layout->addWidget(view);
  layout->addWidget(detail, 1);
}

void EvalDebug::setStatus(const osl::state::SimpleState &state)
{
  model->setStatus(state);
  detail->clear();
}

void EvalDebug::itemClicked(const QModelIndex &index)
{
  if (index.column() == 2)
    detail->showAttackDefenseBonus(static_cast<osl::state::SimpleState>(model->getState()));
  else if (index.column() == 3)
    detail->showProgressBonus(*model);
  else if (index.column() == 4)
    detail->showProgressIndependentBonus(*model);
  else if (index.column() == 5)
    detail->showProgressDependentBonus(*model);
  else if (index.column() == 6)
    detail->showMinorPiecesBonus(*model);
  else
    detail->clear();
}

EvalDebugDialog::EvalDebugDialog(
  const osl::state::SimpleState &s,
  QWidget *parent, const char *name)
  : QDialog(parent, name)
{
  QVBoxLayout *layout = new QVBoxLayout(this);
  debug = new EvalDebug(osl::state::NumEffectState(s), this);
  layout->addWidget(debug);

  QPushButton *button = new QPushButton(this);
  button->setText("&OK");
  layout->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(accept()));
}

void EvalDebugDialog::setStatus(
  const osl::state::SimpleState &state,
  const osl::stl::vector<osl::Move> &,
  int, osl::Move)
{
  debug->setStatus(state);
}



class OpenMidEndingEvalDebugModel : public QAbstractItemModel
{
public:
  OpenMidEndingEvalDebugModel(const osl::state::NumEffectState &s,
		     QObject *parent = 0);
  int rowCount(const QModelIndex &parent = QModelIndex()) const;
  int columnCount(const QModelIndex &parent = QModelIndex()) const;
  QVariant data(const QModelIndex &index, int role) const;
  QVariant headerData(int section, Qt::Orientation orientation,
		      int role = Qt::DisplayRole) const;

  QModelIndex index(int row, int column,
		    const QModelIndex &parent = QModelIndex()) const {
    if (!parent.isValid() && row < rowCount() && column < columnCount())
      return createIndex(row, column);
    else
      return QModelIndex();
  }
  QModelIndex parent(const QModelIndex &/*index*/) const {
    return QModelIndex();
  }
  void setStatus(const osl::state::SimpleState &state);
  osl::eval::ml::OpenMidEndingEvalDebugInfo debugInfo() const {
    return debug_info;
  }
  const osl::state::NumEffectState &getState() const { return state; }
private:
  QStringList headers;
  osl::eval::ml::OpenMidEndingEvalDebugInfo debug_info;
  osl::state::NumEffectState state;
};

OpenMidEndingEvalDebugModel::OpenMidEndingEvalDebugModel(const osl::state::NumEffectState &s,
				       QObject *parent)
  : QAbstractItemModel(parent), state(s)
{
  osl::eval::ml::OpenMidEndingEval eval(state);
  debug_info = eval.debugInfo(s);
  headers << "Eval (normalized)" << "Opening" << "Endgame"
	  << "Progress Independent"
	  << "Progress"
	  << "Piece" << "Rook Mobility" << "Bishop Mobility"
	  << "Lance Mobility"
	  << "King25 Effect" << "Piece Pair"
	  << "King Piece O" << "King Piece E"
	  << "Piece Stand O" << "Piece Stand E"
	  << "Pawn Drop O" << "Pawn Drop E"
	  << "Ptype X O" << "Ptype X E"
	  << "Ptype Y O" << "Ptype Y E"
	  << "King25 Each O" << "King25 Each E"
	  << "No Pawn O" << "No Pawn E"
	  << "Gold Retreat O" << "Gold Retreat E"
	  << "Silver Retreat O" << "Silver Retreat E"
	  << "Knight Blocked O" << "Knight Blocked E"
	  << "King X Blocked O" << "King X Blocked E"
	  << "Rook Pawn O" << "Rook Pawn E"
	  << "Pin O" << "Pin E"
	  << "All Gold O" << "All Gold E"
	  << "Anaguma Empty O" << "Anaguma Empty E"
	  << "Stand Except Pawn O" << "Stand Except Pawn E";
}

int OpenMidEndingEvalDebugModel::rowCount(const QModelIndex &) const
{
  return 1;
}

int OpenMidEndingEvalDebugModel::columnCount(const QModelIndex &) const
{
  return headers.size();
}

QVariant OpenMidEndingEvalDebugModel::data(const QModelIndex &index, int role) const
{
  if (!index.isValid() || index.row() > rowCount() ||
      index.column() >  columnCount())
    return QVariant();

  if (role == Qt::DisplayRole)
  {
    if (index.column() == OpenMidEndingEvalDebugInfo::EVAL)
    {
      return debug_info.values[index.column()] / OpenMidEndingEval::seeScale();
    }
    else if (index.column() == OpenMidEndingEvalDebugInfo::OPENING)
    {
      return QString("%1 (%2)").arg(debug_info.values[index.column()] * (16 - debug_info.values[OpenMidEndingEvalDebugInfo::PROGRESS])).arg(debug_info.values[index.column()]);
    }
    else if (index.column() == OpenMidEndingEvalDebugInfo::ENDGAME)
    {
      return QString("%1 (%2)").arg(debug_info.values[index.column()] * debug_info.values[OpenMidEndingEvalDebugInfo::PROGRESS]).arg(debug_info.values[index.column()]);
    }
    else if (index.column() == OpenMidEndingEvalDebugInfo::PROGRESS_INDEPENDENT)
    {
      return QString("%1 (%2)").arg(debug_info.values[index.column()] * 16).arg(debug_info.values[index.column()]);
    }
    else
    {
      return debug_info.values[index.column()];
    }
  }
  else
  {
    return QVariant();
  }
}

QVariant OpenMidEndingEvalDebugModel::headerData(int section,
					Qt::Orientation /*orientation*/,
					int role) const
{
  if (role != Qt::DisplayRole || section > headers.size())
    return QVariant();

  return headers[section];
}

void OpenMidEndingEvalDebugModel::setStatus(const osl::state::SimpleState &s_state)
{
  state = osl::state::NumEffectState(s_state);
  osl::eval::ml::OpenMidEndingEval eval(state);
  debug_info = eval.debugInfo(state);
  emit dataChanged(createIndex(0, 0), createIndex(0, headers.size()));
}

void OpenMidEndingEvalDebug::setStatus(const osl::state::SimpleState &state)
{
  model->setStatus(state);
}

OpenMidEndingEvalDebug::OpenMidEndingEvalDebug(const osl::state::NumEffectState &s,
			     QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  model = new OpenMidEndingEvalDebugModel(s, this);
  QTreeView *view = new QTreeView(this);
  view->setModel(model);
  QHBoxLayout *layout = new QHBoxLayout(this);
  layout->addWidget(view);
}

OpenMidEndingEvalDebugDialog::OpenMidEndingEvalDebugDialog(
  const osl::state::SimpleState &s,
  QWidget *parent, const char *name)
  : QDialog(parent, name)
{
  QVBoxLayout *layout = new QVBoxLayout(this);
  debug = new OpenMidEndingEvalDebug(osl::state::NumEffectState(s), this);
  layout->addWidget(debug);

  QPushButton *button = new QPushButton(this);
  button->setText("&OK");
  layout->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(accept()));
}

void OpenMidEndingEvalDebugDialog::setStatus(
  const osl::state::SimpleState &state,
  const osl::stl::vector<osl::Move> &,
  int, osl::Move)
{
  debug->setStatus(state);
}

