#include "board.h"
#include "commands.h"
#ifndef MINIMAL_GPSSHELL
#  include "SReadline.h"
#endif
#include "book.h"

#include "osl/oslConfig.h"
#include "osl/eval/progressEval.h"
#include "osl/eval/ml/openMidEndingEval.h"
#include "osl/record/csa.h"
#include "osl/record/csaRecord.h"
#include "osl/record/record.h"
#include "gpsshogi/revision.h"

#ifndef MINIMAL_GPSSHELL
#  include <Poco/Net/HTTPStreamFactory.h>
#endif
#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>

#include <iostream>
#include <vector>
#include <string>
#include <sstream>
#include <cstdlib>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#ifndef MINIMAL_GPSSHELL
using namespace swift;
#endif
    

/*========== Global variables ==========*/

boost::filesystem::path cache_dir;
boost::scoped_ptr<gpsshell::Board> board;
boost::scoped_ptr<gpsshell::Book> the_book;
    

/*========== Classes and Funcitons ==========*/


class MyObserver : public osl::record::RecordVisitorObserver
{
  gpsshell::MySession& session;
public:
  MyObserver(gpsshell::MySession& _session)
    : session(_session)
  {}
  ~MyObserver() {}
  void update(osl::record::RecordVisitor* rv);
};

void MyObserver::update(osl::record::RecordVisitor* rv)
{
  const osl::record::MoveRecord *move_record = rv->getLastMove();
  const osl::Move& new_move = move_record->getMove();
  std::vector<std::string> params;
  params.push_back("move");
  params.push_back(osl::record::csa::show(new_move));
  session.move(params);
  std::cout << "time: " << move_record->getTime() << std::endl;
}


/*========== MAIN ==========*/

void printUsage(std::ostream& out, 
                char **argv,
                const boost::program_options::options_description& command_line_options)
{
  out << 
    "Usage: " << argv[0] << " [options]" << "\n"
      << command_line_options 
      << std::endl;
}

int parseCommandLine(int argc, char **argv, 
                     boost::program_options::variables_map& vm)
{
  namespace bp = boost::program_options;
  bp::options_description command_line_options;
  command_line_options.add_options()
    ("color",   bp::value<std::string>(),
                "Specify three comma-separated colors to show colorful pieces for BLACK, WHITE and the last move, respectively. eX. --color blue,brown,red")
    ("stdin",   bp::value<std::string>(),
                "Read a file from a specified type from the stdin.")
    ("version", "Show version information")
    ("help,h",  "Show help message");
  bp::positional_options_description p;

  try
  {
    bp::store(
      bp::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    bp::notify(vm);
    if (vm.count("help"))
    {
      printUsage(std::cout, argv, command_line_options);
      return 0;
    }
    if (vm.count("version")) 
    {
      std::string name = "gpsshell ";
#ifdef OSL_SMP
      name += "(smp) ";
#endif
      std::cout << name << gpsshogi::gpsshogi_revision << "\n\n"
        << gpsshogi::gpsshogi_copyright << "\n";
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << "\n"
	      << e.what() << std::endl;
    printUsage(std::cerr, argv, command_line_options);
    return 1;
  }

  if (vm.count("color"))
  {
    const std::string color = vm["color"].as<std::string>();
    typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
    boost::char_separator<char> sep(",");
    tokenizer tokens(color, sep);
    tokenizer::iterator each = tokens.begin();
    assert(each != tokens.end());
    board->setBlackColor(osl::record::Color::colorFor(*each++));
    assert(each != tokens.end());
    board->setWhiteColor(osl::record::Color::colorFor(*each++));
    assert(each != tokens.end());
    board->setLastMoveColor(osl::record::Color::colorFor(*each++));
    assert(each == tokens.end());
  }
  return 1;
}

void processGpsshellHome(boost::filesystem::path& readline_home)
{
  std::string GPSSHELL_HOME = "/tmp/gpsshell"; // default value
  if (getenv("HOME"))
    GPSSHELL_HOME = std::string(getenv("HOME")) + "/.gpsshell";
  const boost::filesystem::path home(GPSSHELL_HOME);
  if (!boost::filesystem::exists(home))
    boost::filesystem::create_directories(home);

  // re-create download cache 
  cache_dir = boost::filesystem::path(GPSSHELL_HOME) / "download_cache";
  if (boost::filesystem::exists(cache_dir))
    boost::filesystem::remove_all(cache_dir);
  boost::filesystem::create_directories(cache_dir);
  
  readline_home = home / "gpsshell";
}

#ifndef MINIMAL_GPSSHELL
template <class SReadline>
void setCommands(SReadline& reader, 
                 gpsshell::MySession *session)
{
  setCommandsMain(session);
  // Now register the completers.
  // Actually it is possible to re-register another set at any time
  reader.RegisterCompletions(session->getContainer());
}
#endif

int main(int argc, char **argv)
{
#ifndef MINIMAL_GPSSHELL
  Poco::Net::HTTPStreamFactory::registerFactory();
#endif
  osl::eval::ProgressEval::setUp();
  osl::eval::ml::OpenMidEndingEval::setUp();
  osl::progress::ml::NewProgress::setUp();
  board.reset(new gpsshell::Board());

  namespace bp = boost::program_options;
  bp::variables_map vm;
  if (!parseCommandLine(argc, argv, vm))
    exit(1);
  
  const static size_t max_stored_commands = 32;
  boost::filesystem::path readline_home;
  processGpsshellHome(readline_home);
  gpsshell::MySession session;
#ifndef MINIMAL_GPSSHELL
  SReadline reader(readline_home.native_file_string(), max_stored_commands);
  setCommands(reader, &session);
#else
  setCommandsMain(&session);
#endif

  if (vm.count("stdin"))
  {
    const std::string& file_type = vm["stdin"].as<std::string>();
    if ("csa" == file_type)
    {
      boost::shared_ptr<osl::record::RecordVisitor> rv(new osl::record::RecordVisitor());
      rv->addObserver(new MyObserver(session)); // ptr_vector
      osl::record::Record record;
      osl::record::csa::InputStream is(std::cin, rv);
      is.load(&record);
      return 0;
    }
    else
    {
      std::cerr << "Not supported file type: " << file_type << std::endl;
      return 1;
    }
  }

  const char *path = osl::OslConfig::openingBook(); // default 
  the_book.reset(new gpsshell::Book(path));
  board->showState();

  bool end_of_input = false; // true when we should exit
  std::vector<std::string> tokens; // List of the entered tokens
  while(true)
  {
    // We get the list of tokens
#ifdef MINIMAL_GPSSHELL
    std::cerr << "> " << std::flush;
    tokens.clear();
    std::string line, word;
    std::getline(cin, line);
    std::istringstream is(line);
    while (is >> word) 
      tokens.push_back(word);
#else
    reader.GetLine( "> ", tokens, end_of_input );
#endif

    if (end_of_input)
    {
      std::cout << "End of the session. Exiting." << std::endl;
      break;
    }
#ifndef MINIMAL_GPSSHELL
    if (tokens.empty())
    {
      // if no command, try the last command.
      std::vector<std::string> history;
      reader.GetHistory(history);
      std::cout << "history size " << history.size() << std::endl;
      if (!history.empty())
      {
        const std::string& last_command = history.back();
        SplitTokens(last_command, tokens);
      }
    }
#endif
    if (tokens.front() == "exit" || tokens.front() == "quit")
      break;

    session.executeCommand(tokens);
  } // while

  return 0;
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
