#include "kifuAnalyzer.h"
#include "osl/sennichite.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/search/simpleHashTable.h"
#include "gpsshogi/gui/util.h"

#if QT_VERSION >= 0x040000
#include <Q3ListView>
#else
#include <qlistview.h>
#define Q3ListView QListView
#define Q3ListViewItem QListViewItem
#endif

#include <qlayout.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qpainter.h>

#include <boost/scoped_ptr.hpp>

class EvaluationGraph : public QWidget
{
public:
  EvaluationGraph(QWidget *parent = 0, const char *name = 0);
  void setResult(const osl::stl::vector<Result>& r) {
    result = r;
  }
protected:
  void paintEvent(QPaintEvent *);
private:
  osl::stl::vector<Result> result;
};

EvaluationGraph::EvaluationGraph(QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  resize(640, 400);
}

void EvaluationGraph::paintEvent(QPaintEvent *)
{
  QPixmap pixmap(width(), height());

  QPainter painter(&pixmap);
  QBrush brush(QColor("white"));
  painter.fillRect(0, 0, width(), height(), brush);
  painter.setPen(QColor("grey"));
  painter.drawLine(0, 100, 640, 100);
  painter.drawLine(0, 200, 640, 200);
  painter.drawLine(0, 300, 640, 300);
  painter.setPen(QColor("black"));
  for (size_t i = 0; i < result.size() - 1; i++)
  {
    painter.drawLine(4 * i, 200 + result[i].value / 200,
		     4 * (i + 1), 200 + result[i + 1].value / 200);
  }

  bitBlt(this, 0, 0, &pixmap, 0, 0, width(), height());
}

class AnalyzeItem : public Q3ListViewItem
{
public:
  AnalyzeItem(Q3ListView *parent, int i, const Result& result);
  int compare(Q3ListViewItem *i, int col, bool ascending) const;
  int getNumber() const { return number; }
private:
  int number;
};

AnalyzeItem::AnalyzeItem(Q3ListView *parent, int n, const Result& result)
  : Q3ListViewItem(parent), number(n)
{
  int i = 0;
  setText(i++, QString("%1").arg(number));
  setText(i++, gpsshogi::gui::Util::moveToString(result.move));
  setText(i++, gpsshogi::gui::Util::moveToString(result.computed_move));
  setText(i++, QString("%1").arg(result.value));
  setText(i++, QString("%1").arg(result.depth));
}

int AnalyzeItem::compare(Q3ListViewItem *i, int, bool) const
{
  AnalyzeItem *item = (AnalyzeItem *)i;
  if (number == item->getNumber())
    return 0;
  else if (number < item->getNumber())
    return -1;
  else
    return 1;
}

KifuAnalyzer::KifuAnalyzer(const osl::state::SimpleState &state,
			   const osl::stl::vector<osl::Move> &moves,
			   QWidget *parent, const char *name)
  : QDialog(parent, name), initial_state(state), moves(moves)
{
  boost::scoped_ptr<osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer>
    player(new osl::game_playing::AlphaBeta2OpenMidEndingEvalPlayer());

  QTabWidget *tab = new QTabWidget(this);

  Q3ListView *list = new Q3ListView(tab);
  list->addColumn("");
  list->addColumn("Move");
  list->addColumn("Computed Move");
  list->addColumn("Value");
  list->addColumn("Depth");
  list->setColumnAlignment(0, Qt::AlignRight);
  list->setColumnAlignment(3, Qt::AlignRight);
  list->setColumnAlignment(4, Qt::AlignRight);
  EvaluationGraph *graph = new EvaluationGraph(tab);

  QVBoxLayout *layout = new QVBoxLayout(this);
  tab->addTab(graph, "Graph");
  tab->addTab(list, "List");
  layout->addWidget(tab);
  QPushButton *button = new QPushButton(this);
  button->setText("&OK");
  layout->addWidget(button);
  connect(button, SIGNAL(clicked()), this, SLOT(accept()));
  resize(640, 480);

  boost::scoped_ptr<osl::game_playing::GameState>
    game_state(new osl::game_playing::GameState(state));
  player->setDepthLimit(1200, 600, 200);

  for (size_t i = 0; i < moves.size(); i++)
  {
    player->selectBestMove(*game_state, 0, 0, 5);
    addResult(moves[i], *player, *game_state);
    game_state->pushMove(moves[i]);
  }

  graph->setResult(result);
  for (size_t i = 0; i < result.size(); i++)
  {
    new AnalyzeItem(list, i + 1, result[i]);
  }
}

void KifuAnalyzer::addResult(const osl::Move move,
			     const osl::game_playing::SearchPlayer& player,
			     const osl::game_playing::GameState& state)
{
  const osl::search::SimpleHashTable& table = *player.table();
  const osl::HashKey key(state.state());
  const osl::search::SimpleHashRecord *record = table.find(key);

  if (record)
  {
    result.push_back(Result(move, record->bestMove().move(),
			    record->lowerBound(), record->lowerLimit()));
  }
  else
    result.push_back(Result(move, osl::Move::INVALID(), 0, -1));
}
