/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "layout_util.h"

#include "bar_exif.h"
#include "bar_sort.h"
#include "cache_maint.h"
#include "collect.h"
#include "collect-dlg.h"
#include "dupe.h"
#include "editors.h"
#include "info.h"
#include "layout_image.h"
#include "preferences.h"
#include "utilops.h"
#include "ui_bookmark.h"
#include "ui_fileops.h"
#include "ui_menu.h"
#include "ui_tabcomp.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "icons/icon_thumb.xpm"
#include "icons/icon_float.xpm"

#include "icons/folder_close.xpm"
#include "icons/folder_open.xpm"
#include "icons/folder_deny.xpm"


#define MENU_EDIT_ACTION_OFFSET 16


/*
 *-----------------------------------------------------------------------------
 * keyboard handler
 *-----------------------------------------------------------------------------
 */

static guint tree_key_overrides[] = {
	GDK_Page_Up,	GDK_KP_Page_Up,
	GDK_Page_Down,	GDK_KP_Page_Down,
	GDK_Home,	GDK_KP_Home,
	GDK_End,	GDK_KP_End
};

static gint layout_key_match(guint keyval)
{
	gint i;

	for (i = 0; i < sizeof(tree_key_overrides) / sizeof(guint); i++)
		{
		if (keyval == tree_key_overrides[i]) return TRUE;
		}

	return FALSE;
}

static gint layout_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	LayoutWindow *lw = data;
	gint stop_signal = FALSE;
	gint x = 0;
	gint y = 0;

	if (lw->path_entry && GTK_WIDGET_HAS_FOCUS(lw->path_entry))
		{
		if (event->keyval == GDK_Escape && lw->path)
			{
			gtk_entry_set_text(GTK_ENTRY(lw->path_entry), lw->path);
			}

		/* the gtkaccelgroup of the window is stealing presses before they get to the entry (and more),
		 * so when the some widgets have focus, give them priority (HACK)
		 */
		if (gtk_widget_event(lw->path_entry, (GdkEvent *)event))
			{
			return TRUE;
			}
		}
	if (lw->vdt && GTK_WIDGET_HAS_FOCUS(lw->vdt->treeview) &&
	    !layout_key_match(event->keyval) &&
	    gtk_widget_event(lw->vdt->treeview, (GdkEvent *)event))
		{
		return TRUE;
		}

	if (lw->image && GTK_WIDGET_HAS_FOCUS(lw->image->widget))
		{
		switch (event->keyval)
			{
			case GDK_Left: case GDK_KP_Left:
				x -= 1;
				stop_signal = TRUE;
				break;
			case GDK_Right: case GDK_KP_Right:
				x += 1;
				stop_signal = TRUE;
				break;
			case GDK_Up: case GDK_KP_Up:
				y -= 1;
				stop_signal = TRUE;
				break;
			case GDK_Down: case GDK_KP_Down:
				y += 1;
				stop_signal = TRUE;
				break;
			case GDK_BackSpace:
			case 'B': case 'b':
				layout_image_prev(lw);
				stop_signal = TRUE;
				break;
			case GDK_space:
			case 'N': case 'n':
				layout_image_next(lw);
				stop_signal = TRUE;
				break;
			case GDK_Menu:
				layout_image_menu_popup(lw);
				stop_signal = TRUE;
				break;
			}
		}

	if (!stop_signal && !(event->state & GDK_CONTROL_MASK) )
	    switch (event->keyval)
		{
		case '+': case GDK_KP_Add:
			layout_image_zoom_adjust(lw, get_zoom_increment());
			stop_signal = TRUE;
			break;
		case GDK_KP_Subtract:
			layout_image_zoom_adjust(lw, -get_zoom_increment());
			stop_signal = TRUE;
			break;
		case GDK_KP_Multiply:
			layout_image_zoom_set(lw, 0.0);
			stop_signal = TRUE;
			break;
		case GDK_KP_Divide:
		case '1':
			layout_image_zoom_set(lw, 1.0);
			stop_signal = TRUE;
			break;
		case '2':
			layout_image_zoom_set(lw, 2.0);
			stop_signal = TRUE;
			break;
		case '3':
			layout_image_zoom_set(lw, 3.0);
			stop_signal = TRUE;
			break;
		case '4':
			layout_image_zoom_set(lw, 4.0);
			stop_signal = TRUE;
			break;
		case '7':
			layout_image_zoom_set(lw, -4.0);
			stop_signal = TRUE;
			break;
		case '8':
			layout_image_zoom_set(lw, -3.0);
			stop_signal = TRUE;
			break;
		case '9':
			layout_image_zoom_set(lw, -2.0);
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
			layout_image_prev(lw);
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
			layout_image_next(lw);
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			layout_image_first(lw);
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			layout_image_last(lw);
			stop_signal = TRUE;
			break;
		case GDK_Delete: case GDK_KP_Delete:
			if (enable_delete_key)
				{
				file_util_delete(NULL, layout_selection_list(lw));
				stop_signal = TRUE;
				}
			break;
		case GDK_Escape:
			/* FIXME:interrupting thumbs no longer allowed */
#if 0
			interrupt_thumbs();
#endif
			stop_signal = TRUE;
			break;
		case 'P': case 'p':
			layout_image_slideshow_pause_toggle(lw);
			break;
		case 'V': case 'v':
			layout_image_full_screen_toggle(lw);
			break;
		}

	if (event->state & GDK_SHIFT_MASK)
		{
		x *= 3;
		y *= 3;
		}

	if (x != 0 || y!= 0)
		{
		keyboard_scroll_calc(&x, &y, event);
		layout_image_scroll(lw, x, y);
		}

	if (stop_signal) g_signal_stop_emission_by_name(G_OBJECT(widget), "key_press_event");

	return stop_signal;
}

void layout_keyboard_init(LayoutWindow *lw)
{
	g_signal_connect(G_OBJECT(lw->window), "key_press_event",
			 G_CALLBACK(layout_key_press_cb), lw);
}

/*
 *-----------------------------------------------------------------------------
 * menu callbacks
 *-----------------------------------------------------------------------------
 */

static void layout_menu_new_window_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;
	LayoutWindow *nw;

	nw = layout_new(NULL, FALSE, FALSE);
	layout_sort_set(nw, file_sort_method, file_sort_ascending);
	layout_set_path(nw, layout_get_path(lw));
}

static void layout_menu_new_cb(gpointer data, guint action, GtkWidget *widget)
{
	collection_window_new(NULL);
}

static void layout_menu_open_cb(gpointer data, guint action, GtkWidget *widget)
{
	collection_dialog_load(NULL);
}

static void layout_menu_dupes_cb(gpointer data, guint action, GtkWidget *widget)
{
	dupe_window_new(DUPE_MATCH_NAME);
}

static void layout_menu_dir_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_create_dir(lw->path);
}

static void layout_menu_copy_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_copy(NULL, layout_selection_list(lw), NULL);
}

static void layout_menu_move_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_move(NULL, layout_selection_list(lw), NULL);
}

static void layout_menu_rename_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_rename(NULL, layout_selection_list(lw));
}

static void layout_menu_delete_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_delete(NULL, layout_selection_list(lw));
}

static void layout_menu_close_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_close(lw);
}

static void layout_menu_exit_cb(gpointer data, guint action, GtkWidget *widget)
{
	exit_gqview();
}

static void layout_menu_alter_90_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_90);
}

static void layout_menu_alter_90cc_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_90_CC);
}

static void layout_menu_alter_180_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_180);
}

static void layout_menu_alter_mirror_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_MIRROR);
}

static void layout_menu_alter_flip_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_FLIP);
}

static void layout_menu_info_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;
	GList *list;
	const gchar *path = NULL;

	list = layout_selection_list(lw);
	if (!list) path = layout_image_get_path(lw);

	layout_image_full_screen_stop(lw);
	info_window_new(path, list);
}

static void layout_menu_select_all_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_select_all(lw);
}

static void layout_menu_unselect_all_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_select_none(lw);
}

static void layout_menu_config_cb(gpointer data, guint action, GtkWidget *widget)
{
	show_config_window();
}

static void layout_menu_remove_thumb_cb(gpointer data, guint action, GtkWidget *widget)
{
	cache_maintain_home(FALSE);
}

static void layout_menu_wallpaper_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_to_root(lw);
}

static void layout_menu_zoom_in_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, get_zoom_increment());
}

static void layout_menu_zoom_out_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, -get_zoom_increment());
}

static void layout_menu_zoom_1_1_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 1.0);
}

static void layout_menu_zoom_fit_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 0.0);
}

static void layout_menu_thumb_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_thumb_set(lw, GTK_CHECK_MENU_ITEM(widget)->active);
}

static void layout_menu_list_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (!GTK_CHECK_MENU_ITEM(widget)->active) return;

	layout_views_set(lw, lw->tree_view, FALSE);
}

static void layout_menu_icon_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (!GTK_CHECK_MENU_ITEM(widget)->active) return;

	layout_views_set(lw, lw->tree_view, TRUE);
}

static void layout_menu_tree_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_views_set(lw, GTK_CHECK_MENU_ITEM(widget)->active, lw->icon_view);
}

static void layout_menu_full_screen_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_full_screen_start(lw);
}

static void layout_menu_refresh_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_refresh(lw);
}

static void layout_menu_float_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (lw->tools_float != GTK_CHECK_MENU_ITEM(widget)->active)
		{
		layout_tools_float_toggle(lw);
		}
}

static void layout_menu_hide_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_tools_hide_toggle(lw);
}

static void layout_menu_toolbar_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (lw->toolbar_hidden != GTK_CHECK_MENU_ITEM(widget)->active)
		{
		layout_toolbar_toggle(lw);
		}
}

static void layout_menu_bar_exif_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (lw->bar_exif_enabled != GTK_CHECK_MENU_ITEM(widget)->active)
		{
		layout_bar_exif_toggle(lw);
		}
}

static void layout_menu_bar_sort_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	if (lw->bar_sort_enabled != GTK_CHECK_MENU_ITEM(widget)->active)
		{
		layout_bar_sort_toggle(lw);
		}
}

static void layout_menu_slideshow_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_slideshow_toggle(lw);
}

static void layout_menu_help_cb(gpointer data, guint action, GtkWidget *widget)
{
	help_window_show("documentation");
}

static void layout_menu_notes_cb(gpointer data, guint action, GtkWidget *widget)
{
	help_window_show("release_notes");
}

static void layout_menu_about_cb(gpointer data, guint action, GtkWidget *widget)
{
	show_about_window();
}

/*
 *-----------------------------------------------------------------------------
 * edit menu
 *-----------------------------------------------------------------------------
 */ 

static void layout_menu_edit_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;
	GList *list;
	gint n;

	n = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "edit_index"));

	list = layout_selection_list(lw);
	start_editor_from_path_list(n, list);
	path_list_free(list);
}

static void layout_menu_edit_update(LayoutWindow *lw)
{
	gint i;

	/* main edit menu */

	for(i = 0; i < 10; i++)
		{
		GtkWidget *item;

		item = gtk_item_factory_get_item_by_action(lw->menu_fact, MENU_EDIT_ACTION_OFFSET + i);
		g_object_set_data(G_OBJECT(item), "edit_index", GINT_TO_POINTER(i));

		if (editor_command[i] && strlen(editor_command[i]) > 0)
			{
			gchar *text;
			if (editor_name[i] && strlen(editor_name[i]) > 0)
				{
				text = g_strdup_printf(_("in %s..."), editor_name[i]);
				}
			else
				{
				text = g_strdup(_("in (unknown)..."));
				}
			gtk_label_set_text(GTK_LABEL(GTK_BIN(item)->child), text);
			g_free(text);
			gtk_widget_show(item);
			}
		else
			{
			gtk_widget_hide(item);
			}
		}
}

void layout_edit_update_all(void)
{
	GList *work;

	work = layout_window_list;
	while (work)
		{
		LayoutWindow *lw = work->data;
		work = work->next;

		layout_menu_edit_update(lw);
		}
}

/*
 *-----------------------------------------------------------------------------
 * recent menu
 *-----------------------------------------------------------------------------
 */

static void layout_menu_recent_cb(GtkWidget *widget, gpointer data)
{
	gint n;
	gchar *path;

	n = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "recent_index"));

	path = g_list_nth_data(history_list_get_by_key("recent"), n);

	if (!path) return;

	/* make a copy of it */
	path = g_strdup(path);
	collection_window_new(path);
	g_free(path);
}

static void layout_menu_recent_update(LayoutWindow *lw)
{
	GtkWidget *menu;
	GtkWidget *recent;
	GtkWidget *item;
	GList *list;
	gint n;

	list = history_list_get_by_key("recent");
	n = 0;

	menu = gtk_menu_new();

	while (list)
		{
		item = menu_item_add_simple(menu, filename_from_path((gchar *)list->data),
					    G_CALLBACK(layout_menu_recent_cb), lw);
		g_object_set_data(G_OBJECT(item), "recent_index", GINT_TO_POINTER(n));
		list = list->next;
		n++;
		}

	if (n == 0)
		{
		menu_item_add(menu, _("Empty"), NULL, NULL);
		}

	recent = gtk_item_factory_get_item(lw->menu_fact, "/File/Open recent");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(recent), menu);
	gtk_widget_set_sensitive(recent, (n != 0));
}

void layout_recent_update_all(void)
{
	GList *work;

	work = layout_window_list;
	while (work)
		{
		LayoutWindow *lw = work->data;
		work = work->next;

		layout_menu_recent_update(lw);
		}
}

void layout_recent_add_path(const gchar *path)
{
	if (!path) return;

	history_list_add_to_key("recent", path, recent_list_max);

	layout_recent_update_all();
}

/*
 *-----------------------------------------------------------------------------
 * menu
 *-----------------------------------------------------------------------------
 */ 

static GtkItemFactoryEntry gqview_menu_items[] =
{
	{ N_("/_File"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/File/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/File/New _window"),		NULL, 		layout_menu_new_window_cb,	0, "<StockItem>", GTK_STOCK_NEW },
	{ N_("/File/_New collection"),		"C",		layout_menu_new_cb,		0, "<StockItem>", GTK_STOCK_INDEX },
	{ N_("/File/_Open collection..."),	"O",		layout_menu_open_cb,		0, "<StockItem>", GTK_STOCK_OPEN },
	{ N_("/File/Open _recent"),		NULL,		NULL,				0, NULL },
	{ N_("/File/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/_Find duplicates..."),	"D",		layout_menu_dupes_cb,		0, "<StockItem>", GTK_STOCK_FIND },
	{ N_("/File/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/N_ew folder..."),		"<control>F",	layout_menu_dir_cb,		0, NULL },
	{ N_("/File/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/_Copy..."),			"<control>C",	layout_menu_copy_cb,		0, "<StockItem>", GTK_STOCK_COPY },
	{ N_("/File/_Move..."),			"<control>M",	layout_menu_move_cb,		0, NULL },
	{ N_("/File/_Rename..."),		"<control>R",	layout_menu_rename_cb,		0, NULL },
	{ N_("/File/_Delete..."),		"<control>D",	layout_menu_delete_cb,		0, "<StockItem>", GTK_STOCK_DELETE },
	{ N_("/File/sep4"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/C_lose window"),		"<control>W",	layout_menu_close_cb,		0, "<StockItem>", GTK_STOCK_CLOSE },
	{ N_("/File/E_xit"),			"<control>Q",	layout_menu_exit_cb,		0, "<StockItem>", GTK_STOCK_QUIT },

	{ N_("/_Edit"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/Edit/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Edit/editor1"),			"<control>1",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 0, NULL },
	{ N_("/Edit/editor2"),			"<control>2",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 1, NULL },
	{ N_("/Edit/editor3"),			"<control>3",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 2, NULL },
	{ N_("/Edit/editor4"),			"<control>4",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 3, NULL },
	{ N_("/Edit/editor5"),			"<control>5",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 4, NULL },
	{ N_("/Edit/editor6"),			"<control>6",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 5, NULL },
	{ N_("/Edit/editor7"),			"<control>7",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 6, NULL },
	{ N_("/Edit/editor8"),			"<control>8",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 7, NULL },
	{ N_("/Edit/editor9"),			"<control>9",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 8, NULL },
	{ N_("/Edit/editor0"),			"<control>0",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 9, NULL},
	{ N_("/Edit/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/_Adjust"),			NULL,		NULL,				0, "<Branch>" },
	{ N_("/Edit/_Properties"),		"<control>P",	layout_menu_info_cb,		0, "<StockItem>", GTK_STOCK_PROPERTIES },
	{ N_("/Edit/Adjust/tear1"),		NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Edit/Adjust/_Rotate clockwise"),		"bracketright",	layout_menu_alter_90_cb,	0, NULL },
	{ N_("/Edit/Adjust/Rotate _counterclockwise"),	"bracketleft",	layout_menu_alter_90cc_cb,	0, NULL },
	{ N_("/Edit/Adjust/Rotate 1_80"),	"<shift>R",	layout_menu_alter_180_cb,	0, NULL },
	{ N_("/Edit/Adjust/_Mirror"),		"<shift>M",	layout_menu_alter_mirror_cb,	0, NULL },
	{ N_("/Edit/Adjust/_Flip"),		"<shift>F",	layout_menu_alter_flip_cb,	0, NULL },
	{ N_("/Edit/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/Select _all"),		"<control>A",	layout_menu_select_all_cb,	0, NULL },
	{ N_("/Edit/Select _none"),	"<control><shift>A",	layout_menu_unselect_all_cb,	0, NULL },
	{ N_("/Edit/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/_Options..."),		"<control>O",	layout_menu_config_cb,		0, "<StockItem>", GTK_STOCK_PREFERENCES },
	{ N_("/Edit/_Remove old thumbnails"),	NULL,		layout_menu_remove_thumb_cb,	0, NULL },
	{ N_("/Edit/sep4"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/Set as _wallpaper"),	NULL,		layout_menu_wallpaper_cb,	0, NULL },

	{ N_("/_View"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/View/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/View/Zoom _in"),			"equal",	layout_menu_zoom_in_cb,		0, "<StockItem>", GTK_STOCK_ZOOM_IN },
	{ N_("/View/Zoom _out"),		"minus",	layout_menu_zoom_out_cb,	0, "<StockItem>", GTK_STOCK_ZOOM_OUT },
	{ N_("/View/Zoom _1:1"),		"Z",		layout_menu_zoom_1_1_cb,	0, "<StockItem>", GTK_STOCK_ZOOM_100 },
	{ N_("/View/_Zoom to fit"),		"X",		layout_menu_zoom_fit_cb,	0, "<StockItem>", GTK_STOCK_ZOOM_FIT },
	{ N_("/View/sep1"),			NULL,		NULL,				0, "<Separator>" },

	{ N_("/View/_Thumbnails"),		"T",		layout_menu_thumb_cb,		0, "<CheckItem>" },
	{ N_("/View/_List"),			"<control>L",	layout_menu_list_cb,		0, "<RadioItem>" },
	{ N_("/View/I_cons"),			"<control>I",	layout_menu_icon_cb,		0, "/View/List" },

	{ N_("/View/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/Tr_ee"),			"<control>T",	layout_menu_tree_cb,		0, "<CheckItem>" },
	{ N_("/View/F_ull screen"),		"F",		layout_menu_full_screen_cb,	0, NULL },

	{ N_("/View/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/_Float file list"),		"L",		layout_menu_float_cb,		0, "<CheckItem>" },
	{ N_("/View/_Hide file list"),		"H",		layout_menu_hide_cb,		0, NULL },
	{ N_("/View/Hide tool_bar"),		NULL,		layout_menu_toolbar_cb,		0, "<CheckItem>" },
	{ N_("/View/sep4"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/E_xif data"),		"<control>E",	layout_menu_bar_exif_cb,	0, "<CheckItem>" },
	{ N_("/View/Sort _manager"),		"<control>S",	layout_menu_bar_sort_cb,	0, "<CheckItem>" },
	{ N_("/View/sep5"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/Toggle _slideshow"),	"S",		layout_menu_slideshow_cb,	0, NULL },
	{ N_("/View/_Refresh Lists"),		"R",		layout_menu_refresh_cb,		0, "<StockItem>", GTK_STOCK_REFRESH },

	{ N_("/_Help"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/Help/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Help/_Keyboard shortcuts"),	"F1",		layout_menu_help_cb,		0, "<StockItem>", GTK_STOCK_HELP },
	{ N_("/Help/_Release notes"),		NULL,		layout_menu_notes_cb,		0, NULL},
	{ N_("/Help/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Help/_About"),			NULL,		layout_menu_about_cb,		0, NULL },
};

static int gqview_nmenu_items = sizeof (gqview_menu_items) / sizeof (gqview_menu_items[0]);

static gchar *menu_translate(const gchar *path, gpointer data)
{
	return _(path);
}

GtkWidget *layout_menu_bar(LayoutWindow *lw)
{
	lw->menu_fact = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", lw->accel_grp);
	gtk_item_factory_set_translate_func(lw->menu_fact, menu_translate, NULL, NULL);

	gtk_item_factory_create_items(lw->menu_fact, gqview_nmenu_items, gqview_menu_items, lw);

	return gtk_item_factory_get_widget(lw->menu_fact, "<main>");
}

/*
 *-----------------------------------------------------------------------------
 * toolbar
 *-----------------------------------------------------------------------------
 */ 

static void layout_button_thumb_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_thumb_set(lw, GTK_TOGGLE_BUTTON(widget)->active);
}

static void layout_button_home_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;
	const gchar *path = homedir();

	if (path) layout_set_path(lw, path);
}

static void layout_button_refresh_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_refresh(lw);
}

static void layout_button_zoom_in_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, get_zoom_increment());
}

static void layout_button_zoom_out_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, -get_zoom_increment());
}

static void layout_button_zoom_fit_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 0.0);
}

static void layout_button_zoom_1_1_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 1.0);
}

static void layout_button_config_cb(GtkWidget *widget, gpointer data)
{
	show_config_window();
}

static void layout_button_float_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_tools_float_toggle(lw);
}

GtkWidget *layout_button(GtkWidget *box, gchar **pixmap_data, const gchar *stock_id, gint toggle,
			 GtkTooltips *tooltips, const gchar *tip_text,
			 GCallback func, gpointer data)
{
	GtkWidget *button;
	GtkWidget *icon;

	if (toggle)
		{
		button = gtk_toggle_button_new();
		}
	else
		{
		button = gtk_button_new();
		}

	g_signal_connect(G_OBJECT(button), "clicked", func, data);
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
	gtk_widget_show(button);
	gtk_tooltips_set_tip(tooltips, button, tip_text, NULL);

	if (stock_id)
		{
		icon = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_BUTTON);
		}
	else
		{
		GdkPixbuf *pixbuf;

		pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)pixmap_data);
		icon = gtk_image_new_from_pixbuf(pixbuf);
		gdk_pixbuf_unref(pixbuf);
		}

	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_widget_show(icon);

	return button;
}

GtkWidget *layout_button_bar(LayoutWindow *lw)
{
	GtkWidget *box;
	GtkTooltips *tooltips;

	tooltips = lw->tooltips;

	box = gtk_hbox_new(FALSE, 0);

	lw->thumb_button = layout_button(box, (gchar **)icon_thumb_xpm, NULL, TRUE,
		      tooltips, _("Show thumbnails"), G_CALLBACK(layout_button_thumb_cb), lw);
	layout_button(box, NULL, GTK_STOCK_HOME, FALSE,
		      tooltips, _("Change to home directory"), G_CALLBACK(layout_button_home_cb), lw);
	layout_button(box, NULL, GTK_STOCK_REFRESH, FALSE,
		      tooltips, _("Refresh file list"), G_CALLBACK(layout_button_refresh_cb), lw);
	layout_button(box, NULL, GTK_STOCK_ZOOM_IN, FALSE,
		      tooltips, _("Zoom in"), G_CALLBACK(layout_button_zoom_in_cb), lw);
	layout_button(box, NULL, GTK_STOCK_ZOOM_OUT, FALSE,
		      tooltips, _("Zoom out"), G_CALLBACK(layout_button_zoom_out_cb), lw);
	layout_button(box, NULL, GTK_STOCK_ZOOM_FIT, FALSE,
		      tooltips, _("Fit image to window"), G_CALLBACK(layout_button_zoom_fit_cb), lw);
	layout_button(box, NULL, GTK_STOCK_ZOOM_100, FALSE,
		      tooltips, _("Set zoom 1:1"), G_CALLBACK(layout_button_zoom_1_1_cb), lw);
	layout_button(box, NULL, GTK_STOCK_PREFERENCES, FALSE,
		      tooltips, _("Configure options"), G_CALLBACK(layout_button_config_cb), lw);
	layout_button(box, (gchar **)icon_float_xpm, NULL, FALSE,
		      tooltips, _("Float Controls"), G_CALLBACK(layout_button_float_cb), lw);

	return box;
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

static void layout_util_sync_views(LayoutWindow *lw)
{
	GtkWidget *item;

	if (!lw->menu_fact) return;

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Tree");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->tree_view);

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Icons");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->icon_view);

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Float file list");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->tools_float);

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Exif data");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->bar_exif_enabled);

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Sort manager");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->bar_sort_enabled);

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Hide toolbar");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), lw->toolbar_hidden);
}

void layout_util_sync_thumb(LayoutWindow *lw)
{
	GtkWidget *item;

	if (lw->icon_view)
		{
		item = gtk_item_factory_get_item(lw->menu_fact, "/View/Icons");
		}
	else
		{
		item = gtk_item_factory_get_item(lw->menu_fact, "/View/List");
		}

	if (!GTK_CHECK_MENU_ITEM(item)->active)
		{
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		}

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Thumbnails");
	GTK_CHECK_MENU_ITEM(item)->active = lw->thumbs_enabled;
	gtk_widget_set_sensitive(item, !lw->icon_view);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lw->thumb_button), lw->thumbs_enabled);
	gtk_widget_set_sensitive(lw->thumb_button, !lw->icon_view);
}

void layout_util_sync(LayoutWindow *lw)
{
	layout_util_sync_views(lw);
	layout_util_sync_thumb(lw);
	layout_menu_recent_update(lw);
	layout_menu_edit_update(lw);
}

/*
 *-----------------------------------------------------------------------------
 * icons (since all the toolbar icons are included here, best place as any)
 *-----------------------------------------------------------------------------
 */

PixmapFolders *folder_icons_new(void)
{
	PixmapFolders *pf;

	pf = g_new0(PixmapFolders, 1);

	pf->close = gdk_pixbuf_new_from_xpm_data((const char **)folder_close_xpm);
	pf->open = gdk_pixbuf_new_from_xpm_data((const char **)folder_open_xpm);
	pf->deny = gdk_pixbuf_new_from_xpm_data((const char **)folder_deny_xpm);

	return pf;
}

void folder_icons_free(PixmapFolders *pf)
{
	if (!pf) return;

	g_object_unref(pf->close);
	g_object_unref(pf->open);
	g_object_unref(pf->deny);

	g_free(pf);
}

/*
 *-----------------------------------------------------------------------------
 * sidebars
 *-----------------------------------------------------------------------------
 */

#define SIDEBAR_WIDTH 288

static void layout_bar_exif_destroyed(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	if (lw->bar_exif)
		{
		lw->bar_exif_advanced = bar_exif_is_advanced(lw->bar_exif);
		}

	lw->bar_exif = NULL;
	if (lw->utility_box)
		{
		/* destroyed from within itself */
		lw->bar_exif_enabled = FALSE;
		layout_util_sync_views(lw);
		}
}

static void layout_bar_exif_sized(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	LayoutWindow *lw = data;

	if (lw->bar_exif)
		{
		lw->bar_exif_size = allocation->width;
		}
}

void layout_bar_exif_new(LayoutWindow *lw)
{
	if (!lw->utility_box) return;

	lw->bar_exif = bar_exif_new(TRUE, layout_image_get_path(lw), lw->bar_exif_advanced);
	g_signal_connect(G_OBJECT(lw->bar_exif), "destroy",
			 G_CALLBACK(layout_bar_exif_destroyed), lw);
	g_signal_connect(G_OBJECT(lw->bar_exif), "size_allocate",
			 G_CALLBACK(layout_bar_exif_sized), lw);
        lw->bar_exif_enabled = TRUE;

	if (lw->bar_exif_size < 1) lw->bar_exif_size = SIDEBAR_WIDTH;
	gtk_widget_set_size_request(lw->bar_exif, lw->bar_exif_size, -1);
	gtk_box_pack_start(GTK_BOX(lw->utility_box), lw->bar_exif, FALSE, FALSE, 0);
	gtk_widget_show(lw->bar_exif);
}

void layout_bar_exif_close(LayoutWindow *lw)
{
	if (lw->bar_exif)
		{
		bar_exif_close(lw->bar_exif);
		lw->bar_exif = NULL;
		}
        lw->bar_exif_enabled = FALSE;
}

void layout_bar_exif_toggle(LayoutWindow *lw)
{
	if (lw->bar_exif_enabled)
		{
		layout_bar_exif_close(lw);
		}
	else
		{
		layout_bar_exif_new(lw);
		}
}

static void layout_bar_exif_new_image(LayoutWindow *lw)
{
	if (!lw->bar_exif || !lw->bar_exif_enabled) return;

	bar_exif_set(lw->bar_exif, layout_image_get_path(lw));
}

static void layout_bar_sort_destroyed(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	lw->bar_sort = NULL;

	if (lw->utility_box)
		{
		/* destroyed from within itself */
		lw->bar_sort_enabled = FALSE;

		layout_util_sync_views(lw);
		}
}

void layout_bar_sort_new(LayoutWindow *lw)
{
	if (!lw->utility_box) return;

	lw->bar_sort = bar_sort_new(lw);
	g_signal_connect(G_OBJECT(lw->bar_sort), "destroy",
			 G_CALLBACK(layout_bar_sort_destroyed), lw);
	lw->bar_sort_enabled = TRUE;

	gtk_box_pack_end(GTK_BOX(lw->utility_box), lw->bar_sort, FALSE, FALSE, 0);
	gtk_widget_show(lw->bar_sort);
}

void layout_bar_sort_close(LayoutWindow *lw)
{
	if (lw->bar_sort)
		{
		bar_sort_close(lw->bar_sort);
		lw->bar_sort = NULL;
		}
	lw->bar_sort_enabled = FALSE;
}

void layout_bar_sort_toggle(LayoutWindow *lw)
{
	if (lw->bar_sort_enabled)
		{
		layout_bar_sort_close(lw);
		}
	else
		{
		layout_bar_sort_new(lw);
		}
}

void layout_bars_new_image(LayoutWindow *lw)
{
	layout_bar_exif_new_image(lw);
}

GtkWidget *layout_bars_prepare(LayoutWindow *lw, GtkWidget *image)
{
	lw->utility_box = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(lw->utility_box), image, TRUE, TRUE, 0);
	gtk_widget_show(image);

	if (lw->bar_sort_enabled)
		{
		layout_bar_sort_new(lw);
		}

	if (lw->bar_exif_enabled)
		{
		layout_bar_exif_new(lw);
		}

	return lw->utility_box;
}

void layout_bars_close(LayoutWindow *lw)
{
	layout_bar_sort_close(lw);
	layout_bar_exif_close(lw);
}
