/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "bar_sort.h"

#include "layout.h"
#include "layout_image.h"
#include "utilops.h"
#include "ui_bookmark.h"
#include "ui_fileops.h"


/*
  *-------------------------------------------------------------------
  * sort bar
  *-------------------------------------------------------------------
  */


typedef struct _SortData SortData;
struct _SortData
{
	GtkWidget *vbox;
	GtkWidget *bookmarks;
	LayoutWindow *lw;

	FileDialog *dialog;
	GtkWidget *dialog_name_entry;

	gint move;

	GtkWidget *undo_button;
	gint undo_move;
	gchar *undo_src;
	gchar *undo_dest;
};


static gint bar_sort_move_state = FALSE;


static void bar_sort_undo_set(SortData *sd, const gchar *src, const gchar *dest)
{
	g_free(sd->undo_src);
	sd->undo_src = g_strdup(src);
	g_free(sd->undo_dest);
	sd->undo_dest = g_strdup(dest);

	sd->undo_move = sd->move;

	if (sd->undo_button)
		{
		gtk_widget_set_sensitive(sd->undo_button,
					 (sd->undo_src != NULL && sd->undo_dest != NULL) );
		}
}

static void bar_sort_undo_cb(GtkWidget *button, gpointer data)
{
	SortData *sd = data;

	if (!sd->undo_src || !sd->undo_dest) return;

	if (sd->undo_move)
		{
		GList *list;
		gchar *src_dir;

		list = g_list_append(NULL, g_strdup(sd->undo_dest));
		src_dir = remove_level_from_path(sd->undo_src);
		file_util_move_simple(list, src_dir);
		g_free(src_dir);
		}
	else
		{
		file_util_delete(sd->undo_dest, NULL);
		}

	layout_refresh(sd->lw);

	if (isfile(sd->undo_src))
		{
		layout_image_set_path(sd->lw, sd->undo_src);
		}

	bar_sort_undo_set(sd, NULL, NULL);
}

static void bar_sort_bookmark_select(const gchar *path, gpointer data)
{
	SortData *sd = data;
	GList *list;
	const gchar *source_path;
	gchar *dest_path;

	if (!isdir(path)) return;
	
	source_path = layout_image_get_path(sd->lw);

	if (!source_path) return;

	dest_path = concat_dir_and_file(path, filename_from_path(source_path));
	bar_sort_undo_set(sd, source_path, dest_path);
	g_free(dest_path);
	
	list = g_list_append(NULL, g_strdup(source_path));

	if (sd->move)
		{
		file_util_move_simple(list, path);
		}
	else
		{
		file_util_copy_simple(list, path);
		layout_image_next(sd->lw);
		}
}

static void bar_sort_set_copy_cb(GtkWidget *button, gpointer data)
{
	SortData *sd = data;

	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) return;
	sd->move = FALSE;
}

static void bar_sort_set_move_cb(GtkWidget *button, gpointer data)
{
	SortData *sd = data;
	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) return;
	sd->move = TRUE;
}

static void bar_sort_add_close(SortData *sd)
{
	if (sd->dialog) file_dialog_close(sd->dialog);
	sd->dialog_name_entry = NULL;
	sd->dialog = NULL;
}

static void bar_sort_add_ok_cb(FileDialog *fd, gpointer data)
{
	SortData *sd = data;
	const gchar *name;

	name = gtk_entry_get_text(GTK_ENTRY(sd->dialog_name_entry));
	if (strlen(name) == 0)
		{
		name = filename_from_path(fd->dest_path);
		}

	bookmark_list_add(sd->bookmarks, name, fd->dest_path);

	bar_sort_add_close(sd);
}

static void bar_sort_add_cancel_cb(FileDialog *fd, gpointer data)
{
	SortData *sd = data;

	bar_sort_add_close(sd);
}

static void bar_sort_add_cb(GtkWidget *button, gpointer data)
{
	SortData *sd = data;
	GtkWidget *hbox;
	GtkWidget *label;

	if (sd->dialog)
		{
		gdk_window_raise(GENERIC_DIALOG(sd->dialog)->dialog->window);
		return;
		}

	sd->dialog = file_util_file_dlg(_("Add Bookmark"), NULL,
				       "gqview", "add_bookmark",
				       bar_sort_add_cancel_cb, sd);
	file_dialog_add_button_stock(sd->dialog, NULL, GTK_STOCK_OK, bar_sort_add_ok_cb, TRUE);

	file_dialog_add_path_widgets(sd->dialog, NULL, NULL, "add_bookmark", NULL, NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(sd->dialog)->vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Name"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	sd->dialog_name_entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), sd->dialog_name_entry, TRUE, TRUE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(sd->dialog), sd->dialog_name_entry);
	gtk_widget_show(sd->dialog_name_entry);

	gtk_widget_show(GENERIC_DIALOG(sd->dialog)->dialog);
}

void bar_sort_close(GtkWidget *bar)
{
	SortData *sd;

	sd = g_object_get_data(G_OBJECT(bar), "bar_sort_data");
	if (!sd) return;

	gtk_widget_destroy(sd->vbox);
}

static void bar_sort_destroy(GtkWidget *widget, gpointer data)
{
	SortData *sd = data;

	bar_sort_add_close(sd);
	bar_sort_move_state = sd->move;

	g_free(sd->undo_src);
	g_free(sd->undo_dest);
	g_free(sd);
}

GtkWidget *bar_sort_new(LayoutWindow *lw)
{
	SortData *sd;
	GtkWidget *button;
	GtkWidget *buttongrp;
	GtkWidget *label;
	GtkWidget *tbar;

	if (!lw) return NULL;

	sd = g_new0(SortData, 1);

	sd->lw = lw;
	sd->move = bar_sort_move_state;
	sd->undo_src = NULL;
	sd->undo_dest = NULL;

	sd->vbox = gtk_vbox_new(FALSE, 0);
	g_object_set_data(G_OBJECT(sd->vbox), "bar_sort_data", sd);
	g_signal_connect(G_OBJECT(sd->vbox), "destroy",
			 G_CALLBACK(bar_sort_destroy), sd);

	label = gtk_label_new(_("Sort Manager"));
	gtk_box_pack_start(GTK_BOX(sd->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	sd->bookmarks = bookmark_list_new("sort_manager", bar_sort_bookmark_select, sd);
	gtk_box_pack_start(GTK_BOX(sd->vbox), sd->bookmarks, TRUE, TRUE, 0);
	gtk_widget_show(sd->bookmarks);

	buttongrp = gtk_radio_button_new_with_label(NULL, _("Copy"));
	if (!sd->move) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(buttongrp), TRUE);
	g_signal_connect(G_OBJECT(buttongrp), "clicked",
			 G_CALLBACK(bar_sort_set_copy_cb), sd);
	gtk_box_pack_start(GTK_BOX(sd->vbox), buttongrp, FALSE, FALSE, 0);
	gtk_widget_show(buttongrp);

	button = gtk_radio_button_new_with_label(gtk_radio_button_get_group(GTK_RADIO_BUTTON(buttongrp)), _("Move"));
	if (sd->move) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(bar_sort_set_move_cb), sd);
	gtk_box_pack_start(GTK_BOX(sd->vbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	tbar = gtk_toolbar_new();
	gtk_toolbar_set_style(GTK_TOOLBAR(tbar), GTK_TOOLBAR_BOTH_HORIZ);
	gtk_box_pack_start(GTK_BOX(sd->vbox), tbar, FALSE, FALSE, 0);
	gtk_widget_show(tbar);

	button = gtk_toolbar_insert_stock(GTK_TOOLBAR(tbar),
					  GTK_STOCK_ADD, _("Add Bookmark"), NULL,
					  G_CALLBACK(bar_sort_add_cb), sd, -1);

	sd->undo_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(tbar),
						   GTK_STOCK_UNDO, _("Undo last image"), NULL,
						   G_CALLBACK(bar_sort_undo_cb), sd, -1);

	bar_sort_undo_set(sd, NULL, NULL);
	
	return sd->vbox;
}

