/*  Copyright (C) 1988-2015 by the Institute of Global Environment and Society (IGES).  
    See file COPYRIGHT for more information.   */

/* Routines related to hardcopy (metafile) output. */

#ifdef HAVE_CONFIG_H
#include "config.h"

/* If autoconfed, only include malloc.h when it's present */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#else /* undef HAVE_CONFIG_H */

#include <malloc.h>

#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "gatypes.h"
#include "gx.h"

void *galloc(size_t,char *);

/* Struct to form linked list for the meta buffer */
/* The buffer area is allocated as float, to insure at least four bytes
   per element.  In some cases, ints and chars will get stuffed into a 
   float (via pointer casting).  */

/* Don't use gafloat or gaint for meta buffer stuffing.  */

struct gxmbuf {
  struct gxmbuf *fpmbuf;         /* Forward pointer */
  float *buff;                   /* Buffer area */
  gaint len;                     /* Length of Buffer area */
  gaint used;                    /* Amount of buffer used */
};

/* Buffer chain anchor here; also a convenience pointer to the last buffer
   in the chain.  Times 2, for double buffering.  mbufanch and mbuflast 
   always point to the buffer currently being added to.  In double buffering
   mode, that will be the background buffer (and mbufanch2 points to the 
   buffer representing the currently display image). */

static struct gxmbuf *mbufanch=NULL;  /* Buffer Anchor */
static struct gxmbuf *mbuflast=NULL;  /* Last struct in chain */

static struct gxmbuf *mbufanch2=NULL;  /* Buffer Anchor */
static struct gxmbuf *mbuflast2=NULL;  /* Last struct in chain */

static gaint dbmode;       /* double buffering flag */

#define BWORKSZ 250000

static gaint mbuferror = 0;    /* Indicate an error state; suspends buffering */

/* Initialize any buffering, etc. when GrADS starts up */

void gxhnew (gadouble xsiz, gadouble ysiz, gaint hbufsz) {
gaint rc;
  mbufanch = NULL;
  mbuflast = NULL;
  mbuferror = 0;
  if (sizeof(int) > sizeof(float)) {
    printf ("Error in gx initialization: Incompatable int and float sizes\n");
    mbuferror = 99;
    return;
  }
  rc = mbufget();
  if (rc) {
    printf ("Error in gx initialization: Unable to allocate meta buffer\n");
    mbuferror = 99;
  }
}


/* Add command with 0 args to metafile buffer */

void hout0 (gaint cmd) {
gaint rc;
signed char *ch;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <3) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
}

/* Add a command with one small integer argument to the metafile buffer.
   The argument is assumed to fit into a signed char (-127 to 128).  */

void hout1c (gaint cmd, gaint opt) {
gaint rc;
signed char *ch;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <4) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  *(ch+2) = (signed char)opt;
  mbuflast->used++;
}

/* Add command with one integer argument to metafile buffer */

void hout1 (gaint cmd, gaint opt) {
gaint rc;
signed char *ch;
int *iii;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <4) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)opt;
  mbuflast->used++;
}

/* Metafile buffer, command plus two double args */

void hout2 (gaint cmd, gadouble x, gadouble y) {
gaint rc;
signed char *ch;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <5) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)x;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)y;
  mbuflast->used++;
}

/* Metafile buffer, command plus two integer args */

void hout2i (gaint cmd, gaint i1, gaint i2) {
gaint rc;
signed char *ch;
int *iii;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <5) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i1;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i2;
  mbuflast->used++;
}

/* Metafile buffer, command plus three integer args */

void hout3i (gaint cmd, gaint i1, gaint i2, gaint i3) {
gaint rc;
signed char *ch;
int *iii;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <6) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i1;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i2;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i3;
  mbuflast->used++;
}

/* Metafile buffer, command plus four integer args */

void hout5i (gaint cmd, gaint i1, gaint i2, gaint i3, gaint i4, gaint i5) {
gaint rc;
signed char *ch;
int *iii;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <8) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i1;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i2;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i3;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i4;
  mbuflast->used++;
  iii = (int *)(mbuflast->buff+mbuflast->used);
  *iii = (int)i5;
  mbuflast->used++;
}

/* Metafile buffer, command plus four double args */

void hout4 (gaint cmd, gadouble xl, gadouble xh, gadouble yl, gadouble yh) {
gaint rc;
signed char *ch;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <7) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ch = (signed char *)(mbuflast->buff+mbuflast->used);
  *ch = (signed char)99;
  *(ch+1) = (signed char)cmd;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)xl;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)xh;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)yl;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)yh;
  mbuflast->used++;
}

/* Add a single character to the metafile buffer, along with the font number (less
   than 100), location (x,y), and size/rotation specs (4 floats).  Uses -21 as a
   cmd value.   */

void houtch (char ch, gaint fn, gadouble x, gadouble y,
         gadouble w, gadouble h, gadouble ang) {
gaint rc;
signed char *ccc;
char *ucc;
  if (mbuferror) return;
  if (mbuflast->len - mbuflast->used <8) {
    rc = mbufget();
    if (rc) {
      gxmbuferr();
      return;
    }
  }
  ccc = (signed char *)(mbuflast->buff+mbuflast->used);
  ucc = (char *)(ccc+2);
  *ccc = (signed char)99;
  *(ccc+1) = (signed char)(-21);
  *ucc = ch; 
  *(ccc+3) = (signed char)fn;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)x;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)y;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)w;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)h;
  mbuflast->used++;
  *(mbuflast->buff+mbuflast->used) = (float)ang;
  mbuflast->used++;
}

/* User has issued a clear.  
   This may also indicate the start or end of double buffering.  
   If we are not double buffering, just free up the memory buffer and return.  
   If we are starting up double buffering, we need another buffer chain.  
   If we are ending double buffering, all memory needs to be released.  
   If we are in the midst of double buffering, do a "swap" and free the foreground buffer. 

   Values for action are:
      0 -- new frame (clear display), wait before clearing.
      1 -- new frame, no wait.
      2 -- New frame in double buffer mode.  If not supported
           has same result as action=1.  Usage involves multiple
           calls with action=2 to obtain an animation effect.  
      7 -- new frame, but just clear graphics.  Do not clear  
           event queue; redraw buttons. 
      8 -- clear only the event queue.
      9 -- clear only the X request buffer
*/ 

void gxhfrm (gaint iact) {
struct gxmbuf *pmbuf, *pmbufl;

  /* Start up double buffering */
  if (iact==2 && dbmode==0) { 
    mbufrel(1); 
    if (mbufanch==NULL) mbufget();
    mbufanch2 = mbufanch;
    mbuflast2 = mbuflast;
    mbufanch = NULL;
    mbuflast = NULL;
    mbufget();
    dbmode = 1; 
  }

  /* End of double buffering */
  if (iact!=2 && dbmode==1) {
    mbufrel(0);
    mbufanch = mbufanch2;
    mbufrel(1);
    dbmode = 0;
    mbuferror = 0;
    return;
  }

  /* If double buffering, swap buffers */
  if (dbmode) {
    pmbuf = mbufanch;     /* Save pointer to background buffer */
    pmbufl = mbuflast;
    mbufanch = mbufanch2;
    mbufrel(1);           /* Get rid of former foreground buffer */
    mbufanch2 = pmbuf;    /* Set foreground to former background */
    mbuflast2 = pmbufl; 
  } 
  else {
    /* Not double buffering, so just free buffers */
    mbufrel(1);
  }
  if (!dbmode) mbuferror = 0;        /* Reset error state on clear command */
}

/* Redraw based on contents of current buffers.  Items that persist from plot
   to plot ARE NOT IN THE META BUFFER; these items are set in the hardware attribute
   database and are queried by the backend. 

   This routine is called from gxX (ie, a lower level of the backend rendering), 
   and this routine calls back into gxX.  This is not, however, implemented as
   true recursion -- events are disabled in gxX during this redraw, so addtional
   levels of recursion are not allowed.  

   If dbflg, draw from the background buffer.  Otherwise draw from the 
   foreground buffer. */

void gxhdrw (gaint dbflg, gaint pflg) {
struct gxmbuf *pmbuf;
float *buff;
int *iii;
gadouble r,s,x,y,w,h,ang;
gadouble *xybuf;
gaint ppp,cmd,op1,op2,op3,op4,op5,fflag,xyc=0,fn,sig;
signed char *ch;
char ccc,*uch;

  if (dbflg && !dbmode) {
    printf ("Logic error 0 in Redraw.  Contact Developer.\n");
    return;
  }
 
  if (dbflg) pmbuf = mbufanch2;
  else pmbuf = mbufanch; 

  fflag = 0;
  xybuf = NULL;

  while (pmbuf) {
    ppp = 0;
    while (ppp < pmbuf->used) {

      /* Get message type */
 
      ch = (signed char *)(pmbuf->buff + ppp);
      cmd = (gaint)(*ch);
      if (cmd != 99) {
        printf ("Metafile buffer is corrupted\n");
        printf ("Unable to complete redraw and/or print operation\n");
        return;
      }
      cmd = (gaint)(*(ch+1));
      ppp++;


      /* Handle various message types */
      /* -9 is end of file.  Should not happen. */

      if (cmd==-9) {
        printf ("Logic Error 4 in Redraw.  Notify Developer\n");
        return;
      }

      /*  -1 indicates start of file.  Should not ocurr. */

      else if (cmd==-1) {
        printf ("Logic Error 8 in Redraw.  Notify Developer\n");
        return;
      }
  
      /* -2 indicates new frame.  Also should not ocurr */

      else if (cmd==-2) {
        printf ("Logic Error 12 in Redraw.  Notify Developer\n");
        return;
      }

      /* -3 indicates new color.  One arg; color number.  */
  
      else if (cmd==-3) {
        iii = (int *)(pmbuf->buff + ppp);
        op1 = (gaint)(*iii);
	if (pflg) 
	  gxpcol (op1);          /* for printing */
	else 
	  gxdcol (op1);          /* for hardware */
        ppp++;
      }

      /* -4 indicates new line thickness.  It has two arguments */
 
      else if (cmd==-4) {
        iii = (int *)(pmbuf->buff + ppp);
        op1 = (gaint)(*iii);
	if (pflg)
	  gxpwid (op1);          /* for printing */
	else
	  gxdwid (op1);          /* for hardware */
        ppp += 2;
      }

      /*  -5 defines a new color, in rgb.  It has five int args */

      else if (cmd==-5){
        iii = (int *)(pmbuf->buff + ppp);
        op1 = (gaint)(*iii);
        iii = (int *)(pmbuf->buff + ppp + 1);
        op2 = (gaint)(*iii);
        iii = (int *)(pmbuf->buff + ppp + 2);
        op3 = (gaint)(*iii);
        iii = (int *)(pmbuf->buff + ppp + 3);
        op4 = (gaint)(*iii);
        iii = (int *)(pmbuf->buff + ppp + 4);
        op5 = (gaint)(*iii);
        gxdbacol (op1,op2,op3,op4,op5);   /* update the data base */
	if (pflg) 
	  gxpacol (op1);                 /* for printing (no-op for cairo) */
	else 
	  gxdacol (op1,op2,op3,op4,op5); /* for hardware (no-op for cairo) */
        ppp += 5;
      }

      /* -6 is for a filled rectangle.  It has four args. */ 
 
      else if (cmd==-6){
        buff = pmbuf->buff + ppp;
        r = (gadouble)(*buff);
        s = (gadouble)(*(buff+1));
        x = (gadouble)(*(buff+2));
        y = (gadouble)(*(buff+3));
	if (pflg) 
	  gxprec(r,s,x,y);          /* for printing */
	else
	  gxdrec(r,s,x,y);          /* for hardware */
        ppp += 4;
      }

      /* -7 indicates the start of a polygon fill.  It has one arg, 
         the length of the polygon.  We allocate an array for the entire
         polygon, so we can present it to the hardware backend in 
         on piece. */

      else if (cmd==-7) {
        iii = (int *)(pmbuf->buff + ppp);
        op1 = (gaint)(*iii);
        xybuf = (gadouble *)galloc(sizeof(gadouble)*op1*2,"gxybuf");
        if (xybuf==NULL) {
          printf ("Memory allocation error: Redraw\n");
          return;
        }
        xyc = 0;
        fflag = 1;
        ppp += 1;
	/* tell printing layer about new polygon. */
	if (pflg) gxpbpoly();  
      }

      /* -8 is to terminate polygon fill.  It has no args */

      else if (cmd==-8) {
        if (xybuf==NULL) {
          printf ("Logic Error 16 in Redraw.  Notify Developer\n");
          return;
        }
	if (pflg) 
	  gxpepoly (xybuf,xyc);  /* for printing */
	else
	  gxdfil (xybuf,xyc);    /* for hardware */
        gree (xybuf,"gxybuf");
        xybuf = NULL;
        fflag = 0;
      }

      /* -10 is a move to instruction.  It has two double args */ 

      else if (cmd==-10) {
        buff = pmbuf->buff + ppp;
        x = (gadouble)(*buff);
        y = (gadouble)(*(buff+1));
	if (fflag) {
	  xybuf[xyc*2] = x;
	  xybuf[xyc*2+1] = y;
	  xyc++;
	}
	if (pflg) 
	  gxpmov(x,y);            /* for printing */
	else         
	  gxdmov(x,y);            /* for hardware */
        ppp += 2;
      }

      /*  -11 is draw to.  It has two double args. */  
        
      else if (cmd==-11) {
        buff = pmbuf->buff + ppp;
        x = (gadouble)(*buff);
        y = (gadouble)(*(buff+1));
	if (fflag) {
	  xybuf[xyc*2] = x;
	  xybuf[xyc*2+1] = y;
	  xyc++;
	}
	if (pflg) 
	  gxpdrw(x,y);            /* for printing */
	else 
	  gxddrw(x,y);            /* for hardware */
        ppp += 2;
      }
      
      /* -12 indicates new fill pattern.  It has three arguments. */
 
      else if (cmd==-12) {
	/* This is a no-op for cairo; X-based pattern drawing only JMA Check that this works?? */
/* 	gxdptn ((gaint)*(poi+1),(gaint)*(poi+2),(gaint)*(poi+3));  */
	if (pflg) gxpflush(); 
        ppp += 3;
      }

      /* -20 is a draw widget.  We will redraw it in current state. */

      else if (cmd==-20) {
	/* This is a no-op for cairo; X-based widget drawing only JMA Check that this works?? */
/* 	gxdpbn ((gaint)*(poi+1),NULL,1,0,-1);  */
	if (pflg) gxpflush(); 
        ppp += 1;
      }

      /* -21 is for drawing a single character in the indicated font and size */

      else if (cmd==-21) {
        ch = (signed char *)(pmbuf->buff + ppp - 1);
        fn = (gaint)(*(ch+3));
        uch = (char *)(pmbuf->buff + ppp - 1);
        ccc = *(uch+2);
        buff = pmbuf->buff + ppp;
        x = (gadouble)(*buff);
        y = (gadouble)(*(buff+1));
        w = (gadouble)(*(buff+2));
        h = (gadouble)(*(buff+3));
        ang = (gadouble)(*(buff+4));
	if (pflg) 
	  r = gxpch (ccc,fn,x,y,w,h,ang);     /* print a character */
	else 
	  r = gxdch (ccc,fn,x,y,w,h,ang);     /* draw a character */
        ppp += 5;
      }

      /* -22 is for a signal. It has one signed character argument */

      else if (cmd==-22) {
	ch = (signed char *)(pmbuf->buff + ppp - 1);
	sig = (gaint)(*(ch+2));
	if (pflg) 
	  gxpsignal(sig); 
	else 
	  gxdsignal(sig);
	ppp++;
      }

      /* -23 is for the clipping area. It has four args. */ 
 
      else if (cmd==-23){
        buff = pmbuf->buff + ppp;
        r = (gadouble)(*buff);
        s = (gadouble)(*(buff+1));
        x = (gadouble)(*(buff+2));
        y = (gadouble)(*(buff+3));
	if (pflg) 
	  gxpclip(r,s,x,y);          /* for printing */
	else
	  gxdclip(r,s,x,y);          /* for hardware */
        ppp += 4;
      }

      /* Any other command would be invalid */

      else {
         printf ("Logic Error 20 in Redraw.  Notify Developer\n");
        return;
      }
    } 
    if (pmbuf == mbuflast) break;
    pmbuf = pmbuf->fpmbuf;
  }
  /* tell hardware and printing layer we are finished */
  if (pflg) gxpflush();  
  gxdopt(4);
}


/* Allocate and chain another buffer area */

gaint mbufget (void) {
struct gxmbuf *pmbuf;

  if (mbufanch==NULL) {
    pmbuf = (struct gxmbuf *)galloc(sizeof(struct gxmbuf),"gxmbuf");  
    if (pmbuf==NULL) return (1);
    mbufanch = pmbuf;                  /* set the new buffer structure as the anchor */
    mbuflast = pmbuf;                  /* ... and also as the last one */
    pmbuf->buff = (float *)galloc(sizeof(float)*BWORKSZ,"mbufbuff");  /* allocate a buffer */
    if (pmbuf->buff==NULL) return(1);
    pmbuf->len = BWORKSZ;              /* set the buffer length */
    pmbuf->used = 0;                   /* initialize the buffer as unused */
    pmbuf->fpmbuf = NULL;              /* terminate the chain */
  }
  else {
    if (mbuflast->fpmbuf==NULL) {      /* no more buffers in the chain */
      pmbuf = (struct gxmbuf *)galloc(sizeof(struct gxmbuf),"gxmbuf");  
      if (pmbuf==NULL) return (1);
      mbuflast->fpmbuf = pmbuf;        /* add the new buffer structure to the chain */
      mbuflast = pmbuf;                /* reset mbuflast to the newest buffer structure in the chain */
      pmbuf->buff = (float *)galloc(sizeof(float)*BWORKSZ,"mbufbuff");  /* allocate a buffer */
      if (pmbuf->buff==NULL) return(1);
      pmbuf->len = BWORKSZ;            /* set the buffer length */
      pmbuf->used = 0;                 /* initialize the buffer as unused */
      pmbuf->fpmbuf = NULL;            /* terminate the chain */
    }
    else {                             /* we'll just re-use what's already been chained up */
      pmbuf = mbuflast->fpmbuf;        /* get the next buffer in the chain */
      pmbuf->used = 0;                 /* reset this buffer to unused */
      mbuflast = pmbuf;                /* set mbuflast to point to this buffer */
    }
  }
  return (0);
}

/* Free buffer chain.  If flag is 1, leave the first buffer, 
   if there is at least one buffer already chained.
   If flag is zero, free all buffers.  */

void mbufrel (gaint flag) {
struct gxmbuf *pmbuf,*pmbuf2;
gaint i;

  i = flag;
  pmbuf = mbufanch;                /* point at the anchor */
  while (pmbuf) {
    pmbuf2 = pmbuf->fpmbuf;        /* get next link in chain */
    if (!i) {                      /* this part only gets executed when flag is 0 */
      if (pmbuf->buff) gree (pmbuf->buff,"gxmbuf");
      gree (pmbuf,"mbufbuff");     /* free the pmbuf link */
      i = 0; 
    }
    pmbuf = pmbuf2;                /* move up the chain */
  }
  if (!flag) {
    mbufanch = NULL;               /* no more metabuffer */
  } 
  else {
    if (mbufanch) mbufanch->used = 0;
  }
  mbuflast = mbufanch;
}

void gxmbuferr() {
  printf ("Error in gxmeta: Unable to allocate meta buffer\n");
  printf ("                 Buffering for the current plot is disabled\n");
  mbuferror = 1;
  mbufrel(0);
}

/* The following should go into its own source file */

/* Keep track of persistent settings for "device backend" attributes, such as 
   settings for custom colors, line widths, fonts, patterns, etc.  This interface
   provides routines to set the values and to query the values. 
   If support is ever added for writing the meta buffer out to a file, 
   the information here should be written to the beginning of the file.  */

static gaint pdcred[16] = {  0,255,250,  0, 30,  0,240,230,240,160,160,  0,230,  0,130,170};
static gaint pdcgre[16] = {  0,255, 60,220, 60,200,  0,220,130,  0,230,160,175,210,  0,170};
static gaint pdcblu[16] = {  0,255, 60,  0,255,200,130, 50, 40,200, 50,255, 45,140,220,170};

static gaint greys[16]  = {  0,255,215,140, 80,110,230,170,200, 50,155, 95,185,125, 65,177};

static gadouble pdcwid[12] = {0.6, 0.8, 1.0, 1.25, 1.5, 1.75, 2.0, 2.2, 2.4, 2.6, 2.8, 3.0};

/* basic colors.  legacy backends will only support the first 256 */
static gaint reds[COLORMAX], greens[COLORMAX], blues[COLORMAX], alphas[COLORMAX];
static gaint tilenum[COLORMAX];

/* tile patterns.  these can include solid colors with transparency.  */
static gaint ptypes[COLORMAX], pxsz[COLORMAX], pysz[COLORMAX];
static gaint pfthick[COLORMAX],pfcols[COLORMAX],pbcols[COLORMAX];
static char *pnames[COLORMAX];

/* custom line widths.  */

static gadouble widths[256];

/* custom font info */

static char *fontname[100];
static char *fn_serif = "serif";
static char *fn_sans = "sans-serif";
static char *fn_mono = "monospace";
static gaint fnstatus[100];
static gaint fnbold[100];
static gaint fnitalic[100];
static gaint hershflag;      /* For fn 1 to 6, use Hershey fonts or not */
static gaint dbdevbck;       /* Device background color */
static gaint dboutbck;       /* Ouput (image or hardcopy) background color */
static gaint dbtransclr;     /* transparent color number (for hardcopy) */



/* Initialize all the device backend persitent info on startup.
   The "basic colors" (0 to 15) are pre-defined
   Line widths (0 to 11) are also pre-defined. */

void gxdbinit () {
gaint i;

  /* Initialize colors (default and user-defined) */
  for (i=0; i<COLORMAX; i++) {
    reds[i]   = 150; 
    greens[i] = 150;
    blues[i]  = 150;
    alphas[i] = 255;
    tilenum[i] = -999;
    ptypes[i] = -999;
    pnames[i] = NULL;
    pxsz[i] = 10; 
    pysz[i] = 10;
    pfthick[i] = 3; 
    pfcols[i] = -999; 
    pbcols[i] = -999;
  }
  for (i=0; i<16; i++) {
    reds[i]   = pdcred[i];
    greens[i] = pdcgre[i];
    blues[i]  = pdcblu[i];
    alphas[i] = 255;
  }

  /* initialize line widths (default and user-defined) */
  for (i=0; i<256; i++) widths[i] = 1.0; 
  for (i=0; i<12; i++) widths[i] = pdcwid[i]; 

  /* Initialize font settings */
  for (i=0; i<100; i++) {
    fontname[i] = NULL;
    fnstatus[i] = 0;
    fnbold[i] = 0;
    fnitalic[i] = 0;
  }

  /* these will be for emulations of hershey fonts 0-5, but not 3 */
  fontname[0] = fn_sans;   fnbold[0] = 0;  fnitalic[0] = 0; /* JMA set fnstatus for these? */
  fontname[1] = fn_serif;  fnbold[1] = 0;  fnitalic[1] = 0;
  fontname[2] = fn_mono;   fnbold[2] = 0;  fnitalic[2] = 1;
  fontname[4] = fn_sans;   fnbold[4] = 1;  fnitalic[4] = 0;
  fontname[5] = fn_serif;  fnbold[5] = 1;  fnitalic[5] = 0;

  /* other flags for this and that */
  hershflag  = 0;     /* zero, use hershey fonts.  1, use emulation. */  
  dbdevbck   = 0;     /* initial device background color is black */
  dboutbck   = -1;    /* initial output background color is 'undefined' */
  dbtransclr = -1;    /* initial transparent color is 'undefined' */
   
}

/* Set a font name for a font from 10 to 99.  
   Use of a font number that has no defined font family name will be handled
   by the device backend, usually by using a default sans-serif font. */

void gxdbsetfn (gaint fn, char *str) {
gaint len,i;
char *newname;

  if (fontname[fn]) {
    /* this font number has been previously assigned. Reset and free memory */
    free (fontname[fn]);
    fontname[fn] = NULL;
    fnstatus[fn] = 0;
    fnbold[fn] = 0;
    fnitalic[fn] = 0;
  }
  if (str) {
    len = 0;
    while (*(str+len)) len++;
    newname = (char *)malloc(len+1);
    if (newname==NULL) return;
    for (i=0; i<len; i++) *(newname+i) = *(str+i);
    *(newname+len) = '\0';
    fontname[fn] = newname;
    fnstatus[fn] = 0; 
  }
  /* No 'else' statement here ... we can allow a font name to be NULL */
}

/* Set the status of a font for a font from 10 to 99.
   The status is 1 if the font file has been opened.  */

void gxdbsetfnstatus (gaint fn, gaint status) {
  if (fn>9 && fn<100) fnstatus[fn] = status;
}


/* Query font settings */

void gxdbqfont (gaint fn, struct gxdbquery *pdbq) {
  if (fn<0 || fn>99) {
    pdbq->fname   = fontname[0];
    pdbq->fstatus = fnstatus[0];
    pdbq->fbold   = fnbold[0];
    pdbq->fitalic = fnitalic[0];
  } else {
    pdbq->fname   = fontname[fn];
    pdbq->fstatus = fnstatus[fn];
    pdbq->fbold   = fnbold[fn];
    pdbq->fitalic = fnitalic[fn];
  }
}

/* Query a color */

void gxdbqcol (gaint colr, struct gxdbquery *pdbq) {
  if (colr<0 || colr>=COLORMAX) {
    pdbq->red = 150;
    pdbq->green = 150;
    pdbq->blue = 150;
    pdbq->alpha = 255;
    pdbq->tile = -999;
  } else {
    pdbq->red = reds[colr];
    pdbq->green = greens[colr];
    pdbq->blue = blues[colr];
    pdbq->alpha = alphas[colr];
    pdbq->tile = tilenum[colr];
  }
}

/* Define a new color */

gaint gxdbacol (gaint clr, gaint red, gaint green, gaint blue, gaint alpha) {
  if (clr<0 || clr>=COLORMAX ) return(1); 
  if (red == -9) {
    /* this is a pattern */
    reds[clr] = red; 
    greens[clr] = green; 
    blues[clr] = blue;  
    alphas[clr] = 255;
    tilenum[clr] = green;
  } else {
    /* this is a color */
    reds[clr] = red;
    greens[clr] = green;
    blues[clr] = blue;
    alphas[clr] = alpha;
    tilenum[clr] = -999;
  }
  return 0;
}

/* Query line width */

void gxdbqwid (gaint lwid, struct gxdbquery *pdbq) {    
  if (lwid<0 || lwid>255) pdbq->wid = 1.0; 
  else pdbq->wid = widths[lwid];
}

/* Set line width */

void gxdbsetwid (gaint lwid, gadouble val) {    
  if (lwid>=1 && lwid<=256) {
    if (val>0) widths[lwid-1] = val;
  }
}

/* Set pattern settings */

void gxdbsetpatt (gaint *itt, char *str) {    
gaint pnum,len,i;
char *newname;

  pnum = *itt;
  if (pnum<0 || pnum>=COLORMAX) return;
  /* trigger a pattern reset in the hardware layer */
  gxsetpatt(pnum);                        

  if (pnames[pnum]) {
    /* reset tile filename and free memory */
    free (pnames[pnum]);
    pnames[pnum] = NULL;
  }

  /* reset all elements */
  ptypes[pnum]  = -999;
  pxsz[pnum]    = -999; 
  pysz[pnum]    = -999;
  pfthick[pnum] = -999; 
  pfcols[pnum]  = -999; 
  pbcols[pnum]  = -999;

  /* now populate data base entries with new values */
  if (str) {
    len = 0;
    while (*(str+len)) len++;
    newname = (char *)malloc(len+1);
    if (newname==NULL) return;
    for (i=0; i<len; i++) *(newname+i) = *(str+i);
    *(newname+len) = '\0';
    pnames[pnum] = newname;
  }
  if (*(itt+1) < -900) return;  ptypes[pnum]  = *(itt+1);
  if (*(itt+2) < -900) return;  pxsz[pnum]    = *(itt+2);
  if (*(itt+3) < -900) return;  pysz[pnum]    = *(itt+3);
  if (*(itt+4) < -900) return;  pfthick[pnum] = *(itt+4);
  if (*(itt+5) < -900) return;  pfcols[pnum]  = *(itt+5);
  if (*(itt+6) < -900) return;  pbcols[pnum]  = *(itt+6);
}

/* Query pattern settings */

void gxdbqpatt (gaint pnum, struct gxdbquery *pdbq) {    
  if (pnum<0 || pnum>=COLORMAX) return;
  pdbq->ptype = ptypes[pnum];
  pdbq->pxs = pxsz[pnum];
  if (pdbq->pxs < -900) pdbq->pxs = 9;
  pdbq->pys = pysz[pnum];
  if (pdbq->pys < -900) pdbq->pys = 9;
  pdbq->pthick = pfthick[pnum];
  if (pdbq->pthick < -900) pdbq->pthick = 3;
  pdbq->pfcol = pfcols[pnum];
  pdbq->pbcol = pbcols[pnum];
  pdbq->fname = pnames[pnum];
};

/* Query hershey flag */

gaint gxdbqhersh (void) {    
  return (hershflag);
}

/* Set hershey flag */

void gxdbsethersh (gaint flag) {    
  if (flag==0 || flag==1) 
    hershflag = flag;
}

/* Query transparent color (for image output) */

gaint gxdbqtransclr (void) {   
  return (dbtransclr);
}

/* Set transparent color (for image output) */

void gxdbsettransclr (gaint clr) {   
  dbtransclr = clr; 
}

/* Set device background color. */

void gxdbck (gaint clr) {    
  dbdevbck = clr;
}

/* Set output background color */

void gxdboutbck (gaint clr) {    
  dboutbck = clr;
}

/* Query background color */

gaint gxdbkq  (void) {    
  /* If the output background color is not set, return device background color */
  if (dboutbck != -1) 
    return (dboutbck);
  else 
    return (dbdevbck);
}


