#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2000-2005  Donald N. Allingham
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# $Id: AsciiDoc.py,v 1.10 2005/02/19 00:49:49 rshura Exp $

#------------------------------------------------------------------------
#
# python modules
#
#------------------------------------------------------------------------
import os
from gettext import gettext as _

#------------------------------------------------------------------------
#
# Gramps modules
#
#------------------------------------------------------------------------
import BaseDoc
import PluginMgr
import Errors
import GrampsMime

LEFT,RIGHT,CENTER = 'LEFT','RIGHT','CENTER'
_WIDTH_IN_CHARS = 72

#------------------------------------------------------------------------
#
# This routine was written by David Mertz and placed into the public
# domain. It is sample code from his book, "Text Processing in Python"
#
# Modified by Alex Roitman: right-pad with spaces, if right_pad==1;
#                           return empty string if no text was given
#
#------------------------------------------------------------------------
def reformat_para(para='',left=0,right=72,just=LEFT,right_pad=0):
    if not para.strip():
        return "\n"
    words = para.split()
    lines = []
    line  = ''
    word = 0
    end_words = 0
    while not end_words:
        if len(words[word]) > right-left: # Handle very long words
            line = words[word]
            word +=1
            if word >= len(words):
                end_words = 1
        else:                             # Compose line of words
            while len(line)+len(words[word]) <= right-left:
                line += words[word]+' '
                word += 1
                if word >= len(words):
                    end_words = 1
                    break
        lines.append(line)
        line = ''
    if just==CENTER:
        if right_pad:
            return '\n'.join([' '*left+ln.center(right-left) for ln in lines])
        else:
            return '\n'.join([' '*left+ln.center(right-left).rstrip() for ln in lines])
    elif just==RIGHT:
        if right_pad:
            return '\n'.join([line.rjust(right) for line in lines])
        else:
            return '\n'.join([line.rjust(right).rstrip() for line in lines])
    else: # left justify 
        if right_pad:
            return '\n'.join([' '*left+line.ljust(right-left) for line in lines])
        else:
            return '\n'.join([' '*left+line for line in lines])

#------------------------------------------------------------------------
#
# Ascii
#
#------------------------------------------------------------------------
class AsciiDoc(BaseDoc.BaseDoc):

    #--------------------------------------------------------------------
    #
    # Opens the file, resets the text buffer.
    #
    #--------------------------------------------------------------------
    def open(self,filename):
        if filename[-4:] != ".txt":
            self.filename = filename + ".txt"
        else:
            self.filename = filename

        try:
            self.f = open(self.filename,"w")
        except IOError,msg:
            errmsg = "%s\n%s" % (_("Could not create %s") % self.filename, msg)
            raise Errors.ReportError(errmsg)
        except:
            raise Errors.ReportError(_("Could not create %s") % self.filename)

        self.in_cell = 0
        self.text = ""

    #--------------------------------------------------------------------
    #
    # Close the file. Call the app if required. 
    #
    #--------------------------------------------------------------------
    def close(self):
        self.f.close()

        if self.print_req:
            apptype = 'text/plain'
            prog = GrampsMime.get_application(apptype)
            os.environ["FILE"] = self.filename
            os.system ('%s "$FILE" &' % prog[0])

    def get_usable_width(self):
        return _WIDTH_IN_CHARS

    #--------------------------------------------------------------------
    #
    # Force a section page break
    #
    #--------------------------------------------------------------------
    def end_page(self):
        self.f.write('\012')

    #--------------------------------------------------------------------
    #
    # Force a line break
    #
    #--------------------------------------------------------------------
    def line_break(self):
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + '\n'
        else:
            self.f.write('\n')

    def start_superscript(self):
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + '['
        else:
            self.f.write('[')

    def end_superscript(self):
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + ']'
        else:
            self.f.write(']')

    #--------------------------------------------------------------------
    #
    # Starts a paragraph. 
    #
    #--------------------------------------------------------------------
    def start_paragraph(self,style_name,leader=None):
        self.p = self.style_list[style_name]

        i = int(abs(self.p.get_first_indent())*4)
        self.leader = leader
        
        this_text = ''

        if leader:
            if i:
                ln = max(i,len(leader))
                t = leader + ' '*ln
                this_text = t[0:ln]
            else:
                this_text = leader
        elif i:
            this_text = ' '*i
        
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + this_text
        else:
            self.f.write(this_text)

    #--------------------------------------------------------------------
    #
    # End a paragraph. First format it to the desired widths. 
    # If not in table cell, write it immediately. If in the cell,
    # add it to the list for this cell after formatting.
    #
    #--------------------------------------------------------------------
    def end_paragraph(self):
        if self.p.get_alignment() == BaseDoc.PARA_ALIGN_RIGHT:
            fmt = RIGHT
        elif self.p.get_alignment() == BaseDoc.PARA_ALIGN_CENTER:
            fmt = CENTER
        else:
            fmt = LEFT

        if self.in_cell:
            right = self.cell_widths[self.cellnum]
        else:
            right = self.get_usable_width()

        margin = 0
        if self.p.get_left_margin():
            margin = int(4*self.p.get_left_margin())

        i = int(abs(self.p.get_first_indent())*4)

        if self.in_cell and self.cellnum < self.ncols - 1:
            right_pad = 1
            the_pad = ' '*right
        else:
            right_pad = 0
            the_pad = ''
        t = reformat_para(self.text,margin,right,fmt,right_pad)

        if i and self.leader:
            this_text = t[i:]
        else:
            this_text = t
        
        this_text = this_text + '\n' + the_pad + '\n' 
        
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + this_text
        else:
            self.f.write(this_text)
            
        self.text = ""

    #--------------------------------------------------------------------
    #
    # Start a table. Grab the table style, and store it. 
    #
    #--------------------------------------------------------------------
    def start_table(self,name,style_name):
        self.tbl_style = self.table_styles[style_name]
        self.ncols = self.tbl_style.get_columns()

    #--------------------------------------------------------------------
    #
    # End a table. Turn off the self.in_cell flag
    #
    #--------------------------------------------------------------------
    def end_table(self):
        self.in_cell = 0

    #--------------------------------------------------------------------
    #
    # Start a row. Initialize lists for cell contents, number of lines,
    # and the widths. It is necessary to keep a list of cell contents
    # that is to be written after all the cells are defined.
    #
    #--------------------------------------------------------------------
    def start_row(self):
        self.cellpars = [''] * self.ncols
        self.cell_lines = [0] * self.ncols
        self.cell_widths = [0] * self.ncols
        self.cellnum = -1
        self.maxlines = 0
        table_width = self.get_usable_width() * self.tbl_style.get_width() / 100.0
        for cell in range(self.ncols):
            self.cell_widths[cell] = int( table_width * \
                                self.tbl_style.get_column_width(cell) / 100.0 )

    #--------------------------------------------------------------------
    #
    # End a row. Write the cell contents. Write the line of spaces
    # if the cell has fewer lines than the maximum number.
    #
    #--------------------------------------------------------------------
    def end_row(self):
        self.in_cell = 0
        cell_text = [None]*self.ncols
        for cell in range(self.ncols):
            if self.cell_widths[cell]:
                blanks = ' '*self.cell_widths[cell] + '\n'
                if self.cell_lines[cell] < self.maxlines:
                    self.cellpars[cell] = self.cellpars[cell] \
                              + blanks * (self.maxlines-self.cell_lines[cell])
                cell_text[cell] = self.cellpars[cell].split('\n')
        for line in range(self.maxlines):
            for cell in range(self.ncols):
                if self.cell_widths[cell]:
                    self.f.write(cell_text[cell][line])
            self.f.write('\n')

    #--------------------------------------------------------------------
    #
    # Start a cell. Set the self.in_cell flag, increment the curren cell number.
    #
    #--------------------------------------------------------------------
    def start_cell(self,style_name,span=1):
        self.in_cell = 1
        self.cellnum = self.cellnum + span
        span = span - 1
        while span:
            self.cell_widths[self.cellnum-span] = 0
            span = span - 1
            

    #--------------------------------------------------------------------
    #
    # End a cell. Find out the number of lines in this cell, correct
    # the maximum number of lines if necessary.
    #
    #--------------------------------------------------------------------
    def end_cell(self):
        self.in_cell = 0
        self.cell_lines[self.cellnum] = self.cellpars[self.cellnum].count('\n')
        if self.cell_lines[self.cellnum] > self.maxlines:
            self.maxlines = self.cell_lines[self.cellnum]

    def add_photo(self,name,pos,x,y):
        this_text = '(photo)'
        if self.in_cell:
            self.cellpars[self.cellnum] = self.cellpars[self.cellnum] + this_text
        else:
            self.f.write(this_text)

    def write_note(self,text,format,style_name):
        if format == 1:
            for line in text.split('\n'):
                self.start_paragraph(style_name)
                self.write_text(line)
                self.end_paragraph()
        elif format == 0:
            for line in text.split('\n\n'):
                self.start_paragraph(style_name)
                line = line.replace('\n',' ')
                line = ' '.join(line.split())
                self.write_text(line)
                self.end_paragraph()

    #--------------------------------------------------------------------
    #
    # Writes text. 
    #--------------------------------------------------------------------
    def write_text(self,text):
        text = text.replace('<super>','[')
        text = text.replace('</super>',']')
        self.text = self.text + text

#------------------------------------------------------------------------
#
# Register the document generator with the GRAMPS plugin system
#
#------------------------------------------------------------------------
print_label = None
try:
    import Utils

    mprog = GrampsMime.get_application("text/plain")
    mtype = GrampsMime.get_description('text/plain')

    if Utils.search_for(mprog[0]):
        print_label=_("Open in %s") % mprog[1]
    else:
        print_label=None

    PluginMgr.register_text_doc(mtype,AsciiDoc,1,1,1,".txt", print_label)
except:
    PluginMgr.register_text_doc(_("Plain Text"),AsciiDoc,1,1,1,".txt", None)

