// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: CardBox.cpp,v 1.47 2005/11/13 02:50:48 vlg Exp $
//------------------------------------------------------------------------------
//                            CardBox.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2004,2005 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Sat Apr 17 23:37:11 EDT 2004
//
//------------------------------------------------------------------------------

#include <limits.h>				// PATH_MAX
#include <libgen.h>				// basename(3), dirname(3)
#include <string.h>				// strcpy(3)

#include <gtkmm/table.h>
#include <gtkmm/eventbox.h>

#include "Granule.h"
#include "CardBox.h"
#include "Config.h"
#include "MainWindow.h"
#include "CardRef.h"
#include "Deck.h"
#include "Intern.h"

/*------------------------------------------------------------------------------
 * static methods
 *------------------------------------------------------------------------------
 */
std::string
CardBox::
calculate_relative_path (const std::string& abs_pname_,
						 const std::string& abs_fname_)
{
    trace_with_mask("CardBox::calculate_relative_path", GUITRACE);

	std::string::iterator ap_iter;
	std::string::iterator af_iter;
	std::string::iterator slash;

	std::string ap (abs_pname_);
	std::string af (abs_fname_);

	if (ap [ap.size ()-1] != G_DIR_SEPARATOR) {
		ap += G_DIR_SEPARATOR_S;
	}

	/* Remove prefix similar in both string (if any)
	 */
	ap_iter = ap.begin ();
	af_iter = af.begin ();

	while (ap_iter != ap.end () || af_iter != af.end ()) {
		if (*ap_iter != *af_iter) {
			break;
		}
		ap_iter++, af_iter++;
	}
	ap.erase (ap.begin (), ap_iter);
	af.erase (af.begin (), af_iter);

	/* In AP string, replace directory names between consequtive '/' with
	   '..'. Then, remove leading '/'.
	*/
	ap_iter = ap.begin ();
	slash = ap_iter;
	do {
		while (*slash != G_DIR_SEPARATOR && slash != ap.end ()) { 
			slash++;
		}
		if (slash == ap.end ()) {
			break;
		}
		ap.replace (ap_iter, slash, "..");
		ap_iter += 3;
		slash = ap_iter;
	} while (ap_iter != ap.end ());

	/* Add AF to AP and return result 
	 */
	return (ap + af);
}

std::string
CardBox::
calculate_absolute_path (const std::string& abs_pname_,
						 const std::string& rel_fname_)
{
    trace_with_mask("CardBox::calculate_absolute_path", GUITRACE);
	
	std::string::size_type idx;

	std::string ap (abs_pname_);
	std::string rf (rel_fname_);

	/** For each '..' found, remove it and its preceding directory name
		(i.e. /Words/Travel/../.. => /Words/..).
	*/
	while (rf.find (".." G_DIR_SEPARATOR_S) != std::string::npos) {
		rf.erase (rf.begin (), rf.begin ()+3);
		idx = ap.rfind (G_DIR_SEPARATOR);
		if (idx != std::string::npos) {
			ap.erase (idx);
		}
	}

	/** If the first character is not '/', then you don't have an absolute
		path as you migh have expected.
	*/
	return (ap + G_DIR_SEPARATOR_S + rf);
}


/*------------------------------------------------------------------------------
 * CardBox class methods
 *------------------------------------------------------------------------------
 */
CardBox::
CardBox ()
{
	trace_with_mask("CardBox::CardBox",GUITRACE);
	Gtk::EventBox* evtbox = NULL;
	int i = 0;
	
	for (i = 0; i < CARD_BOX_SIZE; i++) {
		m_cdplayers [i] = NULL;
	}

	m_drawer_empty = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/drawer_empty.xpm");
	m_drawer_one   = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/drawer_one.xpm");
	m_drawer_many  = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/drawer_many.xpm");
	m_sched_green  = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/sched_green.xpm");
	m_sched_grey   =
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/sched_grey.xpm");
	m_sched_red    = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/sched_red.xpm");
	m_sched_yellow = 
		Gdk::Pixbuf::create_from_file(GRAPPDATADIR "/pixmaps/sched_yellow.xpm");

	for (i = 0; i < CARD_BOX_SIZE; i++) {
		m_cbs.push_back (CardBoxIF ());
		m_cbs [i].m_sched_light = manage (new Gtk::Image);
		m_cbs [i].m_sched_count = manage (new Gtk::Label);
		m_cbs [i].m_sched_vbox  = manage (new Gtk::VBox (false, 0));
		m_cbs [i].m_box_pic     = manage (new Gtk::Image);
		m_cbs [i].m_total_count = manage (new Gtk::Label);

		m_cbs [i].m_sched_light->set (m_sched_grey);
		m_cbs [i].m_sched_light->set_alignment (0.5,0.5);
		m_cbs [i].m_sched_light->set_padding (0,2);

		m_cbs [i].m_sched_count->set_alignment(0.5,0.5);
		m_cbs [i].m_sched_count->set_padding(0,0);
		m_cbs [i].m_sched_count->set_justify(Gtk::JUSTIFY_CENTER);
		m_cbs [i].m_sched_count->set_line_wrap(false);
		m_cbs [i].m_sched_count->set_text ("0");

		m_cbs [i].m_sched_vbox->pack_start (
			*m_cbs [i].m_sched_light, Gtk::PACK_SHRINK, 2);

		m_cbs [i].m_sched_vbox->pack_start (
			*m_cbs [i].m_sched_count, Gtk::PACK_SHRINK, 0);

		m_cbs [i].m_box_pic->set (m_drawer_empty);
		m_cbs [i].m_box_pic->set_alignment (0.5,0.5);
		m_cbs [i].m_box_pic->set_padding (2,2);
		
		m_cbs [i].m_total_count->set_alignment (0,0.5);
		m_cbs [i].m_total_count->set_padding (8,0);
		m_cbs [i].m_total_count->set_justify (Gtk::JUSTIFY_RIGHT);
		m_cbs [i].m_total_count->set_line_wrap (false);
		m_cbs [i].m_total_count->set_text (_("0 cards"));
	}

	Gtk::Table* cbt = manage (new Gtk::Table (5, 3, false));
	
	cbt->set_row_spacings(1);
	cbt->set_col_spacings(1);

	/** Also enable card box icon as DND destination.
	 */
	std::list<Gtk::TargetEntry> list_targets;
	list_targets.push_back (Gtk::TargetEntry ("STRING"));
	list_targets.push_back (Gtk::TargetEntry ("text/plain"));

	for (i = 0; i < 5; i++) {
		evtbox = manage (new Gtk::EventBox);
		evtbox->add (*(m_cbs [i].m_box_pic));
		evtbox->set_events (Gdk::BUTTON_PRESS_MASK);

		evtbox->signal_button_press_event ().connect (
			sigc::bind<int>(sigc::mem_fun(*this, &CardBox::play_in_box_cb), i));

		evtbox->drag_dest_set (list_targets);
		evtbox->signal_drag_data_received ().connect (
			sigc::bind(sigc::mem_fun (*this, &CardBox::on_deck_dropped_cb), i));

		cbt->attach (*evtbox, 1,2,i,i+1, Gtk::FILL, Gtk::FILL, 0,0);
	}

	for (i = 0; i < 5; i++) {
		cbt->attach (*m_cbs[i].m_total_count, 
					 2,3,i,i+1, 
					 Gtk::FILL,
					 Gtk::AttachOptions(), 
					 0,0);
	}

	for (i = 0; i < 5; i++) {
		cbt->attach (*m_cbs [i].m_sched_vbox, 0,1,i,i+1, 
					 Gtk::FILL, Gtk::FILL,4,0);
	}
  	pack_start (*cbt, Gtk::PACK_SHRINK, 4);
	show_all ();
}

void
CardBox::
update_count (int idx_)
{
	std::ostringstream os;
	
	os << m_card_deck [idx_].size () << " cards";
	m_cbs [idx_].m_total_count->set_text (os.str ());

	m_cbs [idx_].m_box_pic->clear ();	
	if (m_card_deck [idx_].size () == 0) {
		m_cbs [idx_].m_box_pic->set (m_drawer_empty);
	}
	else if (m_card_deck [idx_].size () == 1) {
		m_cbs [idx_].m_box_pic->set (m_drawer_one);
	}
	else {
		m_cbs [idx_].m_box_pic->set (m_drawer_many);
	}
}

void
CardBox::
new_file ()
{
	trace_with_mask("CardBox::new_file",GUITRACE);

	m_fname = UNKNOWN_FNAME;
	CONFIG->set_proj_name (m_fname);
	for (int i = 0; i < CARD_BOX_SIZE; i++) {
		m_card_deck [i].set_name (i);
		m_cdplayers [i] = new DeckPlayer (m_card_deck [i]);
	}
	MAINWIN->set_mw_name ();
}

int
CardBox::
load (const string& fname_)
{
	trace_with_mask("CardBox::load",GUITRACE);

	int ret = 0;
	string error_msg;
	xmlParserCtxtPtr context;
	xmlDocPtr parser;

	/** Test to see if file exists
	 */
    struct stat file_stat;
    if (stat (fname_.c_str (), &file_stat) < 0 ||
		S_ISREG (file_stat.st_mode) == false)           // invalid project file
	{
		error_msg = "Invalid CardBox file:\n";
		error_msg += fname_;
		Gtk::MessageDialog md_error (error_msg, false, Gtk::MESSAGE_ERROR);
		md_error.run ();
		return -1;
    }

    CONFIG->set_proj_name (fname_);

	context = xmlCreateFileParserCtxt (fname_.c_str ()); // new
	if (!context) {
		DL ((GRAPP,"Failed to create file context\n"));
		xmlFreeParserCtxt (context);
		return -1;
	}
	// Initialize context
	context->linenumbers = 1;
	context->validate = 1;
	context->replaceEntities = 1;

	// Parse the document
	xmlParseDocument (context);

	if(!context->wellFormed || context->errNo != 0) {
		error_msg = "File is not well-formed:\n";
		error_msg += fname_;
		Gtk::MessageDialog md_error (error_msg, false, Gtk::MESSAGE_ERROR);
		md_error.run ();
		xmlFreeParserCtxt (context); 
		return -1;
	}

	parser = context->myDoc;
	xmlFreeParserCtxt (context); // not needed anymore?

	DL((GRAPP,"Parsing the card file ...\n"));

	if (parse_xml_tree (parser, error_msg) == 0) {
		m_fname = fname_;
		MAINWIN->set_mw_name ();
		CONFIG->set_filesel_path (fname_);
	}
	else {
 		string msg (_("Failed to parse file\n   "));
 		msg += fname_ + _("\nReason: invalid file syntax/DTD");
 		Gtk::MessageDialog e (msg, false, MESSAGE_ERROR);
 		e.run ();
		DL ((ASSA::ERROR,"Failed to parse file: %s\n", error_msg.c_str ()));
		ret = -1;
	}

	xmlFreeDoc (parser);
	return ret;
}

int
CardBox::
parse_xml_tree (const xmlDocPtr parser_, string error_msg_)
{
	trace_with_mask("CardBox::parse_xml_tree",GUITRACE);

	char total_count [32];
	char xpath [64];
	xmlChar* result;

	string   attr_node;
	long     card_id;
	string   abs_path;			// absolute deck path
	string   deck_path;			// deck path as recorded in CDF file
	int      ccount;			// card counter
	int      cloaded;			// count of cards loaded
	Card*    cardptr;
	CardRef* card_ref;

	for (int i = 0; i < CARD_BOX_SIZE; i++) 
	{
		DL((GRAPP,"Filling Card Deck # %d ...\n", i+1));
		m_card_deck [i].set_name (i);
		cloaded = 0;
		for (ccount = 1; ccount < MAX_CARDS; ccount++) 
		{
			DL((GRAPP,"Parsing Card # %d/%d ...\n", i+1, ccount));

			sprintf(xpath, "/cardfile/carddeck[%d]/card[%d]", i+1, ccount);
			result = Granule::get_node_by_xpath (parser_, xpath);
			if (result == NULL) {
				break;
			}
			xmlFree (result);

			/** Parse card's ID
			 */
			sprintf (xpath, "/cardfile/carddeck[%d]/card[%d]/@id", i+1, ccount);
			result = Granule::get_node_by_xpath (parser_, xpath);
			if (result == NULL) {
				error_msg_  = "Failed to retrieve card id for ";
				error_msg_ += xpath;
				return -1;
			}
			attr_node = (const char*) result;
			card_id = ::atol ((attr_node.substr (1)).c_str ());
			xmlFree (result);

			/** Parse the expiration time of the card (expdate tag). If
				tag is not found, then we are dealing with an older file.
			 */
			ASSA::TimeVal exp_time;
			sprintf (xpath, "/cardfile/carddeck[%d]/card[%d]/expdate", 
					 i+1, ccount);
			result = Granule::get_node_by_xpath (parser_, xpath);
			if (result == NULL) {
				exp_time = CONFIG->get_load_time (); // card expires today
			}
			else {
				exp_time.sec (::atol ((const char*) result));
				xmlFree (result);
			}

			/** Parse the source Deck path of the card
			 */
			sprintf (xpath, "/cardfile/carddeck[%d]/card[%d]/fromdeck", 
					 i+1, ccount);
			result = Granule::get_node_by_xpath (parser_, xpath);
			if (result == NULL) {
				return -1;
			}
			deck_path = Glib::locale_from_utf8 ((const char*) result);
			xmlFree (result);

			if (deck_path[0] != G_DIR_SEPARATOR) {
				abs_path = calculate_absolute_path (CONFIG->get_proj_path (), 
													deck_path);
			}
			else {
				abs_path = deck_path;
			}

			DL((GRAPP,"Card parsed ID=%ld,\n path = \"%s\"\n", 
				card_id, deck_path.c_str ()));

			/** Look up the card in the list of decks by (ID, PATH) pair.
				If deck is found, it is loaded (if it is not already in the
				memory) and a newly created CardRef object is returned.
			 */
			card_ref = DECKMGR->lookup_card (card_id, abs_path);
			if (card_ref != NULL) {
				DL((GRAPP,"Card (ID=%ld) found\n", card_id));
				card_ref->set_expiration (exp_time);
				if (card_ref->is_expired ()) {
					m_cbs [i].add_one ();
				}
				m_card_deck [i].push_back (card_ref);
				cloaded++;
			}
		} // for (each card) 

		DL((GRAPP,"Processed %d cards\n", m_card_deck [i].size ()));
		m_card_deck [i].sort_by_expiration_time ();
		sprintf (total_count, "%d cards", cloaded);
		update_count (i);
		set_expiration_light (i);
		m_cdplayers [i] = new DeckPlayer (m_card_deck [i]);

	} // for (each card box)

	return 0;
}
	
int
CardBox::
save ()
{
	trace_with_mask("CardBox::save",GUITRACE);

	static const char dtd_url[] = "http://granule.sourceforge.net/cardfile.dtd";

	std::string abs_fname;
	const std::string proj_path (CONFIG->get_proj_path ());
	std::string deck_fname;
	VDeck::cardlist_iterator citer;
	CardRef* card_ref = NULL;
	int ret = 0;

	xmlTextWriterPtr writer;

	DL((GRAPP,"Saving to \"%s\"\n", m_fname.c_str ()));
	::unlink (m_fname.c_str ());

	writer = xmlNewTextWriterFilename (m_fname.c_str (), 0);
	if (writer == NULL) {
		return -1;
	}

	ret = xmlTextWriterStartDocument (writer, NULL, "UTF-8", NULL);
	if (ret < 0) {
		xmlFreeTextWriter (writer);
		return -1;
	}

	xmlTextWriterSetIndent(writer, 0);
	xmlTextWriterStartDTD (writer, BAD_CAST "cardfile", NULL, BAD_CAST dtd_url);
	xmlTextWriterSetIndent(writer, 1);
	xmlTextWriterEndDTD   (writer);

	ret = xmlTextWriterStartElement (writer, BAD_CAST "cardfile");

	for (int i = 0; i < CARD_BOX_SIZE; i++) {
		ret = xmlTextWriterStartElement (writer, BAD_CAST "carddeck");
		citer = m_card_deck [i].begin ();

		while (citer != m_card_deck [i].end ()) {

			if ((card_ref = dynamic_cast<CardRef*> (*citer)) != NULL) {
				ret = xmlTextWriterStartElement (writer, BAD_CAST "card");
				ret = xmlTextWriterWriteAttribute (writer, BAD_CAST "id",
						   BAD_CAST (*citer)->get_id_str ().c_str ());

				abs_fname = card_ref->get_deck ().get_name ();
				if (CONFIG->with_relpaths ()) {
					deck_fname = calculate_relative_path (proj_path, abs_fname);
				}
				else {
					deck_fname = abs_fname;
				}
				ret = xmlTextWriterWriteElement (writer, BAD_CAST "fromdeck",
						 BAD_CAST Glib::locale_to_utf8 (deck_fname).c_str ());

				ret = xmlTextWriterWriteElement (writer, BAD_CAST "expdate",
						 BAD_CAST card_ref->get_expiration_str ().c_str ());
				ret = xmlTextWriterEndElement (writer);
			}
			citer++;
		}
		ret = xmlTextWriterEndElement (writer);	// carddeck
	}

	xmlTextWriterEndElement (writer); // @end cardfile
	xmlFreeTextWriter (writer);	      // cleanup

	return ret;
}

int
CardBox::
save_as (const string& fname_)
{
	trace_with_mask("CardBox::save_as",GUITRACE);

	/** If extension is not given, add it to the filepath.
	 */
	m_fname = fname_;
    string ext (CARD_FILE_EXT);
    if (m_fname.find (ext) == string::npos) {
		m_fname += ext;
    }

	if (Glib::file_test (fname_, Glib::FILE_TEST_IS_REGULAR)) {
		string msg = N_("File \n") + fname_ 
			+ _("\nalready exists.\nWould you like to delete it first?");
		Gtk::MessageDialog qm (*MAINWIN, 
							   msg, 
							   false,
							   Gtk::MESSAGE_QUESTION, 
							   Gtk::BUTTONS_NONE,
							   true);
		qm.add_button (_("Yes"), Gtk::RESPONSE_YES);
		qm.add_button (_("No"),  Gtk::RESPONSE_NO );
		if (qm.run () == Gtk::RESPONSE_NO) {
			return -1;
		}
		::unlink (fname_.c_str ());
	}
	CONFIG->set_proj_name (fname_);
	save ();
}

void
CardBox::
close ()
{
	trace_with_mask("CardBox::close",GUITRACE);

	for (int i = 0; i < CARD_BOX_SIZE; i++) {
		delete m_cdplayers [i];
		m_cdplayers [i] = NULL;
		m_card_deck [i].clear ();
		m_cbs [i].reset ();
		m_cbs [i].m_box_pic->clear ();
		m_cbs [i].m_box_pic->set (m_drawer_empty);
		m_cbs [i].m_sched_light->set (m_sched_grey);
	}
	m_fname = "";				// No cardbox is loaded any more
}

void
CardBox::
play_deck (int idx_)
{
	trace_with_mask("CardBox::play_deck",GUITRACE);
	
	if (idx_ < 0 || idx_ > CARD_BOX_SIZE-1) {
		Gtk::MessageDialog w (_("Trying to access index out of range!"), 
							  false, MESSAGE_ERROR);
		w.run ();
	}
	if (m_card_deck [idx_].size ()) 
	{
		m_card_deck [idx_].reset_progress ();
		m_cdplayers [idx_]->run ();
		update_count (idx_);	// User might have deleted some cards.
	}
	else {
		Gtk::MessageDialog w (_("Card Deck you've selected is empty"), 
							  false, MESSAGE_WARNING);
		w.run ();
	}
}

/** 
	Add all cards from_deck_ to the CardDeck number idx_.
	This method is called from DeckMgr::add_deck_to_box (id) callback.
 */
void
CardBox::
add_deck_to_box (int idx_, Deck* from_deck_)
{
	trace_with_mask("CardBox::add_deck_to_box",GUITRACE);
	Assure_exit (idx_ < CARD_BOX_SIZE);
	Assure_exit (from_deck_);

	if (m_fname.empty ()) {
		Gtk::MessageDialog em (_("First, open or create\na Card Box file."), 
							   false, Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}
	
	CardRef* card_ref;
	VDeck::cardlist_const_iterator iter = from_deck_->begin ();

	while (iter != from_deck_->end ()) {
		card_ref = new CardRef (**iter, *from_deck_);
		card_ref->set_expiration (CONFIG->get_expiration (idx_));
		m_card_deck [idx_].push_back (card_ref);
		iter++;
	}
	if (CONFIG->get_remove_duplicates ()) {
		m_card_deck [idx_].remove_duplicates ();
	}
	m_card_deck [idx_].reset_progress ();
	update_count (idx_);
	m_cdplayers [idx_]->reset ();
	MAINWIN->go_state (Dirty);
}

/**
   Add card from DeckView to the CardBox. This function can be called 
   both from DeckView to add selected card to CardBox #1 or when 
   a new card is created and the configuration option 
   { new_card_to_cardbox } is set to true.

   @param card_ref_ New card reference to add. Assume ownership.
*/
void 
CardBox::
add_card_to_box (CardRef* card_ref_)
{
	trace_with_mask("CardBox::add_card_to_box",GUITRACE);

	if (m_fname.empty ()) {
		Gtk::MessageDialog em (_("First, open or create a Card File."), 
							   false, Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}
	card_ref_->set_expiration (CONFIG->get_expiration (0));
	m_card_deck [0].push_back (card_ref_);
	m_card_deck [0].reset_progress ();
	update_count (0);
	m_cdplayers [0]->reset ();
	MAINWIN->go_state (Dirty);
}

/** Move a card from one Card Box to another.
	This method is invoked by the DeckPlayer (know_answer_cb(), no_clue_cb())
	when the user is either guesses the card right or not.
*/
void
CardBox::
move_card_to_box (VDeck::cardlist_iterator card_ref_, 
				  int from_deck_, 
				  int to_deck_,
				  bool is_expired_)
{
	trace_with_mask("CardBox::move_card_to_box",GUITRACE);

	if (to_deck_ >= CARD_BOX_SIZE) {
		to_deck_ = CARD_BOX_SIZE - 1;
	}

	/** Get hold of CardRef itself
	 */
	VCard* vcard = *card_ref_;

	/** First, remove the card from the old CardDeck.
	 */ 
	DL((GRAPP,"Removing card from the old CardDeck ...\n"));
	m_card_deck [from_deck_].erase (card_ref_);
	update_count (from_deck_);

	if (is_expired_) {
		m_cbs [from_deck_].m_expired_count--;
		set_expiration_light (from_deck_);
		if (m_cbs [from_deck_].m_expired_count == 0) {
			Gtk::MessageDialog msg (_("You have learnt all\nexpired cards in this deck."), false, MESSAGE_INFO);
			msg.run ();
		}
	}

	if (to_deck_ != from_deck_) {
		m_cdplayers [from_deck_]->reset ();
	}

	/** Insert the CardRef in the back of the next CardDeck.
		In case of the same CardDeck (#1 if 'don't know' and #5 if 'know it'),
		the card goes back to the end of the CardDeck.
	*/
	DL((GRAPP,"Placing card in next/first CardDeck ...\n"));
	vcard->set_expiration (CONFIG->get_expiration (to_deck_));
	m_card_deck [to_deck_].push_back (vcard);
	if (to_deck_ != from_deck_) {
		m_card_deck [to_deck_].reset_progress ();
	}
	m_cdplayers [to_deck_]->reset ();
	update_count (to_deck_);
	set_expiration_light (to_deck_);

	MAINWIN->go_state (Dirty);
}

bool 
CardBox::
play_in_box_cb (GdkEventButton* event, int idx_) 
{
	trace_with_mask("CardBox::play_in_box_cb",GUITRACE);

	if (m_fname.empty ()) {
		Gtk::MessageDialog emsg (_("First Open or Create\na New Card File!"), 
								 false, MESSAGE_ERROR);
		emsg.run ();
	}
	else {
		MAINWIN->play_in_box (idx_);
		set_expiration_light (idx_);
	}
	return true;
}

/** Set the expiration light according to the ratio of expired
	cards to the total count:

        [0%;   25%[ - grey; 
        [25%;  50%[ - green; 
        [50%;  75%[ - yellow;
        [75%; 100%] - red.

	@param idx_ Index of the CardDeck
*/
void 
CardBox::
set_expiration_light (int idx_) 
{
	trace_with_mask("CardBox::set_expiration_light",GUITRACE);

	std::ostringstream os;
	os << m_cbs [idx_].m_expired_count;
	m_cbs [idx_].m_sched_count->set_text (os.str ());
	
	if (m_card_deck [idx_].size () == 0) {
		m_cbs [idx_].m_sched_light->set (m_sched_grey);
		return;
	}

	float x = (m_cbs [idx_].m_expired_count * 100.0) /
		m_card_deck [idx_].size();

	DL((GRAPP,"m_expired_count/deck size : %d/%d (x = %f)\n",
		m_cbs [idx_].m_expired_count, m_card_deck[idx_].size(), x)); 

	if (x >= 0 && x < 25.0) {                                  // grey
		m_cbs [idx_].m_sched_light->set (m_sched_grey);
	}
	else if (x >= 25.0 && x < 50.0) {                          // green
		m_cbs [idx_].m_sched_light->set (m_sched_green);
	}
	else if (x >= 50.0 && x < 75.0) {                          // yellow
		m_cbs [idx_].m_sched_light->set (m_sched_yellow);
	}
	else {				                                       // red
		m_cbs [idx_].m_sched_light->set (m_sched_red);
	}
}

void 
CardBox::
on_deck_dropped_cb (const Glib::RefPtr<Gdk::DragContext>& context_,
					int drop_x_coordinate_,
					int drop_y_coordinate_,
					const Gtk::SelectionData& selection_data_,
					guint type_id_,
					guint drop_timestamp_,
					int card_deck_index_)
{
	std::ostringstream os;
	Deck* src_deck;

	trace_with_mask("CardBox::on_deck_dropped_cd",GUITRACE);
	
    /** Doesn't work - I get bogus pointer back! (investigate)
	    Deck* src_deck = (Deck*)(selection_data_.get_data ());
	*/

	src_deck = MAINWIN->get_dnd_source ();

	os << "Proceed with adding Deck \n\"" << src_deck->get_name () << "\"\n"
	   << " to CardBox #" << (card_deck_index_+1) << " ?";

	Gtk::MessageDialog qm (*MAINWIN,
						   _ (os.str ().c_str ()),
						   false,
						   Gtk::MESSAGE_QUESTION, 
						   Gtk::BUTTONS_NONE,
						   true);
	qm.add_button (_("Yes"), Gtk::RESPONSE_YES);
	qm.add_button (_("No"),  Gtk::RESPONSE_NO );
	if (qm.run () == Gtk::RESPONSE_NO) {
		return;
	}

	DL((GRAPP,"Adding Deck \"%s\" to CardBox # %d\n", 
		src_deck->get_name ().c_str (), card_deck_index_));
		
	add_deck_to_box (card_deck_index_, src_deck);
	context_->drag_finish (false, false, drop_timestamp_);
}

void
CardBox::
repaint_deck_players ()
{
	trace_with_mask("CardBox::repaint_deck_players",GUITRACE);

	for (int idx = 0; idx < CARD_BOX_SIZE; idx++) {
		if (m_cdplayers [idx]) {
			m_cdplayers [idx]->repaint ();
		}
	}
}

