// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: EditControls.cpp,v 1.4 2005/11/27 04:59:22 vlg Exp $
//------------------------------------------------------------------------------
//                            EditControls.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2005 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Nov 13 2005
//
//------------------------------------------------------------------------------

#include <gtkmm/table.h>

#include "Granule-main.h"
#include "EditControls.h"

using Gtk::Image;

EditControls::
EditControls (Glib::RefPtr<Gtk::TextBuffer> tb_ref_)
	: Gtk::HBox (false, 2), 
	  m_textbuf (tb_ref_)
{
	trace_with_mask("EditControls::EditControls", GUITRACE);

	Glib::RefPtr<Gdk::Pixbuf> pixbuf_ref;

	vector<Glib::ustring> widget_names;
	widget_names.push_back ("TextBold");
	widget_names.push_back ("TextItalic");
	widget_names.push_back ("TextUnderline");
	widget_names.push_back ("TextMonospace");
	widget_names.push_back ("TextSubscript");
	widget_names.push_back ("TextSuperscript");
	widget_names.push_back ("TextBig");
	widget_names.push_back ("TextSmall");
	widget_names.push_back ("TextRemoveTags");

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_bold.png");
	m_icon_list.push_back (icon_t ("bold", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_italic.png");
	m_icon_list.push_back (icon_t ("italic", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_underlined.png");
	m_icon_list.push_back (icon_t ("underline", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_monospaced.png");
	m_icon_list.push_back (icon_t ("monospace", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_subscript.png");
	m_icon_list.push_back (icon_t ("subscript", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_superscript.png");
	m_icon_list.push_back (icon_t ("superscript", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_big.png");
	m_icon_list.push_back (icon_t ("big", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_small.png");
	m_icon_list.push_back (icon_t ("small", 
								   manage (new Image (pixbuf_ref))));

	pixbuf_ref = Gdk::Pixbuf::create_from_file (
		GRAPPDATADIR "pixmaps/text_remove_tags.png");
	m_icon_list.push_back (icon_t ("remove_tags", 
								   manage (new Image (pixbuf_ref))));

	icon_list_t::iterator iter = m_icon_list.begin ();

	Gtk::Button* button;
	Gtk::Table* elems_table = Gtk::manage (new Gtk::Table (2, 5, false));

	elems_table->set_border_width (0);
	elems_table->set_row_spacings (1);
	elems_table->set_col_spacings (1);
	elems_table->set_name ("EditControlsTable");

	int row = 0;
	int col = 0;
	
	/** unset_flags () takes buttons out of the dialog's focus chain.
		There is no other way it can be achieved with set_focus_chain()
		in CardView dialog. I tried it and it doesn't work.
	*/

	while (iter != m_icon_list.end ()) {
		button = manage (new Gtk::Button ());
		button->unset_flags (Gtk::CAN_FOCUS);
		button->add (*((*iter).second));
		elems_table->attach (*button, col, col+1, row, row+1, 
							 Gtk::FILL, Gtk::FILL, 0, 0);
		if (++col == 2) {
			row++;
			col = 0;
		}
		button->signal_clicked ().connect (
			bind<string>(mem_fun(*this, &EditControls::on_edit_button_clicked),
						 (*iter).first));
		iter++;
	}
//	elems_table->unset_focus_chain ();

	this->pack_start (*elems_table, Gtk::PACK_SHRINK);
}

void
EditControls::
on_edit_button_clicked (const string& action_)
{
	trace_with_mask("EditControls::on_edit_button_clicked",GUITRACE);
	
	DL ((GRAPP,"Inserting <%s> tag.\n", action_.c_str ()));

	Gtk::TextBuffer::iterator iter;
	Gtk::TextBuffer::iterator start_iter;
	Gtk::TextBuffer::iterator end_iter;

	/** Iterators are invalidated between text changes.
		But marks are not. They are placed in the place where
		the text used to be it it were deleted. 
		Comparing the marks, however, turned out to be useless - 
		you ought to compares iterators only.
	*/
	if (!m_textbuf->get_selection_bounds (start_iter, end_iter)) {
		DL ((GRAPP,"No selection has been made\n"));
		return;
	}

	if (start_iter == end_iter) {
		DL ((GRAPP,"Selection size is 0.\n"));
		return;
	}

	/** We got a valid selection - mark it now. We cannot use
		predefined selection marks "insert" and "selection_bound"
		because if the user made the selection by dragging pointer
		in the opposite direction, we would get iterators in reverse
		and Gtkmm doesn't allow for iterators comparison.
	 */
	Glib::RefPtr<Gtk::TextBuffer::Mark> mb;
	Glib::RefPtr<Gtk::TextBuffer::Mark> me;

	mb = m_textbuf->create_mark ("start_selection", start_iter);
	me = m_textbuf->create_mark ("end_selection",   end_iter);
	iter = start_iter;

	DL ((GRAPP,"selection: from='%c' to='%c'\n", 
		 display_unichar_at (iter),
		 display_unichar_at (me->get_iter ())));

	if (action_ == "bold") {
		m_textbuf->insert (iter, "<b>");
		m_textbuf->insert (me->get_iter (), "</b>");
	}
	else if (action_ == "italic") {
		m_textbuf->insert (iter, "<i>");
		m_textbuf->insert (me->get_iter (), "</i>");
	}
	else if (action_ == "underline") {
		m_textbuf->insert (iter, "<u>");
		m_textbuf->insert (me->get_iter (), "</u>");
	}
	else if (action_ == "monospace") {
		m_textbuf->insert (iter, "<tt>");
		m_textbuf->insert (me->get_iter (), "</tt>");
	}
	else if (action_ == "subscript") {
		m_textbuf->insert (iter, "<sub>");
		m_textbuf->insert (me->get_iter (), "</sub>");
	}
	else if (action_ == "superscript") {
		m_textbuf->insert (iter, "<sup>");
		m_textbuf->insert (me->get_iter (), "</sup>");
	}
	else if (action_ == "big") {
		m_textbuf->insert (iter, "<big>");
		m_textbuf->insert (me->get_iter (), "</big>");
	}
	else if (action_ == "small") {
		m_textbuf->insert (iter, "<small>");
		m_textbuf->insert (me->get_iter (), "</small>");
	}
	else if (action_ == "remove_tags") {
		remove_tags (mb, me);
	}

	else {
		DL ((GRAPP,"Unexpected action!\n"));
	}
	m_textbuf->place_cursor (m_textbuf->begin ());

	m_textbuf->delete_mark (mb);
	m_textbuf->delete_mark (me);
}

/** We remove all complete individual tags. The incomplete tags
	are ignored. The tag context is also ignored - we don't match
	open/close tags at all.
 */
void 
EditControls::
remove_tags (Glib::RefPtr<Gtk::TextBuffer::Mark> begin_mark_,
			 Glib::RefPtr<Gtk::TextBuffer::Mark> end_mark_)
{
    trace_with_mask("EditControls::remove_tags",GUITRACE);

	gunichar open_brace = '<';
	gunichar close_brace = '>';

	Gtk::TextBuffer::iterator next_ch = begin_mark_->get_iter ();
	Gtk::TextBuffer::iterator last_ch = end_mark_->get_iter ();

	DL ((GRAPP,"scanning from='%c' to='%c'\n", 
		 display_unichar_at (next_ch),
		 display_unichar_at (last_ch)));
	
	while (next_ch && next_ch != last_ch) {
		if (next_ch.get_char () == open_brace) {
			Glib::RefPtr<Gtk::TextBuffer::Mark> start_mark;
			start_mark = m_textbuf->create_mark ("start", next_ch);
			next_ch++;

			DL ((GRAPP,"outer: next_ch='%c', last_ch='%c'\n", 
				 display_unichar_at (next_ch),
				 display_unichar_at (last_ch)));

			while (next_ch && next_ch != last_ch) {
				if (next_ch.get_char () == close_brace) {
					DL ((GRAPP,"found tag - erasing from '%c' to '%c' ...\n",
						 display_unichar_at (start_mark->get_iter ()),
						 display_unichar_at (next_ch)));

					next_ch++;
					m_textbuf->erase (start_mark->get_iter (), next_ch);

					next_ch = start_mark->get_iter ();
					last_ch = end_mark_->get_iter ();

					DL ((GRAPP,"erase: next_ch='%c', last_ch='%c'\n", 
						 display_unichar_at (next_ch), 
						 display_unichar_at (last_ch)));
					break;
				}
				next_ch++;

				DL ((GRAPP,"inner: next_ch='%c', last_ch='%c'\n", 
					 display_unichar_at (next_ch),
					 display_unichar_at (last_ch)));
			}
			m_textbuf->delete_mark (start_mark);
		}
		next_ch++;

		DL ((GRAPP,"outer: next_ch='%c', last_ch='%c'\n", 
			 display_unichar_at (next_ch), 
			 display_unichar_at (last_ch)));
	}
}

void
EditControls::
on_focus_changed (Gtk::TextView* text_view_)
{
    trace_with_mask("EditControls::on_focus_changed",GUITRACE);

	m_textbuf->place_cursor (m_textbuf->begin ()); // cancel old select
	m_textbuf = text_view_->get_buffer ();
}

