// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: DeckView.cpp,v 1.41 2006/03/19 15:39:44 vlg Exp $
//------------------------------------------------------------------------------
//                            DeckView.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2004,2006 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Fri Feb  6 23:37:11 EST 2004
//
//------------------------------------------------------------------------------

#include <sstream>
#include <gtkmm/messagedialog.h>
#include <gtkmm/stock.h>
#include <gtkmm/buttonbox.h>

#include <assa/AutoPtr.h>

#include "Granule-main.h"
#include "Granule.h"
#include "MainWindow.h"
#include "Card.h"
#include "CardView.h"
#include "CardDeck.h"
#include "Deck.h"
#include "DeckView.h"
#include "DeckInfo.h"
#include "ButtonWithImageLabel.h"

using sigc::bind;
using sigc::mem_fun;

DeckView::
DeckView (VDeck& deck_, int selected_row_) 
		:
#ifdef IS_HILDON
	Hildon::AppView ("Deck View"),
#endif
	m_add_button    (manage (new Gtk::Button (Gtk::StockID ("gtk-add")))),
	m_edit_button   (
		manage (new ButtonWithImageLabel (Gtk::Stock::DND, "Edit"))),
	m_delete_button (manage (new Gtk::Button (Gtk::StockID("gtk-delete")))),
	m_save_deck_button (
		manage (new ButtonWithImageLabel(Gtk::Stock::SAVE,"Save Deck"))),
	m_label_count   (manage (new Gtk::Label ("0 cards"))),
	m_more_button   (
		manage (new ButtonWithImageLabel (
					Gtk::Stock::DIALOG_INFO, "Deck Info"))),
	m_card_down_button (
		manage (new Gtk::Button (Gtk::Stock::GO_DOWN))),
	m_card_up_button (
		manage (new Gtk::Button (Gtk::Stock::GO_UP))),
	m_add_card_to_cardbox (
		manage (new ButtonWithImageLabel (Gtk::Stock::UNDO, "Add to Box 1 "))),
	m_deck           (deck_),
	m_row_idx        (selected_row_),
	m_modified       (false)
{
	trace_with_mask("DeckView::DeckView", GUITRACE);

	set_title("Deck View");
	set_border_width (2);

#ifdef IS_HILDON
	/**
	 * Not a dialog any more - it is a Bin (AppView) instead.
	 * We handle packaging ourselves and mute all attributes of a 
	 * Dialog everywhere down the line.
	 */
    m_vbox = manage (new Gtk::VBox (false, 2));
	add (*m_vbox);

	set_size_request (672, 396);
//	set_transient_for (*HILDONAPPWIN);
#else
	set_resizable(true);
	set_position (Gtk::WIN_POS_CENTER_ALWAYS);
	set_size_request (633, 410);	// width; height 
	set_transient_for (*MAINWIN);
#endif

	Gtk::Label* label_total = manage (new Gtk::Label ("Total: "));
	Gtk::VBox* vbox         = manage (new Gtk::VBox(false, 0));
	Gtk::HBox* h_infobox    = manage (new Gtk::HBox(false, 0));
	Gtk::HBox* h_viewbox    = manage (new Gtk::HBox(false, 0));

	m_add_button         ->set_flags  (Gtk::CAN_FOCUS);
	m_add_button         ->set_relief (Gtk::RELIEF_NORMAL);
	m_edit_button        ->set_flags  (Gtk::CAN_FOCUS);
	m_edit_button        ->set_relief (Gtk::RELIEF_NORMAL);
    m_save_deck_button   ->set_flags  (Gtk::CAN_FOCUS);
	m_save_deck_button   ->set_relief (Gtk::RELIEF_NORMAL);
	m_delete_button      ->set_flags  (Gtk::CAN_FOCUS);
	m_delete_button      ->set_relief (Gtk::RELIEF_NORMAL);
	m_more_button        ->set_flags  (Gtk::CAN_FOCUS);
	m_more_button        ->set_relief (Gtk::RELIEF_NORMAL);
	m_card_down_button   ->set_flags  (Gtk::CAN_FOCUS);
	m_card_down_button   ->set_relief (Gtk::RELIEF_NORMAL);
	m_card_up_button     ->set_flags  (Gtk::CAN_FOCUS);
	m_card_up_button     ->set_relief (Gtk::RELIEF_NORMAL);
	m_add_card_to_cardbox->set_flags  (Gtk::CAN_FOCUS);
	m_add_card_to_cardbox->set_relief (Gtk::RELIEF_NORMAL);

	vbox->set_size_request (122,-1);
	vbox->pack_start (*m_add_button,          Gtk::PACK_SHRINK, 4);
	vbox->pack_start (*m_edit_button,         Gtk::PACK_SHRINK, 4);
	vbox->pack_start (*m_delete_button,       Gtk::PACK_SHRINK, 4);

	vbox->pack_start (*m_card_up_button,      Gtk::PACK_SHRINK, 4);
	vbox->pack_start (*m_card_down_button,    Gtk::PACK_SHRINK, 4);
	vbox->pack_start (*m_add_card_to_cardbox, Gtk::PACK_SHRINK, 4);

	vbox->pack_start (*m_more_button,         Gtk::PACK_SHRINK, 4);
	vbox->pack_start (*m_save_deck_button,    Gtk::PACK_SHRINK, 4);

	label_total->set_alignment  (0.5,0.5);
	label_total->set_padding    (0,0);
	label_total->set_justify    (Gtk::JUSTIFY_LEFT);
	label_total->set_line_wrap  (false);
	label_total->set_use_markup (false);
	label_total->set_selectable (false);

	m_label_count->set_alignment  (0.5,0.5);
	m_label_count->set_padding    (0,0);
	m_label_count->set_justify    (Gtk::JUSTIFY_LEFT);
	m_label_count->set_line_wrap  (false);
	m_label_count->set_use_markup (false);
	m_label_count->set_selectable (false);

	h_infobox->pack_start (*label_total,   Gtk::PACK_SHRINK, 9);
	h_infobox->pack_start (*m_label_count, Gtk::PACK_SHRINK, 0);

	/** <Close> button
	 */
#ifdef IS_HILDON
	m_close_button = manage (new ButtonWithImageLabel (Gtk::Stock::CLOSE, 
													   "Close"));
#else
	m_close_button = add_button (Gtk::StockID ("gtk-close"), 
								 Gtk::RESPONSE_CLOSE );
#endif
	m_close_button->set_flags  (Gtk::CAN_FOCUS);
	m_close_button->set_relief (Gtk::RELIEF_NORMAL);

	m_close_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_close_clicked));

	/** List View 
	 */
	m_scrollwin = manage (new Gtk::ScrolledWindow);
	m_scrollwin->set_shadow_type (Gtk::SHADOW_ETCHED_OUT);
	m_scrollwin->set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	Gtk::TreeModel::Row row;
	Card* card;
	m_list_store_ref = Gtk::ListStore::create (m_columns);
	VDeck::cardlist_iterator iter = m_deck.begin ();

	while (iter != m_deck.end ()) {
		row = *m_list_store_ref->append ();
		row [m_columns.m_front_row] = 
			Granule::trim_multiline ((*iter)->get_question ());
		row [m_columns.m_back_row ] = 
			Granule::trim_multiline ((*iter)->get_answer ());
		row [m_columns.m_card     ] = *iter;
		iter++;
	}

	/** Make both Front and Back columns sortable.
	 */
	m_list_store_ref->set_sort_func (
		0, (mem_fun (*this, &DeckView::on_sort_compare_front)));

	m_list_store_ref->set_sort_func (
		1, (mem_fun (*this, &DeckView::on_sort_compare_back)));

	/** Create TreeView
	 */
	DL((GRAPP,"Create TreeView\n"));
	m_tree_view.set_model (m_list_store_ref);
	m_tree_view.set_rules_hint ();

	Gtk::TreeView::Column* column;
	Gtk::CellRendererText* renderer;

	/** Front: Column
	 */
	column = Gtk::manage (new Gtk::TreeView::Column ("Front"));
	m_tree_view.append_column (*column);
	renderer = Gtk::manage (new Gtk::CellRendererText);

	column->pack_start (*renderer, false);
	column->add_attribute (renderer->property_markup (), 
						   m_columns.m_front_row);
						  
	column->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
	column->set_fixed_width (140);
	column->set_resizable ();

	/** Enable column sorting
	 */
	column->set_sort_column_id (m_columns.m_front_row);
	column->set_sort_indicator (true);
	column->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_column_sorted));

	/** Back: Column
	 */
	column = Gtk::manage (new Gtk::TreeView::Column ("Back"));
	m_tree_view.append_column (*column);
	renderer = Gtk::manage (new Gtk::CellRendererText);

	column->pack_start (*renderer, false);
	column->add_attribute (renderer->property_markup (), 
						   m_columns.m_back_row);

	column->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
	column->set_fixed_width (140);

	/** Enable column sorting
	 */
	column->set_sort_column_id (m_columns.m_back_row);
	column->set_sort_indicator (true);
	column->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_column_sorted));

	m_tree_sel_ref = m_tree_view.get_selection ();
	m_tree_sel_ref->set_mode (Gtk::SELECTION_SINGLE);
	m_tree_sel_ref->signal_changed ().connect (
		mem_fun (*this, &DeckView::on_card_selected));

	m_scrollwin->add (m_tree_view);

	/** Initialize callbacks 
	 */
	m_add_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_add_clicked));
	m_edit_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_edit_clicked));
	m_save_deck_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_save_deck_clicked));
	m_delete_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_delete_clicked));
	m_more_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_more_clicked));
	m_card_up_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_card_up_clicked));
	m_card_down_button->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_card_down_clicked));
	m_add_card_to_cardbox->signal_clicked ().connect (
		mem_fun (*this, &DeckView::on_add_card_to_cardbox));

	/** Pack 'em all
	 */
	h_viewbox->pack_start (*m_scrollwin);
	h_viewbox->pack_start (*vbox, Gtk::PACK_SHRINK, 8);

	get_vbox ()->pack_start (*h_viewbox);
	get_vbox ()->pack_start (*h_infobox, Gtk::PACK_SHRINK, 4);

#ifdef IS_HILDON
	/**
	 * Add separator and [Close] button in the ButtonBox
	 */
	Gtk::HSeparator* hseparator = manage (new Gtk::HSeparator);
	get_vbox ()->pack_start (*hseparator, false, true, 0);

	Gtk::HButtonBox*  hbuttonbox = manage (new Gtk::HButtonBox);
	hbuttonbox->set_spacing (6);
	hbuttonbox->set_border_width (6);

	get_vbox ()->pack_start (*hbuttonbox, false, true, 0);
	hbuttonbox->pack_end (*m_close_button, false, true, 0);
#endif

	set_cards_count ();

	show_all ();

	/** Select the same row (card) that the DeckPlayer has selected,
		and scroll to it. 

		NOTE: There are two stunning points worth noting here.
		First, upon executing show_all(), Gtkmm internals set the
		selection to row # 0 and then emmit *changed* signal!
		Bottom line - if you want to select a row in c'tor, do 
		your selection AFTER show_all(). 
		Secondly, select (iter) doesn't work, but select (row) does.
	 */
	DL((APP,"m_row_idx = %d\n", m_row_idx));
	if (m_row_idx >= 0) {
		m_row_idx = selected_row_;
		children_t children = m_list_store_ref->children();
		chiter_t iter = children [m_row_idx];
		row = *iter;
		m_tree_sel_ref->select (row);
	}
	set_sensitivity ();
	set_win_name ();

#ifndef IS_HILDON
	set_icon_from_file (GRAPPDATADIR "/pixmaps/deckview_32x32.png");
#endif

}

DeckView::
~DeckView ()
{
	trace_with_mask("DeckView::~DeckView", GUITRACE);
}

void
DeckView::
set_cards_count ()
{
	std::ostringstream os;
	os << m_deck.size () << " cards.";
	m_label_count->set_text (os.str ());
}

//==============================================================================
//                              Callbacks
//------------------------------------------------------------------------------

#ifdef IS_HILDON
gint
DeckView::
run ()
{
    trace_with_mask("DeckView::run",GUITRACE);

	GRANULE->register_appview (this);
	Gtk::Main::run ();
	return (Gtk::RESPONSE_NONE);
}
#endif

void
DeckView::
on_close_clicked ()  
{
	trace_with_mask("DeckView::on_close_clicked", GUITRACE);

#ifdef IS_HILDON
	hide ();
	GRANULE->unregister_appview (this);
	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
#endif
}

void
DeckView::
on_column_sorted ()
{
	trace_with_mask("DeckView::on_column_sorted", GUITRACE);

	/** Swapping elements works only with unsorted stores!
	 */
	m_card_up_button  ->set_sensitive (false);
	m_card_down_button->set_sensitive (false);

	on_card_selected ();
}

void
DeckView::
on_card_selected ()
{
	trace_with_mask("DeckView::on_card_selected", GUITRACE);

	Gtk::TreeModel::iterator iter = m_tree_sel_ref->get_selected ();
	if (!iter) {
		return;
	}
	Gtk::TreeModel::Path path = m_list_store_ref->get_path (iter);
	m_row_idx = *(path.get_indices ().begin ());
	DL((GRAPP,"Row selected : %d\n", m_row_idx));
}

void
DeckView::
add_new_card (Card* card_)
{
	trace_with_mask("DeckView::add_new_card", GUITRACE);

	m_deck.push_back (card_);

	Gtk::TreeModel::Row row = *m_list_store_ref->append ();
	row [m_columns.m_front_row] = 
		Granule::trim_multiline (card_->get_question ());
	row [m_columns.m_back_row ] = 
		Granule::trim_multiline (card_->get_answer ());
	row [m_columns.m_card     ] = card_;
	m_tree_sel_ref->select (row);
	set_cards_count ();
	autoscroll_down ();
	m_modified = true;
}

void 
DeckView::
on_add_clicked ()
{
	trace_with_mask("DeckView::on_add_clicked", GUITRACE);

	CardView cview (new Card);

	int ret = cview.run (*this);
	DL((GRAPP,"cview.run () = %d\n", ret));

	if (ret == Gtk::RESPONSE_OK) {
		DL((GRAPP,"Card accepted\n"));
	}
	else if (ret == Gtk::RESPONSE_CANCEL) {
		DL((GRAPP,"Card cancelled\n"));
	}
	else if (ret == Gtk::RESPONSE_REJECT) {
		DL((GRAPP,"Text entry syntax error (wrong markup?)\n"));
	}
	else {						
		DL((GRAPP|ASSA::ERROR, "Unexpected response = %d, "
			"see /path/to/gtkmm/dialog.h for details.\n", ret));
	}
}

void 
DeckView::
on_edit_clicked ()
{
	trace_with_mask("DeckView::on_edit_clicked", GUITRACE);

	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}
	Gtk::TreeModel::Row row = *m_list_store_ref->children ()[m_row_idx];
	VCard* card_iter = row [m_columns.m_card];

#ifdef IS_HILDON
	hide ();
#endif

	CardView cview (card_iter);
	int ret = cview.run ();

	if (ret == Gtk::RESPONSE_OK) {
		row [m_columns.m_front_row] = 
			Granule::trim_multiline(card_iter->get_question());
		row [m_columns.m_back_row ] = 
			Granule::trim_multiline(card_iter->get_answer  ());
	}

#ifdef IS_HILDON
	show ();
#endif
}

void 
DeckView::
on_delete_clicked ()
{
	trace_with_mask("DeckView::on_delete_clicked", GUITRACE);
	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}
	/** Delete card from the deck. We can be looking at either Deck
		or CardDeck. Either way the card is completely deleted from
		the Deck (even via CardDeck->VCard->CardRef->Deck route).
	 */
	Gtk::TreeModel::Row row = *m_list_store_ref->children ()[m_row_idx];
	VCard* vcard = row [m_columns.m_card];
	m_deck.erase (vcard);

	Deck* real_deck = NULL;
	if ((real_deck = dynamic_cast<Deck*>(&m_deck)) != NULL) {
		real_deck->mark_as_modified (); 
	}

	/** Delete card from the view list
	 */
	children_t children = m_list_store_ref->children();
	chiter_t chiter = children [m_row_idx];
	m_list_store_ref->erase (chiter);

	if (m_row_idx == m_deck.size ()) {
		m_row_idx--;
	}
	if (m_row_idx >= 0) {
		chiter = children [m_row_idx];
		Gtk::TreeModel::Row row = *chiter;
		m_tree_sel_ref->select (row);
		set_cards_count ();
		autoscroll_down ();
	}
	m_modified = true;
}

void 
DeckView::
on_more_clicked ()
{
	trace_with_mask("DeckView::on_more_clicked", GUITRACE);

#ifdef IS_HILDON
	DeckInfo deck_info (*HILDONAPPWIN, dynamic_cast<Deck&>(m_deck));
	deck_info.run ();
#else
	DeckInfo deck_info (*this, dynamic_cast<Deck&>(m_deck));
	deck_info.run ();
#endif

}

void
DeckView::
autoscroll_up ()
{
    trace_with_mask("DeckView::autoscroll_up",GUITRACE);
                                                                                
    Gtk::Adjustment* adj = m_scrollwin->get_vadjustment ();
                                                                                
    double pg_inc = adj->get_page_increment ();
    double value  = adj->get_value ();
    double prh    = adj->get_upper () / m_list_store_ref->children().size ();

    if (((m_row_idx + 1) * prh - value) < prh) {
        adj->set_value (value - pg_inc/2);
    }
}

void
DeckView::
autoscroll_down ()
{
    trace_with_mask("DeckView::autoscroll_down",GUITRACE);

    Gtk::Adjustment* adj = m_scrollwin->get_vadjustment ();

	double tmp;
    double pg_inc    = adj->get_page_increment ();
    double value     = adj->get_value ();
    double prh       = adj->get_upper () / m_list_store_ref->children().size ();
    double pg_sz     = adj->get_page_size ();
	double last_page = adj->get_upper () - pg_sz;
                                                                                
    if (((m_row_idx + 1) * prh - value) > pg_sz) {
		if ((tmp = value + pg_inc/2) > last_page) {
			adj->set_value (last_page);
		}
		else {
			adj->set_value (tmp);
		}
    }
}

void
DeckView::
on_card_up_clicked ()
{
    trace_with_mask("DeckView::on_card_up_clicked",GUITRACE);

	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}

	if (m_row_idx == 0) {
		return;
	}

	children_t children = m_list_store_ref->children();
	chiter_t a = children [m_row_idx];	

	Gtk::TreeModel::Row row = *a;
	VCard* vcard = row [m_columns.m_card];

	/** Swap adjacent VCard elements in the Deck
	 */
	Deck& deck = dynamic_cast<Deck&> (m_deck); // might throw an exception
	deck.swap_with_prev (vcard);

	/** Update ListStore
	 */
	chiter_t b = children [--m_row_idx];
	m_list_store_ref->iter_swap (a, b);
	autoscroll_up ();
	m_modified = true;
}
	
void
DeckView::
on_card_down_clicked ()
{
    trace_with_mask("DeckView::on_card_down_clicked",GUITRACE);

	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}

	if (m_row_idx == m_deck.size ()-1) {
		return;
	}
	children_t children = m_list_store_ref->children();
	chiter_t a = children [m_row_idx];	

	Gtk::TreeModel::Row row = *a;
	VCard* vcard = row [m_columns.m_card];

	/** Swap adjacent VCard elements in the Deck
	 */
	Deck& deck = dynamic_cast<Deck&> (m_deck);     // might throw an exception
	deck.swap_with_next (vcard);                  

	/** Update ListStore
	 */
	chiter_t b = children [++m_row_idx];
	m_list_store_ref->iter_swap (a, b);
	autoscroll_down ();
	m_modified = true;
}

void
DeckView::
on_add_card_to_cardbox ()
{
    trace_with_mask("DeckView::on_add_card_to_cardbox",GUITRACE);

	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}

	if (m_deck.get_name ().find ("Untitled") != std::string::npos) {
		Gtk::MessageDialog em1 ("Operation is not permitted!\n"
								"Save the 'Untitled' Deck first.",
								Gtk::MESSAGE_ERROR);
		em1.run ();
		return;
	}

	VDeck::cardlist_iterator card_iter = m_deck.begin () + m_row_idx;
	CARDBOX->add_card_to_box (
		new CardRef (**card_iter, dynamic_cast<Deck&>(m_deck)));
}

/**
   Sort Front column of words.
   The parts of the text may be marked up with Pango markup language.
   Remove markup. Also, remove 'to ', 'a ', 'an ' prefixes.
   Then compare the first word.
*/
int 
DeckView::
sort_compare_impl (const Gtk::TreeModel::iterator& a_, 
				   const Gtk::TreeModel::iterator& b_,
				   int column_)
{
    trace_with_mask("DeckView::sort_compare_impl",GUITRACE);

	gchar* txt_a;
	gchar* txt_b;
	gchar* pa;
	gchar* pb;
	int ret;
	int i;

	const Gtk::TreeModel::Row row_a = *a_;
	const Gtk::TreeModel::Row row_b = *b_;
	
	Glib::ustring a;
	Glib::ustring b;

//	DL ((GRAPP, "Sorting %s column\n", (column_ == 0 ? "Front" : "Back")));

	if (column_ == 0) {
		a = row_a [m_columns.m_front_row];
		b = row_b [m_columns.m_front_row];
	}
	else {
		a = row_a [m_columns.m_back_row];
		b = row_b [m_columns.m_back_row];
	}

	pa = txt_a = Granule::strip_pango_markup (a.c_str ());
	pb = txt_b = Granule::strip_pango_markup (b.c_str ());

	Granule::remove_common_prefixes (txt_a);
	Granule::remove_common_prefixes (txt_b);

//	DL ((APP,"Compare: \"%s\" vs. \"%s\"\n", txt_a, txt_b));

	ret = strcmp (txt_a, txt_b);
	g_free (pa);
	g_free (pb);

	return ret;
}

//==============================================================================
//                              Utilities
//------------------------------------------------------------------------------
void
DeckView::
set_sensitivity ()
{
	if (dynamic_cast<Deck*>(&m_deck) != NULL) {
		return;
	}

	m_add_button          ->set_sensitive (false);
	m_more_button         ->set_sensitive (false);
	m_card_up_button      ->set_sensitive (false);
	m_card_down_button    ->set_sensitive (false);
	m_add_card_to_cardbox ->set_sensitive (false);
}

void
DeckView::
set_win_name ()
{
	string title = m_deck.get_name ();
	string::size_type idx = title.rfind (G_DIR_SEPARATOR);
    if (idx != string::npos) {
        title.replace (0, idx+1, "");
    }
	set_title ("Deck View : " + title);
}

void
DeckView::
on_save_deck_clicked ()
{
	if (m_row_idx < 0) {
		Gtk::MessageDialog em ("The deck is empty!",Gtk::MESSAGE_ERROR);
		em.run ();
		return;
	}

	DECKMGR->save_deck_cb ();
}
