// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: CardBox.h,v 1.29 2006/09/30 20:42:27 vlg Exp $
//------------------------------------------------------------------------------
//                            CardBox.h
//------------------------------------------------------------------------------
//  Copyright (c) 2004-2006 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Sat Apr 17 23:37:11 EDT 2004
//
//------------------------------------------------------------------------------
#ifndef CARD_BOX_H
#define CARD_BOX_H

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <vector>

#include <gdkmm/pixbuf.h>
#include <gdkmm/image.h>
#include <gtkmm/box.h>
#include <gtkmm/label.h>

#include "Granule-main.h"
#include "CardDeck.h"
#include "DeckPlayer.h"
#include "VDeck.h"

class Deck;
class VCard;

/**
 * CardBox is a GUI tray that holds all 5 rows of widgets that 
 * graphically represent card boxes. Each box is filled with a CardDeck
 * that holds a list of references to the real cards. Thus CardBox is
 * in essence a DeckList of virtual decks.
 *
 * It is responsible for loading and saving the card file as requested 
 * by MainWindow.
 */

class CardBox : public Gtk::HBox
{
public:
	CardBox ();
	~CardBox ();

	void new_file ();
	int  load (const string& fname_);
	int  save ();
	int  save_as (const string& fname_);
	void close ();				// Clean up internals

	void play_deck (int idx_);

	void add_deck_to_box  (int idx_, Deck* from_deck_);
	int  add_cards_to_box (std::vector<CardRef*>& cardref_list_);
	void move_card_to_box (VDeck::cardlist_iterator card_ref_, 
						   int from_deck_, 
						   int to_deck_,
						   bool is_expired_);

	bool play_in_box_cb (GdkEventButton* event, int idx_);

	void on_deck_dropped_cb (const Glib::RefPtr<Gdk::DragContext>& context_,
							 int drop_x_coordinate_,
							 int drop_y_coordinate_,
							 const Gtk::SelectionData& selection_data_,
							 guint type_id_,
							 guint drop_timestamp_,
							 int card_deck_index_);

	std::string get_fname () const { return m_fname; }

	/** Some preferences might have changed. 
		Ask each opened DeckPlayer repaint itself according
		to the new configuration.
	*/
	void repaint_deck_players ();

	void export_sound_bits (const std::string& path_);
	void mark_as_modified ();

private:
	struct CardBoxIF 
	{
		Gtk::Image* m_sched_light; // Light button to indicate % of overdue 
		Gtk::Label* m_sched_count; // How many cards are overdue
		Gtk::VBox*  m_sched_vbox;  // Vbox to hold scheduling widgets
		Gtk::Image* m_box_pic;     // Image of the box with(out) cards
		Gtk::Label* m_total_count; // Number of cards in the cardbox
		int         m_expired_count; // Count of cards that have expired

		CardBoxIF () : m_sched_light (NULL), m_sched_count (NULL),
					   m_sched_vbox (NULL),  m_box_pic (NULL), 
					   m_total_count (NULL), m_expired_count (0) 
			{ /* no-op */ }

		void reset () 
		{
			m_sched_count->set_text ("0");
			m_total_count->set_text ("0 cards");
			m_expired_count = 0;
		}
		void add_one () { m_expired_count++; }
	};

	int  parse_xml_tree (xmlDocPtr parser_, string error_msg_);
	void update_count   (int idx_);

	/** Remove duplicates across all CardDecks if enabled in
		preferences.
	*/
	void remove_all_duplicates (int idx_);

	/** Search for CardRef in all CardDecks.
	 */
	bool find_in_all_carddecks (CardRef* card_ref_);

	/** Take an absolute path and absolute file name and calculate resultant 
		file name in relation to the absolute path.
	*/
	static string calculate_relative_path (const string& abs_pname_,
										   const string& abs_fname_);

	/** Take an absolute path to the directory and a relative file name and 
		calculate resultant absolute file name.
	*/
	static string calculate_absolute_path (const string& abs_pname_,
										   const string& rel_fname_);

private:
	std::vector<CardBoxIF> m_cbs;       // GUI representation of a box

	Glib::RefPtr<Gdk::Pixbuf> m_drawer_empty;
	Glib::RefPtr<Gdk::Pixbuf> m_drawer_one;
	Glib::RefPtr<Gdk::Pixbuf> m_drawer_many;

	Glib::RefPtr<Gdk::Pixbuf> m_sched_green;
	Glib::RefPtr<Gdk::Pixbuf> m_sched_grey;
	Glib::RefPtr<Gdk::Pixbuf> m_sched_red;
	Glib::RefPtr<Gdk::Pixbuf> m_sched_yellow;

	void set_expiration_light (int idx_);

private:
	/** Name of the XML file to load. When m_fname is empty, no file is
		loaded.
	*/
	string      m_fname;
	CardDeck    m_card_deck [CARD_BOX_SIZE];
	DeckPlayer* m_cdplayers [CARD_BOX_SIZE];
};

inline CardBox::
~CardBox ()
{
	trace_with_mask("CardBox::~CardBox",GUITRACE);
}

#endif /* CARD_BOX_H */
