/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <panel-applet.h>
#include "config.h"
#include "grdcpixmaps.h"
#include "grdcappletmenuitem.h"

#define GRDC_APPLET_FACTORY_IID "OAFIID:Grdc_Applet_Factory"
#define GRDC_APPLET_IID         "OAFIID:Grdc_Applet"

#define MAX_PATH_LEN 255

typedef struct _GrdcAppletData
{
    PanelApplet *applet;

    GtkWidget *image;
} GrdcAppletData;

typedef enum
{
    GRDC_LAUNCH_MAIN,
    GRDC_LAUNCH_QUICK,
    GRDC_LAUNCH_FILE
} GrdcLaunchType;

static void
grdc_applet_launcher (GrdcLaunchType launch_type, const gchar *filename)
{
    gint argc;
    gchar *argv[50];
    gint i;
    GError *error = NULL;
    gboolean ret;
    GtkWidget *dialog;

    argc = 0;
    argv[argc++] = g_strdup ("grdc");

    switch (launch_type)
    {
    case GRDC_LAUNCH_MAIN:
        break;

    case GRDC_LAUNCH_QUICK:
        argv[argc++] = g_strdup ("-q");
        break;

    case GRDC_LAUNCH_FILE:
        argv[argc++] = g_strdup ("-c");
        argv[argc++] = g_strdup (filename);
        break;
    }    

    argv[argc++] = NULL;

    ret = g_spawn_async (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        NULL, /* child_pid */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (!ret)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            error->message);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }
}

static void
grdc_applet_destroy (GtkWidget *widget, GrdcAppletData *appdata)
{
    g_free (appdata);
}

static void
grdc_applet_change_size (GtkWidget *w, gint size, GrdcAppletData *appdata)
{
	GdkPixbuf *scaled;

	scaled = gdk_pixbuf_scale_simple (grdc_pixmap_logo (), size - 2, size - 2, GDK_INTERP_HYPER);
	gtk_image_set_from_pixbuf (GTK_IMAGE (appdata->image), scaled);
	g_object_unref (scaled);
	gtk_widget_set_size_request (GTK_WIDGET (appdata->image), MAX (11, size), MAX (11, size));
}

static void
grdc_applet_menu_open_main (GrdcAppletMenuItem *menuitem, gpointer data)
{
    grdc_applet_launcher (GRDC_LAUNCH_MAIN, NULL);
}

static void
grdc_applet_menu_open_quick (GrdcAppletMenuItem *menuitem, gpointer data)
{
    grdc_applet_launcher (GRDC_LAUNCH_QUICK, NULL);
}

static void
grdc_applet_menu_open_file (GrdcAppletMenuItem *menuitem, gpointer data)
{
    grdc_applet_launcher (GRDC_LAUNCH_FILE, menuitem->filename);
}

static void
grdc_applet_popup_menu_position (GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer data)
{
    GtkRequisition req;
    gint tx, ty, size;
    GrdcAppletData *appdata;

    appdata = (GrdcAppletData*) data;
    size = panel_applet_get_size (appdata->applet);
    gdk_window_get_origin (GTK_WIDGET (appdata->applet)->window, &tx, &ty);
    gtk_widget_size_request (GTK_WIDGET (menu), &req);

    switch (panel_applet_get_orient (appdata->applet))
    {
    case PANEL_APPLET_ORIENT_UP:
        ty -= req.height;
        break;
    case PANEL_APPLET_ORIENT_DOWN:
        ty += size;
        break;
    case PANEL_APPLET_ORIENT_LEFT:
        tx -= req.width;
        break;
    case PANEL_APPLET_ORIENT_RIGHT:
        tx += size;
        break;
    }

    *x = tx;
    *y = ty;
    *push_in = TRUE;
}

static void
grdc_applet_popdown_menu (GtkWidget *widget, GrdcAppletData *appdata)
{
    gtk_widget_set_state (GTK_WIDGET (appdata->applet), GTK_STATE_NORMAL);
}

static gchar *grdc_applet_menu_group = NULL;

static void
grdc_applet_popup_menu_add_item (gpointer data, gpointer user_data)
{
    GtkWidget *group_item;
    GtkWidget *hbox;
    GtkWidget *widget;
    GrdcAppletMenuItem *item;

    item = GRDC_APPLET_MENU_ITEM (data);

    if (item->group && item->group[0] != '\0')
    {
        if (!grdc_applet_menu_group || g_strcmp0 (grdc_applet_menu_group, item->group) != 0)
        {
            grdc_applet_menu_group = item->group;

            group_item = gtk_menu_item_new ();
            gtk_widget_show (group_item);
            gtk_widget_set_sensitive (group_item, FALSE);
            gtk_menu_append (GTK_MENU (user_data), group_item);

            hbox = gtk_hbox_new (FALSE, 2);
            gtk_widget_show (hbox);
            gtk_container_add (GTK_CONTAINER (group_item), hbox);

            widget = gtk_hseparator_new ();
            gtk_widget_show (widget);
            gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);

            widget = gtk_label_new (grdc_applet_menu_group);
            gtk_widget_show (widget);
            gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);

            widget = gtk_hseparator_new ();
            gtk_widget_show (widget);
            gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);
        }
    }
    else
    {
        if (grdc_applet_menu_group != NULL)
        {
            grdc_applet_menu_group = NULL;
            group_item = gtk_separator_menu_item_new ();
            gtk_menu_append (GTK_MENU (user_data), group_item);
            gtk_widget_show (group_item);
        }
    }
    gtk_menu_append (GTK_MENU (user_data), GTK_WIDGET (data));
}

static gchar*
grdc_applet_get_pref (const gchar *key)
{
    gchar filename[MAX_PATH_LEN];
    GKeyFile *gkeyfile;
    gchar *value;

    g_snprintf (filename, sizeof (filename), "%s/.grdc/grdc.pref", g_get_home_dir ());
    gkeyfile = g_key_file_new ();
    if (g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL))
    {
        value = g_key_file_get_string (gkeyfile, "grdc_pref", key, NULL);
    }
    else
    {
        value = NULL;
    }
    g_key_file_free (gkeyfile);

    return value;
}

static void
grdc_applet_popup_menu (GtkWidget *widget, GdkEventButton *event, GrdcAppletData *appdata)
{
    GtkWidget *menu;
    GtkWidget *menuitem;
    GPtrArray *menuitem_array;
    gint button, event_time;
    gchar dirname[MAX_PATH_LEN];
    gchar filename[MAX_PATH_LEN];
    GDir *dir;
    gchar *groups;
    const gchar *name;

    menu = gtk_menu_new ();

    g_snprintf (dirname, MAX_PATH_LEN, "%s/.grdc", g_get_home_dir ());
    dir = g_dir_open (dirname, 0, NULL);
    if (dir != NULL)
    {
        menuitem_array = g_ptr_array_new ();
        while ((name = g_dir_read_name (dir)) != NULL)
        {
            if (!g_str_has_suffix (name, ".grdc")) continue;
            g_snprintf (filename, MAX_PATH_LEN, "%s/%s", dirname, name);

            menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_FILE, filename);
	        g_signal_connect (G_OBJECT (menuitem), "activate", G_CALLBACK (grdc_applet_menu_open_file), NULL);
	        gtk_widget_show (menuitem);

            g_ptr_array_add (menuitem_array, menuitem);
        }

        groups = grdc_applet_get_pref ("groups");
        g_ptr_array_sort_with_data (menuitem_array, grdc_applet_menu_item_compare, groups);
        g_free (groups);

        grdc_applet_menu_group = NULL;
        g_ptr_array_foreach (menuitem_array, grdc_applet_popup_menu_add_item, menu);

        g_ptr_array_free (menuitem_array, FALSE);
    }

    menuitem = gtk_separator_menu_item_new ();
    gtk_menu_append (GTK_MENU (menu), menuitem);
    gtk_widget_show (menuitem);

    menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_QUICK, NULL);
    g_signal_connect (menuitem, "activate", G_CALLBACK (grdc_applet_menu_open_quick), NULL);
    gtk_menu_append (GTK_MENU (menu), menuitem);
    gtk_widget_show (menuitem);

    if (event)
    {
        button = event->button;
        event_time = event->time;
    }
    else
    {
        button = 0;
        event_time = gtk_get_current_event_time ();
    }

    gtk_widget_set_state (GTK_WIDGET (appdata->applet), GTK_STATE_SELECTED);
    g_signal_connect (menu, "deactivate", G_CALLBACK (grdc_applet_popdown_menu), appdata);

    gtk_menu_attach_to_widget (GTK_MENU (menu), widget, NULL);
    gtk_widget_realize (menu);
    gtk_menu_popup (GTK_MENU (menu), NULL, NULL, grdc_applet_popup_menu_position, appdata, button, event_time);
}

static gboolean
grdc_applet_button_press_event (GtkWidget *widget, GdkEventButton *event, GrdcAppletData *appdata)
{
    if (event->button == 1)
    {
        switch (event->type)
        {
        case GDK_BUTTON_PRESS:
            grdc_applet_popup_menu (widget,  event, appdata);
            break;
        case GDK_2BUTTON_PRESS:
            grdc_applet_menu_open_main (NULL, NULL);
            break;
        default:
            break;
        }
        return TRUE;
    }
    return FALSE;
}

static void
grdc_applet_menu_about (BonoboUIComponent *uic, GrdcAppletData *appdata, const char *verb)
{
    const gchar *authors[] =
    {
        "Vic Lee <llyzs@163.com>",
        NULL
    };
    const gchar *licenses[] =
    {
        N_("Grdc is free software; you can redistribute it and/or modify "
           "it under the terms of the GNU General Public License as published by "
           "the Free Software Foundation; either version 2 of the License, or "
           "(at your option) any later version."),
        N_("Grdc is distributed in the hope that it will be useful, "
           "but WITHOUT ANY WARRANTY; without even the implied warranty of "
           "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the "
           "GNU General Public License for more details."),
        N_("You should have received a copy of the GNU General Public License "
           "along with this program; if not, write to the Free Software "
           "Foundation, Inc., 59 Temple Place, Suite 330, "
           "Boston, MA 02111-1307, USA.")
    };
    gchar *license = g_strjoin ("\n\n", _(licenses[0]), _(licenses[1]), _(licenses[2]), NULL);

    gtk_show_about_dialog (NULL,
        "program-name", "Grdc Applet",
        "version", PACKAGE_VERSION,
        "comments", _("GTK+/Gnome Remote Desktop Client Applet"),
        "authors", authors,
        "translator-credits", _("translator-credits"),
        "copyright", "Copyright (C) 2009 - Vic Lee",
        "license", license,
        "wrap-license", TRUE,
        "logo", grdc_pixmap_logo (),
        "website", "http://grdc.sourceforge.net",
        NULL);

    g_free (license);
}

static const BonoboUIVerb grdc_applet_menu_verbs [] =
{   
	BONOBO_UI_UNSAFE_VERB ("OpenMain", grdc_applet_menu_open_main),
	BONOBO_UI_UNSAFE_VERB ("About", grdc_applet_menu_about),
	BONOBO_UI_VERB_END
};

static const gchar *grdc_applet_menu_xml = 
"    <popup name='button3'>"
"      <menuitem name='OpenMain' verb='OpenMain' _label='%s' pixtype='stock' pixname='gtk-execute'  />"
"      <menuitem name='About'    verb='About'    _label='%s' pixtype='stock' pixname='gtk-about' />"
"    </popup>"
;

static void
grdc_applet_create (PanelApplet *applet)
{
    gchar buf[1000];
    GrdcAppletData *appdata;

    appdata = g_new0 (GrdcAppletData, 1);
    appdata->applet = applet;

    appdata->image = gtk_image_new_from_pixbuf (NULL);
    gtk_widget_show (appdata->image);

    gtk_container_add (GTK_CONTAINER (applet), appdata->image);

    g_signal_connect (GTK_WIDGET (applet), "destroy", G_CALLBACK (grdc_applet_destroy), appdata);
    g_signal_connect (GTK_WIDGET (applet), "change-size", G_CALLBACK (grdc_applet_change_size), appdata);
    g_signal_connect (GTK_WIDGET (applet), "button-press-event", G_CALLBACK (grdc_applet_button_press_event), appdata);

    g_snprintf (buf, sizeof (buf), grdc_applet_menu_xml, _("Open Main Window"), _("About"));
    panel_applet_setup_menu (applet, buf, grdc_applet_menu_verbs, appdata);

    grdc_applet_change_size (GTK_WIDGET (applet), panel_applet_get_size (applet), appdata);

    gtk_widget_show (GTK_WIDGET (applet));
}

static gboolean
grdc_applet_factory (PanelApplet *applet, const gchar *iid, gpointer data)
{
    if (strcmp (iid, GRDC_APPLET_IID) != 0) return FALSE;

    bindtextdomain (GETTEXT_PACKAGE, GRDC_GNOME_LOCALEDIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);

    grdc_applet_create (applet);

    return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY (GRDC_APPLET_FACTORY_IID,
                            PANEL_TYPE_APPLET,
                            _("Grdc Remote Desktop Client Applet"),
                            "0", 
                            grdc_applet_factory, 
                            NULL)

