/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "config.h"
#include "grdcpublic.h"
#include "grdcfile.h"
#include "grdcinitdialog.h"
#include "grdcplug.h"
#include "grdcplug_rdp.h"
#include "grdcplug_vnc.h"
#include "grdcpixmaps.h"
#include "grdcpref.h"
#include "grdcconnectionwindow.h"

enum
{
    SCROLLED_WINDOW_MODE = 1,
    FULLSCREEN_MODE,
    SCROLLED_FULLSCREEN_MODE,
    VIEWPORT_FULLSCREEN_MODE
};

typedef struct _GrdcConnectionWindow
{
    GtkWidget *window;
    GrdcFile *grdc_file;

    GtkWidget *floating_toolbar;
    /* To avoid strange event-loop */
    guint32 floating_toolbar_hide_time;

    /* Containers for GrdcPlug sub-classes: GrdcPlug->viewport->...->window */
    GtkWidget *grdc_plug;
    GtkWidget *viewport;

    /* Motion activates in Viewport Fullscreen mode */
    gboolean viewport_motion;

    /* Toolitems that need to be handled */
    GtkToolItem *toolitem_grab;
    GtkToolItem *toolitem_preferences;

    gboolean connected;

    gboolean quit_when_close;
    gboolean keyboard_grabbed;
    gboolean sticky;
} GrdcConnectionWindow;

void grdc_connection_window_create_scrolled (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_fullscreen (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_scrolled_fullscreen (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_viewport_fullscreen (GrdcConnectionWindow *cnnwin);

static GPtrArray *grdc_connection_window_array = NULL;

static void
grdc_connection_window_register (GrdcConnectionWindow *cnnwin)
{
    if (grdc_connection_window_array == NULL)
    {
        grdc_connection_window_array = g_ptr_array_new ();
    }
    g_ptr_array_add (grdc_connection_window_array, cnnwin);
}

static void
grdc_connection_window_unregister (GrdcConnectionWindow *cnnwin)
{
    if (grdc_connection_window_array)
    {
        g_ptr_array_remove (grdc_connection_window_array, cnnwin);
    }
}

static void
grdc_connection_window_disconnect (GrdcConnectionWindow *cnnwin)
{
    /* Notify the GrdcPlug to disconnect, but not to close the window here.
       The window will be destroyed in GrdcPlug "disconnect" signal */
    grdc_plug_close_connection (GRDC_PLUG (cnnwin->grdc_plug));
}

static void
grdc_connection_window_keyboard_grab (GrdcConnectionWindow *cnnwin)
{
    if (cnnwin->keyboard_grabbed)
    {
        gdk_keyboard_grab (cnnwin->window->window, TRUE, GDK_CURRENT_TIME);
    }
    else
    {
        gdk_keyboard_ungrab (GDK_CURRENT_TIME);
    }
}

static gboolean
grdc_connection_window_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    grdc_connection_window_disconnect ((GrdcConnectionWindow*) data);
    return TRUE;
}

static void
grdc_connection_window_destroy (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    if (!cnnwin->connected)
    {
        grdc_file_free (cnnwin->grdc_file);
        if (cnnwin->floating_toolbar != NULL)
        {
            gtk_widget_destroy (cnnwin->floating_toolbar);
            cnnwin->floating_toolbar = NULL;
        }
        grdc_connection_window_unregister (cnnwin);
        g_free (cnnwin);
        if (cnnwin->quit_when_close)
        {
            gtk_main_quit ();
        }
    }
}

static void
grdc_connection_window_floating_toolbar_hide (GrdcConnectionWindow *cnnwin)
{
    GtkRequisition req;
    gint x;

    if (cnnwin->floating_toolbar == NULL) return;

    gtk_widget_size_request (cnnwin->floating_toolbar, &req);
    gtk_window_get_position (GTK_WINDOW (cnnwin->floating_toolbar), &x, NULL);
    gtk_window_move (GTK_WINDOW (cnnwin->floating_toolbar), x, 2 - req.height);
    cnnwin->floating_toolbar_hide_time = gtk_get_current_event_time ();
}

static void
grdc_connection_window_toolbar_scrolled (GtkWidget *widget, gpointer data)
{
    grdc_connection_window_create_scrolled ((GrdcConnectionWindow*) data);
}

static void
grdc_connection_window_toolbar_scrolled_fullscreen (GtkWidget *widget, gpointer data)
{
    grdc_connection_window_create_scrolled_fullscreen ((GrdcConnectionWindow*) data);
}

static void
grdc_connection_window_toolbar_viewport_fullscreen (GtkWidget *widget, gpointer data)
{
    grdc_connection_window_create_viewport_fullscreen ((GrdcConnectionWindow*) data);
}

static void
grdc_connection_window_toolbar_preferences_popdown (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    cnnwin->sticky = FALSE;

    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_preferences), FALSE);
    grdc_connection_window_floating_toolbar_hide (cnnwin);
}

static void
grdc_connection_window_call_plug_feature_radio (GtkMenuItem *menuitem, GrdcConnectionWindow *cnnwin)
{
    GrdcPlugFeature type;
    gpointer value;
    gboolean changed;

    if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menuitem)))
    {
        type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menuitem), "feature-type"));
        value = g_object_get_data (G_OBJECT (menuitem), "feature-value");

        grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), type, value);

        if (grdc_pref.save_view_mode)
        {
            changed = FALSE;
            switch (type)
            {
                case GRDC_PLUG_FEATURE_PREF_QUALITY:
                    g_free (cnnwin->grdc_file->quality);
                    cnnwin->grdc_file->quality = g_strdup ((const gchar*) value);
                    changed = TRUE;
                    break;
                default:
                    break;
            }
            if (changed) grdc_file_save (cnnwin->grdc_file);
        }
    }
}

static void
grdc_connection_window_call_plug_feature_check (GtkMenuItem *menuitem, GrdcConnectionWindow *cnnwin)
{
    GrdcPlugFeature type;
    gboolean value;

    type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menuitem), "feature-type"));
    value = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menuitem));
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), type, (value ? GINT_TO_POINTER (1) : NULL));
}

static void
grdc_connection_window_toolbar_preferences (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    GtkWidget *menu;
    GtkWidget *menuitem;
    GSList *group;
    gint i;
    gboolean firstgroup;

    if (!gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget))) return;

    cnnwin->sticky = TRUE;

    firstgroup = TRUE;

    menu = gtk_menu_new ();

    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_QUALITY))
    {
        group = NULL;
        for (i = 0; quality_list[i]; i += 2)
        {
            menuitem = gtk_radio_menu_item_new_with_label (group, quality_list[i + 1]);
            group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (menuitem));
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

            g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_PREF_QUALITY));
            g_object_set_data (G_OBJECT (menuitem), "feature-value", (gpointer) quality_list[i]);

            if (g_strcmp0 (quality_list[i], cnnwin->grdc_file->quality) == 0)
            {
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), TRUE);
            }

            g_signal_connect (G_OBJECT (menuitem), "toggled",
                G_CALLBACK (grdc_connection_window_call_plug_feature_radio), cnnwin);
        }
        firstgroup = FALSE;
    }

    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_VIEWONLY))
    {
        if (!firstgroup)
        {
            menuitem = gtk_separator_menu_item_new ();
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
        }
        menuitem = gtk_check_menu_item_new_with_label (_("View Only"));
        gtk_widget_show (menuitem);
        gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

        g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_PREF_VIEWONLY));

        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), cnnwin->grdc_file->viewonly);

        g_signal_connect (G_OBJECT (menuitem), "toggled",
            G_CALLBACK (grdc_connection_window_call_plug_feature_check), cnnwin);

        firstgroup = FALSE;
    }

    g_signal_connect (G_OBJECT (menu), "deactivate",
        G_CALLBACK (grdc_connection_window_toolbar_preferences_popdown), cnnwin);

    gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
        grdc_public_popup_position, widget,
        0, gtk_get_current_event_time());
}

static void
grdc_connection_window_toolbar_minimize (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    grdc_connection_window_floating_toolbar_hide (cnnwin);
    gtk_window_iconify (GTK_WINDOW (cnnwin->window));
}

static void
grdc_connection_window_toolbar_disconnect (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_disconnect (cnnwin);
}

static void
grdc_connection_window_toolbar_grab (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    cnnwin->keyboard_grabbed = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget));
    grdc_connection_window_keyboard_grab (cnnwin);
}

static GtkWidget*
grdc_connection_window_create_toolbar (GrdcConnectionWindow *cnnwin, gint mode)
{
    GtkWidget *toolbar;
    GtkToolItem *toolitem;

    toolbar = gtk_toolbar_new ();
    gtk_widget_show (toolbar);
    gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
    gtk_toolbar_set_show_arrow (GTK_TOOLBAR (toolbar), FALSE);
    if (grdc_pref.small_toolbutton)
    {
        gtk_toolbar_set_icon_size (GTK_TOOLBAR (toolbar), GTK_ICON_SIZE_MENU);
    }

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_LEAVE_FULLSCREEN);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Scrolled Window"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle scrolled window mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_scrolled), cnnwin);
    if (mode == SCROLLED_WINDOW_MODE) gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_FULLSCREEN);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Scrolled Fullscreen"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle scrolled fullscreen mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_scrolled_fullscreen), cnnwin);
    if (mode == FULLSCREEN_MODE || mode == SCROLLED_FULLSCREEN_MODE)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_ZOOM_FIT);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Viewport Fullscreen"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle viewport fullscreen mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_viewport_fullscreen), cnnwin);
    if (mode == FULLSCREEN_MODE || mode == VIEWPORT_FULLSCREEN_MODE)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_DIALOG_AUTHENTICATION);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Grab Keyboard"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Grab all keyboard events"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "toggled",
        G_CALLBACK(grdc_connection_window_toolbar_grab), cnnwin);
    if (mode == SCROLLED_WINDOW_MODE)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
        cnnwin->toolitem_grab = NULL;
    }
    else
    {
        cnnwin->toolitem_grab = toolitem;
    }

    toolitem = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_PREFERENCES);
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF))
    {
        g_signal_connect (G_OBJECT (toolitem), "toggled",
            G_CALLBACK(grdc_connection_window_toolbar_preferences), cnnwin);
    }
    else
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }
    cnnwin->toolitem_preferences = toolitem;

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_GOTO_BOTTOM);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Minimize"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Minimize Window"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_minimize), cnnwin);

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_DISCONNECT);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Disconnect"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Disconnect"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_disconnect), cnnwin);

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    return toolbar;
}

static gboolean
grdc_connection_window_toolbar_enter (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    gint x;

    /* I hardcoded a 100ms timeout here, so that the toolbar won't show up immediately after it hides */
    if (gtk_get_current_event_time () - cnnwin->floating_toolbar_hide_time < 100) return TRUE;

    gtk_window_get_position (GTK_WINDOW (cnnwin->floating_toolbar), &x, NULL);
    gtk_window_move (GTK_WINDOW (cnnwin->floating_toolbar), x, 0);
    return TRUE;
}

static gboolean
grdc_connection_window_toolbar_leave (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    if (!cnnwin->sticky && event->mode == GDK_CROSSING_NORMAL)
    {
        grdc_connection_window_floating_toolbar_hide (cnnwin);
        return TRUE;
    }
    return FALSE;
}

static gboolean
grdc_connection_window_focus_in (GtkWidget *widget, GdkEventFocus *event, GrdcConnectionWindow *cnnwin)
{
    if (cnnwin->floating_toolbar) gtk_widget_show (cnnwin->floating_toolbar);
    /*grdc_connection_window_keyboard_grab (cnnwin);*/
    return FALSE;
}

static gboolean
grdc_connection_window_focus_out (GtkWidget *widget, GdkEventFocus *event, GrdcConnectionWindow *cnnwin)
{
    if (!cnnwin->sticky && cnnwin->floating_toolbar) gtk_widget_hide (cnnwin->floating_toolbar);
    cnnwin->viewport_motion = FALSE;
    /*gdk_keyboard_ungrab (GDK_CURRENT_TIME);*/
    if (cnnwin->toolitem_grab)
    {
        gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_grab), FALSE);
    }
    return FALSE;
}

static void
grdc_connection_window_create_floating_toolbar (GrdcConnectionWindow *cnnwin, gint mode)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *widget;
    GtkWidget *eventbox;
    GtkRequisition req;
    GdkScreen *screen;

    screen = gdk_screen_get_default ();

    if (cnnwin->floating_toolbar != NULL)
    {
        gtk_widget_destroy (cnnwin->floating_toolbar);
        cnnwin->floating_toolbar = NULL;
    }

    /* This has to be a popup window to become visible in fullscreen mode */
    window = gtk_window_new (GTK_WINDOW_POPUP);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    widget = grdc_connection_window_create_toolbar (cnnwin, mode);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);

    /* An event box is required to wrap the label to avoid infinite "leave-enter" event loop */
    eventbox = gtk_event_box_new ();
    gtk_widget_show (eventbox);
    gtk_box_pack_start (GTK_BOX (vbox), eventbox, FALSE, FALSE, 0);
    widget = gtk_label_new (cnnwin->grdc_file->filename ? cnnwin->grdc_file->name : cnnwin->grdc_file->server);
    gtk_label_set_max_width_chars (GTK_LABEL (widget), 50);
    gtk_widget_show (widget);
    gtk_container_add (GTK_CONTAINER (eventbox), widget);

    gtk_widget_size_request (window, &req);
    gtk_window_move (GTK_WINDOW (window), (gdk_screen_get_width (screen) - req.width) / 2, 2 - req.height);

    cnnwin->floating_toolbar = window;

    g_signal_connect (G_OBJECT (window), "enter-notify-event", 
        G_CALLBACK (grdc_connection_window_toolbar_enter), cnnwin);
    g_signal_connect (G_OBJECT (window), "leave-notify-event", 
        G_CALLBACK (grdc_connection_window_toolbar_leave), cnnwin);

    if (cnnwin->connected) gtk_widget_show (window);
}

static GtkWidget*
grdc_connection_window_create_main (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), cnnwin->grdc_file->name);
    gtk_window_set_default_size (GTK_WINDOW (window), 700, 500);
    gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
    gtk_container_set_border_width (GTK_CONTAINER (window), 0);
    gtk_window_set_icon (GTK_WINDOW (window), grdc_pixmap_logo ());

    g_signal_connect (G_OBJECT (window), "delete-event",
        G_CALLBACK (grdc_connection_window_delete_event), cnnwin);
    g_signal_connect (G_OBJECT (window), "destroy",
        G_CALLBACK (grdc_connection_window_destroy), cnnwin);

    g_signal_connect (G_OBJECT (window), "focus-in-event",
        G_CALLBACK (grdc_connection_window_focus_in), cnnwin);
    g_signal_connect (G_OBJECT (window), "focus-out-event",
        G_CALLBACK (grdc_connection_window_focus_out), cnnwin);

    return window;
}

/* Create a scrolled window container */
void
grdc_connection_window_create_scrolled (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *toolbar;
    GtkWidget *scrolledwindow;

    if (grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = SCROLLED_WINDOW_MODE;

    cnnwin->viewport_motion = FALSE;

    if (cnnwin->floating_toolbar != NULL)
    {
        gtk_widget_destroy (cnnwin->floating_toolbar);
        cnnwin->floating_toolbar = NULL;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);

    /* Create the vbox container */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    /* Create the toolbar */
    toolbar = grdc_connection_window_create_toolbar (cnnwin, SCROLLED_WINDOW_MODE);
    gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);

    /* Create the scrolled window */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

    /* Add the viewport into the scrolled window */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (scrolledwindow), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, scrolledwindow);
        gtk_widget_destroy (cnnwin->window);
    }

    gtk_container_set_focus_chain (GTK_CONTAINER (vbox), g_list_append (NULL, scrolledwindow));

    gtk_window_maximize (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);

    cnnwin->window = window;
}

void
grdc_connection_window_create_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (window), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, window);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

static gboolean
grdc_connection_window_scrolled_required (GrdcConnectionWindow *cnnwin)
{
    GdkScreen *screen;
    gint screen_width, screen_height;
    gint server_width, server_height;

    gtk_widget_get_size_request (cnnwin->grdc_plug, &server_width, &server_height);

    if (server_width <= 0 || server_height <= 0) return TRUE;

    screen = gdk_screen_get_default ();
    screen_width = gdk_screen_get_width (screen);
    screen_height = gdk_screen_get_height (screen);

    if (screen_width >= server_width && screen_height >= server_height) return FALSE;

    return TRUE;
}

/* Create a scrolled fullscreen container */
void
grdc_connection_window_create_scrolled_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GtkWidget *scrolledwindow;

    if (grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = SCROLLED_FULLSCREEN_MODE;

    cnnwin->viewport_motion = FALSE;

    if (!grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
        return;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);

    /* Create the scrolled window */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow), 0);
    gtk_container_add (GTK_CONTAINER (window), scrolledwindow);

    /* Link the containers to the main window and the viewport */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (scrolledwindow), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, scrolledwindow);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, SCROLLED_FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

/* Create a viewport fullscreen container */

/* Event handler when mouse move on borders */
static gboolean
grdc_connection_window_viewport_motion_timeout (gpointer data)
{
    GrdcConnectionWindow *cnnwin;
    GdkDisplay *display;
    GdkScreen *screen;
    gint x, y, mx, my, w, h;
    GtkAdjustment *adj;
    gdouble value;

    cnnwin = (GrdcConnectionWindow*) data;
    if (!cnnwin->viewport_motion) return FALSE;
    if (!GTK_IS_VIEWPORT (cnnwin->viewport)) return FALSE;

    display = gdk_display_get_default ();
    gdk_display_get_pointer (display, &screen, &x, &y, NULL);

    w = gdk_screen_get_width (screen);
    h = gdk_screen_get_height (screen);
    mx = (x == 0 ? -1 : (x >= w - 1 ? 1 : 0));
    my = (y == 0 ? -1 : (y >= h - 1 ? 1 : 0));
    if (mx != 0)
    {
        adj = gtk_viewport_get_hadjustment (GTK_VIEWPORT (cnnwin->viewport));
        value = gtk_adjustment_get_value (GTK_ADJUSTMENT (adj)) + (gdouble) (mx * 10);
        value = MAX (0, MIN (value, adj->upper - (gdouble) w + 2.0));
        gtk_adjustment_set_value (GTK_ADJUSTMENT (adj), value);
    }
    if (my != 0)
    {
        adj = gtk_viewport_get_vadjustment (GTK_VIEWPORT (cnnwin->viewport));
        value = gtk_adjustment_get_value (GTK_ADJUSTMENT (adj)) + (gdouble) (my * 10);
        value = MAX (0, MIN (value, adj->upper - (gdouble) h + 2.0));
        gtk_adjustment_set_value (GTK_ADJUSTMENT (adj), value);
    }
    return TRUE;
}

static gboolean
grdc_connection_window_viewport_enter (
    GtkWidget *widget,
    GdkEventCrossing *event,
    gpointer user_data)
{
    GrdcConnectionWindow *cnnwin;

    cnnwin = (GrdcConnectionWindow*) user_data;
    cnnwin->viewport_motion = FALSE;
    return FALSE;
}

static gboolean
grdc_connection_window_viewport_leave (
    GtkWidget *widget,
    GdkEventCrossing *event,
    gpointer user_data)
{
    GrdcConnectionWindow *cnnwin;

    cnnwin = (GrdcConnectionWindow*) user_data;
    cnnwin->viewport_motion = TRUE;
    TIMEOUT_ADD (20, grdc_connection_window_viewport_motion_timeout, cnnwin);
    return FALSE;
}

void
grdc_connection_window_create_viewport_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GtkWidget *eventbox;

    if (grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = VIEWPORT_FULLSCREEN_MODE;

    cnnwin->viewport_motion = FALSE;

    if (!grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
        return;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);
    gtk_container_set_border_width (GTK_CONTAINER (window), 1);

    eventbox = gtk_event_box_new ();
    gtk_widget_show (eventbox);
    gtk_container_add (GTK_CONTAINER (window), eventbox);
    gtk_widget_add_events (eventbox, GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
    g_signal_connect (G_OBJECT (eventbox), "enter-notify-event", 
        G_CALLBACK (grdc_connection_window_viewport_enter), cnnwin);
    g_signal_connect (G_OBJECT (eventbox), "leave-notify-event", 
        G_CALLBACK (grdc_connection_window_viewport_leave), cnnwin);

    /* Create the eventbox to trap events */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (eventbox), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, eventbox);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, VIEWPORT_FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

static void
grdc_connection_window_on_connect (GrdcPlug *gp, GrdcConnectionWindow *cnnwin)
{
    grdc_connection_window_register (cnnwin);
    cnnwin->connected = TRUE;

    /* If "Use client resolution" is selected, we switch to fullscreen by default */
    if (cnnwin->grdc_file->viewmode != SCROLLED_WINDOW_MODE &&
        cnnwin->grdc_file->resolution && g_strcmp0 (cnnwin->grdc_file->resolution, "AUTO") == 0 &&
        !grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
    }

    gtk_widget_show (cnnwin->window);
    if (cnnwin->floating_toolbar)
    {
        gtk_widget_show (cnnwin->floating_toolbar);
    }
}

static void
grdc_connection_window_on_disconnect (GrdcPlug *gp, GrdcConnectionWindow *cnnwin)
{
    GtkWidget *dialog;

    cnnwin->connected = FALSE;

    if (grdc_pref.save_view_mode)
    {
        grdc_file_save (cnnwin->grdc_file);
    }

    if (GRDC_PLUG (cnnwin->grdc_plug)->has_error)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            GRDC_PLUG (cnnwin->grdc_plug)->error_message);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }
    gtk_widget_destroy (cnnwin->window);
}

gboolean
grdc_connection_window_open (GtkWindow *parent, const gchar *filename, gboolean quit_when_close)
{
    GrdcFile *grdcfile;
    GtkWidget *dialog;

    grdcfile = grdc_file_load (filename);
    if (grdcfile)
    {
        grdc_connection_window_open_struct (parent, grdcfile, quit_when_close);
        return TRUE;
    }
    else
    {
        dialog = gtk_message_dialog_new (parent, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            _("File %s not found."), filename);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return FALSE;
    }
}

void
grdc_connection_window_open_struct (GtkWindow *parent, GrdcFile *grdcfile, gboolean quit_when_close)
{
    GrdcConnectionWindow *cnnwin;

    grdc_file_update_screen_resolution (grdcfile);

    cnnwin = (GrdcConnectionWindow*) g_malloc (sizeof (GrdcConnectionWindow));

    cnnwin->grdc_file = grdcfile;
    cnnwin->window = NULL;
    cnnwin->floating_toolbar = NULL;
    cnnwin->floating_toolbar_hide_time = 0;
    cnnwin->connected = FALSE;
    cnnwin->viewport_motion = FALSE;
    cnnwin->quit_when_close = quit_when_close;
    cnnwin->keyboard_grabbed = FALSE;
    cnnwin->sticky = FALSE;

    /* Create the GrdcPlug */
    if (g_strcmp0 (grdcfile->protocol, "RDP") == 0)
    {
        cnnwin->grdc_plug = grdc_plug_rdp_new ();
    }
    else if (g_strcmp0 (grdcfile->protocol, "VNC") == 0)
    {
        cnnwin->grdc_plug = grdc_plug_vnc_new ();
    }
    else
    {
        g_print ("Fatel: Protocol %s not supported\n", grdcfile->protocol);
        g_free (cnnwin);
        return;
    }
    gtk_widget_show (cnnwin->grdc_plug);
    g_signal_connect (G_OBJECT (cnnwin->grdc_plug), "connect",
        G_CALLBACK (grdc_connection_window_on_connect), cnnwin);
    g_signal_connect (G_OBJECT (cnnwin->grdc_plug), "disconnect",
        G_CALLBACK (grdc_connection_window_on_disconnect), cnnwin);

    /* Create the viewport to make the GrdcPlug scrollable */
    cnnwin->viewport = gtk_viewport_new (NULL, NULL);
    gtk_widget_show (cnnwin->viewport);
    gtk_container_set_border_width (GTK_CONTAINER (cnnwin->viewport), 0);
    gtk_viewport_set_shadow_type (GTK_VIEWPORT (cnnwin->viewport), GTK_SHADOW_NONE);
    gtk_container_add (GTK_CONTAINER (cnnwin->viewport), cnnwin->grdc_plug);

    if (!grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = 0;

    switch (cnnwin->grdc_file->viewmode)
    {
        case SCROLLED_FULLSCREEN_MODE:
            grdc_connection_window_create_scrolled_fullscreen (cnnwin);
            break;

        case VIEWPORT_FULLSCREEN_MODE:
            grdc_connection_window_create_viewport_fullscreen (cnnwin);
            break;

        case SCROLLED_WINDOW_MODE:
        default:
            grdc_connection_window_create_scrolled (cnnwin);
            break;
    }

    gtk_widget_realize (cnnwin->window);

    if (!grdc_plug_open_connection (GRDC_PLUG (cnnwin->grdc_plug), grdcfile))
    {
        gtk_widget_destroy (cnnwin->window);
    }
}

gint
grdc_connection_window_count (void)
{
    if (grdc_connection_window_array == NULL) return 0;
    return grdc_connection_window_array->len;
}

void
grdc_connection_window_quit_all (void)
{
    gpointer *pdata;
    gint i, len;

    if (grdc_connection_window_array == NULL) return;

    /* We need to duplicate the array, because the elements in the original array will get removed while killing */
    len = grdc_connection_window_count ();
    pdata = g_memdup (grdc_connection_window_array->pdata, sizeof(gpointer) * len);

    for (i = 0; i < len; i++)
    {
        grdc_connection_window_disconnect ((GrdcConnectionWindow*) pdata[i]);
        gtk_main_iteration ();
    }
    g_free (pdata);
}

