/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "config.h"
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcconnectionwindow.h"
#include "grdcprefdialog.h"
#include "grdcfile.h"

#define COLUMN_WIDTH_1 150
#define COLUMN_ALIGN_1 0.8
#define COLUMN_WIDTH_2 250

static const gchar *protocol_list[] =
{
    "RDP",
#ifdef HAVE_LIBVNCCLIENT
    "VNC",
#endif /* HAVE_LIBVNCCLIENT */
    NULL
};

typedef struct _GrdcFileEditor
{
    GtkWindow *parent;
    GrdcFile *grdc_file;
    GtkWidget *editor_dialog;

    GtkWidget *name_entry;
    GtkWidget *group_combo;
    GtkWidget *protocol_combo;

    GtkWidget *config_frame;
    GtkWidget *config_container;

    GtkWidget *server_entry;
    GtkWidget *username_entry;
    GtkWidget *password_entry;
    GtkWidget *domain_entry;
    GtkWidget *clientname_entry;
    GtkWidget *resolution_auto_radio;
    GtkWidget *resolution_custom_radio;
    GtkWidget *resolution_custom_combo;
    GtkWidget *colordepth_combo;
    GtkWidget *quality_combo;
    GtkWidget *bitmapcaching_check;
    GtkWidget *compression_check;
    GtkWidget *showcursor_check;

    gboolean quit_when_close;
} GrdcFileEditor;

static void
grdc_file_editor_on_realize (GtkWidget *widget, gpointer user_data)
{
    GrdcFileEditor *gfe;
    GtkWidget *defaultwidget;

    gfe = (GrdcFileEditor*) user_data;

    defaultwidget = (gfe->grdc_file->filename == NULL ? gfe->server_entry : gfe->name_entry);

    gtk_editable_select_region (GTK_EDITABLE (defaultwidget), 0, -1);
    gtk_widget_grab_focus (defaultwidget);
}

static void
grdc_file_editor_resolution_radio_on_toggled (GtkToggleButton *togglebutton, GrdcFileEditor *gfe)
{
    gtk_widget_set_sensitive (gfe->resolution_custom_combo, 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->resolution_custom_radio)));
}

static void
grdc_file_editor_create_rdp_config (GrdcFileEditor *gfe)
{
    GtkWidget *tablabel, *tabbody;
    GtkWidget *table;
    GtkWidget *widget;
    GtkWidget *hbox;

    /* Create the notebook */
    gfe->config_container = gtk_notebook_new ();
    gtk_container_add (GTK_CONTAINER (gfe->config_frame), gfe->config_container);
    gtk_container_set_border_width (GTK_CONTAINER (gfe->config_container), 4);
    gtk_widget_show (gfe->config_container);

    /* The Basic tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_INFO, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (_("Basic"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    tabbody = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (tabbody);
    gtk_notebook_append_page (GTK_NOTEBOOK (gfe->config_container), tabbody, tablabel);

    table = gtk_table_new (7, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 8);
    gtk_table_set_col_spacings (GTK_TABLE (table), 8);
    gtk_container_set_border_width (GTK_CONTAINER (table), 15);
    gtk_box_pack_start (GTK_BOX (tabbody), table, FALSE, FALSE, 0);

    /* The Basic content */
    widget = gtk_label_new (_("Server"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    if (gfe->grdc_file->server) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->server);
    gfe->server_entry = widget;

    widget = gtk_label_new (_("User Name"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 1, 2);
    if (gfe->grdc_file->username) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->username);
    gfe->username_entry = widget;

    widget = gtk_label_new (_("Password"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gtk_entry_set_visibility (GTK_ENTRY (widget), FALSE);
    if (gfe->grdc_file->password) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->password);
    gfe->password_entry = widget;

    widget = gtk_label_new (_("Domain"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 3, 4);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 3, 4);
    if (gfe->grdc_file->domain) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->domain);
    gfe->domain_entry = widget;

    widget = gtk_label_new (_("Resolution"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 4, 5);

    widget = gtk_radio_button_new_with_label (NULL, _("Use Client Resolution"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 4, 5);
    gfe->resolution_auto_radio = widget;

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 5, 6);

    widget = gtk_radio_button_new_with_label_from_widget (
        GTK_RADIO_BUTTON (gfe->resolution_auto_radio), _("Custom"));
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "toggled", 
        G_CALLBACK (grdc_file_editor_resolution_radio_on_toggled), gfe);
    gfe->resolution_custom_radio = widget;

    widget = grdc_public_create_combo_text_d (grdc_pref.resolutions, gfe->grdc_file->resolution, NULL);
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);
    gfe->resolution_custom_combo = widget;

    if (!gfe->grdc_file->resolution || gfe->grdc_file->resolution[0] == '\0' ||
        g_strcmp0 (gfe->grdc_file->resolution, "AUTO") == 0)
    {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gfe->resolution_auto_radio), TRUE);
        gtk_widget_set_sensitive (gfe->resolution_custom_combo, FALSE);
    }
    else
    {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gfe->resolution_custom_radio), TRUE);
    }

    widget = gtk_label_new (_("Color Depth"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 6, 7);

    widget = grdc_public_create_combo_map (colordepth_list, gfe->grdc_file->colordepth);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 6, 7);
    gfe->colordepth_combo = widget;

    /* The Advanced tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);

    widget = gtk_label_new (_("Advanced"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);

    tabbody = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (tabbody);
    gtk_notebook_append_page (GTK_NOTEBOOK (gfe->config_container), tabbody, tablabel);

    table = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 8);
    gtk_table_set_col_spacings (GTK_TABLE (table), 8);
    gtk_container_set_border_width (GTK_CONTAINER (table), 15);
    gtk_box_pack_start (GTK_BOX (tabbody), table, FALSE, FALSE, 0);

    /* The Advanced content */
    widget = gtk_label_new (_("Client Name"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_1, -1);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    if (gfe->grdc_file->clientname) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->clientname);
    gfe->clientname_entry = widget;

    widget = gtk_check_button_new_with_label (_("Enable bitmap caching"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 1, 2);
    if (gfe->grdc_file->bitmapcaching) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->bitmapcaching_check = widget;

    widget = gtk_check_button_new_with_label (_("Enable compression"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 2, 3);
    if (gfe->grdc_file->compression) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->compression_check = widget;
}

static void
grdc_file_editor_create_vnc_config (GrdcFileEditor *gfe)
{
    GtkWidget *table;
    GtkWidget *widget;

    table = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 8);
    gtk_table_set_col_spacings (GTK_TABLE (table), 8);
    gtk_container_set_border_width (GTK_CONTAINER (table), 15);

    gtk_container_add (GTK_CONTAINER (gfe->config_frame), table);
    gfe->config_container = table;

    widget = gtk_label_new (_("Server"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    if (gfe->grdc_file->server) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->server);
    gfe->server_entry = widget;

    widget = gtk_label_new (_("Password"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 1, 2);
    gtk_entry_set_visibility (GTK_ENTRY (widget), FALSE);
    if (gfe->grdc_file->password) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->password);
    gfe->password_entry = widget;

    widget = gtk_label_new (_("Quality"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = grdc_public_create_combo_map (quality_list, gfe->grdc_file->quality);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gfe->quality_combo = widget;

    widget = gtk_check_button_new_with_label (_("Show remote cursor"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 3, 4);
    if (gfe->grdc_file->showcursor) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->showcursor_check = widget;

}

static void
grdc_file_editor_protocol_combo_on_changed (GtkComboBox *combo, GrdcFileEditor *gfe)
{
    gchar *protocol;

    if (gfe->config_container)
    {
        gtk_container_remove (GTK_CONTAINER (gfe->config_frame), gfe->config_container);
        gfe->config_container = NULL;
    }

    gfe->server_entry = NULL;
    gfe->username_entry = NULL;
    gfe->password_entry = NULL;
    gfe->domain_entry = NULL;
    gfe->clientname_entry = NULL;
    gfe->resolution_auto_radio = NULL;
    gfe->resolution_custom_radio = NULL;
    gfe->resolution_custom_combo = NULL;
    gfe->colordepth_combo = NULL;
    gfe->quality_combo = NULL;
    gfe->bitmapcaching_check = NULL;
    gfe->compression_check = NULL;
    gfe->showcursor_check = NULL;

    protocol = gtk_combo_box_get_active_text (combo);
    if (g_strcmp0 (protocol, "VNC") == 0)
    {
        grdc_file_editor_create_vnc_config (gfe);
    }
    else /* RDP */
    {
        grdc_file_editor_create_rdp_config (gfe);
    }
    g_free (protocol);

    /* This call shrink the dialog to the minimum size */
    gtk_window_resize (GTK_WINDOW (gfe->editor_dialog), 1, 1);
}

static void
grdc_file_editor_update (GrdcFileEditor *gfe)
{
    g_free (gfe->grdc_file->name);
    gfe->grdc_file->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->name_entry)));

    g_free (gfe->grdc_file->group);
    gfe->grdc_file->group = (gfe->group_combo ?
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->group_combo)) : NULL);

    g_free (gfe->grdc_file->protocol);
    gfe->grdc_file->protocol = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->protocol_combo));

    g_free (gfe->grdc_file->server);
    gfe->grdc_file->server = (gfe->server_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->server_entry))) : NULL);

    g_free (gfe->grdc_file->username);
    gfe->grdc_file->username = (gfe->username_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->username_entry))) : NULL);

    g_free (gfe->grdc_file->password);
    gfe->grdc_file->password = (gfe->password_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->password_entry))) : NULL);

    g_free (gfe->grdc_file->domain);
    gfe->grdc_file->domain = (gfe->domain_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->domain_entry))) : NULL);

    g_free (gfe->grdc_file->clientname);
    gfe->grdc_file->clientname = (gfe->clientname_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->clientname_entry))) : NULL);

    g_free (gfe->grdc_file->resolution);
    gfe->grdc_file->resolution = (gfe->resolution_auto_radio ? g_strdup (
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->resolution_auto_radio)) ? "AUTO" :
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->resolution_custom_combo))) : NULL);

    g_free (gfe->grdc_file->colordepth);
    gfe->grdc_file->colordepth = (gfe->colordepth_combo ?
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->colordepth_combo)) : NULL);

    g_free (gfe->grdc_file->quality);
    gfe->grdc_file->quality = (gfe->quality_combo ?
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->quality_combo)) : NULL);

    gfe->grdc_file->bitmapcaching = (gfe->bitmapcaching_check ?
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->bitmapcaching_check)) : FALSE);
    gfe->grdc_file->compression = (gfe->compression_check ?
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->compression_check)) : FALSE);
    gfe->grdc_file->showcursor = (gfe->showcursor_check ?
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->showcursor_check)) : FALSE);
}

static void
grdc_file_editor_set_default (GtkWidget *button, GrdcFileEditor *gfe)
{
    GrdcFile *gf;
    GtkWidget *dialog;

    grdc_file_editor_update (gfe);

    gf = grdc_file_dup (gfe->grdc_file);

    g_free (gf->filename);
    gf->filename = g_strdup (grdc_pref_file);

    /* Clear properties that should never be default */
    g_free (gf->name); gf->name = NULL;
    g_free (gf->server); gf->server = NULL;
    g_free (gf->password); gf->password = NULL;

    grdc_file_save (gf);

    grdc_file_free (gf);

    dialog = gtk_message_dialog_new (GTK_WINDOW (gfe->editor_dialog),
        GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        _("Default settings saved."));
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

static void
grdc_file_editor_open_groups (GtkWidget *button, GrdcFileEditor *gfe)
{
    if (grdc_pref_dialog_run (GTK_WINDOW (gfe->editor_dialog), GRDC_PREF_GROUPS_TAB))
    {
        grdc_public_load_combo_text_d (gfe->group_combo, grdc_pref.groups, gfe->grdc_file->group, _("<None>"));
    }
}

static void
grdc_file_editor_init (GrdcFileEditor *gfe)
{
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *hbox;
    GtkWidget *widget;

    /* Create the editor dialog */
    gfe->editor_dialog = gtk_dialog_new_with_buttons (_("Remote Desktop Preference"),
        gfe->parent, GTK_DIALOG_MODAL,
        GTK_STOCK_CONNECT, GTK_RESPONSE_OK,
        GTK_STOCK_SAVE, GTK_RESPONSE_APPLY,
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        NULL);
    gtk_dialog_set_default_response (GTK_DIALOG (gfe->editor_dialog), GTK_RESPONSE_OK);
    gtk_dialog_set_has_separator (GTK_DIALOG (gfe->editor_dialog), FALSE);
    if (gfe->grdc_file->filename == NULL)
    {
        gtk_dialog_set_response_sensitive (GTK_DIALOG (gfe->editor_dialog), GTK_RESPONSE_APPLY, FALSE);
    }

    g_signal_connect (G_OBJECT (gfe->editor_dialog),
        "realize",G_CALLBACK (grdc_file_editor_on_realize), gfe);

    /* The Set As Default button */
    widget = gtk_button_new_with_label (_("Default"));
    gtk_widget_show (widget);
    gtk_button_set_image (GTK_BUTTON (widget), gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_BUTTON));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->action_area), widget, FALSE, TRUE, 0);
    gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (GTK_DIALOG (gfe->editor_dialog)->action_area), widget, TRUE);

    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_file_editor_set_default), gfe);

    /* Create the Profile frame on the top (for name and protocol) */
    frame = gtk_frame_new (_("Profile"));
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->vbox), frame, TRUE, TRUE, 2);

    table = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 4);
    gtk_table_set_col_spacings (GTK_TABLE (table), 4);
    gtk_container_set_border_width (GTK_CONTAINER (table), 8);
    gtk_container_add (GTK_CONTAINER (frame), table);

    /* Profile: Name */
    widget = gtk_label_new (_("Name"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, 100, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, 300, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    gfe->name_entry = widget;
    if (gfe->grdc_file->filename == NULL)
    {
        gtk_widget_set_sensitive (widget, FALSE);
        gtk_entry_set_text (GTK_ENTRY (widget), _("Quick Connect"));
    }
    else
    {
        if (gfe->grdc_file->name) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->name);
    }

    /* Profile: Group */
    if (gfe->grdc_file->filename == NULL)
    {
        gfe->group_combo = NULL;
    }
    else
    {
        widget = gtk_label_new (_("Group"));
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 1, 2);

        gfe->group_combo = grdc_public_create_combo_text_d (grdc_pref.groups, gfe->grdc_file->group, _("<None>"));
        gtk_widget_show (gfe->group_combo);
        gtk_box_pack_start (GTK_BOX (hbox), gfe->group_combo, TRUE, TRUE, 0);

        widget = gtk_button_new_with_label ("...");
        gtk_widget_show (widget);
        gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
        g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_file_editor_open_groups), gfe);
    }

    /* Profile: Protocol */
    widget = gtk_label_new (_("Protocol"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = grdc_public_create_combo_text (protocol_list, gfe->grdc_file->protocol);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    g_signal_connect (G_OBJECT (widget), "changed", G_CALLBACK (grdc_file_editor_protocol_combo_on_changed), gfe);
    gfe->protocol_combo = widget;

    /* Create the Preference frame */
    frame = gtk_frame_new (_("Settings"));
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->vbox), frame, TRUE, TRUE, 2);
    gfe->config_frame = frame;

    gfe->config_container = NULL;

    grdc_file_editor_protocol_combo_on_changed (GTK_COMBO_BOX (gfe->protocol_combo), gfe);
}

static gint
grdc_file_editor_run (GrdcFileEditor *gfe)
{
    gint ret;
    GrdcFile *gf;

    ret = gtk_dialog_run (GTK_DIALOG (gfe->editor_dialog));
    switch (ret)
    {
        case GTK_RESPONSE_OK:
            grdc_file_editor_update (gfe);
            gf = grdc_file_dup (gfe->grdc_file);
            grdc_connection_window_open_struct (gfe->parent, gf, gfe->quit_when_close);
            break;
        case GTK_RESPONSE_APPLY:
            grdc_file_editor_update (gfe);
            break;
        default:
            break;
    }
    return ret;
}

static gint
grdc_file_editor_main (GrdcFileEditor *gfe)
{
    gint ret;

    grdc_file_editor_init (gfe);
    ret = grdc_file_editor_run (gfe);
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_file_save (gfe->grdc_file);
    }
    gtk_widget_destroy (gfe->editor_dialog);
    return ret;
}

gint
grdc_file_editor_create (GtkWindow *parent)
{
    GrdcFileEditor editor;
    gint ret;

    editor.parent = parent;
    editor.grdc_file = grdc_file_new ();
    editor.quit_when_close = FALSE;

    ret = grdc_file_editor_main (&editor);

    grdc_file_free (editor.grdc_file);

    return ret;
}

gint
grdc_file_editor_open (GtkWindow *parent, gchar *filename, gboolean quit_when_close)
{
    GrdcFileEditor editor;
    gint ret;

    editor.parent = parent;
    editor.grdc_file = grdc_file_load (filename);
    editor.quit_when_close = quit_when_close;

    ret = grdc_file_editor_main (&editor);

    grdc_file_free (editor.grdc_file);

    return ret;
}

gint
grdc_file_editor_open_temp (GtkWindow *parent, gboolean quit_when_close)
{
    GrdcFileEditor editor;
    gint ret;

    editor.parent = parent;
    editor.grdc_file = grdc_file_new_temp ();
    editor.quit_when_close = quit_when_close;

    ret = grdc_file_editor_main (&editor);

    grdc_file_free (editor.grdc_file);

    return ret;
}

