/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "config.h"
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcsftpwindow.h"
#include "grdcplug.h"

G_DEFINE_TYPE (GrdcPlug, grdc_plug, GTK_TYPE_EVENT_BOX)

enum {
    CONNECT_SIGNAL,
    DISCONNECT_SIGNAL,
    DESKTOP_RESIZE_SIGNAL,
    LAST_SIGNAL
};

typedef struct _GrdcPlugSignalData
{
    GrdcPlug *gp;
    const gchar *signal_name;
} GrdcPlugSignalData;

static guint grdc_plug_signals[LAST_SIGNAL] = { 0 };

static void
grdc_plug_class_init (GrdcPlugClass *klass)
{
    grdc_plug_signals[CONNECT_SIGNAL] =
        g_signal_new ("connect",
            G_TYPE_FROM_CLASS (klass),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            G_STRUCT_OFFSET (GrdcPlugClass, connect),
            NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    grdc_plug_signals[DISCONNECT_SIGNAL] =
        g_signal_new ("disconnect",
            G_TYPE_FROM_CLASS (klass),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            G_STRUCT_OFFSET (GrdcPlugClass, disconnect),
            NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    grdc_plug_signals[DESKTOP_RESIZE_SIGNAL] =
        g_signal_new ("desktop-resize",
            G_TYPE_FROM_CLASS (klass),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            G_STRUCT_OFFSET (GrdcPlugClass, desktop_resize),
            NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
grdc_plug_init_cancel (GrdcInitDialog *dialog, gint response_id, GrdcPlug *gp)
{
    if ((response_id == GTK_RESPONSE_CANCEL || response_id == GTK_RESPONSE_DELETE_EVENT)
        && dialog->mode == GRDC_INIT_MODE_CONNECTING)
    {
        grdc_plug_close_connection (gp);
    }
}

static void
grdc_plug_show_init_dialog (GrdcPlug *gp, GtkWindow *parent, const gchar *name)
{
    if (gp->init_dialog)
    {
        gtk_widget_destroy (gp->init_dialog);
    }
    gp->init_dialog = grdc_init_dialog_new (parent, _("Connecting to '%s'..."), (name ? name : "*"));
    g_signal_connect (G_OBJECT (gp->init_dialog), "response", G_CALLBACK (grdc_plug_init_cancel), gp);
    gtk_widget_show (gp->init_dialog);
}

static void
grdc_plug_hide_init_dialog (GrdcPlug *gp)
{
    if (gp->init_dialog && GTK_IS_WIDGET (gp->init_dialog))
    {
        gtk_widget_destroy (gp->init_dialog);
    }
    gp->init_dialog = NULL;
}

static void
grdc_plug_destroy (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_connect (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_disconnect (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_init (GrdcPlug *gp)
{
    g_signal_connect (G_OBJECT (gp), "destroy", G_CALLBACK (grdc_plug_destroy), NULL);
    g_signal_connect (G_OBJECT (gp), "connect", G_CALLBACK (grdc_plug_connect), NULL);
    g_signal_connect (G_OBJECT (gp), "disconnect", G_CALLBACK (grdc_plug_disconnect), NULL);

    gp->init_dialog = NULL;
    gp->grdc_file = NULL;
    gp->has_error = FALSE;
    gp->width = 0;
    gp->height = 0;
    gp->scale = FALSE;
    gp->error_message[0] = '\0';
    gp->ssh_tunnel = NULL;
    gp->sftp_window = NULL;
    gp->closed = FALSE;
}

gboolean
grdc_plug_open_connection (GrdcPlug *gp, GtkWindow *parent, GrdcFile *grdcfile)
{
    gp->grdc_file = grdcfile;

    /* Show "server" instead of "name" for quick connect */
    grdc_plug_show_init_dialog (gp, parent, (grdcfile->filename ? grdcfile->name : grdcfile->server));
    return (* (GRDC_PLUG_GET_CLASS (gp)->open_connection)) (gp);
}

gboolean
grdc_plug_close_connection (GrdcPlug *gp)
{
    if (!GTK_IS_WIDGET (gp) || gp->closed) return FALSE;
    gp->closed = TRUE;

#ifdef HAVE_LIBSSH
    if (gp->ssh_tunnel)
    {
        grdc_ssh_tunnel_free (gp->ssh_tunnel);
        gp->ssh_tunnel = NULL;
    }
#endif
    return (* (GRDC_PLUG_GET_CLASS (gp)->close_connection)) (gp);
}

#ifdef HAVE_LIBSSH
static void
grdc_plug_sftp_window_destroy (GtkWidget *widget, GrdcPlug *gp)
{
    gp->sftp_window = NULL;
}
#endif

static gboolean
grdc_plug_emit_signal_timeout (gpointer user_data)
{
    GrdcPlugSignalData *data = (GrdcPlugSignalData*) user_data;
    g_signal_emit_by_name (G_OBJECT (data->gp), data->signal_name);
    g_free (data);
    return FALSE;
}

void
grdc_plug_emit_signal (GrdcPlug *gp, const gchar *signal_name)
{
    GrdcPlugSignalData *data;

    data = g_new (GrdcPlugSignalData, 1);
    data->gp = gp;
    data->signal_name = signal_name;
    TIMEOUT_ADD (0, grdc_plug_emit_signal_timeout, data);
}

const gpointer
grdc_plug_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    switch (feature)
    {
        case GRDC_PLUG_FEATURE_TOOL:
            return GINT_TO_POINTER (1);
#ifdef HAVE_LIBSSH
        case GRDC_PLUG_FEATURE_TOOL_SFTP:
        case GRDC_PLUG_FEATURE_TOOL_SSHTERM:
            return (gp->grdc_file->ssh_enabled ? GINT_TO_POINTER (1) : NULL);
#endif
        default:
            return (* (GRDC_PLUG_GET_CLASS (gp)->query_feature)) (gp, feature);
    }
}

void
grdc_plug_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, const gpointer data)
{
    switch (feature)
    {
#ifdef HAVE_LIBSSH
        case GRDC_PLUG_FEATURE_TOOL_SFTP:
            if (!gp->sftp_window)
            {
                GrdcSFTP *sftp;

                if (!gp->ssh_tunnel) return;

                sftp = grdc_sftp_new_from_ssh (GRDC_SSH (gp->ssh_tunnel));
                gp->sftp_window = grdc_sftp_window_new_init (NULL, sftp, FALSE);
                if (!gp->sftp_window) return;

                g_signal_connect (G_OBJECT (gp->sftp_window), "destroy",
                    G_CALLBACK (grdc_plug_sftp_window_destroy), gp);
            }
            gtk_window_present (GTK_WINDOW (gp->sftp_window));
            break;

        case GRDC_PLUG_FEATURE_TOOL_SSHTERM:
            if (gp->ssh_tunnel)
            {
                GrdcSSHTerminal *term;
                GtkWidget *dialog;

                term = grdc_ssh_terminal_new_from_ssh (GRDC_SSH (gp->ssh_tunnel));
                if (!grdc_ssh_terminal_open (term))
                {
                    dialog = gtk_message_dialog_new (GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (gp))),
                        GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                        (GRDC_SSH (term))->error, NULL);
                    gtk_dialog_run (GTK_DIALOG (dialog));
                    gtk_widget_destroy (dialog);
                    grdc_ssh_free (GRDC_SSH (term));
                    return;
                }
            }
            break;

#endif
        default:
            (* (GRDC_PLUG_GET_CLASS (gp)->call_feature)) (gp, feature, data);
            break;
    }
}

gchar*
grdc_plug_start_tunnel (GrdcPlug *gp, gint default_port)
{
    gchar dest[200];
#ifdef HAVE_LIBSSH
    GrdcSSHTunnel *tunnel;
    gint ret;
#endif

    if (grdc_file_is_incoming (gp->grdc_file))
    {
        dest[0] = '\0';
    }
    else
    {
        if (g_strrstr (gp->grdc_file->server, ":") == NULL)
        {
            g_snprintf (dest, sizeof (dest), "%s:%i", gp->grdc_file->server, default_port);
        }
        else
        {
            g_strlcpy (dest, gp->grdc_file->server, sizeof (dest));
        }
    }

#ifdef HAVE_LIBSSH
    if (!gp->grdc_file->ssh_enabled || grdc_file_is_incoming (gp->grdc_file))
    {
        return g_strdup (dest);
    }

    /* Reuse existing SSH connection if it's reconnecting to destination */
    if (gp->ssh_tunnel == NULL)
    {
        tunnel = grdc_ssh_tunnel_new_from_file (gp->grdc_file);

        THREADS_ENTER
        grdc_init_dialog_set_status (GRDC_INIT_DIALOG (gp->init_dialog),
            _("Connecting to SSH server %s..."), GRDC_SSH (tunnel)->server);
        THREADS_LEAVE

        if (!grdc_ssh_init_session (GRDC_SSH (tunnel)))
        {
            g_strlcpy (gp->error_message, GRDC_SSH (tunnel)->error, MAX_ERROR_LENGTH);
            grdc_ssh_tunnel_free (tunnel);
            gp->has_error = TRUE;
            return NULL;
        }

        ret = grdc_ssh_auth_gui (GRDC_SSH (tunnel), GRDC_INIT_DIALOG (gp->init_dialog), TRUE);
        if (ret <= 0)
        {
            if (ret == 0)
            {
                g_strlcpy (gp->error_message, GRDC_SSH (tunnel)->error, MAX_ERROR_LENGTH);
                gp->has_error = TRUE;
            }
            grdc_ssh_tunnel_free (tunnel);
            return NULL;
        }

        gp->ssh_tunnel = tunnel;
    }

    THREADS_ENTER
    grdc_init_dialog_set_status (GRDC_INIT_DIALOG (gp->init_dialog),
        _("Connecting to %s through SSH tunnel..."), gp->grdc_file->server);
    THREADS_LEAVE

    if (!grdc_ssh_tunnel_start (gp->ssh_tunnel, dest, grdc_pref.sshtunnel_port))
    {
        g_strlcpy (gp->error_message, GRDC_SSH (gp->ssh_tunnel)->error, MAX_ERROR_LENGTH);
        grdc_ssh_tunnel_free (gp->ssh_tunnel);
        gp->has_error = TRUE;
        return NULL;
    }

    return g_strdup_printf ("127.0.0.1:%i", grdc_pref.sshtunnel_port);

#else

    return g_strdup (dest);

#endif
}


