/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>
#include "grdcpixmaps.h"
#include "grdcpublic.h"

GtkWidget*
grdc_public_create_combo_entry (const gchar *text, const gchar *def, gboolean descending)
{
    GtkWidget *combo;
    gboolean found;
    gchar *buf, *ptr1, *ptr2;
    gint i;

    combo = gtk_combo_box_entry_new_text ();
    found = FALSE;

    if (text && text[0] != '\0')
    {
        buf = g_strdup (text);
        ptr1 = buf;
        i = 0;
        while (ptr1 && *ptr1 != '\0')
        {
            ptr2 = strchr (ptr1, STRING_DELIMITOR);
            if (ptr2) *ptr2++ = '\0';

            if (descending)
            {
                gtk_combo_box_prepend_text (GTK_COMBO_BOX (combo), ptr1);
                if (!found && g_strcmp0 (ptr1, def) == 0)
                {
                    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
                    found = TRUE;
                }
            }
            else
            {
                gtk_combo_box_append_text (GTK_COMBO_BOX (combo), ptr1);
                if (!found && g_strcmp0 (ptr1, def) == 0)
                {
                    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
                    found = TRUE;
                }
            }

            ptr1 = ptr2;
            i++;
        }

        g_free (buf);
    }

    if (!found && def && def[0] != '\0')
    {
        gtk_entry_set_text (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (combo))), def);
    }

    return combo;
}

GtkWidget*
grdc_public_create_combo_text_d (const gchar *text, const gchar *def, const gchar *empty_choice)
{
    GtkWidget *combo;
    GtkListStore *store;
    GtkCellRenderer *text_renderer;

    store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
    combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL(store)); 

    text_renderer = gtk_cell_renderer_text_new (); 
    gtk_cell_layout_pack_end (GTK_CELL_LAYOUT(combo), text_renderer, TRUE); 
    gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT(combo), text_renderer, "text", 1);

    grdc_public_load_combo_text_d (combo, text, def, empty_choice);

    return combo;
}

void
grdc_public_load_combo_text_d (GtkWidget *combo, const gchar *text, const gchar *def, const gchar *empty_choice)
{
    GtkListStore *store;
    GtkTreeIter iter;
    gint i;
    gchar *buf, *ptr1, *ptr2;

    store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (combo)));
    gtk_list_store_clear (store);

    i = 0;

    if (empty_choice)
    {
        gtk_list_store_append (store, &iter); 
        gtk_list_store_set (store, &iter, 0, "", 1, empty_choice, -1);
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
        i++;
    }

    if (text == NULL || text[0] == '\0') return;

    buf = g_strdup (text);
    ptr1 = buf;
    while (ptr1 && *ptr1 != '\0')
    {
        ptr2 = strchr (ptr1, STRING_DELIMITOR);
        if (ptr2) *ptr2++ = '\0';

        gtk_list_store_append (store, &iter); 
        gtk_list_store_set (store, &iter, 0, ptr1, 1, ptr1, -1);

        if (i == 0 || g_strcmp0 (ptr1, def) == 0)
        {
            gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
        }

        i++;
        ptr1 = ptr2;
    }

    g_free (buf);
}

GtkWidget*
grdc_public_create_combo_map (const gpointer *key_value_list, const gchar *def, gboolean use_icon)
{
    gint i;
    GtkWidget *combo;
    GtkListStore *store; 
    GtkTreeIter iter; 
    GtkCellRenderer *text_renderer;
    GtkCellRenderer *pixbuf_renderer;
    GdkPixbuf *pixbuf;
    gint image_size = 0;

    if (use_icon)
    {
        store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, GDK_TYPE_PIXBUF);
    }
    else
    {
        store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
    }
    combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL(store)); 

    if (use_icon)
    {
        pixbuf_renderer = gtk_cell_renderer_pixbuf_new ();
        gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(combo), pixbuf_renderer, FALSE); 
        gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT(combo), pixbuf_renderer, "pixbuf", 2);
        gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &image_size, NULL);
    }
    text_renderer = gtk_cell_renderer_text_new (); 
    gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(combo), text_renderer, TRUE); 
    gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT(combo), text_renderer, "text", 1);
    if (use_icon) g_object_set (G_OBJECT (text_renderer), "xpad", 5, NULL);

    for (i = 0; key_value_list[i]; i += (use_icon ? 3 : 2))
    {
        gtk_list_store_append (store, &iter); 
        gtk_list_store_set (store, &iter, 0, key_value_list[i], 1, _(key_value_list[i + 1]), -1);
        if (use_icon)
        {
            pixbuf = grdc_pixmap (GPOINTER_TO_INT (key_value_list[i + 2]), image_size);
            gtk_list_store_set (store, &iter, 2, pixbuf, -1);
            g_object_unref (pixbuf);
        }
        if (i == 0 || g_strcmp0 (key_value_list[i], def) == 0)
        {
            gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i / (use_icon ? 3 : 2));
        }
    }
    return combo;
}

GtkWidget*
grdc_public_create_combo_mapint (const gpointer *key_value_list, gint def, gboolean use_icon)
{
    gchar buf[20];
    g_snprintf (buf, sizeof (buf), "%i", def);
    return grdc_public_create_combo_map (key_value_list, buf, use_icon);
}

void
grdc_public_popup_position (GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data)
{
    GtkWidget *widget;
    gint tx, ty;

    widget = GTK_WIDGET (user_data);
    if (widget->window == NULL)
    {
        *x = 0; *y = 0; *push_in = TRUE; return;
    }
    gdk_window_get_origin (widget->window, &tx, &ty);
    if (GTK_WIDGET_NO_WINDOW (widget))
    {
        tx += widget->allocation.x;
        ty += widget->allocation.y;
    }

    *x = tx;
    *y = ty + widget->allocation.height - 1;
    *push_in = TRUE;
}

gchar*
grdc_public_combine_path (const gchar *path1, const gchar *path2)
{
    if (!path1 || path1[0] == '\0') return g_strdup (path2);
    if (path1[strlen (path1) - 1] == '/') return g_strdup_printf ("%s%s", path1, path2);
    return g_strdup_printf ("%s/%s", path1, path2);
}

void
grdc_public_threads_leave (void* data)
{
    gdk_flush();
    gdk_threads_leave();
}

