# $Id: Greenwich.pm,v 1.4 2003/04/10 09:47:13 jodrell Exp $
package Greenwich;

use Gnome;
use Greenwich::L10N;
use Greenwich::WhoisClient;
use Greenwich::WhoisMap;
use Greenwich::AboutBox;
use Greenwich::HelpWindow;
use Greenwich::ServerList;
use Greenwich::Preferences;
use strict;

sub new {
	my $self		= {};
	$self->{PACKAGE}	= shift;
	$self->{NAME}		= 'Greenwich';
	$self->{VERSION}	= '0.5.2';
	$self->{RCFILE}		= sprintf('%s/.%src', $ENV{HOME}, $self->{NAME});
	$main::INSTANCES++;
	bless($self, $self->{PACKAGE});
	return $self;
}

sub init {
	my $self = shift;
	Gnome->init($self->{NAME});
	$self->{client} = Greenwich::WhoisClient->new();
	$self->load_options();
	$self->load_lexicon();
	$self->build_ui();
	$self->configure();
	Gtk->main();
	return;
}

sub load_lexicon {
	my $self = shift;
	my $lexicon;
	($lexicon = Greenwich::L10N->get_handle()) || ($lexicon = Greenwich::L10N->get_handle('en')) || die "Can't find an acceptable language module";
	$self->{lexicon} = $lexicon;
	return;
}

sub load_options {
	my $self = shift;
	$self->{OPTIONS} = {
		auto	=> 'true',
		follow	=> 'true',
		timeout	=> 10,
		font	=> '-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1',
	};
	open(RCFILE, $self->{RCFILE}) or return undef;
	while (<RCFILE>) {
		chomp;
		unless (/^\s?([;#])/) {
			my ($name, $value) = split(/=/, $_, 2);
			$self->{OPTIONS}{lc($name)} = $value;
		}
	}
	close(RCFILE);
	return;
}

sub save_options {
	my $self = shift;
	$self->{OPTIONS}{history} = join(',', grep { defined } @{$self->{HISTORY}});
	$self->{OPTIONS}{servers} = join(',', @{$self->{SERVERS}});
	open(RCFILE, ">$self->{RCFILE}") or $self->error("$self->{RCFILE}: $!") and return undef;
	foreach my $name (keys %{$self->{OPTIONS}}) {
		printf(RCFILE "%s=%s\n", lc($name), $self->{OPTIONS}->{$name});
	}
	close(RCFILE);
	return;
}

sub configure {
	my $self = shift;
	@{$self->{SERVERS}} = split(/,/, $self->{OPTIONS}{servers});
	@{$self->{DEFAULT_SERVERS}} = qw(
		whois.networksolutions.com
		whois.ripe.net
		whois.centralnic.net
		whois.nic.us
		whois.nic.uk
		whois.aunic.net
		whois.cnnic.net.cn
		whois.crsnic.net
		whois.krnic.net
		whois.nic.ad.jp
		whois.nic.br
		whois.nic.fr
		whois.nic.nu
		whois.ripe.net
		whois.tonic.to
		whois.twnic.net
	);
	@{$self->{HISTORY}} = grep { defined } split(/,/, $self->{OPTIONS}{history});
	$self->{FONT} = Gtk::Gdk::Font->load($self->{OPTIONS}{font});
	$self->set_output($self->get_output());
	$self->set_servers();
	$self->set_history();

	return;
}

sub build_ui {
	my $self = shift;
	$self->{app} = Gnome::App->new($self->{NAME}, $self->{NAME});
	$self->{app}->signal_connect('delete_event', sub { $self->close_program() });
	$self->build_menu();
	$self->build_toolbar();

	# build the control bar:
	$self->{controlbox} = Gtk::HBox->new(0, 0);
	$self->{controlbox}->set_spacing(5);

	$self->{controls}{auto} = Gtk::CheckButton->new($self->string('autodetect'));
	$self->{controls}{auto}->signal_connect('clicked', sub { $self->auto_toggle() });
	$self->{controlbox}->pack_start($self->{controls}{auto}, 0, 0, 0);

	$self->{controlbox}->pack_start($self->label('server'), 0, 0, 0);

	$self->{controls}{server} = Gtk::Combo->new();
	$self->{controls}{server}->disable_activate();
	$self->{controlbox}->pack_start($self->{controls}{server}, 0, 0, 0);

	$self->{controlbox}->pack_start(Gtk::VSeparator->new(), 0, 0, 0);

	$self->{controlbox}->pack_start($self->label('port'), 0, 0, 0);

	$self->{controls}{port} = Gtk::Entry->new();
	$self->{controls}{port}->set_usize(30, 0);
	$self->{controls}{port}->set_text(43);
	$self->{controlbox}->pack_start($self->{controls}{port}, 0, 0, 0);

	$self->{controlbox}->pack_start(Gtk::VSeparator->new(), 0, 0, 0);

	$self->{controlbox}->pack_start($self->label('query'), 0, 0, 0);

	$self->{controls}{query} = Gtk::Combo->new();
	$self->{controls}{query}->disable_activate();
	$self->{controls}{query}->entry()->signal_connect('activate', sub { $self->make_query() });
	$self->{controlbox}->pack_start($self->{controls}{query}, 1, 0, 0);

	$self->{controls}{button} = Gtk::Button->new();
	$self->{controls}{button}->add( Gnome::Stock->new_with_icon('Jump To'));
	$self->{controls}{button}->set_relief('half');
	$self->{controls}{button}->signal_connect('clicked', sub { $self->make_query() });
	$self->{controlbox}->pack_start($self->{controls}{button}, 0, 0, 0);

	$self->{controlbar} = Gtk::HandleBox->new();
	$self->{controlbar}->set_shadow_type('none');
	$self->{controlbar}->add($self->{controlbox});

	# build the output window:
	$self->{output} = Gtk::Text->new();
	$self->{output}->set_editable(0);
	$self->{scrollbar} = Gtk::VScrollbar->new($self->{output}->vadj());
	$self->{window} = Gtk::HBox->new(0, 0);
	$self->{window}->pack_start($self->{output}, 1, 1, 0);
	$self->{window}->pack_start($self->{scrollbar}, 0, 0, 0);

	# bring everything together:
	$self->{vbox} = Gtk::VBox->new(0, 0);
	$self->{vbox}->pack_start($self->{controlbar}, 0, 1, 0);
	$self->{vbox}->pack_start($self->{window},     1, 1, 0);

	# set up the user interface settings:
	$self->{controls}{auto}->signal_emit('clicked') if ($self->{OPTIONS}{auto} eq 'true');

	$self->{app}->set_contents($self->{vbox});

	$self->{status} = Gnome::AppBar->new(0, 1, 'never');
	$self->status($self->string('ready'));
	$self->{app}->set_statusbar($self->{status});

	$self->{app}->set_default_size(400, 500);

	$self->{app}->show_all();

	$self->{controls}{query}->entry()->grab_focus();

	return;
}

sub build_menu {
	my $self = shift;
	$self->{app}->create_menus(
		{
			type	=> 'subtree',
			label	=> '_File',
			subtree	=> [
				{
					type		=> 'item',
					label		=> '_New Window',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_New',
					callback	=> sub { Greenwich->new()->init() },
				},
				{
					type		=> 'item',
					label		=> '_Save...',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Save',
					callback	=> sub { $self->save_output() },
				},
				{	type		=> 'separator' },
				{
					type		=> 'item',
					label		=> 'E_xit',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Quit',
					callback	=> sub { $self->close_program() },
				}
			]
		},
		{
			type	=> 'subtree',
			label	=> '_Edit',
			subtree	=> [
				{
					type		=> 'item',
					label		=> 'C_opy',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Copy',
					callback	=> sub { $self->{output}->copy_clipboard() },
				},
				{
					type		=> 'item',
					label		=> '_Paste',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Paste',
					callback	=> sub { $self->{controls}{query}->entry()->copy_clipboard() },
				},
				{	type		=> 'separator' },
				{
					type		=> 'item',
					label		=> '_Preferences...',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Preferences',
					callback	=> sub { $self->edit_preferences() },
				},
				{
					type		=> 'item',
					label		=> '_'.$self->string('serverlist').'...',
					pixmap_type	=> 'stock',
					pixmap_info	=> 'Menu_Index',
					callback	=> sub { $self->edit_server_list() },
				},
			]
		},
		{
			type	=> 'subtree',
			label	=> '_Help',
			subtree	=> [
				{
					type		=> 'item',
					label		=> '_Help',
					pixmap_type 	=> 'stock',
					pixmap_info	=> 'Menu_Help',
					callback	=> sub { Greenwich::HelpWindow->new() },
				},
				{
					type		=> 'item',
					label		=> '_About...',
					pixmap_type 	=> 'stock',
					pixmap_info	=> 'Menu_About',
					callback	=> sub { Greenwich::AboutBox->new($self) },
				}
			]
		}
	);
	return;
}

sub build_toolbar {
	my $self = shift;
	$self->{app}->create_toolbar(
		{
			type		=> 'item',
			label		=> 'Save',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Save',
			hint		=> $self->string('savehint'),
			callback	=> sub { $self->save_output() },
		},
		{
			type		=> 'item',
			label		=> 'Preferences',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Preferences',
			hint		=> $self->string('prefshint'),
			callback	=> sub { $self->edit_preferences() },
		},
		{
			type		=> 'item',
			label		=> 'Server List',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Index',
			hint		=> $self->string('serverlist'),
			callback	=> sub { $self->edit_server_list() },
		},
		{
			type		=> 'item',
			label		=> 'Query',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Jump To',
			hint		=> $self->string('queryhint'),
			callback	=> sub { $self->make_query() },
		},
		{
			type		=> 'item',
			label		=> 'Help',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Help',
			hint		=> $self->string('helphint'),
			callback	=> sub { Greenwich::HelpWindow->new() },
		},
		{
			type		=> 'item',
			label		=> 'Exit',
			pixmap_type	=> 'stock',
			pixmap_info	=> 'Quit',
			hint		=> $self->string('quithint'),
			callback	=> sub { $self->close_program() },
		}
	);
	return;
}

sub auto_toggle {
	my $self = shift;
	if ($self->{controls}{auto}->active()) {
		$self->{OPTIONS}{auto} = 'true';
		$self->{controls}{server}->set_sensitive(0);
	} else {
		$self->{OPTIONS}{auto} = 'false';
		$self->{controls}{server}->set_sensitive(1);
	}
	return;
}

sub server_list {
	my $self = shift;
	my @servers = @{$self->{DEFAULT_SERVERS}};
	unshift(@servers, undef);
	if (scalar(@{$self->{SERVERS}}) > 0) {
		push(@servers, undef);
		push(@servers, @{$self->{SERVERS}});
	}
	return @servers;
}

sub set_servers {
	my $self = shift;
	$self->{controls}{server}->set_popdown_strings($self->server_list());
	return;
}

sub set_history {
	my ($self, $query) = @_;
	my @history = @{$self->{HISTORY}};
	if ($query ne '') {
		unshift(@history, $query);
		$self->{HISTORY} = \@history;
	}
	unshift(@history, undef);
	$self->{controls}{query}->set_popdown_strings(@history);
	return;
}

sub string {
	my ($self, $str) = @_;
	return $self->{lexicon}->maketext($str);
}

sub label {
	my ($self, $str) = @_;
	return Gtk::Label->new($self->{lexicon}->maketext($str));
}

sub status {
	my ($self, $str) = @_;
	$self->{status}->set_status($str);
	return;
}

sub error {
	my ($self, $error) = @_;
	$self->{app}->error($error);
	return;
}

sub make_query {
	my $self = shift;
	my $t0 = time();
	my $query = $self->{controls}{query}->entry()->get_text();
	my %params = (
		query	=> $query,
		port	=> $self->{controls}{port}->get_text(),
		server	=> ($self->{controls}{auto}->active() ? whois_server($query) : $self->{controls}{server}->entry()->get_text()),
		timeout	=> $self->{OPTIONS}{timeout},
		follow	=> $self->{OPTIONS}{follow},
	);
	if ($params{server} eq '') {
		$self->error($self->string('nohost'));
		return undef;
	}
	$self->set_history($query);
	$self->{controls}{query}->entry()->set_text($query);
	$self->status(sprintf($self->string('connecting'), $params{server}));
	Gtk->main_iteration() while (Gtk->events_pending());
	$params{program} = $self->{OPTIONS}{external} if ($self->{OPTIONS}{use_external} eq 'true');
	my ($response, $error) = $self->{client}->whois(%params);
	if ($error ne '') {
		$self->error($error);
		$self->status($self->string('error'));
	} else {
		$self->status(sprintf($self->string('done'), (time() - $t0)));
	}
	$self->set_output($response);
	return;
}

sub set_output {
	my ($self, $output) = @_;
	$output =~ s/\r//g;
	$self->{output}->freeze();
	$self->{output}->backward_delete($self->{output}->get_length());
	$self->{output}->insert($self->{FONT}, undef, undef, $output);
	$self->{output}->thaw();
	return;
}

sub get_output {
	my $self = shift;
	return $self->{output}->get_chars(0, -1);;
}

sub save_output {
	my $self = shift;
	my $output = $self->get_output();
	print $output;
	my $file_selector = Gtk::FileSelection->new($self->string('selectfile'));
	$file_selector->set_position('center');
	$file_selector->signal_connect('destroy', sub { $file_selector->destroy() });
	$file_selector->cancel_button->signal_connect('clicked', sub { $file_selector->destroy() });
	$file_selector->ok_button->signal_connect('clicked', sub {
		my $file = $file_selector->get_filename();
		$file_selector->destroy();
		if (-e $file) {
			$self->{app}->question_modal(
				$self->string('confirmoverwrite'),
				sub {
					if ($_[0] == 0) {
						open(FILE, ">$file") or $self->warning("$file: $!") and return undef;
						print FILE $output;
						$self->status(sprintf($self->string('bufferwritten'), $file))
					}
				}
			);
		} else {
			open(FILE, ">$file") or $self->warning("$file: $!") and return undef;
			print FILE $output;
			$self->status(sprintf($self->string('bufferwritten'), $file))
		}
	});
	$file_selector->show();
	return;

}

sub edit_server_list {
	my $self = shift;
	my $server_list = Greenwich::ServerList->new(lexicon => $self->{lexicon}, app => $self->{app});
	$server_list->set_servers(@{$self->{SERVERS}});
	$server_list->ok_button()->signal_connect('clicked', sub {
		@{$self->{SERVERS}} = $server_list->servers();
		$self->set_servers();
		$server_list->destroy();
	});
	$server_list->show();
	return;
}

sub edit_preferences {
	my $self = shift;
	my $prefs = Greenwich::Preferences->new(lexicon => $self->{lexicon}, options => $self->{OPTIONS});
	$prefs->ok_button()->signal_connect('clicked', sub {
		my %options = $prefs->get_options();
		$self->{OPTIONS} = \%options;
		$self->configure();
		$prefs->destroy();
	});
	$prefs->show();
	return;
}

sub close_program {
	my $self = shift;
	$self->save_options();
	if ($main::INSTANCES > 1) {
		$self->{app}->hide_all();
		$main::INSTANCES--;
		return;
	} else {
		exit;
	}
}

1;
