/*
 *  Copyright (c) by Ramu Ramanathan and Allin Cottrell
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*  monte_carlo.c - loop simulation procedures
*/  

#include "libgretl.h" 
#include "gretl_private.h"
#include "libset.h"
#include "compat.h"

#include <time.h>
#include <unistd.h>

#undef LOOP_DEBUG

#if defined(ENABLE_GMP)
# include <gmp.h>
typedef mpf_t bigval;
#elif defined(HAVE_LONG_DOUBLE)
typedef long double bigval;
#else
typedef double bigval;
#endif

enum inequalities {
    GT = 1,
    LT,
    GTE,
    LTE
};

enum loop_types {
    COUNT_LOOP,
    WHILE_LOOP,
    INDEX_LOOP,
    DATED_LOOP,
    FOR_LOOP,
    EACH_LOOP
};

#define indexed_loop(l) (l->type == INDEX_LOOP || \
                         l->type == DATED_LOOP || \
			 l->type == EACH_LOOP)

typedef struct {
    int ID;
    int *list;
    bigval *sum;
    bigval *ssq;
} LOOP_PRINT;  

/* below: used for special "progressive" loop */ 

typedef struct {
    int ID;                      /* ID number for model */
    int ci;                      /* command index for model */
    int t1, t2, nobs;            /* starting observation, ending
                                    observation, and number of obs */
    int ncoeff, dfn, dfd;        /* number of coefficents; degrees of
                                    freedom in numerator and denominator */
    int *list;                   /* list of variables by ID number */
    int ifc;                     /* = 1 if the equation includes a constant,
                                    else = 0 */
    bigval *sum_coeff;      /* sums of coefficient estimates */
    bigval *ssq_coeff;      /* sums of squares of coeff estimates */
    bigval *sum_sderr;      /* sums of estimated std. errors */
    bigval *ssq_sderr;      /* sums of squares of estd std. errs */
} LOOP_MODEL;

enum loop_flags {
    LOOP_PROGRESSIVE = 1 << 0,
    LOOP_VERBOSE     = 1 << 1
};

struct LOOPSET_ {
    char type;
    char flags;
    int level;
    int err;
    int ntimes;
    int index;
    double initval;
    int lvar;
    int rvar;
    double rval;
    double incr;
    char ineq;
    int ncmds;
    int nmod;
    int nprn;
    int nstore;
    int next_model;
    int next_print;
    char **lines;
    int *ci;
    char **eachstrs;
    MODEL **models;
    LOOP_MODEL *lmodels;
    LOOP_PRINT *prns;
    char storefile[MAXLEN];
    char **storename;
    char **storelbl;
    double *storeval;
    LOOPSET *parent;
    LOOPSET **children;
    int n_children;
};

#define loop_is_progressive(l) (l->flags & LOOP_PROGRESSIVE)
#define loop_set_progressive(l) (l->flags |= LOOP_PROGRESSIVE)
#define loop_is_verbose(l) (l->flags & LOOP_VERBOSE)
#define loop_set_verbose(l) (l->flags |= LOOP_VERBOSE)

static void gretl_loop_init (LOOPSET *loop);
static int prepare_loop_for_action (LOOPSET *loop);
static void free_loop_model (LOOP_MODEL *lmod);
static void free_loop_print (LOOP_PRINT *lprn);
static void print_loop_model (LOOP_MODEL *lmod, int loopnum,
			      const DATAINFO *pdinfo, PRN *prn);
static void print_loop_coeff (const DATAINFO *pdinfo, const LOOP_MODEL *lmod, 
			      int c, int n, PRN *prn);
static void print_loop_prn (LOOP_PRINT *lprn, int n,
			    const DATAINFO *pdinfo, PRN *prn);
static int print_loop_store (LOOPSET *loop, PRN *prn);
static int get_prnnum_by_id (LOOPSET *loop, int id);
static int get_modnum_by_cmdnum (LOOPSET *loop, int cmdnum);

#define LOOP_BLOCK 32

/**
 * ok_in_loop:
 * @ci: command index.
 * @loop: pointer to loop structure
 *
 * Returns: 1 if the given command is acceptable inside the loop construct,
 * 0 otherwise.
 */

int ok_in_loop (int ci, const LOOPSET *loop)
{
    if (ci == OLS || 
	ci == GENR ||
	ci == LOOP ||
	ci == STORE ||
	ci == PRINT ||
	ci == PRINTF ||
	ci == PVALUE ||
	ci == SIM ||
	ci == SMPL ||
	ci == SUMMARY ||
	ci == IF ||
	ci == ELSE ||
	ci == ENDIF ||
	ci == ENDLOOP) { 
	return 1;
    }

    if (ci == LAD || ci == HSK || ci == HCCM || ci == WLS ||
	ci == GARCH || ci == ARMA) {
	return 1;
    }

    if (ci == ADF || ci == KPSS) {
	return 1;
    }

    return 0;
}

/* ......................................................  */

static int loop_attach_child (LOOPSET *loop, LOOPSET *child)
{
    LOOPSET **children;
    int nc = loop->n_children + 1;

    children = realloc(loop->children, nc * sizeof *loop->children);
    if (children == NULL) {
	return 1;
    } 

    loop->children = children;
    loop->children[nc - 1] = child;
    child->parent = loop;
    child->level = loop->level + 1;

#ifdef LOOP_DEBUG
    fprintf(stderr, "child loop %p has parent %p\n", 
	    (void *) child, (void *) child->parent);
#endif

    loop->n_children += 1;

    return 0;
}

static LOOPSET *gretl_loop_new (LOOPSET *parent, int loopstack)
{
    LOOPSET *loop;

    loop = malloc(sizeof *loop);
    if (loop == NULL) {
	return NULL;
    }

    gretl_loop_init(loop);

    if (parent != NULL) {
	int err = loop_attach_child(parent, loop);

	if (err) {
	    free(loop);
	    loop = NULL;
	} 
    }
	
    return loop;
}

int opstr_to_op (const char *s)
{
    int op = 0;

    if (strstr(s, ">=")) {
	op = GTE;
    } else if (strstr(s, ">")) {
	op = GT;
    } else if (strstr(s, "<=")) {
	op = LTE;
    } else if (strstr(s, "<")) {
	op = LT;
    } 
    
    return op;
}

static int 
ok_loop_var (const DATAINFO *pdinfo, const char *vname)
{
    int v = varindex(pdinfo, vname);

    if (v == 0 || v >= pdinfo->v) {
	sprintf(gretl_errmsg, 
		_("Undefined variable '%s' in loop condition."), vname);
	v = 0;
    } else if (pdinfo->vector[v]) {
	strcpy(gretl_errmsg, _("The loop control variable "
	       "must be a scalar"));
	v = 0;
    }
	
    return v;
}

static int parse_as_while_loop (LOOPSET *loop,
				const DATAINFO *pdinfo,
				char *lvar, char *rvar, 
				char *opstr)
{
    int err = 0;

    loop->ineq = opstr_to_op(opstr);
    if (loop->ineq == 0) {
	err = 1;
    }

    if (!err) {
	loop->lvar = ok_loop_var(pdinfo, lvar);
	if (loop->lvar == 0) {
	    err = 1;
	} 
    }

    if (!err) {
	if (numeric_string(rvar)) {
	   loop->rval = dot_atof(rvar);
	} else { 
	    /* try a varname: in this case "rvar" is a true
	       variable */
	    loop->rvar = ok_loop_var(pdinfo, rvar);
	    if (loop->rvar == 0) {
		loop->lvar = 0;
		err = 1;
	    } 
	}
    }

    if (!err) {
	loop->type = WHILE_LOOP;
    }

    return err;
}

static int get_int_value (const char *s, const DATAINFO *pdinfo,
			  const double **Z, int *err)
{
    int v, ret = 0;

    if (numeric_string(s)) {
	ret = atoi(s);
    } else if ((v = ok_loop_var(pdinfo, s)) > 0) {
	ret = Z[v][0];
    } else {
	*err = 1;
    }

    return ret;
}

#define maybe_date(s) (strchr(s, ':') || strchr(s, '/'))

static int parse_as_indexed_loop (LOOPSET *loop,
				  const DATAINFO *pdinfo,
				  const double **Z,
				  const char *lvar, 
				  const char *start,
				  const char *end)
{
    int nstart = -1, nend = -1;
    int dated = 0;
    int err = 0;

    if (lvar != NULL && strcmp(lvar, "i")) {
	sprintf(gretl_errmsg, 
		_("The index variable in a 'for' loop must be the "
		  "special variable 'i'"));
	err = 1;
    }

    if (!err) {
	if (maybe_date(start)) {
	    dated = 1;
	    nstart = dateton(start, pdinfo);
	}
	if (nstart < 0) {
	    nstart = get_int_value(start, pdinfo, Z, &err);
	}
    }

    if (!err) {
	if (maybe_date(end)) {
	    dated = 1;
	    nend = dateton(end, pdinfo);
	}
	if (nend < 0) {
	    nend = get_int_value(end, pdinfo, Z, &err);
	}
    }

    if (!err && nend <= nstart) {
	strcpy(gretl_errmsg, _("Ending value for loop index must be greater "
			       "than starting value."));
	err = 1;
    }

    if (!err) {
	/* initialize loop index to starting value */
	loop->initval = nstart;
	loop->lvar = 0;
	loop->rvar = 0;
	loop->ntimes = nend - nstart + 1; 
	if (dated) {
	    loop->type = DATED_LOOP;
	} else {
	    loop->type = INDEX_LOOP;
	}
    }

    return err;
}

static int parse_as_count_loop (LOOPSET *loop, 
				const DATAINFO *pdinfo,
				const double **Z,
				const char *lvar)
{
    int nt, err = 0;

    /* note: "lvar" may be a numeric constant or a variable:
       if it is a variable it is evaluated only once,
       at loop compile time 
    */
    nt = get_int_value(lvar, pdinfo, Z, &err);

    if (!err && nt <= 0) {
	strcpy(gretl_errmsg, _("Loop count must be positive."));
	err = 1;
    }

    if (!err) {
	loop->ntimes = nt;
	loop->type = COUNT_LOOP;
    }

    return err;
}

static int 
test_forloop_element (const char *s, LOOPSET *loop,
		      DATAINFO *pdinfo, double ***pZ,
		      int i)
{
    char lhs[9], opstr[3], rhs[9];
    int ngot, err = 0;

    if (s == NULL) return 1;

#ifdef LOOP_DEBUG
    fprintf(stderr, "testing forloop element '%s'\n", s);
#endif

    if (i == 0) {
	ngot = sscanf(s, "%8[^=]=%8s", lhs, rhs) + 1;
	strcpy(opstr, "=");
    } else {
	ngot = sscanf(s, "%8[^+-*/=<>]%2[+-*/=<>]%8[^+-*/=<>]", 
		      lhs, opstr, rhs);
    }

    if (ngot != 3) {
	err = E_PARSE;
    } else {
	int v = varindex(pdinfo, lhs);

	/* examine the LHS */
	if (i == 0) {
	    if (v == pdinfo->v) {
		err = dataset_add_scalar(pZ, pdinfo);
		if (err) {
		    strcpy(gretl_errmsg, _("Out of memory!"));
		} else {
		    strcpy(pdinfo->varname[v], lhs);
		    (*pZ)[v][0] = 0.0;
		    loop->lvar = v;
		}
	    } else {
		loop->lvar = ok_loop_var(pdinfo, lhs);
		if (loop->lvar == 0) {
		    err = 1;
		}
	    }
	} else if (v != loop->lvar) {
	    strcpy(gretl_errmsg, _("No valid loop condition was given."));
	    err = 1;
	}
	    
	if (!err) {
	    /* examine the RHS */
	    if (numeric_string(rhs)) {
		double x = dot_atof(rhs);

		if (i == 0) {
		    loop->initval = x;
		} else if (i == 1) {
		    loop->rval = x;
		} else {
		    loop->incr = x;
		}
	    } else {
		v = ok_loop_var(pdinfo, rhs);
		if (v > 0) {
		    if (i == 0) {
			loop->initval = (*pZ)[v][0];
		    } else if (i == 1) {
			loop->rvar = v;
		    } else {
			loop->incr = (*pZ)[v][0];
		    }
		} else {
		    err = 1;
		}
	    } 
	}
	
	/* examine operator(s) */
	if (!err) {
	    if (i == 1) {
		loop->ineq = opstr_to_op(opstr);
		if (loop->ineq == 0) {
		    err = 1;
		}
	    } else if (i == 2) {
		if (!strcmp(opstr, "-=")) {
		    loop->incr *= -1.0;
		} else if (strcmp(opstr, "+=")) {
		    sprintf(gretl_errmsg, "Invalid operator '%s'", opstr);
		    err = 1;
		}
	    }
	}
    }

    return err;
}

static int
parse_as_each_loop (LOOPSET *loop, const char *s)
{
    char ivar[3];
    int i, nf, err = 0;

    s += strlen("loop foreach");

    if (*s == '\0') {
	return 1;
    }

    s++;
    if (sscanf(s, "%2s", ivar) != 1) {
	err = 1;
    } else if (strcmp(ivar, "i") && strcmp(ivar, "$i")) {
	err = 1;
    }

    if (err) {
	return 1;
    }

    s += strlen(ivar);
    nf = count_fields(s);
    if (nf == 0) {
	return 1;
    }

    loop->eachstrs = malloc(nf * sizeof *loop->eachstrs);
    if (loop->eachstrs == NULL) {
	err = E_ALLOC;
    }

    for (i=0; i<nf && !err; i++) {
	int len;

	while (isspace((unsigned char) *s)) s++;
	len = strcspn(s, " ");

	loop->eachstrs[i] = gretl_strndup(s, len);
	if (loop->eachstrs[i] == NULL) {
	    int j;
	    
	    for (j=0; j<i; j++) {
		free(loop->eachstrs[j]);
	    }
	    free(loop->eachstrs);
	    loop->eachstrs = NULL;
	    err = E_ALLOC;
	} else {
	    s += len;
	}
    }

    if (!err) {
	loop->type = EACH_LOOP;
	loop->ntimes = nf;
    }

    return err;
}

static int 
parse_as_for_loop (LOOPSET *loop,
		   DATAINFO *pdinfo, double ***pZ,
		   const char *s)
{
    char *p = strchr(s, '(');
    int err = 0;

    if (p == NULL) {
	err = 1;
    } else {
	char *forstr = NULL;
	int len = strcspn(p, ")");

	if (len < 4 || (forstr = malloc(len)) == NULL) {
	    err = 1;
	} else {
	    char *forbits[3];
	    int i = 0;

	    /* make compressed copy of string */
	    p++;
	    while (*p) {
		if (*p == ')') break;
		if (*p != ' ') {
		    forstr[i++] = *p;
		}
		p++;
	    }
	    forstr[i] = '\0';

	    /* split terms separated by ';' */
	    for (i=0; i<3; i++) {
		forbits[i] = strtok((i == 0)? forstr : NULL, ";");
		err = test_forloop_element(forbits[i], loop, 
					   pdinfo, pZ, i);
		if (err) break;
	    }

	    free(forstr);
	}
    }

    if (!err) {
	loop->type = FOR_LOOP;
    }

    return err;
}

/**
 * parse_loopline:
 * @line: command line.
 * @ploop: current loop struct pointer, or %NULL.
 * @loopstack: stacking level for the loop.
 * @pdinfo: data information struct.
 * @pZ: pointer to data array.
 *
 * Parse a line specifying a loop condition.
 *
 * Returns: loop pointer on successful completion, %NULL on error.
 */

static LOOPSET *
parse_loopline (char *line, LOOPSET *ploop, int loopstack,
		DATAINFO *pdinfo, double ***pZ)
{
    LOOPSET *loop;
    char lvar[VNAMELEN], rvar[VNAMELEN], op[VNAMELEN];
    int err = 0;

#ifdef LOOP_DEBUG
    fprintf(stderr, "parse_loopline: ploop = %p, loopstack = %d\n",
	    (void *) ploop, loopstack);
#endif

    if (ploop == NULL) {
	/* starting from scratch */
#ifdef LOOP_DEBUG
	fprintf(stderr, "parse_loopline: starting from scratch\n");
#endif
	loop = gretl_loop_new(NULL, 0);
	if (loop == NULL) {
	    gretl_errmsg_set(_("Out of memory!"));
	    return NULL;
	}
    } else if (loopstack > ploop->level) {
	/* have to nest this loop */
#ifdef LOOP_DEBUG
	fprintf(stderr, "parse_loopline: adding child\n");
#endif
	loop = gretl_loop_new(ploop, loopstack);
	if (loop == NULL) {
	    gretl_errmsg_set(_("Out of memory!"));
	    return NULL;
	}
    } else {
	/* shouldn't happen: need error message? */
	loop = ploop;
    }

    *gretl_errmsg = '\0';
    
    if (sscanf(line, "loop while %8[^ <>=]%8[ <>=] %8s", lvar, op, rvar) == 3) {
	err = parse_as_while_loop(loop, pdinfo, lvar, rvar, op);
    }

    else if (sscanf(line, "loop i = %8[^.]..%8s", op, rvar) == 2) {
	err = parse_as_indexed_loop(loop, pdinfo, (const double **) *pZ, 
				    NULL, op, rvar);
    }	

    else if (sscanf(line, "loop for %8[^= ] = %8[^.]..%8s", lvar, op, rvar) == 3) {
	err = parse_as_indexed_loop(loop, pdinfo, (const double **) *pZ, 
				    lvar, op, rvar);
    }

    else if (strstr(line, "loop foreach") != NULL) {
	err = parse_as_each_loop(loop, line);
    }    

    else if (strstr(line, "loop for") != NULL) {
	err = parse_as_for_loop(loop, pdinfo, pZ, line);
    }

    else if (sscanf(line, "loop %8s", lvar) == 1) {
	err = parse_as_count_loop(loop, pdinfo, (const double **) *pZ, 
				  lvar);
    }

    /* out of options, complain */
    else {
	strcpy(gretl_errmsg, _("No valid loop condition was given."));
	err = 1;
    }

    if (!err && loop->lvar == 0 && loop->ntimes < 2) {
	strcpy(gretl_errmsg, _("Loop count missing or invalid\n"));
	err = 1;
    }

    if (!err) {
	/* allocates loop->lines, loop->ci */
	err = prepare_loop_for_action(loop);
    }

    if (err) {
	if (loop != ploop) {
	    free(loop->lines);
	    free(loop->ci);
	    free(loop);
	}
	loop = NULL;
    }

    return loop;
}

#define DEFAULT_MAX_ITER 250
#define MAX_FOR_TIMES  100000

static int get_max_iters (void)
{
    static int ml = 0;

    if (ml == 0) {
	char *mlstr = getenv("GRETL_MAX_ITER");

	if (mlstr != NULL && sscanf(mlstr, "%d", &ml)) ;
	else ml = DEFAULT_MAX_ITER;
    }

    return ml;
}

static int loop_count_too_high (LOOPSET *loop)
{
    static int max_iters = 0;
    int nt = loop->ntimes + 1;

    if (loop->type == FOR_LOOP) {
	/* FIXME */
	if (nt >= MAX_FOR_TIMES) {
	    sprintf(gretl_errmsg, _("Reached maximum interations, %d"),
		    MAX_FOR_TIMES);
	    loop->err = 1;
	}
    } else {  
	if (max_iters == 0) {
	    max_iters = get_max_iters();
	}

	if (nt >= max_iters) {
	    sprintf(gretl_errmsg, _("Warning: no convergence after %d interations"),
		    max_iters);
	    loop->err = 1;
	}
    }

    return loop->err;
}

static int 
eval_numeric_condition (int op, double xl, double xr)
{
    int cont = 0;

    if (op == GT) {
	cont = (xl > xr);
    } else if (op == GTE) {
	cont = (xl >= xr);
    } else if (op == LTE) {
	cont = (xl <= xr);
    } else if (op == LT) {
	cont = (xl < xr);
    }

    return cont;
}

/**
 * loop_condition:
 * @k: in case of a simple count loop, the number of iterations so far.
 * @loop: pointer to loop commands struct.
 * @Z: data matrix.
 * @pdinfo: data information struct.
 *
 * Check whether a looping condition is still satisfied.
 *
 * Returns: 1 to indicate looping should continue, 0 to terminate.
 */

static int 
loop_condition (int k, LOOPSET *loop, double **Z, DATAINFO *pdinfo)
{
    int cont = 0;
    int oldtimes = loop->ntimes;

    /* simple count loop */
    if (loop->type == COUNT_LOOP) {
	if (k < loop->ntimes) {
	    cont = 1;
	}
    }

    /* a loop indexed by 'i' */
    else if (indexed_loop(loop)) {
	if (k > 0) {
	    loop->index += 1;
	}
	if (loop->index < loop->ntimes) {
	    cont = 1;
	}
    } 

    /* more complex forms */
    else {

	/* safeguard */
	if (loop_count_too_high(loop)) {
	    cont = 0;
	}
	
	/* inequality between variables */
	else if (loop->rvar > 0) {
	    if (loop->type == FOR_LOOP && k > 0) {
		Z[loop->lvar][0] += loop->incr;
	    }
	    cont = eval_numeric_condition(loop->ineq,
					  Z[loop->lvar][0],
					  Z[loop->rvar][0]);
	} 

	/* inequality between var and constant */
	else if (loop->lvar > 0) {
	    if (loop->type == FOR_LOOP && k > 0) {
		Z[loop->lvar][0] += loop->incr;
	    }
	    cont = eval_numeric_condition(loop->ineq,
					  Z[loop->lvar][0],
					  loop->rval);
	}

	if (cont) {
	    loop->ntimes += 1;
	}
    }

    if (!cont && oldtimes == 0) {
	strcpy(gretl_errmsg, _("Loop condition not satisfied at first round"));
	loop->err = 1;
	loop->ntimes = 0;
    }

    return cont;
}

/* ......................................................  */

static void gretl_loop_init (LOOPSET *loop)
{
#ifdef LOOP_DEBUG
    fprintf(stderr, "gretl_loop_init: initing loop at %p\n", (void *) loop);
#endif

    loop->flags = 0;
    loop->level = 0;

    loop->ntimes = 0;
    loop->index = 0;
    loop->initval = 0.0;
    loop->err = 0;
    loop->lvar = 0;
    loop->rvar = 0;
    loop->rval = 0.0;
    loop->incr = 0.0;
    loop->ineq = 0;

    loop->ncmds = 0;
    loop->nmod = 0;
    loop->nprn = 0;
    loop->nstore = 0;

    loop->next_model = 0;
    loop->next_print = 0;

    loop->lines = NULL;
    loop->ci = NULL;

    loop->eachstrs = NULL;

    loop->models = NULL;
    loop->lmodels = NULL;
    loop->prns = NULL;

    loop->storefile[0] = '\0';
    loop->storename = NULL;
    loop->storelbl = NULL;
    loop->storeval = NULL;

    loop->parent = NULL;
    loop->children = NULL;
    loop->n_children = 0;
}

static int prepare_loop_for_action (LOOPSET *loop)
{
#ifdef ENABLE_GMP
    mpf_set_default_prec(256);
#endif

    /* allocate some initial lines/commands for loop */
    loop->lines = malloc(LOOP_BLOCK * sizeof *loop->lines); 
    loop->ci = malloc(LOOP_BLOCK * sizeof *loop->ci);
    
    if (loop->lines == NULL || loop->ci == NULL) {
	return 1;
    }

    return 0;
}

void gretl_loop_destroy (LOOPSET *loop)
{
    int i;

    for (i=0; i<loop->n_children; i++) {
	gretl_loop_destroy(loop->children[i]);
    }

    if (loop->lines != NULL) {
	for (i=0; i<loop->ncmds; i++) {
	    free(loop->lines[i]);
	}
	free(loop->lines);
    }

    if (loop->ci != NULL) { 
	free(loop->ci);
    }

    if (loop->eachstrs != NULL) {
	for (i=0; i<loop->ntimes; i++) {
	    free(loop->eachstrs[i]);
	}
	free(loop->eachstrs);
    }    

    if (loop->models != NULL) {
	for (i=0; i<loop->nmod; i++) {
	    free_model(loop->models[i]);
	}
	free(loop->models);
    } 

    if (loop->lmodels != NULL) {
	for (i=0; i<loop->nmod; i++) {
	    free_loop_model(&loop->lmodels[i]);
	}
	free(loop->lmodels);
    }

    if (loop->prns != NULL) {
	for (i=0; i<loop->nprn; i++) { 
	    free_loop_print(&loop->prns[i]);
	}
	free(loop->prns);
    }

    if (loop->storename != NULL) {
	for (i=0; i<loop->nstore; i++) {
	    free(loop->storename[i]);
	}
	free(loop->storename);
    }

    if (loop->storelbl != NULL) {
	for (i=0; i<loop->nstore; i++) {
	    free(loop->storelbl[i]);
	}
	free(loop->storelbl);
    }

    if (loop->storeval != NULL) { 
	free(loop->storeval);
    }

    if (loop->children != NULL) {
	free(loop->children);
    }

    free(loop);
}

/**
 * loop_model_init:
 * @lmod: pointer to struct to initialize.
 * @pmod: model to take as basis.
 * @id: ID number to assign to @lmod.
 *
 * Initialize a #LOOP_MODEL struct, based on @pmod.
 *
 * Returns: 0 on successful completion, 1 on error.
 */

static int loop_model_init (LOOP_MODEL *lmod, const MODEL *pmod,
			    int id)
{
    int i, ncoeff = pmod->ncoeff;

    lmod->sum_coeff = malloc(ncoeff * sizeof *lmod->sum_coeff);
    if (lmod->sum_coeff == NULL) return 1;

    lmod->ssq_coeff = malloc(ncoeff * sizeof *lmod->ssq_coeff);
    if (lmod->ssq_coeff == NULL) goto cleanup;

    lmod->sum_sderr = malloc(ncoeff * sizeof *lmod->sum_sderr);
    if (lmod->sum_sderr == NULL) goto cleanup;

    lmod->ssq_sderr = malloc(ncoeff * sizeof *lmod->ssq_sderr);
    if (lmod->ssq_sderr == NULL) goto cleanup;

    lmod->list = copylist(pmod->list);
    if (lmod->list == NULL) goto cleanup;

    for (i=0; i<ncoeff; i++) {
#ifdef ENABLE_GMP
	mpf_init(lmod->sum_coeff[i]);
	mpf_init(lmod->ssq_coeff[i]);
	mpf_init(lmod->sum_sderr[i]);
	mpf_init(lmod->ssq_sderr[i]);
#else
	lmod->sum_coeff[i] = lmod->ssq_coeff[i] = 0.0;
	lmod->sum_sderr[i] = lmod->ssq_sderr[i] = 0.0;
#endif
    }

    lmod->ncoeff = ncoeff;
    lmod->t1 = pmod->t1;
    lmod->t2 = pmod->t2;
    lmod->nobs = pmod->nobs;
    lmod->ID = id;
    lmod->ci = pmod->ci;

    return 0;

 cleanup:
    free(lmod->ssq_coeff);
    free(lmod->sum_sderr);
    free(lmod->ssq_sderr);

    return 1;
}

/**
 * loop_print_init:
 * @lprn: pointer to struct to initialize.
 * @list: list of variables to be printed.
 * @id: ID number to assign to @lprn.
 *
 * Initialize a #LOOP_PRINT struct.
 *
 * Returns: 0 on successful completion, 1 on error.
 */

static int loop_print_init (LOOP_PRINT *lprn, const int *list, int id)
{
    int i;

    lprn->list = copylist(list);
    if (lprn->list == NULL) return 1;

    lprn->sum = malloc(list[0] * sizeof *lprn->sum);
    if (lprn->sum == NULL) goto cleanup;

    lprn->ssq = malloc(list[0] * sizeof *lprn->ssq);
    if (lprn->ssq == NULL) goto cleanup;

    for (i=0; i<list[0]; i++) { 
#ifdef ENABLE_GMP
	mpf_init(lprn->sum[i]);
	mpf_init(lprn->ssq[i]);
#else
	lprn->sum[i] = lprn->ssq[i] = 0.0;
#endif
    }

    lprn->ID = id;

    return 0;

 cleanup:
    free(lprn->list);
    free(lprn->sum);
    free(lprn->ssq);

    return 1;
}

/**
 * loop_store_init:
 * @loop: pointer to loop struct.
 * @fname: name of file in which to store data.
 * @list: list of variables to be stored (written to file).
 * @pdinfo: data information struct.
 *
 * Set up @loop for saving a set of variables.
 *
 * Returns: 0 on successful completion, 1 on error.
 */

static int loop_store_init (LOOPSET *loop, const char *fname, 
			    const int *list, DATAINFO *pdinfo)
{
    int i, tot = list[0] * loop->ntimes;

    loop->storefile[0] = '\0';
    strncat(loop->storefile, fname, MAXLEN - 1);

    loop->storename = malloc(list[0] * sizeof *loop->storename);
    if (loop->storename == NULL) return 1;

    loop->storelbl = malloc(list[0] * sizeof *loop->storelbl);
    if (loop->storelbl == NULL) goto cleanup;

    loop->storeval = malloc(tot * sizeof *loop->storeval);
    if (loop->storeval == NULL) goto cleanup;

    for (i=0; i<list[0]; i++) {
	char *p;

	loop->storename[i] = malloc(VNAMELEN);
	if (loop->storename[i] == NULL) goto cleanup;

	strcpy(loop->storename[i], pdinfo->varname[list[i+1]]);

	loop->storelbl[i] = malloc(MAXLABEL);
	if (loop->storelbl[i] == NULL) goto cleanup;

	strcpy(loop->storelbl[i], VARLABEL(pdinfo, list[i+1]));
	if ((p = strstr(loop->storelbl[i], "(scalar)"))) {
	    *p = 0;
	}
    }

    return 0;

 cleanup:
    free(loop->storename);
    free(loop->storelbl);
    free(loop->storeval);

    return 1;
}

static int add_loop_model_record (LOOPSET *loop, int cmdnum)
{
    MODEL **lmods;
    int err = 0;
    int nm = loop->nmod + 1;

    lmods = realloc(loop->models, nm * sizeof *lmods);
    if (lmods == NULL) {
	err = 1;
    } else {
	loop->models = lmods;
	loop->models[loop->nmod] = gretl_model_new();
	if (loop->models[loop->nmod] == NULL) {
	    err = 1;
	} else {
	    (loop->models[loop->nmod])->ID = cmdnum;
	}
    }

    if (!err) {
	loop->nmod += 1;
    }

    return err;
}

static int add_loop_model (LOOPSET *loop)
{
    int err = 0;
    int nm = loop->nmod + 1;

    loop->lmodels = realloc(loop->lmodels, nm * sizeof *loop->lmodels);
    if (loop->lmodels == NULL) {
	err = 1;
    } else {
	loop->nmod += 1;
    }

    return err;
}

/**
 * update_loop_model:
 * @loop: pointer to loop struct.
 * @cmdnum: sequential index number of the command within @loop.
 * @pmod: contains estimates from the current iteration.
 *
 * Update a #LOOP_MODEL belonging to @loop, based on the results
 * in @pmod.
 *
 * Returns: 0 on successful completion.
 */

static int update_loop_model (LOOPSET *loop, int cmdnum, MODEL *pmod)
{
    int j, i = get_modnum_by_cmdnum(loop, cmdnum);
    LOOP_MODEL *lmod;
#ifdef ENABLE_GMP
    mpf_t m;

    mpf_init(m);
#endif

    lmod = &loop->lmodels[i];

    for (j=0; j<pmod->ncoeff; j++) {
#ifdef ENABLE_GMP
	mpf_set_d(m, pmod->coeff[j]);
	mpf_add(lmod->sum_coeff[j], lmod->sum_coeff[j], m); 
	mpf_mul(m, m, m);
	mpf_add(lmod->ssq_coeff[j], lmod->ssq_coeff[j], m);

	mpf_set_d(m, pmod->sderr[j]);
	mpf_add(lmod->sum_sderr[j], lmod->sum_sderr[j], m);
	mpf_mul(m, m, m);
	mpf_add(lmod->ssq_sderr[j], lmod->ssq_sderr[j], m);
#else
	lmod->sum_coeff[j] += pmod->coeff[j];
	lmod->ssq_coeff[j] += pmod->coeff[j] * pmod->coeff[j];
	lmod->sum_sderr[j] += pmod->sderr[j];
	lmod->ssq_sderr[j] += pmod->sderr[j] * pmod->sderr[j];
#endif
    }

#ifdef ENABLE_GMP
    mpf_clear(m);
#endif

    return 0;
}

static int add_loop_print (LOOPSET *loop, const int *list, int cmdnum)
{
    LOOP_PRINT *prns;
    int np = loop->nprn + 1;
    int err = 0;

    prns = realloc(loop->prns, np * sizeof *prns);
    if (prns == NULL) {
	return 1;
    }

    loop->prns = prns;

    if (loop_print_init(&loop->prns[loop->nprn], list, cmdnum)) { 
	strcpy(gretl_errmsg, _("Failed to initalize print struct for loop\n"));
	err = 1;
    }

    if (!err) {
	loop->nprn += 1;
    }

    return err;
}

/**
 * update_loop_print:
 * @loop: pointer to loop struct.
 * @cmdnum: sequential index number of the command within @loop.
 * @list: list of variables to be printed.
 * @pZ: pointer to data matrix.
 * @pdinfo: pointer to data information struct.
 *
 * Update a #LOOP_PRINT belonging to @loop, based on the current
 * data values.
 *
 * Returns: 0 on successful completion.
 */


static int update_loop_print (LOOPSET *loop, int cmdnum, 
			      const int *list, double ***pZ, 
			      const DATAINFO *pdinfo)
{
    int j, t, i = get_prnnum_by_id(loop, cmdnum);
    LOOP_PRINT *lprn = &loop->prns[i];
#ifdef ENABLE_GMP
    mpf_t m;

    mpf_init(m);
#endif
    
    for (j=1; j<=list[0]; j++) {
	if (pdinfo->vector[list[j]]) t = pdinfo->t1;
	else t = 0;
#ifdef ENABLE_GMP
	mpf_set_d(m, (*pZ)[list[j]][t]); 
	mpf_add(lprn->sum[j-1], lprn->sum[j-1], m);
	mpf_mul(m, m, m);
	mpf_add(lprn->ssq[j-1], lprn->ssq[j-1], m);
#else
	lprn->sum[j-1] += (*pZ)[list[j]][t];
	lprn->ssq[j-1] += (*pZ)[list[j]][t] * (*pZ)[list[j]][t];
#endif
    }

#ifdef ENABLE_GMP
    mpf_clear(m);
#endif

    return 0;
}

/**
 * print_loop_results:
 * @loop: pointer to loop struct.
 * @pdinfo: data information struct.
 * @prn: gretl printing struct.
 *
 * Print out the results after completion of the loop @loop.
 *
 */

static void print_loop_results (LOOPSET *loop, const DATAINFO *pdinfo, 
				PRN *prn)
{
    char linecpy[MAXLINE];
    int i;

    if (loop->type != COUNT_LOOP) {
	pprintf(prn, _("\nNumber of iterations: %d\n\n"), loop->ntimes);
    }

    for (i=0; i<loop->ncmds; i++) {
#ifdef LOOP_DEBUG
	fprintf(stderr, "print_loop_results: loop command %d (i=%d): %s\n", 
		i+1, i, loop->lines[i]);
#endif
	if (!loop_is_progressive(loop) && loop->ci[i] == OLS) {
	    gretlopt opt;

	    strcpy(linecpy, loop->lines[i]);
	    opt = get_gretl_options(linecpy, NULL);
	    
	    if (opt & OPT_P) {
		/* deferred printing of model was requested */
		MODEL *pmod = loop->models[loop->next_model];

		set_model_id(pmod);
		printmodel(pmod, pdinfo, opt, prn);
		loop->next_model += 1;
	    }	    
	}

	if (loop_is_progressive(loop)) {
	    if (loop->ci[i] == OLS || loop->ci[i] == LAD ||
		loop->ci[i] == HSK || loop->ci[i] == HCCM || 
		loop->ci[i] == WLS) {
		print_loop_model(&loop->lmodels[loop->next_model], 
				 loop->ntimes, pdinfo, prn);
		loop->next_model += 1;
	    } else if (loop->ci[i] == PRINT) {
		print_loop_prn(&loop->prns[loop->next_print], 
			       loop->ntimes, pdinfo, prn);
		loop->next_print += 1;
	    } else if (loop->ci[i] == STORE) {
		print_loop_store(loop, prn);
	    }
	}
    }
}

/* ......................................................  */

static void free_loop_model (LOOP_MODEL *lmod)
{
#ifdef ENABLE_GMP
    int i;

    for (i=0; i<lmod->ncoeff; i++) {
	mpf_clear(lmod->sum_coeff[i]);
	mpf_clear(lmod->sum_sderr[i]);
	mpf_clear(lmod->ssq_coeff[i]);
	mpf_clear(lmod->ssq_sderr[i]);
    }
#endif

    free(lmod->sum_coeff);
    free(lmod->sum_sderr);
    free(lmod->ssq_coeff);
    free(lmod->ssq_sderr);
    free(lmod->list);
}

/* ......................................................  */

static void free_loop_print (LOOP_PRINT *lprn)
{
#ifdef ENABLE_GMP
    int i;

    for (i=0; i<lprn->list[0]; i++) {
	mpf_clear(lprn->sum[i]);
	mpf_clear(lprn->ssq[i]);
    }
#endif

    free(lprn->sum);
    free(lprn->ssq);
    free(lprn->list);    
}

static int add_more_loop_lines (LOOPSET *loop)
{
    int nb = 1 + (loop->ncmds + 1) / LOOP_BLOCK;
    char **lines;
    int *ci;
    
    lines = realloc(loop->lines, (nb * LOOP_BLOCK) * sizeof *lines); 
    ci = realloc(loop->ci, (nb * LOOP_BLOCK) * sizeof *ci);
    
    if (lines == NULL || ci == NULL) {
	return 1;
    }

    loop->lines = lines;
    loop->ci = ci;

    return 0;
}    

/**
 * add_to_loop:
 * @line: command line.
 * @ci: command index number.
 * @oflags: option flag(s) associated with the command.
 * @pdinfo: dataset information.
 * @pZ: pointer to data matrix.
 * @loopstack: pointer to integer stacking level.
 * @pointer to integer switch for running loop.
 *
 * Add line and command index to accumulated loop buffer.
 *
 * Returns: pointer to loop struct on success, %NULL on failure.
 */

LOOPSET *add_to_loop (char *line, int ci, gretlopt opt,
		      DATAINFO *pdinfo, double ***pZ,
		      LOOPSET *loop, int *loopstack, int *looprun)
{
    LOOPSET *lret = loop;

    *gretl_errmsg = '\0';

#ifdef LOOP_DEBUG
    fprintf(stderr, "add_to_loop: loop = %p, loopstack = %d, line = '%s'\n", 
	    (void *) loop, *loopstack, line);
#endif

    if (ci == LOOP) {
	/* starting a new loop, possibly inside another */
	lret = parse_loopline(line, loop, *loopstack, pdinfo, pZ);
	if (lret == NULL) {
	    if (*gretl_errmsg == '\0') {
		gretl_errmsg_set(_("No valid loop condition was given."));
	    }
	} else {
	    if (opt & OPT_P) {
		loop_set_progressive(lret);
	    }
	    if (opt & OPT_V) {
		loop_set_verbose(lret);
	    }
	    *loopstack += 1;
	}
    } else if (ci == ENDLOOP) {
	*loopstack -= 1;
	if (*loopstack == 0) {
	    *looprun = 1;
	} else {
	    lret = loop->parent;
	}
    } 

    if (loop != NULL) {
	int nc = loop->ncmds;

	if ((nc + 1) % LOOP_BLOCK == 0) {
	    if (add_more_loop_lines(loop)) {
		gretl_errmsg_set(_("Out of memory!"));
		goto bailout;
	    }
	}

	loop->lines[nc] = malloc(MAXLEN);
	if (loop->lines[nc] == NULL) {
	    gretl_errmsg_set(_("Out of memory!"));
	    goto bailout;
	}

	top_n_tail(line);

	if (ci == PRINT && loop->type != COUNT_LOOP) {
	    /* fixme: what's going on here? */
	    loop->ci[nc] = 0;
	} else {
	    loop->ci[nc] = ci;
	}

	loop->lines[nc][0] = '\0';

	if (opt) {
	    const char *flagstr = print_flags(opt, ci);

	    if (strlen(line) + strlen(flagstr) >= MAXLEN) {
		goto bailout;
	    } else {
		sprintf(loop->lines[nc], "%s%s", line, flagstr);
	    }
	} else {
	    strcpy(loop->lines[nc], line);
	}

	loop->ncmds += 1;

#ifdef LOOP_DEBUG
	fprintf(stderr, "loop: ncmds=%d, line[%d] = '%s'\n",
		loop->ncmds, nc, loop->lines[nc]);
#endif
    }

    return lret;

 bailout:
    
    if (loop != NULL) {
	gretl_loop_destroy(loop);
    }

    return NULL;
}

/* ......................................................... */ 

static void print_loop_model (LOOP_MODEL *lmod, int loopnum,
			      const DATAINFO *pdinfo, PRN *prn)
{
    int i;
    char startdate[OBSLEN], enddate[OBSLEN];

    ntodate(startdate, lmod->t1, pdinfo);
    ntodate(enddate, lmod->t2, pdinfo);

    pputc(prn, '\n');
    pprintf(prn, _("%s estimates using the %d observations %s-%s\n"),
	    _(estimator_string(lmod->ci, prn->format)), lmod->t2 - lmod->t1 + 1, 
	    startdate, enddate);
    pprintf(prn, _("Statistics for %d repetitions\n"), loopnum); 
    pprintf(prn, _("Dependent variable: %s\n\n"), 
	    pdinfo->varname[lmod->list[1]]);

    pputs(prn, _("                     mean of      std. dev. of     mean of"
		 "     std. dev. of\n"
		 "                    estimated      estimated"
		 "      estimated      estimated\n"
		 "      Variable     coefficients   coefficients   std. errors"
		 "    std. errors\n\n"));

    for (i=0; i<lmod->ncoeff; i++) {
	print_loop_coeff(pdinfo, lmod, i, loopnum, prn);
    }
    pputc(prn, '\n');
}

/* ......................................................... */ 

static void print_loop_coeff (const DATAINFO *pdinfo, 
			      const LOOP_MODEL *lmod, 
			      int c, int n, PRN *prn)
{
#ifdef ENABLE_GMP
    mpf_t c1, c2, m, sd1, sd2;
    unsigned long ln = n;

    mpf_init(c1);
    mpf_init(c2);
    mpf_init(m);
    mpf_init(sd1);
    mpf_init(sd2);

    mpf_div_ui(c1, lmod->sum_coeff[c], ln);
    mpf_mul(m, c1, c1);
    mpf_mul_ui(m, m, ln);
    mpf_sub(m, lmod->ssq_coeff[c], m);
    mpf_div_ui(sd1, m, ln);
    if (mpf_cmp_d(sd1, 0.0) > 0) {
	mpf_sqrt(sd1, sd1);
    } else {
	mpf_set_d(sd1, 0.0);
    }

    mpf_div_ui(c2, lmod->sum_sderr[c], ln);
    mpf_mul(m, c2, c2);
    mpf_mul_ui(m, m, ln);
    mpf_sub(m, lmod->ssq_sderr[c], m);
    mpf_div_ui(sd2, m, ln);
    if (mpf_cmp_d(sd2, 0.0) > 0) {
	mpf_sqrt(sd2, sd2);
    } else {
	mpf_set_d(sd2, 0.0);
    }

    pprintf(prn, " %3d) %8s ", lmod->list[c+2], 
	   pdinfo->varname[lmod->list[c+2]]);

    pprintf(prn, "%#14g %#14g %#14g %#14g\n", mpf_get_d(c1), mpf_get_d(sd1), 
	    mpf_get_d(c2), mpf_get_d(sd2));

    mpf_clear(c1);
    mpf_clear(c2);
    mpf_clear(m);
    mpf_clear(sd1);
    mpf_clear(sd2);
#else /* non-GMP */
    bigval m1, m2, var1, var2, sd1, sd2;
    
    m1 = lmod->sum_coeff[c] / n;
    var1 = (lmod->ssq_coeff[c] - n * m1 * m1) / n;
    sd1 = (var1 <= 0.0)? 0.0 : sqrt((double) var1);

    m2 = lmod->sum_sderr[c] / n;
    var2 = (lmod->ssq_sderr[c] - n * m2 * m2) / n;
    sd2 = (var2 <= 0.0)? 0 : sqrt((double) var2);

    pprintf(prn, " %3d) %8s ", lmod->list[c+2], 
	   pdinfo->varname[lmod->list[c+2]]);

    pprintf(prn, "%#14g %#14g %#14g %#14g\n", (double) m1, (double) sd1, 
	    (double) m2, (double) sd2);
#endif
}

/* ......................................................... */ 

static void print_loop_prn (LOOP_PRINT *lprn, int n,
			    const DATAINFO *pdinfo, PRN *prn)
{
    int i;
    bigval mean, m, sd;

    if (lprn == NULL) return;

    pputs(prn, _("   Variable     mean         std. dev.\n"));

#ifdef ENABLE_GMP
    mpf_init(mean);
    mpf_init(m);
    mpf_init(sd);
    
    for (i=1; i<=lprn->list[0]; i++) {
	mpf_div_ui(mean, lprn->sum[i-1], (unsigned long) n);
	mpf_mul(m, mean, mean);
	mpf_mul_ui(m, m, (unsigned long) n);
	mpf_sub(sd, lprn->ssq[i-1], m);
	mpf_div_ui(sd, sd, (unsigned long) n);
	if (mpf_cmp_d(sd, 0.0) > 0) {
	    mpf_sqrt(sd, sd);
	} else {
	    mpf_set_d(sd, 0.0);
	}
	pprintf(prn, " %8s ", pdinfo->varname[lprn->list[i]]);
	pprintf(prn, "%#14g %#14g\n", mpf_get_d(mean), mpf_get_d(sd));
    }

    mpf_clear(mean);
    mpf_clear(m);
    mpf_clear(sd);
#else
    for (i=1; i<=lprn->list[0]; i++) {
	mean = lprn->sum[i-1] / n;
	m = (lprn->ssq[i-1] - n * mean * mean) / n;
	sd = (m < 0)? 0 : sqrt((double) m);
	pprintf(prn, " %8s ", pdinfo->varname[lprn->list[i]]);
	pprintf(prn, "%#14g %#14g\n", (double) mean, (double) sd);
    }
#endif
    pputc(prn, '\n');
}

/* ......................................................... */ 

static int print_loop_store (LOOPSET *loop, PRN *prn)
{
    int i, t;
    FILE *fp;
    char gdtfile[MAXLEN], infobuf[1024];
    char *xmlbuf = NULL;
    time_t writetime;

    /* organize filename */
    if (loop->storefile[0] == '\0') {
	sprintf(gdtfile, "%sloopdata.gdt", gretl_user_dir());	
    } else {
	strcpy(gdtfile, loop->storefile);
    }

    if (strchr(gdtfile, '.') == NULL) {
	strcat(gdtfile, ".gdt");
    }

    fp = fopen(gdtfile, "w");
    if (fp == NULL) return 1;

    writetime = time(NULL);

    pprintf(prn, _("printing %d values of variables to %s\n"), 
	    loop->ntimes, gdtfile);

    fprintf(fp, "<?xml version=\"1.0\"?>\n"
	    "<!DOCTYPE gretldata SYSTEM \"gretldata.dtd\">\n\n"
	    "<gretldata name=\"%s\" frequency=\"1\" "
	    "startobs=\"1\" endobs=\"%d\" ", 
	    gdtfile, loop->ntimes);

    fprintf(fp, "type=\"cross-section\">\n");

    sprintf(infobuf, "%s %s", _("simulation data written"),
	    print_time(&writetime)); 
    xmlbuf = gretl_xml_encode(infobuf);
    fprintf(fp, "<description>\n%s\n</description>\n", xmlbuf);
    free(xmlbuf);

#ifdef ENABLE_NLS
    setlocale(LC_NUMERIC, "C");
#endif

    /* print info on variables */
    fprintf(fp, "<variables count=\"%d\">\n", loop->nstore);

    for (i=0; i<loop->nstore; i++) {
	xmlbuf = gretl_xml_encode(loop->storename[i]);
	fprintf(fp, "<variable name=\"%s\"", xmlbuf);
	free(xmlbuf);
	xmlbuf = gretl_xml_encode(loop->storelbl[i]);
	fprintf(fp, "\n label=\"%s\"/>\n", xmlbuf);
	free(xmlbuf);
    }

    fputs("</variables>\n", fp);

    /* print actual data */
    fprintf(fp, "<observations count=\"%d\" labels=\"false\">\n",
	    loop->ntimes);

    for (t=0; t<loop->ntimes; t++) {
	double x;

	fputs("<obs>", fp);
	for (i=0; i<loop->nstore; i++) {
	    x = loop->storeval[loop->ntimes * i + t];
	    if (na(x)) {
		fputs("NA ", fp);
	    } else {
		fprintf(fp, "%g ", x);
	    }
	}
	fputs("</obs>\n", fp);
    }

    fprintf(fp, "</observations>\n</gretldata>\n");

#ifdef ENABLE_NLS
    setlocale(LC_NUMERIC, "");
#endif

    fclose(fp);

    return 0;
}

/* ......................................................... */ 

static int get_prnnum_by_id (LOOPSET *loop, int id)
{
    int i;

    for (i=0; i<loop->nprn; i++) {
	if (loop->prns[i].ID == id) return i;
    }
    return -1;
}

/**
 * get_modnum_by_cmdnum:
 * @loop: pointer to loop struct.
 * @cmdnum: sequential index of command within @loop.
 *
 * Determine the ID number of a model within a "while" loop construct.
 *
 * Returns: model ID number, or -1 in case of no match.
 */

static int get_modnum_by_cmdnum (LOOPSET *loop, int cmdnum)
{
    int i;

    if (loop_is_progressive(loop)) {
	for (i=0; i<loop->nmod; i++) {
	    if (loop->lmodels[i].ID == cmdnum) {
		return i;
	    }
	}
    } else {
	for (i=0; i<loop->nmod; i++) {
	    if ((loop->models[i])->ID == cmdnum) {
		return i;
	    }
	}
    }

    return -1;
}

/**
 * get_cmd_ci:
 * @line: command line.
 * @command: pointer to gretl command struct.
 *
 * Parse @line and assign to @command->ci the index number of
 * the command embedded in @line.
 */

void get_cmd_ci (const char *line, CMD *command)
{
    /* allow for leading spaces */
    while (isspace(*line)) line++;

    if (*line == '#') {
	command->nolist = 1;
	command->ci = CMD_COMMENT;
	return;
    }

    if (sscanf(line, "%s", command->cmd) != 1 || 
	*line == '(' || *line == '#') {
	command->nolist = 1;
	command->ci = -1;
	return;
    }

    if ((command->ci = gretl_command_number(command->cmd)) == 0) {
	command->errcode = 1;
	sprintf(gretl_errmsg, _("command \"%s\" not recognized"), 
		command->cmd);
	return;
    }

    if (!strcmp(line, "end loop")) {
	command->ci = ENDLOOP;
    }
} 

static int substitute_dollar_i (char *str, const LOOPSET *loop,
				const DATAINFO *pdinfo)
{
    char *p;
    int i = loop->initval + loop->index;
    int err = 0;

    if (!indexed_loop(loop)) {
	return 1;
    }

    while ((p = strstr(str, "$i")) != NULL) {
	char ins[OBSLEN];
	char *pins = ins;
	char *q;

	q = malloc(strlen(p));
	if (q == NULL) {
	    err = 1;
	    break;
	}
	strcpy(q, p + 2);
	if (loop->type == INDEX_LOOP) {
	    sprintf(ins, "%d", i);
	} else if (loop->type == DATED_LOOP) {
	    ntodate(ins, i, pdinfo);
	} else if (loop->type == EACH_LOOP) {
	    pins = loop->eachstrs[i];
	}
	strcpy(p, pins);
	strcpy(p + strlen(pins), q);
	free(q);	
    }

    return err;
}

static void 
loop_add_storevals (const int *list, LOOPSET *loop, int lround,
		    const double **Z, DATAINFO *pdinfo)
{
    int i, sv;

    for (i=0; i<list[0]; i++) {
	sv = i * loop->ntimes + lround;
	if (pdinfo->vector[list[i+1]]) { 
	    loop->storeval[sv] = Z[list[i+1]][pdinfo->t1 + 1];
	} else {
	    loop->storeval[sv] = Z[list[i+1]][0];
	}
    }
}

static void top_of_loop (LOOPSET *loop, double **Z)
{
    if (indexed_loop(loop)) {
	loop->index = 0;
    } else if (loop->type == FOR_LOOP) {
	Z[loop->lvar][0] = loop->initval;
    }
}

static void 
print_loop_progress (const LOOPSET *loop, const DATAINFO *pdinfo,
		     PRN *prn)
{
    int i = loop->initval + loop->index;

    if (loop->type == INDEX_LOOP) {
	pprintf(prn, "loop: i = %d\n\n", i);
    } else if (loop->type == DATED_LOOP) {
	char obs[OBSLEN];

	ntodate(obs, i, pdinfo);
	pprintf(prn, "loop: i = %s\n\n", obs);
    }
}

int loop_exec (LOOPSET *loop, char *line,
	       double ***pZ, DATAINFO **ppdinfo, 
	       MODEL **models, int *echo_off, 
	       PRN *prn)
{
    CMD cmd;
    MODEL *lastmod = models[0];
    char linecpy[MAXLINE];
    int m = 0, lround = 0, ignore = 0;
    int err = 0;

    if (loop == NULL) {
	pputs(prn, "Got a NULL loop\n");
	return 1;
    }

    if (loop->ncmds == 0) {
	pputs(prn, _("No commands in loop\n"));
	return 0;
    }

    err = gretl_cmd_init(&cmd);
    if (err) {
	return err;
    }

    gretl_set_text_pause(0);

#ifdef LOOP_DEBUG
    fprintf(stderr, "loop_exec: loop = %p\n", (void *) loop);
#endif

    top_of_loop(loop, *pZ);

    while (!err && loop_condition(lround, loop, *pZ, *ppdinfo)) {
	DATAINFO *pdinfo = *ppdinfo;
	int childnum = 0;
	int modnum = 0;
	int j;

#ifdef LOOP_DEBUG
	fprintf(stderr, "top of loop: lround = %d\n", lround);
#endif

	if (!(*echo_off) && indexed_loop(loop)) {
	    print_loop_progress(loop, *ppdinfo, prn);
	}

	for (j=0; !err && j<loop->ncmds; j++) {
#ifdef LOOP_DEBUG
	    fprintf(stderr, "loop->lines[%d] = '%s'\n", j, loop->lines[j]);
#endif
	    strcpy(linecpy, loop->lines[j]);

	    cmd.opt = get_gretl_options(linecpy, &err);
	    if (err) {
		break;
	    }

	    if (indexed_loop(loop)) {
		err = substitute_dollar_i(linecpy, loop, *ppdinfo);
		if (err) {
		    break;
		}
	    } 	

	    /* We already have the "ci" index recorded, but this line
	       will do some checking that hasn't been done earlier.
	    */

	    getcmd(linecpy, pdinfo, &cmd, &ignore, pZ, NULL);

	    if (cmd.ci < 0) {
		continue;
	    }

	    if (cmd.errcode) {
		err = cmd.errcode;
		break;
	    }

	    if (!(*echo_off) && indexed_loop(loop)) {
		if (cmd.ci == ENDLOOP) {
		    pputc(prn, '\n');
		} else {
		    echo_cmd(&cmd, pdinfo, linecpy, 0, 1, 0, prn);
		}
	    }

	    switch (cmd.ci) {

	    case SUMMARY:
	    case SIM:
	    case ADF:
	    case KPSS:
		err = simple_commands(&cmd, linecpy, pZ, pdinfo, prn);
		break;

	    case LOOP:
		err = loop_exec(loop->children[childnum++], NULL,
				pZ, ppdinfo, models, echo_off, prn);
		break;

	    case ENDLOOP:
		/* no-op */
		break;

	    case GENR:
		err = generate(pZ, pdinfo, linecpy, lastmod);
		if (loop_is_verbose(loop) && !err) { 
		    print_gretl_msg(prn);
		}
		break;

	    case OLS:
	    case WLS:
	    case LAD:
	    case HSK:
	    case HCCM:
	    case GARCH:
	    case ARMA:
		/* if this is the first time round, allocate space
		   for each loop model */
		if (lround == 0) {
		    if (loop_is_progressive(loop)) {
			err = add_loop_model(loop);
		    } else if (cmd.opt & OPT_P) {
			err = add_loop_model_record(loop, j);
		    }
		    if (err) {
			break;
		    }
		} 

		/* estimate the model called for */
		clear_model(models[0]);

		if (cmd.ci == OLS || cmd.ci == WLS) {
		    *models[0] = lsq(cmd.list, pZ, pdinfo, cmd.ci, cmd.opt, 0.0);
		} else if (cmd.ci == LAD) {
		    *models[0] = lad(cmd.list, pZ, pdinfo);
		} else if (cmd.ci == HSK) {
		    *models[0] = hsk_func(cmd.list, pZ, pdinfo);
		} else if (cmd.ci == HCCM) {
		    *models[0] = hccm_func(cmd.list, pZ, pdinfo);
		} else if (cmd.ci == GARCH) {
		    *models[0] = garch(cmd.list, pZ, pdinfo, cmd.opt, prn);
		}

		if ((err = (models[0])->errcode)) {
		    break;
		}

		if (loop_is_progressive(loop)) {
		    if (lround == 0 && loop_model_init(&loop->lmodels[loop->nmod - 1], 
						       models[0], j)) { 
			gretl_errmsg_set(_("Failed to initialize model for loop\n"));
			err = 1;
			break;
		    } else if (update_loop_model(loop, j, models[0])) {
			gretl_errmsg_set(_("Failed to add results to loop model\n"));
			err = 1;
			break;
		    }
		    lastmod = models[0];
		} else if (cmd.opt & OPT_P) {
		    /* deferred printing of model results */
		    m = get_modnum_by_cmdnum(loop, j);
		    swap_models(&models[0], &loop->models[m]);
		    (loop->models[m])->ID = j;
		    lastmod = loop->models[m];
		    model_count_minus();
		} else {
		    (models[0])->ID = ++modnum; /* FIXME? */
		    printmodel(models[0], pdinfo, cmd.opt, prn);
		    lastmod = models[0];
		}
		break;

	    case PRINT:
		if (cmd.param[0] != '\0') {
		    err = simple_commands(&cmd, linecpy, pZ, pdinfo, prn);
		} else if (loop_is_progressive(loop)) {
		    if (lround == 0) {
			if ((err = add_loop_print(loop, cmd.list, j))) {
			    break;
			}
		    }
		    if (update_loop_print(loop, j, cmd.list, pZ, pdinfo)) {
			gretl_errmsg_set(_("Failed to add values to print loop\n"));
			err = 1;
		    }
		} else {
		    err = printdata(cmd.list, (const double **) *pZ, pdinfo, 
				    cmd.opt, prn);
		}
		break;

	    case PRINTF:
		err = do_printf(linecpy, pZ, pdinfo, models[0], prn);
		break;

	    case SMPL:
		if (cmd.opt) {
		    err = restore_full_sample(pZ, ppdinfo, cmd.opt);
		    if (err) {
			errmsg(err, prn);
			break;
		    } else {
			err = restrict_sample(linecpy, pZ, ppdinfo, 
					      cmd.list, cmd.opt);
		    }
		} else if (!strcmp(linecpy, "smpl full") ||
			   !strcmp(linecpy, "smpl --full")) {
		    err = restore_full_sample(pZ, ppdinfo, OPT_C);
		} else { 
		    err = set_sample(linecpy, *ppdinfo);
		}

		if (err) {
		    errmsg(err, prn);
		} else if (1 || !(*echo_off)) {
		    print_smpl(*ppdinfo, get_full_length_n(), prn);
		}
		break;

	    case STORE:
		if (loop_is_progressive(loop)) {
		    if (lround == 0) {
			loop->nstore = cmd.list[0];
			if (loop_store_init(loop, cmd.param, cmd.list, pdinfo)) {
			    err = 1;
			}
		    }
		    if (!err) {
			loop_add_storevals(cmd.list, loop, lround,
					   (const double **) *pZ, pdinfo);
		    }
		} else {
		    simple_commands(&cmd, linecpy, pZ, pdinfo, prn);
		}
		break;

	    case PVALUE:
		batch_pvalue(linecpy, (const double **) *pZ, pdinfo, prn);
		break;

	    default: 
		/* not reachable (because loop commands were screened in advance) */
		pprintf(prn, _("command: '%s'\nThis is not available in a loop.\n"),
			linecpy);
		err = 1;
		break;

	    } /* end switch on specific command number */

	} /* end execution of commands within loop */

	if (get_halt_on_error() == 0) {
	    errmsg(err, prn);
	    err = 0;
	}

	lround++;

    } /* end iterations of loop */

    if (err) {
	errmsg(err, prn);
	pprintf(prn, ">> %s\n", linecpy);
    } else if (loop->err) {
	errmsg(loop->err, prn);
	err = loop->err;
    }

    if (!err && lround > 0) {
	print_loop_results(loop, *ppdinfo, prn); 
    }

    if (lastmod != models[0]) {
	/* to get genr commands that reference model statistics --
	   after the loop has finished -- to come out right
	*/
	swap_models(&models[0], &loop->models[m]);
    }

    gretl_cmd_free(&cmd);

    if (line != NULL) {
	*line = '\0';
    } 

    if (get_halt_on_error()) {
	return err;
    } else {
	return 0;
    }
}

/* if-then stuff - conditional execution */

int if_eval (const char *line, double ***pZ, DATAINFO *pdinfo)
{
    char formula[MAXLEN];
    int err, ret = -1;

#ifdef LOOP_DEBUG
    printf("if_eval: line = '%s'\n", line);
#endif

    /* + 2 below to omit "if" */
    sprintf(formula, "__iftest=%s", line + 2);
    err = generate(pZ, pdinfo, formula, NULL);
    if (!err) {
	int v = varindex(pdinfo, "iftest");
	
	if (v < pdinfo->v) {
	    ret = (*pZ)[v][0];
	    dataset_drop_vars(1, pZ, pdinfo);
	}
    }

    return ret;
}

#define IF_DEPTH 9

int ifstate (int code)
{
    static unsigned char T[IF_DEPTH];
    static unsigned char got_if[IF_DEPTH];
    static unsigned char got_else[IF_DEPTH];
    static unsigned char indent;

    if (code == RELAX) {
	indent = 0;
    }
    else if (code == SET_FALSE || code == SET_TRUE) {
	indent++;
	if (indent >= IF_DEPTH) {
	    fprintf(stderr, "if depth (%d) exceeded\n", IF_DEPTH);
	    return 1; /* too deeply nested */
	}
	T[indent] = (code == SET_TRUE);
	got_if[indent] = 1;
	got_else[indent] = 0;
    }
    else if (code == SET_ELSE) {
	if (got_else[indent] || !got_if[indent]) {
	    sprintf(gretl_errmsg, "Unmatched \"else\"");
	    return 1; 
	}
	T[indent] = !T[indent];
	got_else[indent] = 1;
    }
    else if (code == SET_ENDIF) {
	if (!got_if[indent] || indent == 0) {
	    sprintf(gretl_errmsg, "Unmatched \"endif\"");
	    return 1; 
	}
	got_if[indent] = 0;
	got_else[indent] = 0;
	indent--;
    }
    else if (code == IS_FALSE) {
	int i;

	for (i=1; i<=indent; i++) {
	    if (T[i] == 0) return 1;
	}
    }

    return 0;
}


