/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/***********************************************
 *  Enrico Fucile
 **********************************************/

#include "grib_api_internal.h"
#include <ctype.h>

/*
   This is used by make_class.pl

   START_CLASS_DEF
   CLASS      = accessor
   SUPER      = grib_accessor_class_codetable
   IMPLEMENTS = init;unpack_string
   IMPLEMENTS = unpack_long;unpack_double
   IMPLEMENTS = value_count;destroy;dump
   MEMBERS = const char* codetable
   MEMBERS = char* filename
   END_CLASS_DEF

 */

/* START_CLASS_IMP */

/*

Don't edit anything between START_CLASS_IMP and END_CLASS_IMP
Instead edit values between START_CLASS_DEF and END_CLASS_DEF
or edit "accessor.class" and rerun ./make_class.pl

*/

static int unpack_double(grib_accessor*, double* val,size_t *len);
static int unpack_long(grib_accessor*, long* val,size_t *len);
static int unpack_string (grib_accessor*, char*, size_t *len);
static long value_count(grib_accessor*);
static void destroy(grib_context*,grib_accessor*);
static void dump(grib_accessor*, grib_dumper*);
static void init(grib_accessor*,const long, grib_arguments* );
static void init_class(grib_accessor_class*);

typedef struct grib_accessor_codetable_units {
    grib_accessor          att;
/* Members defined in gen */
/* Members defined in long */
/* Members defined in unsigned */
	long nbytes;
	grib_arguments* arg;
/* Members defined in codetable */
	const char* tablename;
	const char* masterDir;
	const char* localDir;
/* Members defined in codetable_units */
	const char* codetable;
	char* filename;
} grib_accessor_codetable_units;

extern grib_accessor_class* grib_accessor_class_codetable;

static grib_accessor_class _grib_accessor_class_codetable_units = {
    &grib_accessor_class_codetable,                      /* super                     */
    "codetable_units",                      /* name                      */
    sizeof(grib_accessor_codetable_units),  /* size                      */
    0,                           /* inited */
    &init_class,                 /* init_class */
    &init,                       /* init                      */
    0,                  /* post_init                      */
    &destroy,                    /* free mem                       */
    &dump,                       /* describes himself         */
    0,                /* get length of section     */
    &value_count,                /* get number of values      */
    0,                 /* get number of bytes      */
    0,                /* get offset to bytes           */
    0,            /* get native type               */
    0,                /* get sub_section                */
    0,               /* grib_pack procedures long      */
    0,               /* grib_pack procedures long      */
    0,                  /* grib_pack procedures long      */
    &unpack_long,                /* grib_unpack procedures long    */
    0,                /* grib_pack procedures double    */
    &unpack_double,              /* grib_unpack procedures double  */
    0,                /* grib_pack procedures string    */
    &unpack_string,              /* grib_unpack procedures string  */
    0,                 /* grib_pack procedures bytes     */
    0,               /* grib_unpack procedures bytes   */
    0,            /* pack_expression */
    0,              /* notify_change   */
    0,                /* update_size   */
    0,            /* preferred_size   */
    0,                    /* resize   */
    0,      /* nearest_smaller_value */
    0,                       /* next accessor    */
    0,                    /* compare vs. another accessor   */
    0,             /* unpack only ith value          */
};


grib_accessor_class* grib_accessor_class_codetable_units = &_grib_accessor_class_codetable_units;


static void init_class(grib_accessor_class* c)
{
	c->next_offset	=	(*(c->super))->next_offset;
	c->byte_count	=	(*(c->super))->byte_count;
	c->byte_offset	=	(*(c->super))->byte_offset;
	c->get_native_type	=	(*(c->super))->get_native_type;
	c->sub_section	=	(*(c->super))->sub_section;
	c->pack_missing	=	(*(c->super))->pack_missing;
	c->is_missing	=	(*(c->super))->is_missing;
	c->pack_long	=	(*(c->super))->pack_long;
	c->pack_double	=	(*(c->super))->pack_double;
	c->pack_string	=	(*(c->super))->pack_string;
	c->pack_bytes	=	(*(c->super))->pack_bytes;
	c->unpack_bytes	=	(*(c->super))->unpack_bytes;
	c->pack_expression	=	(*(c->super))->pack_expression;
	c->notify_change	=	(*(c->super))->notify_change;
	c->update_size	=	(*(c->super))->update_size;
	c->preferred_size	=	(*(c->super))->preferred_size;
	c->resize	=	(*(c->super))->resize;
	c->nearest_smaller_value	=	(*(c->super))->nearest_smaller_value;
	c->next	=	(*(c->super))->next;
	c->compare	=	(*(c->super))->compare;
	c->unpack_double_element	=	(*(c->super))->unpack_double_element;
}
/* END_CLASS_IMP */
/*
static grib_codetable* load_table(grib_accessor_codetable_units* self) {
  size_t size = 0;
  grib_handle*    h = ((grib_accessor*)self)->parent->h;
  grib_context*   c = h->context;
  grib_codetable* t = c->codetable;
  grib_accessor* codetable=0;
  char line[1024];
  char filename[1024]={0,};

  if (!self->filename) {
    grib_recompose_name(h, NULL,self->tablename, line,1);
    grib_context_full_path(c,line,filename);

    self->filename=grib_context_strdup( c,filename);
  }

  while(t)
  {
    if(strcmp(self->filename,t->filename[0]) == 0)
      return t;
    t = t->next;
  }

  codetable=grib_find_accessor( h,self->codetable);

  size = grib_byte_count(codetable) * 8;
  size = grib_power(size,2);

  return grib_load_codetable(c,self->filename,line,size);

}
*/

static grib_codetable* load_table(grib_accessor_codetable_units* self)
{
  size_t size = 0;
  grib_handle*    h = ((grib_accessor*)self)->parent->h;
  grib_context*   c = h->context;
  grib_codetable* t = c->codetable;
  grib_accessor* a=(grib_accessor*)self;
  char filename[1024]={0,};
  char recomposed[1024]={0,};
  char localRecomposed[1024]={0,};
  char localFilename[1024]={0,};
  char masterDir[1024]={0,};
  char localDir[1024]={0,};
  size_t len=1024;

  if (self->masterDir != NULL)
    grib_get_string(h,self->masterDir,masterDir,&len);

  len=1024;
  if (self->localDir != NULL)
    grib_get_string(h,self->localDir,localDir,&len);

  if (*masterDir!=0) {
    sprintf(filename,"%s/%s",masterDir,self->tablename);
    grib_recompose_name(h, NULL,filename, recomposed,1);
    grib_context_full_path(c,recomposed,filename);
  } else {
    grib_recompose_name(h, NULL,self->tablename, recomposed,1);
    grib_context_full_path(c,recomposed,filename);
  }

  if (*localDir!=0) {
    sprintf(localFilename,"%s/%s",localDir,self->tablename);
    grib_recompose_name(h, NULL,localFilename, localRecomposed,1);
    grib_context_full_path(c,localRecomposed,localFilename);
  }
  
  while(t) {
    if(strcmp(filename,t->filename[0]) == 0 &&
       ((*localFilename==0 && t->filename[1]==NULL)
           || strcmp(localFilename,t->filename[1]) ==0) )
      return t;
    t = t->next;
  }

  if (a->flags & GRIB_ACCESSOR_FLAG_TRANSIENT) {
    Assert(a->vvalue!=NULL);
    size=a->vvalue->length*8;
  } else {
    size = grib_byte_count((grib_accessor*)self) * 8;
  }
  size = grib_power(size,2);

  t = (grib_codetable*)grib_context_malloc_clear_persistent(c,sizeof(grib_codetable) +
      (size-1)*sizeof(code_table_entry));

  if (*filename!=0) grib_load_codetable(c,filename,recomposed,size,t);

  if (*localFilename!=0) grib_load_codetable(c,localFilename,localRecomposed,size,t);

  if (t->filename[0]==NULL && t->filename[1]==NULL)
    grib_context_free_persistent(c,t);
  
  return t;

}


static void init(grib_accessor* a, const long len, grib_arguments* params)
{
  grib_accessor_codetable_units* self = (grib_accessor_codetable_units*)a;


  self->codetable = grib_arguments_get_name(a->parent->h,params,1);

  a->length = len;
  a->flags  |= GRIB_ACCESSOR_FLAG_READ_ONLY;
  Assert(a->length>=0);
}

static int unpack_string (grib_accessor* a, char* buffer, size_t *len)
{
  grib_accessor_codetable_units* self = (grib_accessor_codetable_units*)a;
  grib_codetable*          table = NULL;

  long   value;
  int err = GRIB_SUCCESS;
  char tmp[1024];
  size_t l = 0;

  if( (err = grib_get_long(a->parent->h,self->codetable,&value)) != GRIB_SUCCESS)
    return err;

  table = load_table(self);


  if(table && (value >= 0) && (value < table->size) && table->entries[value].units)
  {
    strcpy(tmp,table->entries[value].units);
  }
  else
  {

#if 1
    sprintf(tmp,"%d",(int)value);
#else
    return GRIB_DECODING_ERROR;
#endif
  }


  l = strlen(tmp) + 1;

  if(*len < l)
  {
    *len = l;
    return GRIB_BUFFER_TOO_SMALL;
  }

  strcpy(buffer,tmp);
  *len = l;

  return GRIB_SUCCESS;
}

static long value_count(grib_accessor* a)
{
  return 1;
}


static void destroy(grib_context* c,grib_accessor* a) {
  grib_accessor_codetable_units* self = (grib_accessor_codetable_units*)a;
  grib_context_free(c,self->filename);
}


static int unpack_double(grib_accessor* a, double* val,size_t *len){
  int ret = GRIB_SUCCESS;
  long value=0;
  grib_accessor_codetable_units* self = (grib_accessor_codetable_units*)a;

  *len=1;

  ret=grib_get_long_internal(a->parent->h,self->codetable,&value);

  *val=value;

  return ret;
}

static int unpack_long(grib_accessor* a, long* val,size_t *len){
  int ret = GRIB_SUCCESS;
  long value=0;
  grib_accessor_codetable_units* self = (grib_accessor_codetable_units*)a;

  *len=1;

  ret=grib_get_long_internal(a->parent->h,self->codetable,&value);

  *val=value;

  return ret;
}

static void dump(grib_accessor* a, grib_dumper* dumper) {
  grib_accessor_codetable_units* self  = (grib_accessor_codetable_units*)a;
  grib_codetable*          table = load_table(self);
  char comment[2048];

  size_t llen = 1;
  long value;


  grib_unpack_long(a, &value,&llen);

  if(value == GRIB_MISSING_LONG)
  {
    if(a->length < 4)
    {
      value = (1L << a->length) - 1;
    }
  }

  if(table && value >= 0 && value < table->size)  {
    if(table->entries[value].units)
     sprintf(comment,"%s", table->entries[value].units);
    else
     strcpy(comment,"------Unknown code table entry");
  }
  else
  {
    strcpy(comment,"Unknown code table entry");
  }


  strcat(comment," (");
  if (table) {
    strcat(comment,table->recomposed_name[0]);
    if (table->recomposed_name[1]!=NULL) {
      strcat(comment," , ");
      strcat(comment,table->recomposed_name[1]);
    }
  }
  strcat(comment,") ");

  grib_dump_long(dumper,a,comment);

}


