/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

#include "grib_api_internal.h"

/*
   This is used by make_class.pl

   START_CLASS_DEF
   CLASS      = accessor
   SUPER      = grib_accessor_class_values
   IMPLEMENTS = init
   IMPLEMENTS = unpack_double
   IMPLEMENTS = pack_double
   IMPLEMENTS = value_count
   MEMBERS=const char*   number_of_values
   MEMBERS=const char*   reference_value
   MEMBERS=const char*   binary_scale_factor
   MEMBERS=const char*   decimal_scale_factor
   MEMBERS=const char*   bits_per_value
   MEMBERS=const char*   type_of_compression_used
   MEMBERS=const char*   target_compression_ratio
   MEMBERS=const char*   ni
   MEMBERS=const char*   nj
   MEMBERS=const char*   list_defining_points
   MEMBERS=const char*   number_of_data_points
   MEMBERS=const char*   scanning_mode
   END_CLASS_DEF

 */

/* START_CLASS_IMP */

/*

Don't edit anything between START_CLASS_IMP and END_CLASS_IMP
Instead edit values between START_CLASS_DEF and END_CLASS_DEF
or edit "accessor.class" and rerun ./make_class.pl

*/

static int pack_double(grib_accessor*, const double* val,size_t *len);
static int unpack_double(grib_accessor*, double* val,size_t *len);
static long value_count(grib_accessor*);
static void init(grib_accessor*,const long, grib_arguments* );
static void init_class(grib_accessor_class*);

typedef struct grib_accessor_data_jpeg2000_packing {
    grib_accessor          att;
/* Members defined in gen */
/* Members defined in values */
	int  carg;
	const char* seclen;
	const char* offsetdata;
	const char* offsetsection;
	int dirty;
/* Members defined in data_jpeg2000_packing */
	const char*   number_of_values;
	const char*   reference_value;
	const char*   binary_scale_factor;
	const char*   decimal_scale_factor;
	const char*   bits_per_value;
	const char*   type_of_compression_used;
	const char*   target_compression_ratio;
	const char*   ni;
	const char*   nj;
	const char*   list_defining_points;
	const char*   number_of_data_points;
	const char*   scanning_mode;
} grib_accessor_data_jpeg2000_packing;

extern grib_accessor_class* grib_accessor_class_values;

static grib_accessor_class _grib_accessor_class_data_jpeg2000_packing = {
    &grib_accessor_class_values,                      /* super                     */
    "data_jpeg2000_packing",                      /* name                      */
    sizeof(grib_accessor_data_jpeg2000_packing),  /* size                      */
    0,                           /* inited */
    &init_class,                 /* init_class */
    &init,                       /* init                      */
    0,                  /* post_init                      */
    0,                    /* free mem                       */
    0,                       /* describes himself         */
    0,                /* get length of section     */
    &value_count,                /* get number of values      */
    0,                 /* get number of bytes      */
    0,                /* get offset to bytes           */
    0,            /* get native type               */
    0,                /* get sub_section                */
    0,               /* grib_pack procedures long      */
    0,               /* grib_pack procedures long      */
    0,                  /* grib_pack procedures long      */
    0,                /* grib_unpack procedures long    */
    &pack_double,                /* grib_pack procedures double    */
    &unpack_double,              /* grib_unpack procedures double  */
    0,                /* grib_pack procedures string    */
    0,              /* grib_unpack procedures string  */
    0,                 /* grib_pack procedures bytes     */
    0,               /* grib_unpack procedures bytes   */
    0,            /* pack_expression */
    0,              /* notify_change   */
    0,                /* update_size   */
    0,            /* preferred_size   */
    0,                    /* resize   */
    0,      /* nearest_smaller_value */
    0,                       /* next accessor    */
    0,                    /* compare vs. another accessor   */
    0,             /* unpack only ith value          */
};


grib_accessor_class* grib_accessor_class_data_jpeg2000_packing = &_grib_accessor_class_data_jpeg2000_packing;


static void init_class(grib_accessor_class* c)
{
	c->dump	=	(*(c->super))->dump;
	c->next_offset	=	(*(c->super))->next_offset;
	c->byte_count	=	(*(c->super))->byte_count;
	c->byte_offset	=	(*(c->super))->byte_offset;
	c->get_native_type	=	(*(c->super))->get_native_type;
	c->sub_section	=	(*(c->super))->sub_section;
	c->pack_missing	=	(*(c->super))->pack_missing;
	c->is_missing	=	(*(c->super))->is_missing;
	c->pack_long	=	(*(c->super))->pack_long;
	c->unpack_long	=	(*(c->super))->unpack_long;
	c->pack_string	=	(*(c->super))->pack_string;
	c->unpack_string	=	(*(c->super))->unpack_string;
	c->pack_bytes	=	(*(c->super))->pack_bytes;
	c->unpack_bytes	=	(*(c->super))->unpack_bytes;
	c->pack_expression	=	(*(c->super))->pack_expression;
	c->notify_change	=	(*(c->super))->notify_change;
	c->update_size	=	(*(c->super))->update_size;
	c->preferred_size	=	(*(c->super))->preferred_size;
	c->resize	=	(*(c->super))->resize;
	c->nearest_smaller_value	=	(*(c->super))->nearest_smaller_value;
	c->next	=	(*(c->super))->next;
	c->compare	=	(*(c->super))->compare;
	c->unpack_double_element	=	(*(c->super))->unpack_double_element;
}
/* END_CLASS_IMP */

static void init(grib_accessor* a,const long v, grib_arguments* args)
{
  grib_accessor_data_jpeg2000_packing *self =(grib_accessor_data_jpeg2000_packing*)a;

  self->number_of_values       = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->reference_value        = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->binary_scale_factor    = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->decimal_scale_factor   = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->bits_per_value           = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->type_of_compression_used  = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->target_compression_ratio = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->ni                     = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->nj                     = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->list_defining_points   = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->number_of_data_points  = grib_arguments_get_name(a->parent->h,args,self->carg++);
  self->scanning_mode      = grib_arguments_get_name(a->parent->h,args,self->carg++);
  a->flags |= GRIB_ACCESSOR_FLAG_DATA;

  /* printf("JPEG length=%ld\n",a->length); */
}

static long value_count(grib_accessor* a)
{
  grib_accessor_data_jpeg2000_packing *self =(grib_accessor_data_jpeg2000_packing*)a;
  long n_vals= 0;

  if(grib_get_long_internal(a->parent->h,self->number_of_values,&n_vals) != GRIB_SUCCESS)
    return 0;

  return n_vals;
}

#define EXTRA_BUFFER_SIZE 10240

#if HAVE_LIBJASPER
#include "jasper/jasper.h"
static int  unpack_double(grib_accessor* a, double* val, size_t *len)
{
  grib_accessor_data_jpeg2000_packing *self =(grib_accessor_data_jpeg2000_packing*)a;

  jas_image_t  *image = NULL;
  jas_stream_t *jpeg  = NULL;
  int code = GRIB_SUCCESS;
  jas_matrix_t *matrix = NULL;
  int i,j,k;
  jas_image_cmpt_t *p;
  size_t buflen = grib_byte_count(a);

  double bscale = 0;
  double dscale = 0;
  unsigned char* buf = NULL;
  size_t n_vals = 0;

  long binary_scale_factor = 0;
  long decimal_scale_factor = 0;
  double reference_value = 0;
  long bits_per_value =0;

  n_vals = grib_value_count(a);

  if((code = grib_get_long_internal(a->parent->h,self->bits_per_value,&bits_per_value)) != GRIB_SUCCESS)
    return code;
  if((code = grib_get_double_internal(a->parent->h,self->reference_value, &reference_value)) != GRIB_SUCCESS)
    return code;
  if((code = grib_get_long_internal(a->parent->h,self->binary_scale_factor, &binary_scale_factor)) != GRIB_SUCCESS)
    return code;
  if((code = grib_get_long_internal(a->parent->h,self->decimal_scale_factor, &decimal_scale_factor)) != GRIB_SUCCESS)
    return code;

  self->dirty=0;

  bscale = grib_power(binary_scale_factor,2);
  dscale = grib_power(-decimal_scale_factor,10);

  /*jas_setdbglevel(99999);*/

  /* TODO: This should be called upstream */
  if(*len < n_vals)
    return GRIB_ARRAY_TOO_SMALL;

  /* Special case */

  if(bits_per_value == 0)
  {
    for(i = 0; i < n_vals; i++)
      val[i] = reference_value;
    *len = n_vals;
    return GRIB_SUCCESS;
  }

  buf = (unsigned char*)a->parent->h->buffer->data;
  buf += grib_byte_offset(a);

  jpeg = jas_stream_memopen((void*)buf,buflen);
  if(!jpeg)
  {
    code = GRIB_DECODING_ERROR;
    goto cleanup;
  }

  image = jpc_decode(jpeg,NULL);
  if(!image) {
    code = GRIB_DECODING_ERROR;
    goto cleanup;
  }
  p = image->cmpts_[0];

  if (image->numcmpts_ != 1 ) {
    /* Image not gray scale */
    code = GRIB_DECODING_ERROR;
    goto cleanup;
  }

  matrix = jas_matrix_create(jas_image_height(image), jas_image_width(image));
  if(!matrix)
  {
    code = GRIB_DECODING_ERROR;
    goto cleanup;
  }

  jas_image_readcmpt(image,0,0,0,jas_image_width(image), jas_image_height(image),matrix);

  Assert(p->height_ * p->width_ == n_vals);

  /*  printf("JPEG w=%d h=%d\n",p->height_ , p->width_);*/

  /*   printf("### %g %g\n",bscale,dscale);*/
  k=0;
  for (i=0;i<p->height_;i++)
    for (j=0;j<p->width_;j++)
    {
      /*  if(k>10000 && k<10010) printf("jpeg raw decode %04x\n",matrix->rows_[i][j]); */
      val[k++] = (matrix->rows_[i][j] * bscale + reference_value) * dscale;
    }


  *len = n_vals;


cleanup:
  if(matrix) jas_matrix_destroy(matrix);
  if(image)  jas_image_destroy(image);
  if(jpeg)   jas_stream_close(jpeg);

  return code;

}

static int pack_double(grib_accessor* a, const double* val, size_t *len)
{
  grib_accessor_data_jpeg2000_packing *self =(grib_accessor_data_jpeg2000_packing*)a;
  size_t i = 0;
  size_t n_vals = *len;
  int err = 0;

  double reference_value = 0;
  long   binary_scale_factor = 0;
  long   bits_per_value = 0;
  long   decimal_scale_factor = 0;

  double d = 1;
  size_t buflen = 0;
  unsigned char*  buf = NULL;
  long bits8;

  double max = 0;

  double min = 0;


  double divisor = 1;
  unsigned char *encoded = NULL;
  unsigned char *p;

  long width;
  long height;

  long ni;
  long nj;
  long target_compression_ratio;
  long type_of_compression_used;
  long scanning_mode;
  long list_defining_points;
  long number_of_data_points;


  int jaserr,jaslen;
  jas_image_t image = {0,};
  jas_stream_t *jpcstream = 0;
  jas_stream_t *istream = 0;
  jas_image_cmpt_t cmpt = {0,};
  jas_image_cmpt_t *pcmpt = 0;


#define MAXOPTSSIZE 1024
  char opts[MAXOPTSSIZE];

  if(*len == 0){
    grib_buffer_replace(a, NULL, 0, 1, 1);
    return GRIB_SUCCESS;
  }

  /*jas_init();*/

  if((err = grib_get_long_internal(a->parent->h,self->bits_per_value,&bits_per_value)) != GRIB_SUCCESS)
    return err;

  self->dirty=1;
  if(bits_per_value == 0)
  {
    int i;
    /* constant field */
    for(i = 1 ; i < n_vals; i++)
      Assert(val[i] == val[0]);

      if (grib_get_nearest_smaller_value(a->parent->h,self->reference_value,val[0],&reference_value)
        !=GRIB_SUCCESS) {
        grib_context_log(a->parent->h->context,GRIB_LOG_ERROR,
         "unable to find nearest_smaller_value of %g for %s",min,self->reference_value);
        exit(GRIB_INTERNAL_ERROR);
      }
    if((err = grib_set_double_internal(a->parent->h,self->reference_value,
        reference_value)) != GRIB_SUCCESS)
      return err;
    {
      /* Make sure we can decode it again */
      double ref = 1e-100;
      grib_get_double_internal(a->parent->h,self->reference_value,&ref);
      Assert(ref == reference_value);
    }

    grib_buffer_replace(a, NULL, 0,1,1);

    return GRIB_SUCCESS;
  }

  if((err = grib_get_long_internal(a->parent->h,self->decimal_scale_factor, &decimal_scale_factor)) != GRIB_SUCCESS)
    return err;


  d = grib_power(decimal_scale_factor,10) ;

  max = val[0];
  min = max;
  for(i=1;i< n_vals;i++)
  {
    if (val[i] > max )
      max = val[i];
    if (val[i] < min )
      min = val[i];
  }
  min *= d;
  max *= d;

  if (grib_get_nearest_smaller_value(a->parent->h,self->reference_value,min,&reference_value)
        !=GRIB_SUCCESS) {
        grib_context_log(a->parent->h->context,GRIB_LOG_ERROR,
         "unable to find nearest_smaller_value of %g for %s",min,self->reference_value);
        exit(GRIB_INTERNAL_ERROR);
  }
  /* printf("reference_value %g %g %g\n",reference_value,min,max); */

  if(reference_value > min)
  {
    fprintf(stderr,"reference_value=%g min_value=%g diff=%g\n",reference_value,min,reference_value-min);
    Assert(reference_value <= min);
  }

  binary_scale_factor = grib_get_binary_scale_fact(max,reference_value,bits_per_value);

  divisor = grib_power(-binary_scale_factor,2);

  bits8   = (bits_per_value+7)/8*8;
  encoded = grib_context_malloc_clear(a->parent->h->context,bits8/8*n_vals);

  if(!encoded) {
    err = GRIB_OUT_OF_MEMORY;
    goto cleanup;
  }

  buflen = 0;
  p     = encoded;

  for(i=0;i< n_vals;i++){
    long blen = bits8;
    unsigned long unsigned_val = (unsigned long)((((val[i]*d)-(reference_value))*divisor)+0.5);

    /*
       if(i > 10000 && i <10010) printf("jpeg raw encode %04lx\n",unsigned_val);*/
    while(blen >= 8)
    {
      blen   -= 8;
      *p = (unsigned_val >> blen);
      p++;
      buflen++;
    }
  }

  grib_context_log(a->parent->h->context, GRIB_LOG_DEBUG,
      "grib_accessor_data_jpeg2000_packing : pack_double : packing %s, %d values", a->name, n_vals);


  buf  = grib_context_malloc_clear(a->parent->h->context,buflen+EXTRA_BUFFER_SIZE);
  if(!buf) {
    err = GRIB_OUT_OF_MEMORY;
    goto cleanup;
  }

  if((err = grib_set_double_internal(a->parent->h,self->reference_value, reference_value)) != GRIB_SUCCESS)
    return err;
  {
    /* Make sure we can decode it again */
    double ref = 1e-100;
    grib_get_double_internal(a->parent->h,self->reference_value,&ref);
    Assert(ref == reference_value);
    /* printf("jpeg %s %g\n",self->reference_value, reference_value); */
  }

  if((err = grib_set_long_internal(a->parent->h,self->binary_scale_factor, binary_scale_factor)) != GRIB_SUCCESS)
    return err;

  if((err = grib_set_long_internal(a->parent->h,self->decimal_scale_factor, decimal_scale_factor)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->ni,&ni)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->nj,&nj)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->type_of_compression_used,&type_of_compression_used)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->target_compression_ratio,&target_compression_ratio)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->scanning_mode,&scanning_mode)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->list_defining_points,&list_defining_points)) != GRIB_SUCCESS)
    return err;

  if((err = grib_get_long_internal(a->parent->h,self->number_of_data_points,&number_of_data_points)) != GRIB_SUCCESS)
    return err;


  /*printf( "JPEG reference_value=%g binary_scale_factor=%ld decimal_scale_factor=%ld ni=%ld nj=%ld type_of_compression_used=%ld target_compression_ratio=%ld\n",
    reference_value,binary_scale_factor,
    decimal_scale_factor,
    ni,nj,type_of_compression_used,target_compression_ratio);*/

  opts[0] = 0;

  switch( type_of_compression_used)
  {

    case 0:
      Assert(target_compression_ratio == 255);
      break;

    case 1:
      Assert(target_compression_ratio != 255);
      snprintf(opts,MAXOPTSSIZE,"mode=real\nrate=%f",1.0/(float)target_compression_ratio);
      break;

    default:
      err = GRIB_NOT_IMPLEMENTED;
      goto cleanup;
  }

  width  = ni;
  height = nj;

  if((scanning_mode & (1<<5)) != 0)
  {
    long tmp = width;
    width    = height;
    height   = tmp;
  }

  /* The grid is not regular */
  if(list_defining_points != 0)
  {
    width  = *len;
    height = 1;
  }

  /* There is a bitmap */
  if(*len != number_of_data_points)
  {
    width  = *len;
    height = 1;
  }


  if(width*height != *len)
  {
    /* fprintf(stderr,"width=%ld height=%ld len=%d\n",(long)width,(long)height,(long)*len); */
    Assert(width*height == *len);
  }
  /*
     printf( "JPEG bits=%ld reference_value=%g binary_scale_factor=%ld decimal_scale_factor=%ld width=%ld heigth=%ld type_of_compression_used=%ld target_compression_ratio=%ld\n",
     bits_per_value,
     reference_value,binary_scale_factor,
     decimal_scale_factor,
     width,height,type_of_compression_used,target_compression_ratio);
   */

  image.tlx_      = 0;
  image.tly_      = 0;
  image.brx_      = width;
  image.bry_      = height;
  image.numcmpts_ = 1;
  image.maxcmpts_ = 1;
  image.clrspc_   = JAS_CLRSPC_SGRAY;
  image.cmprof_   = 0;
  image.inmem_    = 1;

  cmpt.tlx_       = 0;
  cmpt.tly_       = 0;
  cmpt.hstep_     = 1;
  cmpt.vstep_     = 1;
  cmpt.width_     = width;
  cmpt.height_    = height;
  cmpt.type_      = JAS_IMAGE_CT_COLOR(JAS_CLRSPC_CHANIND_GRAY_Y);
  cmpt.prec_      = bits_per_value;
  cmpt.sgnd_      = 0;
  cmpt.cps_       = bits8/8;

  Assert(cmpt.width_ * cmpt.height_ * cmpt.cps_ == buflen);

  pcmpt           = &cmpt;
  image.cmpts_    = &pcmpt;

  istream         = jas_stream_memopen((char *)encoded,buflen);
  cmpt.stream_    = istream;

  jpcstream       = jas_stream_memopen((char*)buf,buflen+EXTRA_BUFFER_SIZE);
  jaserr          = jpc_encode(&image,jpcstream,opts);

  if(jaserr != 0)
  {
    /* increase the number of guard bits */
    strcat(opts,"\nnumgbits=4");
    grib_context_log(a->parent->h->context, GRIB_LOG_ERROR,
        "grib_accessor_data_jpeg2000_packing : pack_double : jasper error %d, increasing the number of guard bits", jaserr);
    jaserr=jas_stream_close(istream);
    istream         = jas_stream_memopen((char *)encoded,buflen);
    jaserr=jas_stream_close(jpcstream);
    jpcstream       = jas_stream_memopen((char*)buf,buflen);
    jaserr          = jpc_encode(&image,jpcstream,opts);
  }

  if ( jaserr != 0 ) {
    grib_context_log(a->parent->h->context, GRIB_LOG_ERROR,
        "grib_accessor_data_jpeg2000_packing : pack_double : jasper error %d", jaserr);
    return GRIB_ENCODING_ERROR;
  }

  jaslen = jpcstream->rwcnt_;
  jaserr=jas_stream_close(istream); istream = 0;
  jaserr=jas_stream_close(jpcstream);jpcstream = 0;

  /*
     grib_context_log(a->parent->h->context, GRIB_LOG_INFO,
     "grib_accessor_data_jpeg2000_packing : pack_double : buf=%d, packed=%d", buflen,jaslen);
   */

  if(jaslen > buflen)
    grib_context_log(a->parent->h->context, GRIB_LOG_WARNING,
    "grib_accessor_data_jpeg2000_packing : jpeg data (%ld) larger than input data (%ld)",
    jaslen, buflen);
  Assert(jaslen <= buflen+EXTRA_BUFFER_SIZE);


  grib_buffer_replace(a, buf, jaslen, 1, 1);
  /* printf("my length=%ld\n",a->length); */



cleanup:

  grib_context_free(a->parent->h->context,buf);
  grib_context_free(a->parent->h->context,encoded);

  if(istream)   jas_stream_close(istream);
  if(jpcstream) jas_stream_close(jpcstream);

  if(err == GRIB_SUCCESS)
    err = grib_set_long_internal(a->parent->h,self->number_of_values, *len);
  return err;





}
#else

static int  unpack_double(grib_accessor* a, double* val, size_t *len)
{
  grib_context_log(a->parent->h->context, GRIB_LOG_ERROR,
      "grib_accessor_data_jpeg2000_packing : jpeg support no enabled. Please rerun configure with --with-jpeg-support");
  return GRIB_NOT_IMPLEMENTED;
}

static int pack_double(grib_accessor* a, const double* val, size_t *len)
{
  grib_context_log(a->parent->h->context, GRIB_LOG_ERROR,
      "grib_accessor_data_jpeg2000_packing : jpeg support no enabled. Please rerun configure with --with-jpeg-support");
  return GRIB_NOT_IMPLEMENTED;
}

#endif
