/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/*
//
// C Implementation: grib_associative
//
// Description: associative array
//
//
// Author: Enrico Fucile <enrico.fucile@ecmwf.int>
//
//
//
*/
/*TODO test destroy*/

#include <string.h>
#include <stdlib.h>
#include "grib_api_internal.h"
#include <errno.h>

#define GRIB_RUNTIMETYPE_CHAR_BUFFER 100

GRIB_INLINE static int grib_inline_strcmp(const char* a,const char* b) {
  if (*a != *b) return 1;
  while((*a!=0 && *b!=0) &&  *(a) == *(b) ) {a++;b++;}
  return (*a==0 && *b==0) ? 0 : 1;
}

extern int errno;

/* ~! RUNTIME TYPES IMPLEMENTATION */

/*  Constructor/Destructor  */
grib_runtime_type* grib_runtimetype_new(grib_context* gc) {
  grib_runtime_type* val;
  if (!gc) return NULL;
  val=(grib_runtime_type*)grib_context_malloc(gc,sizeof(grib_runtime_type));
  val->type=0;
  val->value=0;
  val->size=0;
  val->context=gc;
  return val;
}

void grib_runtimetype_destroy(grib_runtime_type* rttv) {
  grib_context_free(rttv->context,rttv);
}

int grib_runtimetype_copy(grib_runtime_type* copy,grib_runtime_type* original) {
  if (original == NULL || copy == NULL)
    return GRIB_RUNTIMETYPE_NULL_POINTER;
  copy->type=original->type;
  copy->value=original->value;
  copy->size=original->size;
  copy->context=original->context;
  return GRIB_RUNTIMETYPE_SUCCESS;
}


/* Runtime type get functions */

int grib_runtimetype_get_long(grib_runtime_type* rttv,long** pv,size_t* size) {
  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;
  if (rttv->type == GRIB_RUNTIMETYPE_LONG) {
    *pv=(long*)rttv->value;
    *size=rttv->size;
    return GRIB_RUNTIMETYPE_SUCCESS;
  }
  return GRIB_RUNTIMETYPE_WRONG_TYPE;
}

int grib_runtimetype_get_double(grib_runtime_type* rttv,double** pv,size_t* size) {
  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;
  if (rttv->type == GRIB_RUNTIMETYPE_DOUBLE) {
    *pv=(double*)rttv->value;
    *size=rttv->size;
    return GRIB_RUNTIMETYPE_SUCCESS;
  }
  return GRIB_RUNTIMETYPE_WRONG_TYPE;
}

int grib_runtimetype_get_char(grib_runtime_type* rttv,char** pv,size_t* size) {
  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;
  if (rttv->type == GRIB_RUNTIMETYPE_CHAR) {
    *pv=(char*)rttv->value;
    *size=rttv->size;
    return GRIB_RUNTIMETYPE_SUCCESS;
  }
  return GRIB_RUNTIMETYPE_WRONG_TYPE;
}

int grib_runtimetype_get(grib_runtime_type* rttv,void** pv,size_t* size) {
  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;
  *pv=rttv->value;
  *size=rttv->size;
  return GRIB_RUNTIMETYPE_SUCCESS;
}

int grib_runtimetype_get_type(grib_runtime_type* rttv,int* type) {
  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;
  *type=rttv->type;
  return GRIB_RUNTIMETYPE_SUCCESS;
}

/* Runtime type set functions */

int grib_runtimetype_set_long(grib_runtime_type* rttv,long* v,size_t size){
  return grib_runtimetype_set(rttv,v,GRIB_RUNTIMETYPE_LONG,size);
}

int grib_runtimetype_set_double(grib_runtime_type* rttv,double* v,size_t size) {
  return grib_runtimetype_set(rttv,v,GRIB_RUNTIMETYPE_DOUBLE,size);
}

int grib_runtimetype_set_char(grib_runtime_type* rttv,char* v,size_t size) {
  return grib_runtimetype_set(rttv,v,GRIB_RUNTIMETYPE_CHAR,size);
}

int grib_runtimetype_set(grib_runtime_type* rttv,void* v, int type, size_t size) {

  if (rttv == NULL) return GRIB_RUNTIMETYPE_NULL_POINTER;

  rttv->type = type;
  rttv->value=v;
  rttv->size=size;

  return GRIB_RUNTIMETYPE_SUCCESS;
}

/* !~ */

/* ~! ASSOCIATIVE ARRAY IMPLEMENTATION */

grib_associative_array_el* grib_associative_array_el_new(grib_context* gc) {
  grib_associative_array_el* e;

  if (gc==NULL) return NULL;

  e=(grib_associative_array_el*)grib_context_malloc(gc,sizeof(grib_associative_array_el));
  e->key=0;
  e->value=0;
  e->context=gc;

  return e;
}

void grib_associative_array_el_destroy(grib_associative_array_el* el) {
  if (!el) return;
  grib_runtimetype_destroy(el->value);
  grib_context_free(el->context,el);
}

grib_associative_array* grib_associative_array_new(grib_context* gc) {
  grib_associative_array *ar;

  if(gc == NULL) return NULL;

  ar=(grib_associative_array*)grib_context_malloc(gc,sizeof(grib_associative_array));

  ar->el=0;
  ar->next=0;
  ar->prev=0;
  ar->context=gc;
  return ar;
}

void grib_associative_array_destroy( grib_associative_array *ar) {
  grib_associative_array* cur;

  if (!ar) return;

  while ( ar->next ) {
    cur=ar;
    ar=ar->next;
    grib_associative_array_el_destroy(cur->el);
    grib_context_free(ar->context,cur);
  }

  grib_associative_array_el_destroy(ar->el);
  grib_context_free(ar->context,ar);

}

/* Set functions */
int grib_associative_array_set(grib_associative_array* ar,char* key,grib_runtime_type* value) {
  grib_runtime_type* oldval=NULL;

  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  if (grib_associative_array_get(ar,key,&oldval)==GRIB_ARRAY_SUCCESS) {
    grib_runtimetype_copy(oldval,value);
    return GRIB_ARRAY_FOUND;
  } else
    return grib_associative_array_push( ar,key,value);
}

int grib_associative_array_set_long(grib_associative_array* ar,char* key,long* v,size_t size) {
  grib_runtime_type *rtv=NULL;

  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  rtv=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_long(rtv,v,size);
  if (grib_associative_array_set(ar,key,rtv) == GRIB_ARRAY_FOUND)
    grib_runtimetype_destroy(rtv);

  return GRIB_ARRAY_SUCCESS;
}

int grib_associative_array_set_double(grib_associative_array* ar,char* key,double* v,size_t size) {
  grib_runtime_type *rtv=NULL;

  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  rtv=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_double(rtv,v,size);
  if (grib_associative_array_set(ar,key,rtv) == GRIB_ARRAY_FOUND)
    grib_runtimetype_destroy(rtv);

  return GRIB_ARRAY_SUCCESS;

}

int grib_associative_array_set_char(grib_associative_array* ar,char* key,char* v,size_t size) {
  grib_runtime_type *rtv=NULL;

  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  rtv=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_char(rtv,v,size);
  if (grib_associative_array_set(ar,key,rtv) == GRIB_ARRAY_FOUND)
    grib_runtimetype_destroy(rtv);

  return GRIB_ARRAY_SUCCESS;
}

/* Push functions */

int grib_associative_array_push(grib_associative_array* ar,char* key,grib_runtime_type* value) {
  grib_associative_array_el *el=NULL;
  grib_associative_array *nar=NULL;
  grib_associative_array *last=NULL;


  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  el=grib_associative_array_el_new(ar->context);
  el->key=key;
  el->value=value;
  if (!ar->el) {
    ar->el=el;
    return GRIB_ARRAY_SUCCESS;
  }
  nar=grib_associative_array_new(ar->context);
  nar->el=el;
  last=ar;
  while (last->next) last=last->next;
  last->next=nar;
  nar->prev=last;
  return GRIB_ARRAY_SUCCESS;
}

int grib_associative_array_push_long(grib_associative_array* ar,char* key,long* lvalue,size_t size) {
  grib_runtime_type *value=NULL;
  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  value=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_long(value,lvalue,size);
  return grib_associative_array_push(ar,key,value);
}

int grib_associative_array_push_double(grib_associative_array* ar,char* key,double* dvalue,size_t size) {
  grib_runtime_type *value=NULL;
  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  value=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_double(value,dvalue,size);
  return grib_associative_array_push(ar,key,value);
}

int grib_associative_array_push_char(grib_associative_array* ar,char* key,char* cvalue,size_t size) {
  grib_runtime_type *value=NULL;
  if (ar == NULL ) return GRIB_RUNTIMETYPE_NULL_POINTER;

  value=grib_runtimetype_new( ar->context);
  grib_runtimetype_set_char(value,cvalue,size);
  return grib_associative_array_push(ar,key,value);
}

/* Get functions */

int grib_associative_array_get(grib_associative_array* ar,const char* key,grib_runtime_type** value) {
  grib_associative_array* cur=0;

  int ret=GRIB_ARRAY_NOT_FOUND;

  if ( !ar || !ar->el )
    return GRIB_RUNTIMETYPE_NULL_POINTER;

  cur=ar;

  while (cur) {
    if (!grib_inline_strcmp(key,cur->el->key)) {
      *value=cur->el->value;
      ret = GRIB_RUNTIMETYPE_SUCCESS;
      break;
    }
    cur=cur->next;
  }

  return ret;
}

int grib_associative_array_get_long(grib_associative_array* ar,const char* key,long** value,size_t* size) {
  grib_associative_array* cur=0;

  int ret=GRIB_ARRAY_NOT_FOUND;

  if (!ar || !ar->el) return GRIB_RUNTIMETYPE_NULL_POINTER;

  cur=ar;

  while (cur) {
    if (!grib_inline_strcmp(key,cur->el->key)) {
      ret=grib_runtimetype_get_long(cur->el->value,value,size);
      ret= (ret == GRIB_RUNTIMETYPE_SUCCESS) ? GRIB_ARRAY_SUCCESS : ret;
      break;
    }
    cur=cur->next;
  }

  return ret;
}

int grib_associative_array_get_double(grib_associative_array* ar,const char* key,double** value,size_t* size) {
  grib_associative_array* cur=0;

  int ret=GRIB_ARRAY_NOT_FOUND;

  if (!ar || !ar->el) return GRIB_RUNTIMETYPE_NULL_POINTER;

  cur=ar;

  while (cur) {
    if (!grib_inline_strcmp(key,cur->el->key)) {
      ret=grib_runtimetype_get_double(cur->el->value,value,size);
      ret= (ret == GRIB_RUNTIMETYPE_SUCCESS) ? GRIB_ARRAY_SUCCESS : ret;
      break;
    }
    cur=cur->next;
  }

  return ret;
}

int grib_associative_array_get_char(grib_associative_array* ar,const char* key,char** value,size_t* size) {
  grib_associative_array* cur=0;

  int ret=GRIB_ARRAY_NOT_FOUND;

  if (!ar || !ar->el) return GRIB_RUNTIMETYPE_NULL_POINTER;

  cur=ar;

  while (cur) {
    if (!grib_inline_strcmp(key,cur->el->key)) {
      ret=grib_runtimetype_get_char(cur->el->value,value,size);
      ret= (ret == GRIB_RUNTIMETYPE_SUCCESS) ? GRIB_ARRAY_SUCCESS : ret;
      break;
    }
    cur=cur->next;
  }

  return ret;
}


int grib_file_to_array(FILE *in,grib_associative_array* ar) {

  char line[1024];
  char *left = NULL;
  char *right = NULL;
  char *pc=NULL;
  char *eq=NULL;
  char *pce=NULL;
  char *pcb=NULL;
  char escape[]="!!!";
  char separator='=';
  char values_name[]="values";
  double* values;
  int len=0;
  int count=0;
  grib_context* gc=NULL;
  int i=0;
  int ret;

  gc=ar->context;

  while (fgets(line,sizeof(line)-1,in)) {

    if (*line==0) break;

    line[strlen(line)-1] = 0;

    pcb=&line[0];
    while( *pcb!='\0' && *pcb == ' ') pcb++;
    if (strncmp(pcb,escape,strlen(escape))==0) {
      grib_context_log(gc,GRIB_LOG_DEBUG, "Escape %s matched",escape);
      break;
    }
    if (strlen(pcb)!=0 && *pcb != '#') {
      if ( *pcb!=separator && (pc=strchr(pcb,separator)) ) {
        eq=pc;
        pc++;
        while( *pc!='\0' && *pc == ' ') pc++;
        len=strlen(pc);
        if (len < 1) continue;
        if ((pce=strchr(pc,' '))){
          len=(pce-pc);
        }
        right=malloc((len+1)*sizeof(char));
        memcpy(right,pc,len);
        *(right+len)=0;

        pc=pcb;
        *eq=0;
        if ((len=strlen(pc))!=1) {
          pce=eq;
          pce--;
          while ( *pce==' ' && pce >= pc) pce--;
          if ((pce==pc) && (*pc == ' ')) continue;
          *(pce+1)=0;
          pcb=pc;
          while (pc < pce && (pc=strchr(pc,' '))) {
            pc++;
            pcb=pc;
          }
          len=strlen(pcb);
        }

        left=malloc((len+1)*sizeof(char));
        memcpy(left,pcb,len);
        *(left+len)=0;
        grib_context_log(gc,GRIB_LOG_DEBUG, "%s = %s",left,right);
        grib_associative_array_push_char(ar,left,right,1);

      } else if (!strncmp(pcb,values_name,strlen(values_name))) {
         while( *pcb!='\0' && *pcb!='(') pcb++;
         sscanf(pcb,"(%d)",&count);
         printf("++++++++++ %d +++++++++++\n",count);
         values=(double*)malloc(sizeof(double)*count);
         for (i=0;i<count-1;i++) {
            ret=fscanf(in,"%lg,",&values[i]);
            if (ret == EOF) return GRIB_END_OF_FILE;
            printf("-- %g ---\n",values[i]);
         }
         i++;
         fscanf(in,"%lg",&values[i]);
      }
    }

  }

  return GRIB_SUCCESS;
}

/* !~ */


