/* Gridlock
Copyright (c) 2002-2003 by Brian Nenninger. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#import "AtaxxGame.h"

@implementation AtaxxGame

-(id)init {
  if (self = [super init]) {
    capturedPieces = [[NSMutableArray alloc] init];
  }
  return self;
}

idAccessor(removedPiece, setRemovedPiece)
idAccessor(newPiece, setNewPiece)

-(void)dealloc {
  [capturedPieces release];
  [self setRemovedPiece:nil];
  [self setNewPiece:nil];
  [super dealloc];
}

-(void)reset {
  [super reset];

  [self setRemovedPiece:nil];
  [self setNewPiece:nil];
  [capturedPieces removeAllObjects];

  // read size and initial positions from configuration
  [self setGrid:[[DCHypergrid alloc] initWithRows:[[[self configurationInfo] objectForKey:@"rows"] intValue]
                                          columns:[[[self configurationInfo] objectForKey:@"cols"] intValue]]];
  [self setGridValuesFromArray:[[self configurationInfo] objectForKey:@"positions"]];

  [self setCurrentPlayerNumber:1];
}

-(BOOL)prepareMoveSequence:(NSArray *)positions {
  // clear change arrays
  [capturedPieces removeAllObjects];
  [self setRemovedPiece:nil];
  [self setNewPiece:nil];
  if ([positions count]==0) return YES;
  else {
    DCHypergridPosition *pos0 = [positions objectAtIndex:0];
    DCHypergridPosition *pos1 = [positions objectAtIndex:1];
    // if this is a jump, remove the starting cell
    if (abs([pos1 row]-[pos0 row])>1 ||
        abs([pos1 column]-[pos0 column])>1) {
      [self setRemovedPiece:pos0];
    }
    // the destination cell is always filled
    [self setNewPiece:pos1];
    // change all neighbors of the destination
    {
      int opponentValue = 3-playerNumber;
      int r = [pos1 row];
      int c = [pos1 column];
      int dr;
      for(dr=-1; dr<=1; dr++) {
        if (r+dr>=0 && r+dr<[self numberOfRows]) {
          int dc;
          for(dc=-1; dc<=1; dc++) {
            if (c+dc>=0 && c+dc<[self numberOfColumns]) {
              if (opponentValue==[self valueAtRow:r+dr column:c+dc]) {
                [capturedPieces addObject:[DCHypergridPosition positionWithRow:r+dr column:c+dc]];
              }
            }
          }
        }
      }
    }
    return YES;
  }
}

-(NSArray *)positionsOfCellsChangingOwnership {
  return capturedPieces;
}

-(NSArray *)positionsOfCellsBeingFilled {
  return [newPiece arrayWithSelf_];
}

-(NSArray *)positionsOfCellsBeingRemoved {
  return [removedPiece arrayWithSelf_];
}

-(int)futureValueAtPosition:(DCHypergridPosition *)pos {
  // check captured, new, removed arrays
  if ([capturedPieces containsObject:pos]) {
    return playerNumber;
  }
  else if ([newPiece isEqual:pos]) {
    return playerNumber;
  }
  else if ([removedPiece isEqual:pos]) {
    return 0;
  }
  // default future value is current value
  return [grid valueAtPosition:pos];
}

-(int)scoreForPlayer:(int)pnum {
  return [grid numberOfCellsWithValue:pnum];
}

-(BOOL)isGameOver {
  // game is over if all cells are filled, or if either player has no cells
  // therefore, game is not over if and only if grid contains at least 1 cell with each of 0,1,2.
  return (![grid hasCellWithValue:0] ||
          ![grid hasCellWithValue:1] ||
          ![grid hasCellWithValue:2]);  
}

-(NSArray *)allValidMoveSequences {
  NSMutableArray *sequences = [NSMutableArray array];
  NSEnumerator *posEnum = [[self grid] enumeratorForPositionsWithValue:[self currentPlayerNumber]];
  id position;
  while (position=[posEnum nextObject]) {
    NSArray *neighbors = [grid neighborsOfPosition:position distance:2];
    NSEnumerator *nenum = [neighbors objectEnumerator];
    id npos;
    while (npos=[nenum nextObject]) {
      if ([self valueAtPosition:npos]==0) {
        [sequences addObject:[NSArray arrayWithObjects:position,npos,nil]];
      }
    }
  }
  
  if ([sequences count]==0) [sequences addObject:[NSArray array]];
  return sequences;
}

@end
