/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  gnome-hamlib-rig.c: Gtk+/Gnome widget that contains all implemented
                      hamlib RIG controls. See the API documentation
		      for details.

  Copyright (C)  2002  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <math.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>
#include "gnome-hamlib-rig.h"
#include "defaults.h"

/* Frequency digits on LCD panel */
#include "../pixmaps/digits_normal.xpm"
#include "../pixmaps/digits_small.xpm"


static void gnome_hamlib_rig_class_init (GnomeHamlibRigClass *class);
static void gnome_hamlib_rig_init       (GnomeHamlibRig      *ghr);

/* callback functions */
static void gnome_hamlib_rig_power_cb  (GtkWidget *tb, gpointer ghr);
static void gnome_hamlib_rig_mode_cb   (GtkWidget *item, gpointer ghr);
static void gnome_hamlib_rig_filter_cb (GtkWidget *item, gpointer ghr);
static void gnome_hamlib_rig_agc_cb    (GtkWidget *item, gpointer ghr);
static void gnome_hamlib_rig_af_cb     (GtkAdjustment *adj, gpointer ghr);
static void gnome_hamlib_rig_rf_cb     (GtkAdjustment *adj, gpointer ghr);
static void gnome_hamlib_rig_nr_cb     (GtkAdjustment *adj, gpointer ghr);
static void gnome_hamlib_rig_sql_cb    (GtkAdjustment *adj, gpointer ghr);
static void gnome_hamlib_rig_apf_cb    (GtkAdjustment *adj, gpointer ghr);
static gint gnome_hamlib_rig_freq_cb   (GtkWidget *digit, GdkEvent *event, gpointer ghr);


/* timeout functions */
static gint gnome_hamlib_rig_freq_timeout   (gpointer ghr);
static gint gnome_hamlib_rig_slider_timeout (gpointer ghr);
static gint gnome_hamlib_rig_misc_timeout   (gpointer ghr);


/* misc private functions */
static GtkWidget *gnome_hamlib_rig_create_modemenu     (GnomeHamlibRig *ghr);
static GtkWidget *gnome_hamlib_rig_create_filtermenu   (GnomeHamlibRig *ghr);
static GtkWidget *gnome_hamlib_rig_create_agcmenu      (GnomeHamlibRig *ghr);
static void       gnome_hamlib_rig_create_lcd          (GnomeHamlibRig *ghr);
static GtkWidget *gnome_hamlib_rig_create_sliders      (GnomeHamlibRig *ghr);
static void       gnome_hamlib_rig_set_lcd_digits      (GnomeHamlibRig *ghr, guint64 freq);


static GtkVBoxClass *parent_class;


guint
gnome_hamlib_rig_get_type (void)
{
	static guint hamlib_rig_type = 0;

	if (!hamlib_rig_type) {
		GtkTypeInfo hamlib_rig_info = {
			"GnomeHamlibRig",
			sizeof (GnomeHamlibRig),
			sizeof (GnomeHamlibRigClass),
			(GtkClassInitFunc) gnome_hamlib_rig_class_init,
			(GtkObjectInitFunc) gnome_hamlib_rig_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		hamlib_rig_type = gtk_type_unique (gtk_vbox_get_type (),
						   &hamlib_rig_info);
	}

	return hamlib_rig_type;
}

static void
gnome_hamlib_rig_class_init (GnomeHamlibRigClass *class)
{
	parent_class = gtk_type_class (gtk_hbox_get_type ());
}


static void
gnome_hamlib_rig_destroy(GnomeHamlibRig *ghr)
{
	gint i;

	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));

	/* hamlib backend */
	if (ghr->rig) {
		rig_close (ghr->rig);    /* close port */
		rig_cleanup (ghr->rig);  /* if you care about memory */
	}

	/* widgets */
	if(ghr->power)
		gtk_widget_destroy (ghr->power);
	if (ghr->mode)
		gtk_widget_destroy (ghr->mode);
	if (ghr->filter)
		gtk_widget_destroy (ghr->filter);
	if (ghr->agc)
		gtk_widget_destroy (ghr->agc);
	if (ghr->canvas)
		gtk_widget_destroy (ghr->canvas);
	if (ghr->hbox1)
		gtk_widget_destroy (ghr->hbox1);
	if (ghr->vbox1)
		gtk_widget_destroy (ghr->vbox1);
/*  	This gives a GtkCritical! */
/*  	if (ghr->af) */
/*  		gtk_widget_destroy (ghr->af); */
	if (ghr->rf)
		gtk_widget_destroy (ghr->rf);
	if (ghr->nr)
		gtk_widget_destroy (ghr->nr);
	if (ghr->sql)
		gtk_widget_destroy (ghr->sql);
	if (ghr->apf)
		gtk_widget_destroy (ghr->apf);
	/* canvas stuff */
	for (i=0; i<12; i++) {
		if (ghr->digits_normal[i])
			gdk_pixbuf_unref (ghr->digits_normal[i]);
		if (ghr->digits_small[i])
			gdk_pixbuf_unref (ghr->digits_small[i]);
	}
/*  	if (ghr->canvas) */
/*  		gtk_widget_destroy (ghr->canvas); */
}


static void
gnome_hamlib_rig_init (GnomeHamlibRig *ghr)
{
	GtkTooltips *kooltip;
	GtkWidget *sliderbox;

	gtk_box_set_spacing (GTK_BOX (ghr), 0);
	
	gtk_signal_connect(GTK_OBJECT(ghr),"destroy",
			   GTK_SIGNAL_FUNC(gnome_hamlib_rig_destroy), NULL);

	/* hbox1 for buttons, LCD and sliders */
	ghr->hbox1 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (ghr), ghr->hbox1, TRUE, TRUE, 0);
	gtk_widget_show (ghr->hbox1);

	/* vbox1 for power, mode etc... */
	ghr->vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (ghr->hbox1), ghr->vbox1, TRUE, TRUE, 0);
	gtk_widget_show (ghr->vbox1);

	/* power button */
	ghr->power = gtk_toggle_button_new_with_label(_("Power"));
	gtk_box_pack_start (GTK_BOX (ghr->vbox1), ghr->power, FALSE, FALSE, 0);
	gtk_widget_show (ghr->power);

	/* AGC selector */
	ghr->agc = gtk_option_menu_new ();
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->agc, _("Automatic Gain Control"), NULL);
	gtk_box_pack_end (GTK_BOX (ghr->vbox1), ghr->agc, FALSE, FALSE, 0);
	gtk_widget_show (ghr->agc);

	/* filter selector */
	ghr->filter = gtk_option_menu_new ();
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->filter, _("FILTER"), NULL);
	gtk_box_pack_end (GTK_BOX (ghr->vbox1), ghr->filter, FALSE, FALSE, 0);
	gtk_widget_show (ghr->filter);

	/* mode selector */
	ghr->mode = gtk_option_menu_new ();
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->mode, _("MODE"), NULL);
	gtk_box_pack_end (GTK_BOX (ghr->vbox1), ghr->mode, FALSE, FALSE, 0);
	gtk_widget_show (ghr->mode);

	/* LCD canvas */
	gnome_hamlib_rig_create_lcd (ghr);
	gtk_box_pack_start (GTK_BOX (ghr->hbox1), ghr->canvas, TRUE, TRUE, 0);
	gtk_widget_show (ghr->canvas);

	/* sliders */
	sliderbox = gnome_hamlib_rig_create_sliders (ghr);
	gtk_box_pack_start (GTK_BOX (ghr->hbox1), sliderbox, FALSE, FALSE, 0);
	gtk_widget_show (sliderbox);

	/* Readback timers */
	ghr->freqtimer_id = 0;
	ghr->slidertimer_id = 0;
	ghr->misctimer_id = 0;
	ghr->freqtimer_val = 0;
	ghr->slidertimer_val = 0;
	ghr->misctimer_val = 0;
}


/**
 * gnome_hamlib_rig_new:
 * @rigid: the hamlib ID of the radio (see 'rigctl --list')
 * @port: the port the radio is attached to
 *
 * Description: Creates a new radio control widget.
 *
 * Returns: A new GnomeHamlibRig widget.
 */
GtkWidget *
gnome_hamlib_rig_new (guint rigid,
		      const gchar *port)
{
	GnomeHamlibRig *ghr;
	
	/* hamlib */
	gint retcode;
	powerstat_t pwr;
	rmode_t mode;
	pbwidth_t width;     
	value_t val;
	setting_t level_rd, level_wr;
	freq_t freq;

	ghr = gtk_type_new (gnome_hamlib_rig_get_type ());
	
	/* initialize rig backend */
	ghr->rig = rig_init (rigid ? rigid : 1);
	g_return_val_if_fail (ghr->rig != NULL, NULL);

	/* set port and open */
	strncpy (ghr->rig->state.rigport.pathname, port ? port : "/dev/ttyS0", FILPATHLEN );
	retcode = rig_open (ghr->rig);
	if (retcode != RIG_OK) {
		rig_cleanup (ghr->rig);
		return NULL;
	}

	/* read power status from radio */
	retcode = rig_get_powerstat (ghr->rig, &pwr);
	if (retcode == RIG_OK) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ghr->power),
					      (pwr==RIG_POWER_ON) ? TRUE : FALSE);
		gtk_signal_connect (GTK_OBJECT (ghr->power), "toggled",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_power_cb),
				    ghr);
	}
	else {
		g_warning (_("gnome_hamlib_rig.c: Could not get power status from RIG.\n"));
	}

	/* try to get current mode and pbwidth */
	retcode = rig_get_mode (ghr->rig, RIG_VFO_CURR, &mode, &width);
	if (retcode != RIG_OK) {
		/* set some defaults */
		mode = 1;
		width = rig_passband_normal (ghr->rig, mode);
		g_warning (_("gnome_hamlib_rig.c: Could not get mode and pbwidth from RIG.\n"));
	}

	/* MODE selector */
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ghr->mode),
				  gnome_hamlib_rig_create_modemenu (ghr));
	/* mode can be 1, 2, 4, 8 ... */
	gtk_option_menu_set_history (GTK_OPTION_MENU (ghr->mode), rint (log (mode)/log (2)));

	/**** FIXME !!!!!! ****/
	gtk_widget_set (ghr->mode, "width", 80, NULL);

	/* FILTER selector */
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ghr->filter),
				  gnome_hamlib_rig_create_filtermenu (ghr));
	if (width == rig_passband_wide (ghr->rig, mode))
		gtk_option_menu_set_history (GTK_OPTION_MENU (ghr->filter), 0);
	if (width == rig_passband_normal (ghr->rig, mode))
		gtk_option_menu_set_history (GTK_OPTION_MENU (ghr->filter), 1);
	if (width == rig_passband_narrow (ghr->rig, mode))
		gtk_option_menu_set_history (GTK_OPTION_MENU (ghr->filter), 2);

	/* AGC selector */
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ghr->agc),
				  gnome_hamlib_rig_create_agcmenu (ghr));
	/* try to get AGC setting */
	retcode = rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_AGC, &val);
	if (retcode != RIG_OK) {
		/* ERROR */
		val.i = 1;
	}
	gtk_option_menu_set_history (GTK_OPTION_MENU (ghr->agc), val.i);
	/* disable AGC selector if RIG doesn't support it */
	if (!rig_has_set_level (ghr->rig, RIG_LEVEL_AGC))
		gtk_widget_set_sensitive (ghr->agc, FALSE);

	/* get frequency from RIG */
	retcode = rig_get_freq (ghr->rig, RIG_VFO_CURR, &freq);
	if (retcode != RIG_OK) {
		g_warning (_("gnome-hamlib-rig.c: Could not get frequency.\n"));
	}
	gnome_hamlib_rig_set_lcd_digits (ghr, freq);


	/* disable sliders if a level setting is not available */
	level_rd = rig_has_get_level (ghr->rig, RIG_LEVEL_AF | RIG_LEVEL_RF | RIG_LEVEL_IF |
				      RIG_LEVEL_NR | RIG_LEVEL_APF | RIG_LEVEL_SQL | RIG_LEVEL_STRENGTH);
	level_wr = rig_has_set_level (ghr->rig, RIG_LEVEL_AF | RIG_LEVEL_RF | RIG_LEVEL_IF |
				      RIG_LEVEL_NR | RIG_LEVEL_APF | RIG_LEVEL_SQL);

	/* AF */
	if (level_rd & RIG_LEVEL_AF) {
		rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_AF, &val);
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->af)), -val.f);
	}
	if (!(level_wr & RIG_LEVEL_AF))
		gtk_widget_set_sensitive (ghr->af, FALSE);
	else
		gtk_signal_connect (GTK_OBJECT (gtk_range_get_adjustment (GTK_RANGE (ghr->af))),
				    "value-changed",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_af_cb),
				    ghr);		
	/* RF */
	if (level_rd & RIG_LEVEL_RF) {
		rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_RF, &val);
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->rf)), -val.f);
	}
	if (!(level_wr & RIG_LEVEL_RF))
		gtk_widget_set_sensitive (ghr->rf, FALSE);
	else
		gtk_signal_connect (GTK_OBJECT (gtk_range_get_adjustment (GTK_RANGE (ghr->rf))),
				    "value-changed",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_rf_cb),
				    ghr);		
	/* NR */
	if (level_rd & RIG_LEVEL_NR) {
		rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_NR, &val);
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->nr)), -val.f);
	}
	if (!(level_wr & RIG_LEVEL_NR))
		gtk_widget_set_sensitive (ghr->nr, FALSE);
	else
		gtk_signal_connect (GTK_OBJECT (gtk_range_get_adjustment (GTK_RANGE (ghr->nr))),
				    "value-changed",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_nr_cb),
				    ghr);		
	/* SQL */
	if (level_rd & RIG_LEVEL_SQL) {
		rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_SQL, &val);
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->sql)), -val.f);
	}
	if (!(level_wr & RIG_LEVEL_SQL))
		gtk_widget_set_sensitive (ghr->sql, FALSE);
	else
		gtk_signal_connect (GTK_OBJECT (gtk_range_get_adjustment (GTK_RANGE (ghr->sql))),
				    "value-changed",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_sql_cb),
				    ghr);		
	/* APF */
	if (level_rd & RIG_LEVEL_APF) {
		rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_APF, &val);
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->apf)), -val.f);
	}
	if (!(level_wr & RIG_LEVEL_APF))
		gtk_widget_set_sensitive (ghr->apf, FALSE);
	else
		gtk_signal_connect (GTK_OBJECT (gtk_range_get_adjustment (GTK_RANGE (ghr->apf))),
				    "value-changed",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_apf_cb),
				    ghr);		



	return GTK_WIDGET (ghr);
}



static void
gnome_hamlib_rig_power_cb (GtkWidget *tb, gpointer ghr)
{
	/* This function is called, when the user clicks on the POWER
	   button.
	*/
 	gint retcode;
	gboolean tbstate;

	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));

	tbstate = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (tb));
	retcode = rig_set_powerstat (GNOME_HAMLIB_RIG (ghr)->rig, tbstate);
	if (retcode != RIG_OK) {
		/* change button state back */
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tb), !tbstate);
		g_warning (_("gnome_hamlib_rig.c: Could not set power status.\n"));
	}
}



static void
gnome_hamlib_rig_mode_cb (GtkWidget *item, gpointer ghr)
{
	/* This function is called when a new item in the mode
	   selector has been chosen. The mode info is attached
	   to the activated item.
	*/
	gint retcode;
	rmode_t mode;
	pbwidth_t pbwidth;
	GtkOptionMenu *modemenu;

	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));

	modemenu = GTK_OPTION_MENU (GNOME_HAMLIB_RIG (ghr)->mode);

	/* get current mode an pbwidth */
	retcode = rig_get_mode ((GNOME_HAMLIB_RIG (ghr))->rig, RIG_VFO_CURR, &mode, &pbwidth);
	if (retcode != RIG_OK) {
		g_warning (_("gnome-hamlib-rig.c: Could not get mode and pbwidth.\n"));
		pbwidth = rig_passband_normal ((GNOME_HAMLIB_RIG (ghr))->rig, mode);
	}
	mode = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "mode"));
	retcode = rig_set_mode ((GNOME_HAMLIB_RIG (ghr))->rig, RIG_VFO_CURR, mode, pbwidth);
	if (retcode != RIG_OK) {
		g_warning (_("gnome-hamlib-rig.c: Could not set mode. Trying to reset.\n"));
		/* reset to current mode */
		retcode = rig_get_mode ((GNOME_HAMLIB_RIG (ghr))->rig, RIG_VFO_CURR, &mode, &pbwidth);
		gtk_option_menu_set_history (modemenu, rint (log (mode)/log (2)));
	}
}



static void
gnome_hamlib_rig_filter_cb (GtkWidget *item, gpointer ghr)
{
	/* This function is called when a new item in the
	   filter selector has been chosen. The passband
	   info is attached to the selected item.
	*/
	gint retcode;
	rmode_t mode;
	pbwidth_t pbwidth;
	guint wd;
	GtkOptionMenu *filter;

	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));

	filter = GTK_OPTION_MENU (GNOME_HAMLIB_RIG (ghr)->filter);

	/* Passband can only be set via the rig_set_mode API call */
	retcode = rig_get_mode (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, &mode, &pbwidth);
	if (retcode != RIG_OK) {
		g_warning (_("gnome-hamlib-rig.c: Could not get mode (filter_cb).\n"));
	}
	else {
		wd = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "filter"));
		switch (wd) {
		case 0: pbwidth = rig_passband_wide (GNOME_HAMLIB_RIG (ghr)->rig, mode);
			break;
		case 1: pbwidth = rig_passband_normal (GNOME_HAMLIB_RIG (ghr)->rig, mode);
			break;
		case 2: pbwidth = rig_passband_narrow (GNOME_HAMLIB_RIG (ghr)->rig, mode);
			break;
		default: /* THIS SHOULD NEVER HAVE HAPPENED ERROR */
		}
		retcode = rig_set_mode (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, mode, pbwidth);
		if (retcode != RIG_OK) {
			g_warning (_("gnome-hamlib-rig.c: Could not set mode. (filter_cb)\n"));
			/* reset to default */
			retcode = rig_get_mode (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, &mode, &pbwidth);
			if (retcode == RIG_OK) {
				if (pbwidth == rig_passband_wide (GNOME_HAMLIB_RIG (ghr)->rig, mode))
					gtk_option_menu_set_history (filter, 0);
				if (pbwidth == rig_passband_normal (GNOME_HAMLIB_RIG (ghr)->rig, mode))
					gtk_option_menu_set_history (filter, 1);
				if (pbwidth == rig_passband_narrow (GNOME_HAMLIB_RIG (ghr)->rig, mode))
					gtk_option_menu_set_history (filter, 2);
			}
		}
	}
}


static void
gnome_hamlib_rig_agc_cb (GtkWidget *item, gpointer ghr)
{
	/* This function is called when a new item in the
	   AGC selector has been chosen. The AGC mode
	   info is attached to the selected item.
	*/
	gint retcode;
	value_t val;
	GtkOptionMenu *menu;
	
	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));
	g_return_if_fail (rig_has_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_LEVEL_AGC));

	menu = GTK_OPTION_MENU (GNOME_HAMLIB_RIG (ghr)->agc);

	val.i = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "agc")); 
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_AGC, val);
	if (retcode != RIG_OK) {
		/* try to reset AGC selector */
		g_warning (_("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_AGC.\n"));
		retcode = rig_get_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_AGC, &val);
		gtk_option_menu_set_history (menu, val.i);		
	}
}


static void
gnome_hamlib_rig_af_cb     (GtkAdjustment *adj, gpointer ghr)
{
	/* This function is called when the user adjusts
	   the AF slider.
	*/
	gint retcode;
	value_t val;

	val.f = -adj->value;
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_AF, val);
	if (retcode != RIG_OK) {
		g_warning ("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_AF.\n");
	}		 
}


static void
gnome_hamlib_rig_rf_cb     (GtkAdjustment *adj, gpointer ghr)
{
	/* This function is called when the user adjusts
	   the RF slider.
	*/
	gint retcode;
	value_t val;

	val.f = -adj->value;
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_RF, val);
	if (retcode != RIG_OK) {
		g_warning ("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_RF.\n");
	}		 
}


static void
gnome_hamlib_rig_nr_cb     (GtkAdjustment *adj, gpointer ghr)
{
	/* This function is called when the user adjusts
	   the NR slider.
	*/
	gint retcode;
	value_t val;

	val.f = -adj->value;
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_NR, val);
	if (retcode != RIG_OK) {
		g_warning ("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_NR.\n");
	}		 
}


static void
gnome_hamlib_rig_sql_cb    (GtkAdjustment *adj, gpointer ghr)
{
	/* This function is called when the user adjusts
	   the SQL slider.
	*/
	gint retcode;
	value_t val;

	val.f = -adj->value;
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_SQL, val);
	if (retcode != RIG_OK) {
		g_warning ("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_SQL.\n");
	}		 
}


static void
gnome_hamlib_rig_apf_cb    (GtkAdjustment *adj, gpointer ghr)
{
	/* This function is called when the user adjusts
	   the APF slider.
	*/
	gint retcode;
	value_t val;

	val.f = -adj->value;
	retcode = rig_set_level (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, RIG_LEVEL_APF, val);
	if (retcode != RIG_OK) {
		g_warning ("gnome-hamlib-rig.c: Could not set level RIG_LEVEL_APF.\n");
	}		 
}


static gint
gnome_hamlib_rig_freq_cb   (GtkWidget *digit,
			    GdkEvent  *event,
			    gpointer   data)
{
	/* This function is called when the user clicks
	   on a digit in the LCD dispay. Left click
	   increases while right click decreases the
	   frequency.
	*/

	GnomeHamlibRig *ghr;
	GdkCursor *cursor;
	gint retcode;
	guint power;
	gchar *str;
	freq_t freq,dfreq;

	g_return_val_if_fail (data != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (data), FALSE);

	ghr = GNOME_HAMLIB_RIG (data);

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		retcode = rig_get_freq (ghr->rig, RIG_VFO_CURR, &freq);
		if (retcode == RIG_OK) {
			power = GPOINTER_TO_UINT (gtk_object_get_user_data (GTK_OBJECT (digit)));
			dfreq = (guint64) pow (10, power);
			/*********** FIXME: Find out whether hamlib checks the frequency ranges.
				     Is it safe to try to set frequency out of range? Don't rely
				     that the RIG can handle it well
			*/
			switch (event->button.button) {
			case 1: case 4: /* LEFT button or wheel up: increase digit */
				freq += dfreq;
				if (freq < 1e12) {
					retcode = rig_set_freq (ghr->rig, RIG_VFO_CURR, freq);
					if (retcode == RIG_OK) {
						gnome_hamlib_rig_set_lcd_digits (ghr, freq);
					}
				}
				break;
			case 2: /* MIDDLE mouse button: set digit to zero */
				/* convert freq to string */
				str = g_strdup_printf ("%12lld", freq);
				str[11-power] = '0';
				freq = (freq_t) g_strtod (str, NULL);
				/* try the new frequency */
				if (freq >= 30000) {
					retcode = rig_set_freq (ghr->rig, RIG_VFO_CURR, freq);
					if (retcode == RIG_OK) {
						gnome_hamlib_rig_set_lcd_digits (ghr, freq);
					}
				}
				break;
			case 3: case 5: /* RIGHT button or wheel down: decrease digit */
				freq -= (dfreq<freq) ? dfreq : dfreq/10;
				if (freq >= 30000) {
					retcode = rig_set_freq (ghr->rig, RIG_VFO_CURR, freq);
					if (retcode == RIG_OK) {
						gnome_hamlib_rig_set_lcd_digits (ghr, freq);
					}
				}
				break;
			default:
				break;
			}
		}
		else {
			/* ERROR */
		}
		return TRUE;
		break;
	case GDK_ENTER_NOTIFY: /* change cursor */
		cursor = gdk_cursor_new (GDK_HAND2);
		gdk_window_set_cursor (ghr->canvas->window, cursor);
		gdk_cursor_destroy (cursor); /* yes, we do care about memory */
		return TRUE;
		break;
	case GDK_LEAVE_NOTIFY: /* change cursor */
		cursor = gdk_cursor_new (GDK_LEFT_PTR);
		gdk_window_set_cursor (ghr->canvas->window, cursor);
		gdk_cursor_destroy (cursor); /* yes, we do care about memory */
		return TRUE;
		break;
	default:
		return TRUE;
	}

	return FALSE;  /* Try parent */
}


static GtkWidget *
gnome_hamlib_rig_create_modemenu (GnomeHamlibRig *ghr)
{
	/* This function creates the Gtk-menu that is used
	   for mode selection. The RIG structure is needed
	   to correctly connect the callback functions.
	*/
	GtkWidget *modemenu;
	GtkWidget *item;

	g_return_val_if_fail (ghr != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), NULL);

	modemenu = gtk_menu_new ();
	item = gtk_menu_item_new_with_label (_("AM"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_AM));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("CW"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_CW));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("USB"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_USB));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("LSB"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_LSB));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("RTTY"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_RTTY));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("FMN"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_FM));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);
	item = gtk_menu_item_new_with_label (_("FMW"));
	gtk_object_set_data (GTK_OBJECT (item), "mode", GUINT_TO_POINTER (RIG_MODE_WFM));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_mode_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (modemenu), item);

	return modemenu;
}


static GtkWidget *
gnome_hamlib_rig_create_filtermenu (GnomeHamlibRig *ghr)
{
	/* This function creates the Gtk-menu that is used
	   for filter selection. The RIG structure is needed
	   to correctly connect the callback functions.
	*/
	GtkWidget *filtermenu;
	GtkWidget *item;

	g_return_val_if_fail (ghr != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), NULL);

	filtermenu = gtk_menu_new ();
	item = gtk_menu_item_new_with_label (_("Wide"));
	gtk_object_set_data (GTK_OBJECT (item), "filter", GUINT_TO_POINTER (0));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_filter_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (filtermenu), item);
	item = gtk_menu_item_new_with_label (_("Normal"));
	gtk_object_set_data (GTK_OBJECT (item), "filter", GUINT_TO_POINTER (1));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_filter_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (filtermenu), item);
	item = gtk_menu_item_new_with_label (_("Narrow"));
	gtk_object_set_data (GTK_OBJECT (item), "filter", GUINT_TO_POINTER (2));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_filter_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (filtermenu), item);

	return filtermenu;
}


static GtkWidget *
gnome_hamlib_rig_create_agcmenu (GnomeHamlibRig *ghr)
{
	/* This function creates the Gtk-menu that is used
	   for AGC selection. The RIG structure is needed
	   to correctly connect the callback functions.
	*/
	GtkWidget *agcmenu;
	GtkWidget *item;

	g_return_val_if_fail (ghr != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), NULL);

	agcmenu = gtk_menu_new ();
	item = gtk_menu_item_new_with_label (_("AGC OFF"));
	gtk_object_set_data (GTK_OBJECT (item), "agc", GUINT_TO_POINTER (RIG_AGC_OFF));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_agc_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (agcmenu), item);
	item = gtk_menu_item_new_with_label (_("Fast"));
	gtk_object_set_data (GTK_OBJECT (item), "agc", GUINT_TO_POINTER (RIG_AGC_SUPERFAST));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_agc_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (agcmenu), item);
	item = gtk_menu_item_new_with_label (_("Medium"));
	gtk_object_set_data (GTK_OBJECT (item), "agc", GUINT_TO_POINTER (RIG_AGC_FAST));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_agc_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (agcmenu), item);
	item = gtk_menu_item_new_with_label (_("Slow"));
	gtk_object_set_data (GTK_OBJECT (item), "agc", GUINT_TO_POINTER (RIG_AGC_SLOW));
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_agc_cb),
			    ghr);
	gtk_menu_append (GTK_MENU (agcmenu), item);

	return agcmenu;
}


static void
gnome_hamlib_rig_create_lcd (GnomeHamlibRig *ghr)
{
	/* This function creates and sets up the LCD panel */
	gint i;
	GdkPixbuf *numbers;
	GnomeCanvasItem *main_qrg_gr;

	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	ghr->canvas = gnome_canvas_new ();
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();
	gtk_widget_set_usize (ghr->canvas, LCD_SIZE_X+5, LCD_SIZE_Y+5);
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (ghr->canvas), 1);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (ghr->canvas), 0.0, 0.0, 
					LCD_SIZE_X, LCD_SIZE_Y);

	/* background */
	ghr->lcdbg.red   = 26749; /* 104 in 8-bit */
	ghr->lcdbg.green = 48544; /* 189 in 8-bit */
	ghr->lcdbg.blue  = 57996; /* 226 in 8-bit */
	gdk_colormap_alloc_color (gdk_rgb_get_cmap (), &ghr->lcdbg, FALSE, TRUE);
	/* foreground */
	ghr->lcdfg.red   = 10280; /*  40 in 8-bit */
	ghr->lcdfg.green = 21845; /*  85 in 8-bit */
	ghr->lcdfg.blue  = 33153; /* 129 in 8-bit */
	gdk_colormap_alloc_color (gdk_rgb_get_cmap (), &ghr->lcdfg, FALSE, TRUE);

	gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (ghr->canvas)),
			       gnome_canvas_rect_get_type (),
			       "x1", (double) 0,
			       "y1", (double) 0,
			       "x2", (double) LCD_SIZE_X,
			       "y2", (double) LCD_SIZE_Y,
			       "fill_color_gdk", &ghr->lcdbg,
			       "outline_color", "black",
			       NULL);

	/* normal size digits */
	numbers = gdk_pixbuf_new_from_xpm_data (digits_normal_xpm);
	/* split pixmap into digits */
	for (i=0; i<11; i++) {
		ghr->digits_normal[i] = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8,
							LCD_DIG_NOR_W, LCD_DIG_NOR_H);
		gdk_pixbuf_copy_area (numbers, LCD_DIG_NOR_W*i, 0,
				      LCD_DIG_NOR_W, LCD_DIG_NOR_H,
				      ghr->digits_normal[i], 0, 0);
	}
	/* decimal point - don't need whole pixmap - cut the first 10 pixels */
	ghr->digits_normal[11] = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8,
						 LCD_DIG_NOR_W-12, LCD_DIG_NOR_H);
	gdk_pixbuf_copy_area (numbers, LCD_DIG_NOR_W*11+12, 0,
			      LCD_DIG_NOR_W-12, LCD_DIG_NOR_H,
			      ghr->digits_normal[11], 0, 0);

	gdk_pixbuf_unref (numbers);

		/* small size digits */
	numbers = gdk_pixbuf_new_from_xpm_data (digits_small_xpm);
	/* split pixmap into digits */
	for (i=0; i<10; i++) {
		ghr->digits_small[i] = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8,
						       LCD_DIG_SMA_W, LCD_DIG_SMA_H);
		gdk_pixbuf_copy_area (numbers, LCD_DIG_SMA_W*i, 0,
				      LCD_DIG_SMA_W, LCD_DIG_SMA_H,
				      ghr->digits_small[i], 0, 0);
	}
	gdk_pixbuf_unref (numbers);

		/* Canvas group for main frequency digits (current VFO) */
	main_qrg_gr = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (ghr->canvas)),
					     gnome_canvas_group_get_type (),
					     "x", (gdouble) LCD_FREQ_MAIN_X, /* consts in defaults.h */
					     "y", (gdouble) LCD_FREQ_MAIN_Y,
					     NULL );

	/* now for the digits... */

	/* Frequency will be displayed as kHz with a resolution of
	   0.001 that is 1 Hz. */
	for (i=0; i<9; i++) { /* from GHz to kHz */ 
		ghr->main_qrg[i] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (main_qrg_gr),
							  gnome_canvas_pixbuf_get_type (),
							  "pixbuf", ghr->digits_normal[8],
							  "x", (gdouble) i*LCD_DIG_NOR_W,
							  NULL);
		gtk_object_set_user_data (GTK_OBJECT (ghr->main_qrg[i]), GUINT_TO_POINTER (11-i));
		gtk_signal_connect (GTK_OBJECT (ghr->main_qrg[i]), "event",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_freq_cb),
				    ghr);
	}
	/* decimal point */
	gnome_canvas_item_new (GNOME_CANVAS_GROUP (main_qrg_gr),
			       gnome_canvas_pixbuf_get_type (),
			       "pixbuf", ghr->digits_normal[11],
			       "x", (gdouble) 9*LCD_DIG_NOR_W,
			       NULL);
	/* Hz */
	for (i=9; i<12; i++) {
		ghr->main_qrg[i] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (main_qrg_gr),
							  gnome_canvas_pixbuf_get_type (),
							  "pixbuf", ghr->digits_small[8],
							  "x", (gdouble) 8*LCD_DIG_NOR_W+15+(i-8)*LCD_DIG_SMA_W,
							  "y", (gdouble) LCD_DIG_NOR_H - LCD_DIG_SMA_H,
							  NULL);
		gtk_object_set_user_data (GTK_OBJECT (ghr->main_qrg[i]), GUINT_TO_POINTER (11-i));
		gtk_signal_connect (GTK_OBJECT (ghr->main_qrg[i]), "event",
				    GTK_SIGNAL_FUNC (gnome_hamlib_rig_freq_cb),
				    ghr);
	}
	/* kHz text */
	gnome_canvas_item_new (GNOME_CANVAS_GROUP (main_qrg_gr),
			       gnome_canvas_text_get_type (),
			       "text", "kHz",
			       "x", (gdouble) 8*LCD_DIG_NOR_W+25+4*LCD_DIG_SMA_W,
			       "y", (gdouble) LCD_DIG_NOR_H,
			       "anchor", GTK_ANCHOR_SOUTH_WEST,
			       "fill_color_gdk", &ghr->lcdfg,
			       "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
			       NULL);
}


static void
gnome_hamlib_rig_set_lcd_digits (GnomeHamlibRig *ghr,
				 guint64 freq)
{
	/* This function sets the digits of the frequency readout on the
	   LCD panel. The numeric value of the frequency is converted to
	   a string with 12 characters (max 999.999999999 GHz) and scanned
	   through one by one.
	*/
	gchar *str;
	guint i;

	g_return_if_fail (ghr != NULL);
	g_return_if_fail (GNOME_IS_HAMLIB_RIG (ghr));

	ghr->freq1 = freq;
	str = g_strdup_printf ("%12lld", freq);
	for (i=0; i<12; i++) {
		switch (str[i]) {
		case '0' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[0] : ghr->digits_small[0],
					       NULL);
			break;
		case '1' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[1] : ghr->digits_small[1],
					       NULL);
			break;
		case '2' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[2] : ghr->digits_small[2],
					       NULL);
			break;
		case '3' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[3] : ghr->digits_small[3],
					       NULL);
			break;
		case '4' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[4] : ghr->digits_small[4],
					       NULL);
			break;
		case '5' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[5] :ghr-> digits_small[5],
					       NULL);
			break;
		case '6' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[6] : ghr->digits_small[6],
					       NULL);
			break;
		case '7' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[7] : ghr->digits_small[7],
					       NULL);
			break;
		case '8' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[8] : ghr->digits_small[8],
					       NULL);
			break;
		case '9' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[9] : ghr->digits_small[9],
					       NULL);
			break;
		case ' ' :
			gnome_canvas_item_set (ghr->main_qrg[i],
					       "pixbuf",
					       i < 9 ? ghr->digits_normal[10] : ghr->digits_small[10],
					       NULL);
			break;
		default  :
			break;
			}
	}
	g_free (str);
}



static GtkWidget *
gnome_hamlib_rig_create_sliders (GnomeHamlibRig *ghr)
{
	/* This function creates the adjustments and vertical
	   scales that are used to adjust various level settings.
	   it returns a box-widget that contains all scales and
	   labels.
	*/
	GtkObject *adj;
	GtkTooltips *kooltip;
	GtkWidget *sliderbox,*labelbox,*vbox,*frame;

	/* create boxes */
	sliderbox = gtk_hbox_new (TRUE, 0);
	labelbox = gtk_hbox_new (TRUE, 0);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), sliderbox);
	gtk_box_pack_start (GTK_BOX (vbox), labelbox, FALSE, FALSE, 0);
	
	/* AF */
	/* we use negative values otherwise the max would be at the bottom of
	   the slider. Go figure...
	*/
	adj = gtk_adjustment_new (0.0, -1.0, 0.0, 0.01, 0.1, 0.0);
/*  	gtk_signal_connect (adj, "value-changed", */
/*  			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_af_cb), */
/*  			    ghr); */
	ghr->af = gtk_vscale_new (GTK_ADJUSTMENT (adj));
	gtk_scale_set_draw_value (GTK_SCALE (ghr->af), FALSE);
	gtk_scale_set_digits (GTK_SCALE (ghr->af), 2);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->af, _("AF Gain"), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (sliderbox), ghr->af);
	gtk_box_pack_start_defaults (GTK_BOX (labelbox), gtk_label_new (_(" AF ")));
	/* RF */
	adj = gtk_adjustment_new (0.0, -1.0, 0.0, 0.01, 0.1, 0.0);
/*  	gtk_signal_connect (adj, "value-changed", */
/*  			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_rf_cb), */
/*  			    ghr); */
	ghr->rf = gtk_vscale_new (GTK_ADJUSTMENT (adj));
	gtk_scale_set_draw_value (GTK_SCALE (ghr->rf), FALSE);
	gtk_scale_set_digits (GTK_SCALE (ghr->rf), 2);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->rf, _("RF Gain"), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (sliderbox), ghr->rf);
	gtk_box_pack_start_defaults (GTK_BOX (labelbox), gtk_label_new (_(" RF ")));
	/* NR */
	adj = gtk_adjustment_new (0.0, -1.0, 0.0, 0.01, 0.1, 0.0);
/*  	gtk_signal_connect (adj, "value-changed", */
/*  			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_nr_cb), */
/*  			    ghr); */
	ghr->nr = gtk_vscale_new (GTK_ADJUSTMENT (adj));
	gtk_scale_set_draw_value (GTK_SCALE (ghr->nr), FALSE);
	gtk_scale_set_digits (GTK_SCALE (ghr->nr), 2);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->nr, _("Noise Reduction"), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (sliderbox), ghr->nr);
	gtk_box_pack_start_defaults (GTK_BOX (labelbox), gtk_label_new (_(" NR ")));
	/* SQL */
	adj = gtk_adjustment_new (0.0, -1.0, 0.0, 0.01, 0.1, 0.0);
/*  	gtk_signal_connect (adj, "value-changed", */
/*  			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_sql_cb), */
/*  			    ghr); */
	ghr->sql = gtk_vscale_new (GTK_ADJUSTMENT (adj));
	gtk_scale_set_draw_value (GTK_SCALE (ghr->sql), FALSE);
	gtk_scale_set_digits (GTK_SCALE (ghr->sql), 2);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->sql, _("Squelch Level"), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (sliderbox), ghr->sql);
	gtk_box_pack_start_defaults (GTK_BOX (labelbox), gtk_label_new (_(" SQL ")));
	/* APF */
	adj = gtk_adjustment_new (0.0, -1.0, 0.0, 0.01, 0.1, 0.0);
/*  	gtk_signal_connect (adj, "value-changed", */
/*  			    GTK_SIGNAL_FUNC (gnome_hamlib_rig_apf_cb), */
/*  			    ghr); */
	ghr->apf = gtk_vscale_new (GTK_ADJUSTMENT (adj));
	gtk_scale_set_draw_value (GTK_SCALE (ghr->apf), FALSE);
	gtk_scale_set_digits (GTK_SCALE (ghr->apf), 2);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, ghr->apf, _("APF Level"), NULL);
	gtk_box_pack_start_defaults (GTK_BOX (sliderbox), ghr->apf);
	gtk_box_pack_start_defaults (GTK_BOX (labelbox), gtk_label_new (_(" APF ")));

	/* frame */
	frame = gtk_frame_new (_("Receiver"));
	gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
/*	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN); */
	gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	gtk_widget_show_all (frame);

	return frame;
}


/**
 * gnome_hamlib_rig_get_power:
 * @ghr: A GnomeHamlibRig widget.
 * @power: Pointer to an integer.
 *
 * Used to aquire the power state of the radio controled by
 * the @ghr widget. The power state can be OFF (0), ON (1) or
 * STANDBY (3). Please note that the STANDBY state is not
 * supported by the GnomeHamlibRig widget.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error) or -1 if @ghr is not a valid GnomeHamlibRig widget.
 */
gint
gnome_hamlib_rig_get_power   (GnomeHamlibRig *ghr,
			      gint *power)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return rig_get_powerstat (ghr->rig, (powerstat_t *) power);
}


/**
 * gnome_hamlib_rig_set_power:
 * @ghr: A GnomeHamlibRig widget.
 * @power: Integer representing the pwoer (0=OFF, 1=ON).
 *
 * Used to set the power state of the radio controled by
 * the @ghr widget. NOTE: Besides RIG_POWER_OFF and RIG_POWER_ON,
 * Hamlib also implements RIG_POWER_STANDBY. This is, however not
 * supported by the GnomeHamlibRig widget.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * the value of @power is invalid.
 */
gint
gnome_hamlib_rig_set_power   (GnomeHamlibRig *ghr,
			      gint power)
{
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	if ((power == 0) || (power == 1)) {
		retcode = rig_set_powerstat (ghr->rig, power);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ghr->power), power);
	} else
		return -2;
	       
	return retcode;
}


/**
 * gnome_hamlib_rig_get_mode:
 * @ghr: A GnomeHamlibRig widget.
 * @mode: Pointer to an integer.
 *
 * Used to aquire the mode of the radio controled by
 * the @ghr widget. @mode can be 0, 1, 2, 4, 8 ... (see <hamlib/rig.h>
 * for details).
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error) or -1 if @ghr is not a valid GnomeHamlibRig widget.
 */
gint
gnome_hamlib_rig_get_mode    (GnomeHamlibRig *ghr,
			      gint *mode)
{
	pbwidth_t wd;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return rig_get_mode (ghr->rig, RIG_VFO_CURR, (rmode_t *) mode, &wd);
}


/**
 * gnome_hamlib_rig_set_mode:
 * @ghr: A GnomeHamlibRig widget.
 * @mode: An integer.
 *
 * Set the mode of the radio controled by the @ghr widget to @mode.
 * @mode can be 0, 1, 2, 4, 8 ... (see <hamlib/rig.h> for details).
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @mode is not a valid mode.
 */
gint
gnome_hamlib_rig_set_mode    (GnomeHamlibRig *ghr,
			      gint mode)
{
	gint retcode;
	rmode_t oldmode;
	pbwidth_t pbwidth;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ((mode%2) || (mode==1), -2);

	/* get current mode */
	retcode = rig_get_mode (ghr->rig, RIG_VFO_CURR, &oldmode, &pbwidth);
	/* set new mode */
	retcode = rig_set_mode (ghr->rig, RIG_VFO_CURR, mode, pbwidth);
	if (retcode == RIG_OK) {
		/* update MODE selector */
		gtk_menu_set_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (ghr->mode))),
				     rint (log (mode)/log (2)));
	}
	return retcode;
}


/**
 * gnome_hamlib_rig_get_filter:
 * @ghr: A GnomeHamlibRig widget.
 * @filter: Pointer to an integer.
 *
 * Get the current filter setting of the radio controled by the @ghr widget.
 * @filter is 0 for WIDE, 1 for NORMAL and 2 for NARROW.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error) or -1 if @ghr is not a valid GnomeHamlibRig widget.
 */
gint
gnome_hamlib_rig_get_filter  (GnomeHamlibRig *ghr,
			      gint *filter)
{
	gint retcode;
	rmode_t mode;
	pbwidth_t pbwidth;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	retcode = rig_get_mode (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, &mode, &pbwidth);
	if (retcode == RIG_OK) {
		if (pbwidth == rig_passband_wide (GNOME_HAMLIB_RIG (ghr)->rig, mode))
			*filter = 0;
		if (pbwidth == rig_passband_normal (GNOME_HAMLIB_RIG (ghr)->rig, mode))
			*filter = 1;
		if (pbwidth == rig_passband_narrow (GNOME_HAMLIB_RIG (ghr)->rig, mode))
			*filter = 2;
	}

	return retcode;
}


/**
 * gnome_hamlib_rig_set_filter:
 * @ghr: A GnomeHamlibRig widget.
 * @filter: An integer.
 *
 * Set the filter of the radio controled by the @ghr widget to @filter.
 * @filter should be 0 for WIDE, 1 for NORMAL or 2 for NARROW.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @filter is not a valid filter.
 */
gint
gnome_hamlib_rig_set_filter  (GnomeHamlibRig *ghr,
			      gint filter)
{
	gint retcode;
	rmode_t mode;
	pbwidth_t pbwidth;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ((filter >= 0) && (filter <=2), -2);


	/* Passband can only be set via the rig_set_mode API call */
	retcode = rig_get_mode (ghr->rig, RIG_VFO_CURR, &mode, &pbwidth);
	g_return_val_if_fail (retcode == RIG_OK, retcode);
	switch (filter) {
	case 0: pbwidth = rig_passband_wide (ghr->rig, mode);
		break;
	case 1: pbwidth = rig_passband_normal (ghr->rig, mode);
		break;
	case 2: pbwidth = rig_passband_narrow (ghr->rig, mode);
		break;
	default: /* THIS SHOULD NEVER HAVE HAPPENED ERROR */
	}
	retcode = rig_set_mode (ghr->rig, RIG_VFO_CURR, mode, pbwidth);
	if (retcode == RIG_OK) {
		/* update FILTER selector */
		gtk_menu_set_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (ghr->filter))),
				     filter);
	}

	return retcode;
}


/**
 * gnome_hamlib_rig_get_agc:
 * @ghr: A GnomeHamlibRig widget.
 * @agc: Pointer to an integer.
 *
 * Get the current AGC setting of the radio controled by the @ghr widget.
 * @agc is 0 for OFF, 1 for FAST, 2 for MEDIUM and 3 for SLOW. Note that
 * the Hamlib definitions for these values are RIG_AGC_OFF, RIG_AGC_SUPERFAST,
 * RIG_AGC_FAST and RIG_AGC_SLOW, thus omitting the MEDIUM definition.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -3 if the
 * radio doesn't support AGC readback.
 */
gint
gnome_hamlib_rig_get_agc     (GnomeHamlibRig *ghr,
			      gint *agc)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail (rig_has_get_level (ghr->rig, RIG_LEVEL_AGC), -3);

	/* try to get AGC setting */
	retcode = rig_get_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_AGC, &val);
	if (retcode != RIG_OK)
		*agc = -1;
	else
		*agc = val.i;

	return retcode;
}


/**
 * gnome_hamlib_rig_set_agc:
 * @ghr: A GnomeHamlibRig widget.
 * @agc: An integer.
 *
 * Set the AGC of the radio controled by the @ghr widget to @agc.
 * @agc should be 0 for OFF, 1 for FAST, 2 for MEDIUM and 3 for
 * SLOW. Note that
 * the Hamlib definitions for these values are RIG_AGC_OFF, RIG_AGC_SUPERFAST,
 * RIG_AGC_FAST and RIG_AGC_SLOW, thus omitting the MEDIUM definition.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget, -2 if
 * @agc is not a valid AGC setting or -3 if the radio doesn't support the
 * AGC setting.
 */
gint
gnome_hamlib_rig_set_agc     (GnomeHamlibRig *ghr,
			      gint agc)
{
	gint retcode;
	value_t val;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ((agc >= 0) && (agc <= 3), -2);
	g_return_val_if_fail (rig_has_set_level (ghr->rig, RIG_LEVEL_AGC), -3);

	val.i = agc;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_AGC, val);
	if (retcode == RIG_OK) {
		/* update AGC selector */
		gtk_menu_set_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (ghr->agc))),
				     agc);
	}

	return retcode;
}


/**
 * gnome_hamlib_rig_get_af:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the current AF Gain level. The returned value is a float between 0 and 1,
 * or a negative value if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A float between 0 and 1 or -1.
 */
gfloat
gnome_hamlib_rig_get_af      (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return ((gtk_range_get_adjustment (GTK_RANGE (ghr->af)))->value);
}


/**
 * gnome_hamlib_rig_set_af:
 * @ghr: A GnomeHamlibRig widget.
 * @af: A float between 0 and 1.
 *
 * Set the AF Gain level of @ghr to @af.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @af is not a valid AF setting.
 */
gint
gnome_hamlib_rig_set_af      (GnomeHamlibRig *ghr,
			      gfloat af)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ( (af>=0.00) && (af<=1.00), -2);

	val.f = af;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_AF, val);
	if (retcode == RIG_OK)
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->af)), af);

	return retcode;
}


/**
 * gnome_hamlib_rig_get_rf:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the current RF Gain level. The returned value is a float between 0 and 1,
 * or a negative value if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A float between 0 and 1 or -1.
 */
gfloat
gnome_hamlib_rig_get_rf      (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return ((gtk_range_get_adjustment (GTK_RANGE (ghr->rf)))->value);
}


/**
 * gnome_hamlib_rig_set_rf:
 * @ghr: A GnomeHamlibRig widget.
 * @rf: A float between 0 and 1.
 *
 * Set the RF Gain level of @ghr to @rf.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @rf is not a valid RF setting.
 */
gint
gnome_hamlib_rig_set_rf      (GnomeHamlibRig *ghr,
			      gfloat rf)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ( (rf>=0.00) && (rf<=1.00), -2);

	val.f = rf;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_RF, val);
	if (retcode == RIG_OK)
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->rf)), rf);

	return retcode;

}


/**
 * gnome_hamlib_rig_get_nr:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the current Noise Reduction level. The returned value is a float between 0 and 1,
 * or a negative value if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A float between 0 and 1 or -1.
 */
gfloat
gnome_hamlib_rig_get_nr      (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return ((gtk_range_get_adjustment (GTK_RANGE (ghr->nr)))->value);
}


/**
 * gnome_hamlib_rig_set_nr:
 * @ghr: A GnomeHamlibRig widget.
 * @nr: A float between 0 and 1.
 *
 * Set the Noise Reduction level of @ghr to @nr.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @nr is not a valid NR setting.
 */
gint
gnome_hamlib_rig_set_nr      (GnomeHamlibRig *ghr,
			      gfloat nr)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ( (nr>=0.00) && (nr<=1.00), -2);

	val.f = nr;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_NR, val);
	if (retcode == RIG_OK)
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->nr)), nr);

	return retcode;
}


/**
 * gnome_hamlib_rig_get_sql:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the current Squelch level. The returned value is a float between 0 and 1,
 * or a negative value if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A float between 0 and 1 or -1.
 */
gfloat
gnome_hamlib_rig_get_sql     (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return ((gtk_range_get_adjustment (GTK_RANGE (ghr->sql)))->value);
}


/**
 * gnome_hamlib_rig_set_sql:
 * @ghr: A GnomeHamlibRig widget.
 * @sql: A float between 0 and 1.
 *
 * Set the Squelch level of @ghr to @sql.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @sql is not a valid Squelch setting.
 */
gint
gnome_hamlib_rig_set_sql     (GnomeHamlibRig *ghr,
			      gfloat sql)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ( (sql>=0.00) && (sql<=1.00), -2);

	val.f = sql;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_SQL, val);
	if (retcode == RIG_OK)
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->sql)), sql);

	return retcode;
}


/**
 * gnome_hamlib_rig_get_apf:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the current APF level, whatever that might be... The returned value is a float between 0 and 1,
 * or a negative value if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A float between 0 and 1 or -1.
 */
gfloat
gnome_hamlib_rig_get_apf     (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return ((gtk_range_get_adjustment (GTK_RANGE (ghr->apf)))->value);
}


/**
 * gnome_hamlib_rig_set_apf:
 * @ghr: A GnomeHamlibRig widget.
 * @apf: A float between 0 and 1.
 *
 * Set the APF Gain level of @ghr to @apf.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @apf is not a valid APF setting.
 */
gint 
gnome_hamlib_rig_set_apf     (GnomeHamlibRig *ghr,
			      gfloat apf)
{
	value_t val;
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail ( (apf>=0.00) && (apf<=1.00), -2);

	val.f = apf;
	retcode = rig_set_level (ghr->rig, RIG_VFO_CURR, RIG_LEVEL_APF, val);
	if (retcode == RIG_OK)
		gtk_adjustment_set_value (gtk_range_get_adjustment (GTK_RANGE (ghr->apf)), apf);

	return retcode;
}


/**
 * gnome_hamlib_rig_get_freq:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the frequency of the current VFO. The returned value is a 64-bit unsigned integer
 * representing the frequency in Hz or 0 if @ghr is not a valid GnomeHamlibRig widget.
 *
 * Returns: A 64-bit unsigned integer representing the frequency in Hz.
 */
guint64
gnome_hamlib_rig_get_freq    (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (ghr != NULL, 0);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), 0);

	return ghr->freq1;
}


/**
 * gnome_hamlib_rig_set_freq:
 * @ghr: A GnomeHamlibRig widget.
 * @freq: A 64-bit unsigned integer representing the frequency in Hz.
 *
 * Set the frequency of @ghr to @freq.
 *
 * Returns: A positive integer representing the Hamlib error code (0 for
 * no error), -1 if @ghr is not a valid GnomeHamlibRig widget or -2 if
 * @freq is not a valid frequency setting (eg. 10 Hz).
 */
gint
gnome_hamlib_rig_set_freq    (GnomeHamlibRig *ghr,
			      guint64 freq)
{
	gint retcode;

	g_return_val_if_fail (ghr != NULL, -1);
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail (freq >= 30000, -2);

	retcode = rig_set_freq (ghr->rig, RIG_VFO_CURR, freq);
	if (retcode == RIG_OK)
		gnome_hamlib_rig_set_lcd_digits (ghr, freq);

	return retcode;
}



/**
 * gnome_hamlib_rig_get_freq_timeout:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the timer interval at which the frequency display is updated. This is the same
 * interval at which the frequency of the current VFO is read back from the radio.
 * The returned value is in milliseconds or 0 if the timer is not running and no update
 * is performed.
 *
 * Returns: An integer value representing the time interval in milliseconds, 0 if the
 * timer is not running or -1 if @ghr is not a valid GnomeHamlibRig widget.
 */
gint
gnome_hamlib_rig_get_freq_timeout (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), 1);

	return (gint) ghr->freqtimer_val;
}


/**
 * gnome_hamlib_rig_set_freq_timeout:
 * @ghr: A GnomeHamlibRig widget.
 * @timeout: An unsigned integer representing the update interval in ms.
 *
 * This function has three puposes. The first is to enable the frequency
 * readback timer and the second is to change the time interval between updates
 * if the timer is already running. Finally, if @timeout is 0, the timer
 * will be disabled and no update of the frequency display will be performed,
 * except when the user changes the frequency from the program.
 * The valid range for @timeout is between 300...10000 milliseconds.
 *
 * Returns: 0 if the timer has been (re)started successfully, -1 if @ghr is not
 * a valid GnomeHamlibRig widget or -2 if @timeout is out of the valid range.
 */
gint  
gnome_hamlib_rig_set_freq_timeout (GnomeHamlibRig *ghr, 
				   guint timeout)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), 1);
	g_return_val_if_fail (((timeout >= 300) || (timeout == 0)) && (timeout <= 10000), -2);

	/* We need to stop the timer no matter what */
	if (ghr->freqtimer_id)
		gtk_timeout_remove (ghr->freqtimer_id);

	if (!timeout) {
		/* caller wants to stop the timer */
		rig_debug (RIG_DEBUG_TRACE, "*** Gnome RIG: stopping frequency readback timer\n");
		ghr->freqtimer_id = 0;
		ghr->freqtimer_val = 0;

		return 0;
	}

	rig_debug (RIG_DEBUG_TRACE, "*** Gnome RIG: starting frequency readback timer\n");

	/* otherwise start a new timer */
	ghr->freqtimer_id = gtk_timeout_add (timeout,
					     gnome_hamlib_rig_freq_timeout,
					     ghr);

	/* I hope we can assume, that if the timer has been started successfully, the
	   timer ID will be non-zero.
	*/
	if (!ghr->freqtimer_id) {
		ghr->freqtimer_val = 0;

		return -3;
	}

	/* Everything is all right, store the timer delay */
	ghr->freqtimer_val = timeout;

	return 0;
}



static gint
gnome_hamlib_rig_freq_timeout (gpointer ghr)
{
	/* This is the timeout function that reads
	   the frequency of the radio periodically
	   and updates the LCD panel.
	*/
	gint retcode;
	freq_t freq;

       	/* get frequency from RIG */
	retcode = rig_get_freq (GNOME_HAMLIB_RIG (ghr)->rig, RIG_VFO_CURR, &freq);
	if (retcode != RIG_OK) {
		g_warning (_("gnome-hamlib-rig.c: Could not get frequency. Stopping timer.\n"));
		return FALSE;
	}
	gnome_hamlib_rig_set_lcd_digits (ghr, freq);

 	return TRUE;
}


/**
 * gnome_hamlib_rig_get_slider_timeout:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the timer interval at which the sliders are updated.
 * This is the same interval at which the values are read back from the radio.
 * The returned value is in milliseconds or 0 if the timer is not running and no update
 * is performed.
 *
 * Returns: A positive integer value representing the time interval in milliseconds, 0 if the
 * timer is not running or -1 if @ghr is not a valid GnomeHamlibRig widget. 
 */
gint
gnome_hamlib_rig_get_slider_timeout (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return (gint) ghr->slidertimer_val;
}


/**
 * gnome_hamlib_rig_set_slider_timeout:
 * @ghr: A GnomeHamlibRig widget.
 * @timeout: An unsigned integer representing the update interval in ms.
 *
 * This function has three puposes. The first is to enable the readback timer
 * for the different controls and the second is to change the time interval
 * between updates if the timer is already running. Finally, if @timeout is 0,
 * the timer will be disabled and no update of the buttons and menus will be
 * performed, except when the user changes the settings from the program.
 * The valid range for @timeout is between 400...10000 milliseconds.
 *
 * Returns: 0 if the timer has been (re)started successfully, -1 if @ghr is not a valid
 * GnomeHamlibRig widget or -2 if @timeout is out of the valid range.
 */
gint
gnome_hamlib_rig_set_slider_timeout (GnomeHamlibRig *ghr, 
				     guint timeout)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail (((timeout >= 400) || (timeout == 0)) && (timeout <= 10000), -2);

	return 0;
}


static gint
gnome_hamlib_rig_slider_timeout (gpointer ghr)
{
	/* This is the timeout function that reads
	   level settings of the radio periodically
	   and updates the slider widgets.
	*/

 	return TRUE;
}



/**
 * gnome_hamlib_rig_get_misc_timeout:
 * @ghr: A GnomeHamlibRig widget.
 *
 * Get the timer interval at which the buttons and menus are updated.
 * This is the same interval at which the values are read back from the radio.
 * The returned value is in milliseconds or 0 if the timer is not running and no update
 * is performed.
 *
 * Returns: A positive integer value representing the time interval in milliseconds, 0 if the
 * timer is not running or -1 if @ghr is not a valid GnomeHamlibRig widget. 
 */
gint
gnome_hamlib_rig_get_misc_timeout (GnomeHamlibRig *ghr)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);

	return (gint) ghr->misctimer_val;
}


/**
 * gnome_hamlib_rig_set_misc_timeout:
 * @ghr: A GnomeHamlibRig widget.
 * @timeout: An unsigned integer representing the update interval in ms.
 *
 * This function has three puposes. The first is to enable the readback timer
 * for the different controls and the second is to change the time interval
 * between updates if the timer is already running. Finally, if @timeout is 0,
 * the timer will be disabled and no update of the buttons and menus will be
 * performed, except when the user changes the settings from the program.
 * The valid range for @timeout is between 700...10000 milliseconds.
 *
 * Returns: 0 if the timer has been (re)started successfully, -1 if @ghr is not a valid
 * GnomeHamlibRig widget or -2 if @timeout is out of the valid range.
 */
gint
gnome_hamlib_rig_set_misc_timeout (GnomeHamlibRig *ghr, 
				   guint timeout)
{
	g_return_val_if_fail (GNOME_IS_HAMLIB_RIG (ghr), -1);
	g_return_val_if_fail (((timeout >= 700) || (timeout == 0)) && (timeout <= 10000), -2);

	return 0;
}


static gint
gnome_hamlib_rig_misc_timeout (gpointer ghr)
{
	/* This is the timeout function that reads
	   misc. values from the radio periodically
	   and updates the button and menu widgets.
	*/

 	return TRUE;
}
