/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  grig-config-rig.c:  Gnome RIG configuration module (radios).

  Copyright (C)  2002  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <hamlib/rig.h>
#include "grig-druid-rig.h"
#include "grig-utils.h"
#include "grig-config-rig.h"
#include "grig-config.h"
#include "hamlib-widgets.h"
#include "riglist-util.h"


extern GConfClient *confclient;     /* shared client connection to GConf in main.c */

/* widgets */
static GtkWidget *newb,*editb,*delb,*defb,*testb,*clist;

/* other */
static gchar *titles[] = { N_(" # "), N_(" Brand "), N_(" Model "), N_(" Port "), N_(" Default ") };

/* private function prototypes */
static gboolean grig_config_rig_exists         (guint i);
static void     grig_config_rig_row_clicked_cb (GtkCList *clist,
						gint *row,
						gint *column,
						GdkEventButton *button,
						gpointer data);
static void     grig_config_rig_add_cb         (GtkWidget *button, gpointer clist);
static void     grig_config_rig_delete_cb      (GtkWidget *button, gpointer clist);
static void     grig_config_rig_edit_cb        (GtkWidget *button, gpointer clist);
static void     grig_config_rig_default_cb     (GtkWidget *button, gpointer clist);
static void     grig_config_rig_test_cb        (GtkWidget *button, gpointer clist);


GtkWidget *
grig_config_rig_create ()
{
	GtkWidget *butbox,*hbox;
	GtkWidget *swin;
	guint i,defrig,max;
	gchar *p1,*p2,*p3,*buff,**bufv;


	/* CList widget */
	clist = gtk_clist_new_with_titles (5, titles);
	gtk_clist_column_titles_passive (GTK_CLIST (clist));

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
			    GTK_SIGNAL_FUNC (grig_config_rig_row_clicked_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (clist), "unselect-row",
			    GTK_SIGNAL_FUNC (grig_config_rig_row_clicked_cb),
			    NULL);

	/* adjust columns automatically */
	for (i=0; i<4; i++)
		gtk_clist_set_column_auto_resize (GTK_CLIST (clist), i, TRUE);

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					 GTK_POLICY_AUTOMATIC,
					 GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), clist);

	/* default radio */
	p1 = g_strdup_printf ("%s/default", GRIG_CONFIG_RIG_DIR);
	defrig = gconf_client_get_int (confclient, p1, NULL);
	g_free (p1);

	/* number of radios */
	max = gconf_client_get_int (confclient, GRIG_CONFIG_RIG_DIR "/number", NULL);

	/* read radios */
	for (i=0; (i<max) && (grig_config_rig_exists (i)); i++) {
		p1 = g_strdup_printf ("%s/%i/Brand", GRIG_CONFIG_RIG_DIR, i);
		p2 = g_strdup_printf ("%s/%i/Model", GRIG_CONFIG_RIG_DIR, i);
		p3 = g_strdup_printf ("%s/%i/port", GRIG_CONFIG_RIG_DIR, i);
		buff = g_strdup_printf ("%i;%s;%s;%s; ", i,
					gconf_client_get_string (confclient, p1, NULL),
					gconf_client_get_string (confclient, p2, NULL),
					gconf_client_get_string (confclient, p3, NULL));
		bufv = g_strsplit (buff, ";", 0);
		gtk_clist_append (GTK_CLIST (clist), bufv);
		g_strfreev (bufv);
		g_free (buff);
		g_free (p1);
		g_free (p2);
		g_free (p3);
		/* attach rigid to row */
		p1 = g_strdup_printf ("%s/%i/ID", GRIG_CONFIG_RIG_DIR, i);
		gtk_clist_set_row_data (GTK_CLIST (clist), i,
					GINT_TO_POINTER (gconf_client_get_int (confclient,
									       p1,
									       NULL)));
		g_free (p1);

		/* Is this the default radio? */
		if (defrig == i)
			gtk_clist_set_text (GTK_CLIST (clist), i, 4, _("YES"));
		else
			gtk_clist_set_text (GTK_CLIST (clist), i, 4, " ");
	}


	/* buttons */
	newb = grig_utils_pixmap_button (_("Add"), GNOME_STOCK_PIXMAP_ADD,
					 _("Add a new radio to the list"));
	delb = grig_utils_pixmap_button (_("Delete"), GNOME_STOCK_PIXMAP_REMOVE,
					 _("Delete the selected radio from the list"));
	editb = grig_utils_pixmap_button (_("Edit"), GNOME_STOCK_PIXMAP_PROPERTIES,
					  _("Edit the selected radio"));
	defb = grig_utils_pixmap_button (_("Default"), GNOME_STOCK_PIXMAP_ABOUT,
					 _("Make the selected radio default"));
	testb = grig_utils_pixmap_button (_("Test"), GNOME_STOCK_PIXMAP_EXEC,
					  _("Test the selected radio")); 

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (newb), "clicked",
			    GTK_SIGNAL_FUNC (grig_config_rig_add_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (delb), "clicked",
			    GTK_SIGNAL_FUNC (grig_config_rig_delete_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (editb), "clicked",
			    GTK_SIGNAL_FUNC (grig_config_rig_edit_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (defb), "clicked",
			    GTK_SIGNAL_FUNC (grig_config_rig_default_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (testb), "clicked",
			    GTK_SIGNAL_FUNC (grig_config_rig_test_cb),
			    clist);

	/* should only be clickable when a row is selected */
	gtk_widget_set_sensitive (delb, FALSE);
	gtk_widget_set_sensitive (editb, FALSE);
	gtk_widget_set_sensitive (defb, FALSE);
	gtk_widget_set_sensitive (testb, FALSE);

	/* vertical button box */
	butbox = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (butbox), GTK_BUTTONBOX_SPREAD);

	gtk_box_pack_start_defaults (GTK_BOX (butbox), newb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), delb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), editb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), defb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), testb);


	/* main horizontal box */
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);

	gtk_box_pack_start (GTK_BOX (hbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (hbox), butbox, FALSE, FALSE, 0);

	return hbox;
}

void
grig_config_rig_close ()
{
	gtk_widget_destroy (newb);
	gtk_widget_destroy (delb);
	gtk_widget_destroy (editb);
	gtk_widget_destroy (defb);
	gtk_widget_destroy (testb);
	gtk_widget_destroy (clist);
}


void
grig_config_rig_apply ()
{
	gint i,rows;
	gchar *key,*buff[] = { " " };
	gboolean has_default = FALSE;
	gboolean ok = TRUE;

	/* save number of rigs */
	rows = (GTK_CLIST (clist)->rows);
	key = g_strdup_printf ("%s/number", GRIG_CONFIG_RIG_DIR);
	gconf_client_set_int (confclient, key, rows, NULL);
	g_free (key);

	for (i=0; i<rows; i++) {
		/* rigid */
		key = g_strdup_printf ("%s/%d/ID", GRIG_CONFIG_RIG_DIR, i);
		ok = ok && gconf_client_set_int (confclient, key,
						 GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (clist), i)),
						 NULL);
		g_free (key);

		/* brand */
		gtk_clist_get_text (GTK_CLIST (clist), i, 1, buff);
		key = g_strdup_printf ("%s/%d/Brand", GRIG_CONFIG_RIG_DIR, i);
		ok = ok && gconf_client_set_string (confclient, key, buff[0], NULL);
		g_free (key);

		/* Model */
		gtk_clist_get_text (GTK_CLIST (clist), i, 2, buff);
		key = g_strdup_printf ("%s/%d/Model", GRIG_CONFIG_RIG_DIR, i);
		ok = ok && gconf_client_set_string (confclient, key, buff[0], NULL);
		g_free (key);

		/* port */
		gtk_clist_get_text (GTK_CLIST (clist), i, 3, buff);
		key = g_strdup_printf ("%s/%d/port", GRIG_CONFIG_RIG_DIR, i);
		ok = ok && gconf_client_set_string (confclient, key, buff[0], NULL);
		g_free (key);

		/* is this one the default? */
		gtk_clist_get_text (GTK_CLIST (clist), i, 4, buff);
		if (!g_strcasecmp (buff[0], "YES")) {
			key = g_strdup_printf ("%s/default", GRIG_CONFIG_RIG_DIR);
			ok = ok && gconf_client_set_int (confclient, key, i, NULL);
			g_free (key);
			has_default = TRUE;
		}
		if (!ok) {
			rig_debug (RIG_DEBUG_TRACE, _("*** Gnome RIG: Error saving preferences\n"));
			ok = TRUE;
		}
	}
	
	/* set default rig to 0 if not already set */
	if (!has_default) {
		key = g_strdup_printf ("%s/default", GRIG_CONFIG_RIG_DIR);
		gconf_client_set_int (confclient, key, 0, NULL);
		g_free (key);
	}

	gconf_client_suggest_sync (confclient, NULL);
}


void
grig_config_rig_cancel ()
{
/*  	if (newb) */
/*  		gtk_widget_destroy (newb); */
/*  	if (delb) */
/*  		gtk_widget_destroy (delb); */
/*  	if (editb) */
/*  		gtk_widget_destroy (editb); */
/*  	if (defb) */
/*  		gtk_widget_destroy (defb); */
/*  	if (testb) */
/*  		gtk_widget_destroy (testb); */
/*  	if (clist) */
/*  		gtk_widget_destroy (clist); */
}

void
grig_config_rig_help ()
{
}



static gboolean
grig_config_rig_exists (guint i)
{
	/* This function checks whether the Gnome RIG
	   configuration contains a radio with grig
	   number 'i'. This is the case if there is
	   a /apps/grig/radios/i directory in the
	   GConf path.
	*/
	gchar *path;
	gboolean result;

	path = g_strdup_printf ("%s/%i", GRIG_CONFIG_RIG_DIR, i);
	result = gconf_client_dir_exists (confclient, path, NULL);
	g_free (path);

	return result;
}


static void
grig_config_rig_row_clicked_cb (GtkCList *clist,
				gint *row,
				gint *column,
				GdkEventButton *button,
				gpointer data)
{
	/* This function is called when a row is selected
	   or unselected. It check whether the CList has a
	   valid selection and enables the buttons if so.
	*/
	GList *selection = clist->selection;

	if (selection) {
		/* Default radio should not be deleted */
		gtk_widget_set_sensitive (delb, TRUE);
		gtk_widget_set_sensitive (editb, TRUE);
		gtk_widget_set_sensitive (defb, TRUE);
/*  		gtk_widget_set_sensitive (testb, TRUE); */
	}
	else {
		gtk_widget_set_sensitive (delb, FALSE);
		gtk_widget_set_sensitive (editb, FALSE);
		gtk_widget_set_sensitive (defb, FALSE);
/*  		gtk_widget_set_sensitive (testb, FALSE); */
	}
}


static void   
grig_config_rig_add_cb         (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Add" button. It opens a new dialog to select a
	   new radio.
	*/
	GtkWidget *dialog,*rigclist,*swin,*entry,*frame;
	GSList *riglist;
	gint diabut=-1,row=-1,rigid,i;
	gchar *newrig[5],*buff[] = {" "};

	/* get list of drivers */
	riglist = riglist_get_list ();

	/* create a columned list */
	rigclist = hamlib_riglist_widget_new (riglist);

	/* pack the clist into a scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), rigclist);
	gtk_widget_show_all (swin);

	/* Entry for port */
	entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (entry), "/dev/ttyS0");
	frame = gtk_frame_new (_("Port"));
	gtk_container_add (GTK_CONTAINER (frame), entry);
	gtk_widget_show_all (frame);

	/* create dialog */
	dialog = gnome_dialog_new (_("Select a radio"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, FALSE, FALSE, 0);

	/* make list more visible */
	gtk_widget_set (dialog, "height", 300, NULL);

	diabut = gnome_dialog_run (GNOME_DIALOG (dialog));

	switch (diabut) {
	case 0: /* OK */
		if (GTK_CLIST (rigclist)->selection)
			row = GPOINTER_TO_INT (GTK_CLIST (rigclist)->selection->data);
		else
			break;

		/* get rig info from rigclist */

		/* row number */
		newrig[0] = g_strdup_printf ("%d", (GTK_CLIST (clist)->rows));

		/* brand */
		gtk_clist_get_text (GTK_CLIST (rigclist), row, 1, buff);
		newrig[1] = g_strdup (buff[0]);

		/* model */
		gtk_clist_get_text (GTK_CLIST (rigclist), row, 2, buff);
		newrig[2] = g_strdup (buff[0]);

		/* port */
		newrig[3] = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

		/* default */
		newrig[4] = g_strdup (" ");

		/* rigid */
		rigid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rigclist), row));

		/* add new rig */
		row = gtk_clist_append (GTK_CLIST (clist), newrig);
		gtk_clist_set_row_data (GTK_CLIST (clist), row,
					GINT_TO_POINTER (rigid));
		/* free some memory */
		for (i=0; i<5; i++)
			if (newrig[i])
			    g_free (newrig[i]);

		/* trigger config changed */
		grig_config_changed ();
		break;
	case 1: /* Cancel */
		break;
	default:
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}



static void
grig_config_rig_delete_cb      (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Delete" button. It deletes the selected radio entry
	   from the clist.
	*/
	GList *selection = (GTK_CLIST (clist))->selection;
	gint num;
	gchar *buff,*dummy[] = { " "," "," "," "," "};
	gboolean def = FALSE;

	if (selection) {
		/* is this the default rig? */
		gtk_clist_get_text (GTK_CLIST (clist),
				    GPOINTER_TO_UINT (selection->data), 4,
				    dummy);
		g_print ("%s\n", dummy[0]);
		def = !g_strcasecmp (dummy[0], "YES");

		gtk_clist_remove (GTK_CLIST (clist),
				  GPOINTER_TO_UINT (selection->data));
		num = (GTK_CLIST (clist)->rows);
		while (--num >= 0) {
			buff = g_strdup_printf ("%d", num);
			gtk_clist_set_text (GTK_CLIST (clist), num, 0, buff);
			g_free (buff);
			/* make radio 0 the default if needed */
			if (!num && def)
				gtk_clist_set_text (GTK_CLIST (clist), num, 4, "YES");
		}
		/* trigger signal */
		grig_config_changed ();
	}
}


static void
grig_config_rig_edit_cb        (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Edit" button. It opens a new dialog to edit the
	   selected radio.
	*/
	GtkWidget *dialog,*rigclist,*swin,*entry,*frame;
	GSList *riglist;
	gint crow,crigid;
	gint diabut=-1,row=-1,rigid,i;
	gchar *newrig,*buff[] = {" "};

	/* get selected row and rigid (needed to select the rig in the popup dialog */
	crow = GPOINTER_TO_INT (GTK_CLIST (clist)->selection->data);
	crigid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (clist), crow));

	/* get list of drivers */
	riglist = riglist_get_list ();

	/* create a columned list */
	rigclist = hamlib_riglist_widget_new (riglist);

	/* select the current rig */
	for (i=0; i<(GTK_CLIST (rigclist))->rows; i++) {
		if (crigid == GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rigclist), i))) {
			gtk_clist_select_row (GTK_CLIST (rigclist), i, 0);
			/* stop iteration */
			i = (GTK_CLIST (rigclist))->rows;
		}
	}
	/* pack the clist into a scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), rigclist);
	gtk_widget_show_all (swin);

	/* Entry for port */
	entry = gtk_entry_new ();

	/* get current port */
	gtk_clist_get_text (GTK_CLIST (clist), crow, 3, buff);
	gtk_entry_set_text (GTK_ENTRY (entry), buff[0]);
	frame = gtk_frame_new (_("Port"));
	gtk_container_add (GTK_CONTAINER (frame), entry);
	gtk_widget_show_all (frame);

	/* create dialog */
	dialog = gnome_dialog_new (_("Select a radio"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, FALSE, FALSE, 0);

	/* make list more visible */
	gtk_widget_set (dialog, "height", 300, NULL);

	diabut = gnome_dialog_run (GNOME_DIALOG (dialog));

	switch (diabut) {
	case 0: /* OK */
		if (GTK_CLIST (rigclist)->selection)
			row = GPOINTER_TO_INT (GTK_CLIST (rigclist)->selection->data);
		else
			break;

		/* get rig info from rigclist */

		/* brand */
		gtk_clist_get_text (GTK_CLIST (rigclist), row, 1, buff);
		newrig = g_strdup (buff[0]);
		gtk_clist_set_text (GTK_CLIST (clist), crow, 1, newrig);
		g_free (newrig);

		/* model */
		gtk_clist_get_text (GTK_CLIST (rigclist), row, 2, buff);
		newrig = g_strdup (buff[0]);
		gtk_clist_set_text (GTK_CLIST (clist), crow, 2, newrig);
		g_free (newrig);

		/* port */
		newrig = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
		gtk_clist_set_text (GTK_CLIST (clist), crow, 3, newrig);
		g_free (newrig);

		/* rigid */
		rigid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rigclist), row));
		gtk_clist_set_row_data (GTK_CLIST (clist), crow,
					GINT_TO_POINTER (rigid));

		/* trigger config changed */
		grig_config_changed ();
		break;
	case 1: /* Cancel */
		break;
	default:
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}


static void 
grig_config_rig_default_cb     (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Default" button. It makes the selected radio the
	   default radio.
	*/
	GList *selection = (GTK_CLIST (clist)->selection);
	gint row,i;

	if (selection) {
		row = GPOINTER_TO_INT (selection->data);
		for (i=0; i < (GTK_CLIST (clist)->rows); i++)
			gtk_clist_set_text (GTK_CLIST (clist), i, 4,
					    (i == row) ? "YES" : " ");
		/* trigger signal */
		grig_config_changed ();
	} 
}


static void
grig_config_rig_test_cb        (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Test" button. It performs a test on the selected
	   radio.
	*/
}
