/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  main.c:  Main program file for Gnome RIG.
           Gnome RIG is a GUI for the Ham Radio Control Libraries.

  Copyright (C)  2001, 2002  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "grig-druid.h"
#include "grig-druid-rig.h"
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
#  include "grig-druid-ant.h"
#endif
#include "hamlib-widgets.h"
#include "riglist-util.h" 

#include "gnome-hamlib-rig.h"

#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_RIG_H)
#  include <hamlib/rig.h>
#endif


GtkWidget *app;
GConfClient *confclient;     /* shared client connection to GConf */
extern GnomeUIInfo grig_menubar[];  /* menubar.c */


guint rigid = 1;  /* hamlib-dummy backend */
guint rotid = 1;  /* hamlib-dummy backend */
gchar *rigport,*rotport;


/* command line arguments */
static gint rignum = -1;
static gint listrigs = 0;
static gint rundruid = 0;
static gint enabletimer = 0;

#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
static gint rotnum = -1;
static gint listrots = 0;
#endif

static const struct poptOption grig_options[] =
{
	{ "radio", 'r', POPT_ARG_INT, &rignum, 0,
	  N_("Use the radio with NUMBER (see --list-rigs)"),
	  N_("NUMBER") },
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
	{ "antenna", 'a', POPT_ARG_INT, &rotnum, 0,
	  N_("Use the rotator with NUMBER (see --list-rots)"), 
	  N_("NUMBER") },
#endif
	{ "list-rigs", 'i', POPT_ARG_NONE, &listrigs, 1,
	  N_("Show a list of configured radios and exit"), NULL },
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
	{ "list-rots", 'o', POPT_ARG_NONE, &listrots, 1,
	  N_("Show a list of configured rotators and exit"), NULL },
#endif
	{ "run-druid", 'd', POPT_ARG_NONE, &rundruid, 1,
	  N_("Run first-time configuration druid"), NULL },
	{ "enable-readback", 'e', POPT_ARG_NONE, &enabletimer, 1,
	  N_("Enable frequency readback (experimental)"), NULL },
	{ NULL, '\0', 0, NULL, 0 }
};



/* private function prototypes */
static gint grig_window_close (GtkWidget *, GdkEvent *, gpointer);
static void grig_destroy      (GtkWidget *, gpointer);


int main (int argc, char *argv[])
{
	GtkWidget *controls;
	gint defrig, defrot, debug, numrigs, i;
	gchar *buff,*model;

	GtkWidget *menushell, *menuitem;
	gint pos;

#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);
#endif

	/* initialize Gnome */
	gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv,
				    grig_options, 0, NULL);

	/* initialize GConf if needed */
	if (!gconf_is_initialized ())
		gconf_init (argc, argv, NULL);

	/* get default GConf client */
	confclient = gconf_client_get_default ();
	if (!confclient) {
		g_print(_("\nERROR: Could not get GConf client.\n"));
		return 1;
	}

	/* get the number of radios */
	numrigs = gconf_client_get_int (confclient,
					GRIG_CONFIG_RIG_DIR "/number",
					NULL);

	/* get default rig id */
	defrig = gconf_client_get_int (confclient,
				       GRIG_CONFIG_RIG_DIR "/default",
				       NULL);

	/* List available radios? */
	if (listrigs) {
		gchar *brand,*model,*port;
		g_print ("\nFollowing radios are configured:\n\n");
		for (i=0; i<numrigs; i++) {
			buff = g_strdup_printf ("%s/%d/Brand", GRIG_CONFIG_RIG_DIR, i);
			brand = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			buff = g_strdup_printf ("%s/%d/Model", GRIG_CONFIG_RIG_DIR, i);
			model = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			buff = g_strdup_printf ("%s/%d/port", GRIG_CONFIG_RIG_DIR, i);
			port = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			g_print (_("%3d. %s %s on %s"), i, brand, model, port);
			if (defrig == i)
				g_print (_(" (default)"));
			g_print ("\n");
			g_free (brand);
			g_free (model);
			g_free (port);
		}
		g_print (_("\nUse the number in the first row with the -r or --radio option.\n"));
		return 1;
	}

#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)

	/* Set debug level from preferences */
	debug = gconf_client_get_int (confclient,
				      "/apps/grig/hamlib/debug",
				      NULL);
	rig_set_debug (debug);

	/* get default rotator id */
	defrot = gconf_client_get_int (confclient,
				       GRIG_CONFIG_ROT_DIR "/default",
				       NULL);

	/* List available rotators? */
	if (listrots) {
		gchar *brand,*model,*port;
		g_print (_("\nFollowing rotators are configured:\n\n"));
		for (i=0; i<numrigs; i++) {
			buff = g_strdup_printf ("%s/%d/Brand", GRIG_CONFIG_RIG_DIR, i);
			brand = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			buff = g_strdup_printf ("%s/%d/Model", GRIG_CONFIG_RIG_DIR, i);
			model = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			buff = g_strdup_printf ("%s/%d/port", GRIG_CONFIG_RIG_DIR, i);
			port = g_strdup (gconf_client_get_string (confclient, buff, NULL));
			g_free (buff);
			g_print (_("%3d. %s %s on %s"), i, brand, model, port);
			if (defrot == i)
				g_print (_(" (default)"));
			g_print ("\n");
			g_free (brand);
			g_free (model);
			g_free (port);
		}
		g_print (_("\nUse the number in the first row with the -a or --antenna option.\n"));
		return 1;
	}
#endif

	gconf_client_add_dir (confclient, "/apps/grig",
			      GCONF_CLIENT_PRELOAD_RECURSIVE,
			      NULL);

	if ((rignum != -1) && (rignum < numrigs)) {
		/* get the requested rig */
		buff = g_strdup_printf ("%s/%i/ID", GRIG_CONFIG_RIG_DIR, rignum);
		rigid = gconf_client_get_int (confclient, buff, NULL);
		g_free (buff);
		buff = g_strdup_printf ("%s/%i/port", GRIG_CONFIG_RIG_DIR, rignum);
		rigport = gconf_client_get_string (confclient, buff, NULL);
		g_free (buff);
	}
	else {
		/* get default rig */
		buff = g_strdup_printf ("%s/%i/ID", GRIG_CONFIG_RIG_DIR, defrig);
		rigid = gconf_client_get_int (confclient, buff, NULL);
		g_free (buff);
		buff = g_strdup_printf ("%s/%i/port", GRIG_CONFIG_RIG_DIR, defrig);
		rigport = gconf_client_get_string (confclient, buff, NULL);
		g_free (buff);
	}

	/* Run first-time druid if necessary or requested by user */
	if (!rigid || !rigport || !numrigs || rundruid) {
		grig_druid_run ();
		/* and try again */
		defrig = gconf_client_get_int (confclient,
					       GRIG_CONFIG_RIG_DIR "/default",
					       NULL);
		buff = g_strdup_printf ("%s/%i/ID", GRIG_CONFIG_RIG_DIR, defrig);
		rigid = gconf_client_get_int (confclient, buff, NULL);
		g_free (buff);
		buff = g_strdup_printf ("%s/%i/port", GRIG_CONFIG_RIG_DIR, defrig);
		rigport = gconf_client_get_string (confclient, buff, NULL);
		g_free (buff);
	}

	/* get rig model */
	buff = g_strdup_printf ("%s/%i/Model", GRIG_CONFIG_RIG_DIR, defrig);
	model = gconf_client_get_string (confclient, buff, NULL);
	g_free (buff);

	buff = g_strconcat (_("Gnome RIG "), VERSION, ": ", model, NULL);
	app = gnome_app_new (PACKAGE, buff);
	g_free (buff);

	/* window icon */
//	path = g_strconcat (PACKAGE_PIXMAPS_DIR,"/icons/gpredict-icon.png",NULL);
//	gnome_window_icon_set_from_file (GTK_WINDOW (app), path);
//	g_free (path);


	/* create control widget */
	controls = gnome_hamlib_rig_new (rigid, rigport);
	if (!controls) {
		/* create an error dialog and exit */
		GtkWidget *err;

		err = gnome_error_dialog(_("The Radio Control Widget could not be\n"\
					   "created. Please make sure that the\n"\
					   "hamlib backend works properly and try\n"\
					   "again!"));
		gnome_dialog_run_and_close (GNOME_DIALOG (err));
		return 1;
	}

	/* connect delete and destroy signals */
	gtk_signal_connect (GTK_OBJECT (app), "delete_event",
			    GTK_SIGNAL_FUNC (grig_window_close),
			    controls);
	gtk_signal_connect (GTK_OBJECT (app), "destroy",
			    GTK_SIGNAL_FUNC (grig_destroy),
			    controls);

	gnome_app_create_menus (GNOME_APP (app), grig_menubar);
	gnome_app_set_contents (GNOME_APP (app), controls);
	gtk_widget_show_all (app);

	/* Set the correct debug level in the menubar */
	switch (debug) {
	case RIG_DEBUG_BUG:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/Bug",
						     &pos);
		break;
	case RIG_DEBUG_ERR:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/Error",
						     &pos);
		break;
	case RIG_DEBUG_WARN:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/Warning",
						     &pos);
		break;
	case RIG_DEBUG_VERBOSE:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/Verbose",
						     &pos);
		break;
	case RIG_DEBUG_TRACE:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/Trace",
						     &pos);
		break;
	default:
		menushell = gnome_app_find_menu_pos (GNOME_APP(app)->menubar,
						     "Settings/Debug Level/None",
						     &pos);
	}
	menuitem = GTK_WIDGET (g_list_nth_data (GTK_MENU_SHELL (menushell)->children, pos-1));
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), TRUE);

	/* start timers */

	if (enabletimer)
		gnome_hamlib_rig_set_freq_timeout (GNOME_HAMLIB_RIG (controls), 500);

	gtk_main ();

	gconf_client_remove_dir (confclient, "/apps/gpredict", NULL);

	return 0;
}






static gint
grig_window_close (GtkWidget *widget, GdkEvent  *event, gpointer data)
{
	/* If you return FALSE in the "delete_event" signal handler,
	   GTK will emit the "destroy" signal. Returning TRUE means
	   you don't want the window to be destroyed.
	   This is useful for popping up 'are you sure you want to quit?'
	   type dialogs.
	*/

	return FALSE;
}


static void
grig_destroy (GtkWidget *widget, gpointer data)
{
	/* This function is called when the
	   grig window is destroyed.
	*/

	/* stop timers if running */
	gnome_hamlib_rig_set_freq_timeout (GNOME_HAMLIB_RIG (data), 0);
	gnome_hamlib_rig_set_slider_timeout (GNOME_HAMLIB_RIG (data), 0);
	gnome_hamlib_rig_set_misc_timeout (GNOME_HAMLIB_RIG (data), 0);

	gtk_main_quit ();
}
