/*
 * Visible region handling module
 * Refer to visregion.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include <gdk/gdkx.h>
#include "visregion.h"
#include "guimisc.h"

/* If a window is destroyed after calling XQueryTree(),
   the return value becomes inconsistent with the actual windows.
   In this case, XGetWindowAttributes() or XGetGeometry() cause an X error.
   To get around an abort, I handle it.
   This follows xroach-1.5. */
static int _x_error;
static int _x_error_handler(Display *display, XErrorEvent *error);


/**
 * visible_region_new:
 * Calculate the visible region on @window, and return it.
 * This is derived from CalcRootVisible() of xroach-1.5.
 * XXX: this looks too expensive, because gdk_region routines basically
 * return a new allocated region. So, I have to keep the old one, and
 * destroy it later.
 * Output:
 * Return value; Allocated GdkRegion, which should be destroyed
 * by visible_region_delete().
 **/
GdkRegion*
visible_region_new(GdkWindow *window)
{
    GdkRegion *visible_region;/* return value */
    GdkRegion *covered_region;
    GdkRegion *whole_region;
	GdkRegion *tmp_region;
    Window *children;
    int n_children;
    Window dummy;
    XWindowAttributes xwa;
    int wx;
    int win_x, win_y;
    unsigned int win_height, win_width;
    unsigned int border_width;
    unsigned int depth;
	GdkRectangle tmp_rect;
	XErrorHandler old_handler;/* may be gdk_x_error() */
	
	old_handler = XSetErrorHandler(_x_error_handler);

    /* Get children of @window */
    XQueryTree(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(window), &dummy, &dummy, &children, &n_children);
    
    /* For each mapped child, add the window rectangle to the covered region. */
    covered_region = gdk_region_new();
    for (wx = 0; wx < n_children; wx++) {
		_x_error = 0;
		XGetWindowAttributes(GDK_DISPLAY(), children[wx], &xwa);
		if (_x_error)
			continue;
		if (xwa.class == InputOutput && xwa.map_state == IsViewable) {
			XGetGeometry(GDK_DISPLAY(), children[wx], &dummy, &win_x, &win_y,
						 &win_width, &win_height, &border_width, &depth);
			if (_x_error)
				continue;

			/* tmp_rect is a child window's area */
			tmp_rect.x = win_x;
			tmp_rect.y = win_y;
			tmp_rect.width = win_width + (border_width * 2);
			tmp_rect.height = win_height + (border_width * 2);

			/* gdk_region_union_with_rect() returns a new allocated region.
			   So, I have to keep the old one and destroy it later. */
			tmp_region = covered_region;
			covered_region = gdk_region_union_with_rect(covered_region, &tmp_rect);
			gdk_region_destroy(tmp_region);
		}
    }
    XFree(children);

	XSetErrorHandler(old_handler);
	
    /* Subtract the covered region from the window region. */
	/* I need tmp_region because gdk_region_union_with_rect() returns
	   a new allocated region. */
    tmp_region = gdk_region_new();
    tmp_rect.x = 0;
    tmp_rect.y = 0;
	tmp_rect.width = _gdk_window_get_width(window);
	tmp_rect.height = _gdk_window_get_height(window);
	whole_region = gdk_region_union_with_rect(tmp_region, &tmp_rect);
	gdk_region_destroy(tmp_region);

	visible_region = gdk_regions_subtract(whole_region, covered_region);

	gdk_region_destroy(covered_region);
	gdk_region_destroy(whole_region);
    
	return visible_region;
}

/**
 * visible_region_delete:
 * Destroy the region.
 **/
void
visible_region_delete(GdkRegion *region)
{
	gdk_region_destroy(region);
}


/* ---The followings are private functions--- */
static int
_x_error_handler(Display *display, XErrorEvent *error)
{
	_x_error = error->error_code;
#ifdef DEBUG
	g_print("_x_error=%d\n", _x_error);
#endif
	return 0;
}
