#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""App for asking for a merge from the owner of the parent."""

# Import standard python libs
import os
import logging

# Various required variables and locations
from GroundControl.gtkviews  import GtkApp, ThreadedWindow
from GroundControl.launchpad import get_launchpad
from GroundControl.base      import LP_WEB_ROOT
from launchpadlib.errors     import HTTPError

def get_lp_name(name):
    """Gets an lp name from a url"""
    if 'bzr+ssh://' in name:
        # Reduce bzrlib bzr+ssh:// urls to lp names
        return '/'.join(name.split('/')[-4:-1])
    return name

def merge_url(branch):
    """Return the merge url for this request"""
    merge_id = branch.config.get_user_option('merge_id')
    lp_name  = get_lp_name(branch.branch.get_push_location())
    # This should be a url join, too lazy to change it,
    return os.path.join(LP_WEB_ROOT, lp_name, '+merge', merge_id)


class RequestMergeWindow(ThreadedWindow):
    """Select a project to import."""
    name = 'request'

    def load(self, branch, **kwargs):
        """Load the requesting merge gui."""
        self._branch    = branch
        self._child     = None
        self._launchpad = None
        self._source_br = None
        self._target_br = None
        super(RequestMergeWindow, self).load(**kwargs)

    @property
    def launchpad(self):
        """Return the launchpad object via OAuth"""
        if not self._launchpad:
            try:
                self._launchpad = get_launchpad()
            except HTTPError:
                logging.warn(_("Unable to authorise Launchpad account."))
        return self._launchpad

    @property
    def branch(self):
        """Return the wrapped branch"""
        return self._branch.branch

    @property
    def source_branch(self):
        """Return the source branch from location (usually push location)"""
        if not self._source_br:
            self._source_br = self.get_branch(self.branch.get_push_location())
        return self._source_br

    @property
    def parent_branch(self):
        """Return the target branch, aka parent branch"""
        if not self._target_br:
            self._target_br = self.get_branch(self.branch.get_parent())
        return self._target_br

    def inital_thread(self):
        """What to run when we execute the loading thread."""
        from_name = self.source_branch.bzr_identity
        to_name   = self.parent_branch.bzr_identity
        reviewer  = self._dname(self.get_reviewer())
        self.call('update_details', from_name, to_name, reviewer)

    def get_reviewer(self):
        """Returns the name of the reviewer of the branch parent."""
        reviewer  = self.parent_branch.reviewer
        if not reviewer:
            # and use the branch owner if specific field isn't set.
            reviewer = self.parent_branch.owner
        return reviewer

    def get_branch(self, name):
        """Request a branch from launchpad"""
        name = get_lp_name(name)
        branches = self.launchpad.launchpad.branches
        result = branches.getByUniqueName(unique_name=name)
        if result:
            return result
        logging.warn(_("Couldn't get branch: '%s'") % name)

    def update_details(self, from_name, to_name, reviewer):
        """start and init the GUI and signals"""
        self.widget("br_from").set_text(from_name)
        self.widget("br_to").set_text(to_name)
        self.widget("reviewer").set_text(reviewer)
        self.widget("br_from").show()
        self.widget("br_to").show()
        self.widget("reviewer").show()

    def _dname(self, person):
        """Return a nice displable name of an lp person object"""
        name = person.name
        disp = person.display_name
        if not disp or name == disp:
            return name
        return "%s (%s)" % (disp, name)

    @property
    def comment(self):
        """Get the comment from the gui, at this point I must question the
           sanity of the developers that throught a simple get_text for
           TextView objects was too much trouble."""
        xbuffer = self.widget('comments').get_buffer()
        return xbuffer.get_text(*xbuffer.get_bounds())

    def is_valid(self):
        """return true if the dialog has everything required."""
        return len(self.comment) > 4

    def post_process(self):
        """Thread out a merge request"""
        # If you don't thread out this request, launchpadlib will die
        # Because it isn't thread safe, but at the same time we don't
        # Want the post process to return as it'll kill the window.
        self.start_thread(self.do_merge, self.comment, self.get_reviewer())
        while self._thread.isAlive():
            pass

    def do_merge(self, comment, reviewer):
        """Make sure we're threading this merge"""
        mreq = self.source_branch.createMergeProposal(
            initial_comment=comment,
            needs_review=True,
            target_branch=self.parent_branch,
            reviewers=(reviewer),
            review_types=('')
        )
        # Get the merge id number used in urls
        merge_id = mreq.self_link.split('/')[-1]
        # Make sure to track that the merge request is there.
        self._branch.merge_revno(True)
        self._branch.config.set_user_option('merge_id', merge_id)


class RequestMergeApp(GtkApp):
    """An application to input all required fields for a merge request"""
    gtkfile = 'request-merge.glade'
    windows = [ RequestMergeWindow ]

