/*-----------------------------------------------------------------+
 |                                                                 |
 |  Copyright (C) 2002-2003 Grubconf                               |
 |                     http://grubconf.sourceforge.net/            | 
 |                                                                 |
 | This program is free software; you can redistribute it and/or   |
 | modify it under the terms of the GNU General Public License     |
 | as published by the Free Software Foundation; either version 2  |
 | of the License, or (at your option) any later version           |
 |                                                                 |
 | This program is distributed in the hope that it will be useful, |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of  |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
 | GNU General Public License for more details.                    |
 |                                                                 |
 | A copy of the GNU General Public License may be found in the    |
 | installation directory named "COPYING"                          |
 |                                                                 |
 +-----------------------------------------------------------------+
 */
/* 
 * This file manages the listing of partions. It will search the
 * partition table of available devices and generate a list of
 * valid partitions
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fstab.h>

#include <include/grubconf_global.h>
#include <include/partitioning.h>

#define EXT2_SUPER_MAGIC 0xEF53

#define VOLNAMSZ 16

struct ext2_super_block {
	char  s_dummy0[56];
	unsigned char  s_magic[2];
	char  s_dummy1[62];
	char  s_volume_name[VOLNAMSZ];
	char  s_last_mounted[64];
	char  s_dummy2[824];
} sb;

/* prototypes for private functions */
void read_parts (char *drive_str, int drv_count, struct disk_desc *z);
void put_dev (struct grubconf_dev *dev);

FILE *mtab_fp;

/* Initializes the list of devices 
 */
void
init_grubconf_dev ()
{
	dev_front = NULL;
	mtab_fp = NULL;
	dev_size = 0;
	reset_grubconf_dev ();
}

/* Probes the system devices to build a list of available paritions
 * that grub can use
 */
void
reset_grubconf_dev ()
{
	int i;
	dev_list_clean ();
	char drive_str[16];
	int drv_count = 0;	// keeps track of how grub numbers disks
	struct disk_desc *z;

	// IDE disks.
	for (i = 0; i < 8; i++)	{
		sprintf (drive_str, "/dev/hd%c", i + 'a');
		if ((z = get_drive_info(drive_str, drv_count)) != NULL) {
			read_parts(drive_str, drv_count, z);
			drv_count++;
		}
	}

	// SCSI disks
	for (i = 0; i < 16; i++, drv_count++) {
		sprintf (drive_str, "/dev/sd%c", i + 'a');
		if ((z = get_drive_info(drive_str, drv_count)) == NULL) {
			break;
		} else {
			read_parts(drive_str, drv_count, z);
			drv_count++;
		}
	}

}


/* Reads the partition table from a given device
 */
void
read_parts (char *drive_str, int drv_count, struct disk_desc *z)
{
	int i;
	char part_str[16];
	struct grubconf_dev *dev;
	struct fstab *dev_tab;
	char buf[40];
	int part_ctr = 1;

	/* loop through disk searching 4 primary partitions 
	 * for now we ignore extended partitions
	 */
	for (i = 1; i <= z->partno; i++) {
		dev = calloc (1, sizeof (struct grubconf_dev));
		if (dev == NULL) {
			printf("could not allocate any more devices");
			return;
		}
		if ((dev->dev_type = valid_type (z->partitions[i-1].p.sys_type)) != 0x00)	{
			sprintf (part_str, "%s%i", drive_str, part_ctr);
			dev->drv_num = drv_count;
			dev->part_num = part_ctr - 1;
			dev->dev_name = strdup (part_str);
			if (dev->dev_type == 0x83)
				dev->fs_label = get_fs_label (dev->dev_name);
			dev_tab = getfsspec (part_str);
			if (!dev_tab) {
				if (dev->fs_label) {
					sprintf (buf, "LABEL=%s", dev->fs_label);
					dev_tab = getfsspec (buf);
					if (dev_tab) {
						dev->mount_point = strdup (dev_tab->fs_file);
					} else {
						dev->mount_point = NULL;
					}
				} else {
					dev->mount_point = NULL;
				}
			} else {
				dev->mount_point = strdup (dev_tab->fs_file);
			}
			dev->mounted = mounted ((dev->mount_point ? dev->mount_point : part_str));
			put_dev (dev);
			if (z->partitions[i].ep)
				part_ctr++;
		} else { // found a nonactive partition, free allocated dev
			free (dev);
		}
		if (i <= 4) part_ctr++;
	}
}


/* Returns the filesystem label for specified device
 * 
 * Adapted from Andries Brouwer's (aeb@cwi.nl) e2label
 */
char *get_fs_label(char *dev_str) {
	int fd;
	char *ret;

	fd = open(dev_str, O_RDONLY);
	if (fd < 0) {
		return NULL;
	}
	if (lseek(fd, 1024, SEEK_SET) != 1024) {
	     return NULL;
	}
	if (read(fd, (char *) &sb, sizeof(sb)) != sizeof(sb)) {
	     return NULL;
	}
	close (fd);
	if (sb.s_magic[0] + 256*sb.s_magic[1] != EXT2_SUPER_MAGIC) {
	     return NULL;
	}
	ret = calloc (1, VOLNAMSZ + 1);
	strncpy (ret, sb.s_volume_name, VOLNAMSZ);
	return ret;
}

/*
 * cleans the list of devices
 */
void 
dev_list_clean()
{
	struct grubconf_dev *dev;
	while (dev_front != NULL) {
		dev = dev_front;
		dev_front = dev_front->next;
		if (dev->need_umount) {
			umount_from_dev (dev);
		}
		free(dev->dev_name);
		free(dev);
		dev_size--;
	}
	if (mtab_fp)
		fclose (mtab_fp);
}

/* A private function to insert a drive into the grubconf_dev structure
 */
void 
put_dev (struct grubconf_dev *dev)
{
	struct grubconf_dev *tmp = dev_front;
	if (dev_front == NULL) {
		dev_front = dev;
		dev_size++;
		return;
	}
	for ( ;tmp->next != NULL; tmp = tmp->next);
	tmp->next = dev;
	dev_size++;
}

/* retrieves the device at index
 */
struct grubconf_dev *get_dev(int index) 
{
	struct grubconf_dev *tmp = dev_front;
	int i;
	for (i = 0; tmp != NULL && i < dev_size && i < index; i++, tmp = tmp->next) { }
	return tmp;
}

/* retrieves the index from GRUB device of format (hd0,0).
 * returns -1 on a device not found, -2 on an invalid format
 */
int 
get_index_grub(char *dev_str) 
{
	int tmp_int = 5;
	int drv_num = -1 , part_num = -1;
	
	// check if valid string
	if (!dev_str)
		return -2;
	
	// check first part of device
	if (dev_str[0] != '(' || dev_str[1] != 'h' || dev_str[2] != 'd')
		return -2;
	
	// check drive number
	if (dev_str[3] < '0' || dev_str[3] > '9') 
		return -2;
	
	// check to see if there are more than 9 drives
	if (dev_str[4] != ',') {
		// check validity of next didget
		if (dev_str[4] < '0' || dev_str[4] > '9') {
			return -1;
		} else {
			drv_num = ((dev_str[3] - '0') * 10) + (dev_str[4] - '0');
			tmp_int++;
		}
	} else {
		drv_num = dev_str[3] - '0';
	}
	
	// check partition number
	if (dev_str[tmp_int] < '0' || dev_str[tmp_int] > '9') 
		return -2;
	
	// check to see if there are more than 9 partitions
	if (dev_str[tmp_int+1] != ')') {
		// check validity of next didget
		if (dev_str[tmp_int+1] < '0' || dev_str[tmp_int+1] > '9') {
			return -1;
		} else {
			part_num = ((dev_str[tmp_int] - '0') * 10) + 
			           (dev_str[tmp_int+1] - '0');
			tmp_int++;
		}
	} else {
		part_num = dev_str[tmp_int] - '0';
	}
	
	tmp_int++;
	
	if (dev_str[tmp_int] != ')')
		return -2;
	
	struct grubconf_dev *tmp = dev_front;
	int i;
	
	for (i = 0; tmp != NULL && i < dev_size; i++, tmp = tmp->next) {
		if (tmp->drv_num == drv_num && tmp->part_num == part_num) {
			return i;
		}
	}
	return -1;
}

/* retrieves the index from GRUB device of format /dev/hda1. 
 */
int 
get_index_dev(char *dev_str) 
{
	int i;
	struct grubconf_dev *tmp = dev_front;
	if (!dev_str || strncmp (dev_str, "/dev/", 5) != 0)
		return -1;
	
	for (i = 0; tmp != NULL && i < dev_size; i++, tmp = tmp->next) {
		if ( *(tmp->dev_name + 5) == *(dev_str + 5) &&
				*(tmp->dev_name + 7) == *(dev_str + 7) && 
				*(tmp->dev_name + 8) == *(dev_str + 8) ) {
			return i;
		}
	}
	return -1;
}

/* retrieves the index from GRUB device matching label
 */
int
get_index_label(char *label) 
{
	int i;
	struct grubconf_dev *tmp = dev_front;
	if (!label)
		return -1;
	
	for (i = 0; tmp != NULL && i < dev_size; i++, tmp = tmp->next) {
		if (tmp->fs_label && strcmp (tmp->fs_label, label) == 0) {
			return i;
		}
	}
	return -1;
}

/* returns true if the given filename of format /dev/hda is mounted
 */
gboolean 
mounted (char *dev_str)
{
	gboolean ret = FALSE;
	char *buf = NULL;
	size_t size = 100;

	if (!mtab_fp)
		mtab_fp = fopen ("/etc/mtab", "r");
	else 
		rewind (mtab_fp);
	
	if (mtab_fp) {
		/* loop through mtab searching for /boot */
		while (getline (&buf, &size, mtab_fp) > 0 && !feof (mtab_fp)) {
			if (strstr (buf, dev_str)) {
				ret = TRUE;		// to say it was found
				break;
			}
		}
		free (buf);
		buf = NULL;
	}
	return ret;
}

/* attempts to mount selected mnt_point. Will return true if is or was mounted.
 */
gboolean
mount (char *mnt_point) {
	if (!mnt_point)
		return FALSE;
	
	int status = 0;
	int i;
	struct grubconf_dev *tmp_dev = dev_front;
	
	// first try to find the device from its mount point
	for (i = 0; tmp_dev != NULL && i < dev_size; i++, 
					tmp_dev = tmp_dev->next) {
		if (tmp_dev->mount_point && 
						strcmp (tmp_dev->mount_point, mnt_point) == 0) {
			status = 1;
			break;
		}
	}
	
	if (!status)
		return FALSE;
	
	return mount_from_dev (tmp_dev);
}

/* attempts to mount selected device. Will return true if is or was mounted.
 */
gboolean
mount_from_dev (struct grubconf_dev *dev) {
	if (!dev) 
		return FALSE;
	
	if (dev->mounted)
		return TRUE;
	
	// attemt a forked mount command
	int status = 0;
	pid_t pid = fork ();
	if (pid == 0) {
		execlp ("mount", "mount", dev->mount_point, NULL);
	} else {
		waitpid (pid, &status, 0);
	}
	
	if (status < 0)
		return FALSE;

	// now that we know it was mounted, save that info in the device
	dev->mounted = TRUE;
	dev->need_umount = TRUE;
	
	return TRUE;
}

/* attempts to umount selected device. Will return true if is or was umounted.
 */
gboolean
umount_from_dev (struct grubconf_dev *dev) {
	if (!dev->mounted)
		return TRUE;
	
	// attemt a forked mount command
	int status = 0;
	pid_t pid = fork ();
	if (pid == 0) {
		execlp ("umount", "umount", dev->mount_point, NULL);
	} else {
		waitpid (pid, &status, 0);
	}
	
	if (status < 0)
		return FALSE;

	// now that we know it was mounted, save that info in the device
	dev->mounted = FALSE;
	dev->need_umount = FALSE;
	
	return TRUE;
}

/* loads mnu with current devices
 */
void 
dev_load_menu(GtkWidget *mnu) 
{
	dev_load_menu_path (mnu, FALSE);
}

/* loads mnu with current devices, optionaly printing path
 */
void 
dev_load_menu_path(GtkWidget *mnu, gboolean path) 
{
	struct grubconf_dev *dev = dev_front;
	GtkWidget *menuitem;
	char buf[50];
	int i;
	for (i = 0; dev != NULL && i < dev_size; i++, dev = dev->next) {
		if (path && dev->mount_point) {
			sprintf (buf, "%s [%s, %s]", dev->dev_name + 5, 
					type_to_name(dev->dev_type), 
					(dev->mounted ? dev->mount_point: "Not Mounted"));
		} else {
			sprintf(buf, "%s [%s]", dev->dev_name + 5, 
					type_to_name(dev->dev_type));
		}
		menuitem = gtk_menu_item_new_with_label (buf);
		gtk_widget_show (menuitem);
		gtk_menu_shell_append (GTK_MENU_SHELL (mnu), menuitem);
	}
}
