/*
 * This file is part of gshutdown.
 *
 * Copyright (c) 2006, GShutdown Team <gshutdown@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <glib/gutils.h> /* g_atexit() */
#include <gtk/gtk.h>
#include "main.h"
#include "gui.h"
#include "misc.h"
#include "values.h"
#include "bacon.h"

/* Variables */
static BaconMessageConnection *bacon_connection = NULL;
static gboolean main_loop = FALSE;
static gboolean values_loaded = FALSE;

/* Prototypes */
static void handle_sigchld_signal(int signal);
static void handle_sigint_signal(int signal);
static void on_message_received (const char *message, gpointer data);

int main(int argc, char **argv)
{
    /* Gettext */
#ifdef ENABLE_NLS
	bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif

	bacon_connection = bacon_message_connection_new ("gshutdown");

	if(bacon_connection != NULL) {
		if(bacon_message_connection_get_is_server(bacon_connection) == FALSE) {
#ifdef DEBUG
			g_warning ("DEBUG: Still opened");
#endif

            bacon_message_connection_send(bacon_connection, "show");
			bacon_message_connection_free(bacon_connection);
            return 0;
		}
		else {
#ifdef DEBUG
		  	g_warning("DEBUG : First opening"); 
#endif
			bacon_message_connection_set_callback(bacon_connection, on_message_received, NULL);
		}
	}
	else
		g_warning("Cannot create the 'gshutdown' connection.");
    
    /* Signal handling */
	signal(SIGINT, handle_sigint_signal);
	signal(SIGCHLD, handle_sigchld_signal);
    g_atexit(called_at_exit);

    /* Init */
    values_init();

    if(gui_init(argc, argv) == TRUE)
        return QUIT_FAILURE;

    if(values_load() == TRUE) {
        /* It's important to load this after gui_init() */
        GtkWidget *dialog;

        dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                        GTK_DIALOG_MODAL,
                                        GTK_MESSAGE_ERROR,
                                        GTK_BUTTONS_OK,
                                        _("The configuration cannot be loaded from %s."), 
                                        "~/.gshutdown");
        gtk_dialog_run(GTK_DIALOG(dialog));

        gtk_widget_destroy(dialog);
    }

    values_loaded = TRUE;

    /* Show if needed detection error warning */
    detect_error_dialog();

    /* Main */
    main_loop = TRUE;
    gtk_main();

    return QUIT_SUCCESS;
}

/*
 * This function are called at quit.
 */
void called_at_exit()
{
    static gboolean first = TRUE;

    if(first) {
        /* Save the configuration */
        if(values_loaded == TRUE && values_save() == TRUE) {
            /* It's important to load this after gui_init() */
            GtkWidget *dialog;
            
            dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                            GTK_DIALOG_MODAL,
                                            GTK_MESSAGE_ERROR,
                                            GTK_BUTTONS_OK,
                                            _("The configuration cannot be saved in %s."),
                                            "~/.gshutdown");

            gtk_dialog_run(GTK_DIALOG(dialog));
            gtk_widget_destroy(dialog);
        }

#ifdef DEBUG
        printf("DEBUG: atexit() function called!\n");
#endif

        if(main_loop == TRUE)
            gtk_main_quit();

	    bacon_message_connection_free(bacon_connection);

        values_destroy();
        first = FALSE;
    }
}

/* 
 * Handle signals.
 */
static void handle_sigint_signal(int signal)
{
    #ifdef DEBUG
    printf("DEBUG: SIGINT handled.\n");
    #endif

    called_at_exit(); /* Free memory allocated, before quit */
}

static void handle_sigchld_signal(int signal)
{
    #ifdef DEBUG
    printf("DEBUG: SIGCHLD handled.\n");
    #endif
	wait(NULL);
}

/* Bacon message handing */
static void on_message_received(const char *message, gpointer data)
{
   if(g_str_equal(message, "show") == TRUE)
      gtk_widget_show(main_window);      
}

/* vim:cindent:et:sw=4:ts=4:sts=4:tw=78:fenc=utf-8
 */
