/*
 * This file is part of gshutdown.
 *
 * Copyright (c) 2006, GShutdown Team <gshutdown@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <glib/gtypes.h>
#include "timer.h"
#include "values.h"
#include "gui.h"
#include "progress_confirm.h"
#include "main.h"

const guint SECS_IN_YEAR = 31536000;
const guint SECS_IN_MONTH = 2628000;
const guint SECS_IN_DAY = 86400;
const guint SECS_IN_HOUR = 3600;
const guint SECS_IN_MIN = 60;

guint nb_days_in_month(guint month, guint year)
{
    gint nb_days[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
   
    /* Leap year */
    if((!(year % 4) && year % 100) || !(year % 400))
        nb_days[1] = 29;

    return nb_days[month];
}


/*
 * Return time_left in seconds
 */
guint time_left()
{
    time_t t = time(0);
    struct tm *tm_time = localtime(&t);

    gint timeleft = 0;
    gint nbdays = 0;
    guint month, year;

    /* At Date Time */
    if(values.time_choice == TIME_CHOICE_TIME_DATE) {
        year = tm_time->tm_year + 1900;
        while(year - values.time_date.year) {
           if(values.time_date.month > 1) {
              if((!((year + 1) % 4) && (year + 1) % 100) || !((year + 1) % 400))
                 nbdays = 366;
              else
                 nbdays = 365;
           } else {
              if((!(year % 4) && year % 100) || !(year % 400))
                 nbdays = 366;
              else
                 nbdays = 365;
           }
           year++;
        }
        month = tm_time->tm_mon;
        while(values.time_date.month > month)
        {
           nbdays += nb_days_in_month(month, year);
           month++;
        }
        while(values.time_date.month < month)
        {
           nbdays -= nb_days_in_month(month, year);
           month--;
        }

        nbdays += values.time_date.day - tm_time->tm_mday;

        timeleft += nbdays * SECS_IN_DAY;
        timeleft += (values.time_date.hour - tm_time->tm_hour) * SECS_IN_HOUR;
        timeleft += (values.time_date.min - tm_time->tm_min) * SECS_IN_MIN;
    }

    /* Time from now */
    if(values.time_choice == TIME_CHOICE_TIME_FROM_NOW) {
        timeleft += (values.time_from_now.hour) * SECS_IN_HOUR;
        timeleft += (values.time_from_now.min) * SECS_IN_MIN;
        timeleft += values.time_from_now.sec;
    }

    timeleft -= tm_time->tm_sec;

    if(timeleft < 0)
        timeleft = 0;


    return timeleft;
}

/*
 * Function called every sec
 *
 * Returne TRUE on error.
 */
gboolean timer_notify()
{
    guint timeleft;
    time_t t = time(0);
    struct tm *tm_time = localtime(&t);
    static guint min = -1;
    gchar message[100];

    if(min == -1)
        min = tm_time->tm_min;

    if(values.time_choice == TIME_CHOICE_TIME_FROM_NOW) {
        /* Minute has changed */
        if(min != tm_time->tm_min) {
            min = tm_time->tm_min;

            if(values.time_from_now.min) {
                values.time_from_now.min--;
            }
            else {
                values.time_from_now.min = 59;
                values.time_from_now.hour--;
            }
        }

    }

    timeleft = time_left();

    if(timeleft >= 60) {
        sprintf(message, _("Time before action :"));
        if(timeleft < SECS_IN_HOUR) {
            if(timeleft/SECS_IN_MIN > 1)
                sprintf(message, _("%s %u minutes"), 
                                 message, 
                                 timeleft/SECS_IN_MIN);
            else
                sprintf(message, _("%s 1 minute"), message);

            if(timeleft < 300 && timeleft%SECS_IN_MIN) {
                if(timeleft%SECS_IN_MIN > 1)
                    sprintf(message, _("%s and %u seconds"), 
                                     message, 
                                     timeleft%SECS_IN_MIN);
                else
                    sprintf(message, _("%s and 1 second"), message);
            }
        } else if(timeleft < SECS_IN_DAY) {
            if(timeleft/SECS_IN_HOUR > 1) {
                sprintf(message, _("%s %u hours"), 
                                 message,
                                 timeleft/SECS_IN_HOUR);
            } else {
                sprintf(message, _("%s 1 hour"), message);
            }        
            if((timeleft%SECS_IN_HOUR)/SECS_IN_MIN) {
                if((timeleft%SECS_IN_HOUR)/SECS_IN_MIN > 1)
                    sprintf(message, _("%s and %u minutes"), 
                                    message,
                                    (timeleft%SECS_IN_HOUR)/SECS_IN_MIN);
                else
                    sprintf(message, _("%s and 1 minute"), message);
            }
        } else {
            if(timeleft/SECS_IN_DAY > 1)
                sprintf(message, _("%s %u days"),
                                message, 
                                timeleft/SECS_IN_DAY);
            else if((timeleft%SECS_IN_DAY)/SECS_IN_HOUR)
            {
               sprintf(message, _("%s %u day"),
                                message, 
                                timeleft/SECS_IN_DAY);
               if((timeleft%SECS_IN_DAY)/SECS_IN_HOUR > 1)
                   sprintf(message, _("%s and %u hours"),
                                    message, 
                                    (timeleft%SECS_IN_DAY)/SECS_IN_HOUR);
               else
                   sprintf(message, _("%s and 1 hour"),
                                    message);
            }
        }
        gtk_tooltips_set_tip(tray_tooltips, tray_eventbox, message, NULL);

        if(timeleft == 60  || timeleft == 120 || timeleft == 300 || 
                timeleft == 900 || timeleft == 1800 || timeleft == 3600)
        {
            gui_notification(message, NOTIFICATION_TYPE_IMPORTANT, NOTIFICATION_ICON_TIME);
        }

    }
    else {
        sprintf(message, _("Time before action : %u seconds"), timeleft);
        gtk_tooltips_set_tip(tray_tooltips, tray_eventbox, message, NULL);

        if(timeleft <= 30 && progress_confirm_open == FALSE)
            progress_confirm();
    }

    return TRUE;
}

/* vim:cindent:et:sw=4:ts=4:sts=4:tw=78:fenc=utf-8
*/
