/*
 * This file is part of gshutdown.
 *
 * Copyright (c) 2006, GShutdown Team <gshutdown@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdlib.h>
#include "main.h"
#include "values.h"
#include "gui.h"
#include "callback.h"
#include "misc.h"
#include "action.h"
#include "timer.h"
#include "progress_confirm.h"
#include <libnotify/notify.h>

/* Config Action (custom command) {{{1 */

/* Action Variables  */ 
static Action *config_action_current_action = NULL;

/* Called when the config action dialog are closed, to save all informations */
gboolean on_config_action_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    const gchar *command;

    gtk_widget_hide(GTK_WIDGET(config_action_dialog));

    /* Apply the action in "values" */
    command = gtk_entry_get_text(GTK_ENTRY(entry_action_custom_command));
    if(config_action_current_action != NULL)
        action_set_command(config_action_current_action, command);

    /* save */
    values_save();

    return gtk_widget_hide_on_delete(widget);
}

void on_btn_config_action_close(GtkWidget *widget, gpointer data)
{
    on_config_action_dialog_delete_event(GTK_WIDGET(config_action_dialog), NULL, NULL);
}

void on_btn_config_action(GtkWidget *widget, gpointer data)
{
    /* It must not be visible */
    if(GTK_WIDGET_VISIBLE(GTK_WIDGET(config_action_dialog)))
        return;

    /* Get the actual action information */
    if(widget == btn_config_shutdown)
        config_action_current_action = values.action_shutdown;

    else if(widget == btn_config_restart)
        config_action_current_action = values.action_restart;

    else if(widget == btn_config_logout) 
        config_action_current_action = values.action_logout;

    else /* WARNING new actions are here */
        return;

    gtk_entry_set_text(GTK_ENTRY(entry_action_custom_command), 
                       action_get_command(config_action_current_action));

    gtk_widget_grab_focus(entry_action_custom_command);

    /* Show the dialog */
    gtk_widget_show(GTK_WIDGET(config_action_dialog));
}

/* }}}1 */

/* Config Action (method) {{{1 */

void on_combo_method_manual_display_manager(GtkWidget *widget, gpointer data)
{
    switch(gtk_combo_box_get_active(GTK_COMBO_BOX(widget))) {
        case 1:  /* GDM */
            values.selected_display_manager = DISPLAY_MANAGER_GDM;
            break;

        case 2:  /* KDM */
            values.selected_display_manager = DISPLAY_MANAGER_KDM;
            break;

        default: /* Auto */
            values.selected_display_manager = DISPLAY_MANAGER_AUTO;
    }
}

void on_combo_method_manual_desktop(GtkWidget *widget, gpointer data)
{
    switch(gtk_combo_box_get_active(GTK_COMBO_BOX(widget))) {
        case 1:  /* GNOME */
            values.selected_desktop = DESKTOP_GNOME;
            break;

        case 2:  /* KDE */
            values.selected_desktop = DESKTOP_KDE;
            break;

        case 3:  /* XFCE */
        { /* TODO remove this block */
            GtkWidget *dialog;
            dialog = gtk_message_dialog_new(NULL,
                                            GTK_DIALOG_MODAL,
                                            GTK_MESSAGE_ERROR,
                                            GTK_BUTTONS_OK,
                                            (gchar *)_("Not yet implemented!"));
            gtk_dialog_run(GTK_DIALOG(dialog));
            gtk_widget_destroy(GTK_WIDGET(dialog));
            gtk_combo_box_set_active(GTK_COMBO_BOX(widget), values.selected_desktop);
        }
        /* TODO xfce support 
        values.selected_desktop = DESKTOP_XFCE;
        break;
        */

        default: /* Auto */
            values.selected_desktop = DESKTOP_AUTO;
    }
}

void on_btn_method_manual_close(GtkWidget *widget, gpointer data)
{
    gtk_widget_hide(method_manual_dialog);
}

/* Method in general config dialog {{{2 */

void on_btn_action_method_manual(GtkWidget *widget, gpointer data)
{
    gtk_widget_show(method_manual_dialog);
}

void on_radio_action_method(GtkWidget *widget, gpointer data)
{
    /* Diable all */
    gtk_widget_set_sensitive(btn_check_system, FALSE);
    gtk_widget_set_sensitive(btn_action_method_manual, FALSE);
    gtk_widget_set_sensitive(btn_config_shutdown, FALSE);
    gtk_widget_set_sensitive(btn_config_restart, FALSE);
    gtk_widget_set_sensitive(btn_config_logout, FALSE);

    /* Enable the required widgets */
    if(widget == radio_action_method_autodetect) {
        values.action_method = ACTION_METHOD_AUTODETECT;
        gtk_widget_set_sensitive(btn_check_system, TRUE);
    }
    else if(widget == radio_action_method_manual) {
        values.action_method = ACTION_METHOD_MANUAL;
        gtk_widget_set_sensitive(btn_action_method_manual, TRUE);
    }
    else {
        values.action_method = ACTION_METHOD_CUSTOM_COMMAND;
        gtk_widget_set_sensitive(btn_config_shutdown, TRUE);
        gtk_widget_set_sensitive(btn_config_restart, TRUE);
        gtk_widget_set_sensitive(btn_config_logout, TRUE);
    }
}

/* }}}2 */

/* }}}1 */

/* Config Action (command before action) {{{1 */

void on_check_command_before_action(GtkWidget *widget, gpointer data)
{
    gboolean enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    gtk_widget_set_sensitive(entry_command_before_action, enabled);
    gtk_widget_grab_focus(entry_command_before_action);

    values.command_before_action_enabled = enabled;
}

/* }}}1 */

/* General configuration {{{1 */

/* 
 * NB: It's update values.
 *
 * Show/Hide notifications
 *
 * */
void on_check_show_notifications(GtkWidget *widget, gpointer data)
{
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) == TRUE) {
        gtk_widget_set_sensitive(GTK_WIDGET(btn_test_notifications), TRUE);
        values.show_notifications = TRUE;
    }
    else {
        gtk_widget_set_sensitive(GTK_WIDGET(btn_test_notifications), FALSE);
        values.show_notifications = FALSE;
    }
}

void on_btn_test_notifications(GtkWidget *widget, gpointer data)
{
    gui_notification(_("Notification test"), NOTIFICATION_TYPE_IMPORTANT, NOTIFICATION_ICON_TIME);
}

/*
 * config dialog.
 */
gboolean on_config_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    const gchar *text;
   
    text = gtk_entry_get_text(GTK_ENTRY(entry_command_before_action));
    values_set_command_before_action(text);

    values_save();
    return gtk_widget_hide_on_delete(widget);
}

void on_btn_cfg_close(GtkWidget *widget, gpointer data)
{
    on_config_dialog_delete_event(config_dialog, NULL, NULL);
}

/* Complete a command */
void on_entry_command_complete(GtkWidget *widget, gpointer data)
{
    gchar *stock_id = NULL;

    if(command_exists(gtk_entry_get_text(GTK_ENTRY(widget))) == TRUE)
        stock_id = GTK_STOCK_APPLY;
    else
        stock_id = GTK_STOCK_CANCEL;
    
    gtk_image_set_from_stock(GTK_IMAGE(data), stock_id,  
                             GTK_ICON_SIZE_BUTTON); 
}


/*
 * Called when user clicked on the notify
 */

void on_notify_click(NotifyNotification *n, const char *action)
{
    gtk_widget_show(main_window);
}

/*
 * Check system dialog.
 */
void on_btn_check_system(GtkWidget *widget, gpointer data)
{
    gtk_widget_show(GTK_WIDGET(data));
}

void on_btn_check_system_close(GtkWidget *widget, gpointer data)
{
    gtk_widget_hide(GTK_WIDGET(data));
}

/*
 * Radio button to show/hide the systray icon.
 */
void on_radio_systray_always(GtkWidget *widget, gpointer data)
{
    values.systray_icon = SYSTRAY_ICON_ALWAYS;
    gtk_widget_set_sensitive(check_iconify, TRUE);
    gui_systray_icon_show();
}

void on_radio_systray_if_active(GtkWidget *widget, gpointer data)
{
    values.systray_icon = SYSTRAY_ICON_IF_ACTIVE;
    gtk_widget_set_sensitive(check_iconify, TRUE);

    if(values.action_scheduled)
        gui_systray_icon_show();
    else
        gui_systray_icon_hide();
}

void on_radio_systray_never(GtkWidget *widget, gpointer data)
{
    values.systray_icon = SYSTRAY_ICON_NEVER;
    gtk_widget_set_sensitive(check_iconify, FALSE);
    gui_systray_icon_hide();
}

void on_check_iconify(GtkWidget *widget, gpointer data)
{
    values.iconify = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}

/* }}}1 */

/* Menu {{{1 */

void on_mnu_about(GtkWidget *widget, gpointer data)
{
    gtk_widget_show(GTK_WIDGET(data));
}

void on_mnu_quit(GtkWidget *widget, gpointer data)
{
    GtkWidget *dialog;
    gint response;
    
    dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                    GTK_DIALOG_MODAL,
                                    GTK_MESSAGE_QUESTION,
                                    GTK_BUTTONS_YES_NO,
                                    (gchar *)_("Are you sure you want to quit ?"));

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    if(response == GTK_RESPONSE_YES)
        quit(QUIT_SUCCESS); /* atexit called */

    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/*
 * "Cancel" in the contextual menu.
 */
void on_mnu_tray_cancel(GtkWidget *widget, gpointer data)
{
    if(values.action_scheduled)
        on_btn_start(btn_start, NULL);
}

void on_mnu_config(GtkComboBox *combo, gpointer data)
{
    gtk_widget_show(config_dialog);
}

/* }}}1 */

/* Main window {{{1 */

/* It modify the "values" */
void on_combo_action(GtkWidget *widget, gpointer data)
{
    gint current = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));

    switch(current) {
        case 0: /* Shutdown */
            values.current_action = values.action_shutdown;
            break;

        case 1: /* Restart */
            values.current_action = values.action_restart;
            break;

        case 2: /* Logout */
            values.current_action = values.action_logout;
            break;
    }
}

/* 
 * NB: It's update "values". [ SYNC ]
 *
 * If the type of the time are changed
 *     - Hide and show some widgets
 *     - Load and save some values
 */
void on_combo_time_type(GtkComboBox *combo, gpointer data)
{
    /* Hide/Show widgets */
    gtk_widget_set_sensitive(GTK_WIDGET(calendar), 0);
    gtk_widget_set_sensitive(GTK_WIDGET(spin_hour), 0);
    gtk_widget_set_sensitive(GTK_WIDGET(spin_minute), 0);
    gtk_widget_set_sensitive(GTK_WIDGET(btn_actual_time), 0);

    /* Update "values" */
    switch(gtk_combo_box_get_active(combo)) {
        case 0:
            values.time_choice = TIME_CHOICE_TIME_DATE;

            /* Set values */
            gtk_spin_button_set_value(spin_hour, values.time_date.hour);
            gtk_spin_button_set_value(spin_minute, values.time_date.min);

            /* Enable widgets */
            gtk_widget_set_sensitive(GTK_WIDGET(btn_actual_time), 1);
            gtk_widget_set_sensitive(GTK_WIDGET(spin_hour), 1);
            gtk_widget_set_sensitive(GTK_WIDGET(spin_minute), 1);
            gtk_widget_set_sensitive(GTK_WIDGET(calendar), 1);
            break;

        case 1:
            values.time_choice = TIME_CHOICE_TIME_FROM_NOW;

            /* Set values */
            gtk_spin_button_set_value(spin_hour, values.time_from_now.hour);
            gtk_spin_button_set_value(spin_minute, values.time_from_now.min);

            /* Enable widgets */
            gtk_widget_set_sensitive(GTK_WIDGET(spin_hour), 1);
            gtk_widget_set_sensitive(GTK_WIDGET(spin_minute), 1);
            break;

        case 2:
            values.time_choice = TIME_CHOICE_NOW;
            break;
    }
}


/*
 * NB: It's update "values". [ SYNC ]
 */
void on_calendar(GtkCalendar *calendar, gpointer data)
{
    gtk_calendar_get_date(calendar, &values.time_date.year, 
                                    &values.time_date.month,
                                    &values.time_date.day);

    mark_day();
}

void on_spin_hour(GtkSpinButton *spin, GtkScrollType *arg1, gpointer data)
{
    gint current_choice;
    gint day, month, year;

    current_choice = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_time_type));

    if(gtk_spin_button_get_value(spin_hour) == 24) {
       gtk_spin_button_set_value(spin_hour, 0);

       if(current_choice == TIME_CHOICE_TIME_DATE) {
          day = values.time_date.day;
          if(day < nb_days_in_month(values.time_date.month ,values.time_date.year)) {
             day += 1;
             gtk_calendar_select_day(calendar, day);
          } else {
             month = values.time_date.month + 1;
             year = values.time_date.year;
             if(month == 12) {
                month = 0;
                year += 1;
             }
             day = 1;
             gtk_calendar_select_day(calendar, day);
             gtk_calendar_select_month(calendar, month,
                                                 year);
          }
          on_calendar(calendar, NULL);
       }
    }
    else if(gtk_spin_button_get_value(spin_hour) == -1) {
       gtk_spin_button_set_value(spin_hour, 23);
       if(current_choice == TIME_CHOICE_TIME_DATE) {
          day = values.time_date.day;
          if(day == 1) {
             month = values.time_date.month - 1;
             year = values.time_date.year;
             if(month == -1) {
                month = 11;
                year -= 1;
             }
             day = nb_days_in_month(month, values.time_date.year);
             gtk_calendar_select_month(calendar, month,
                                                 year);
             gtk_calendar_select_day(calendar, day);
          } else {
             day -= 1;
             gtk_calendar_select_day(calendar, day);
          }
          on_calendar(calendar, NULL);
       }
    }

    switch(current_choice) {
        case 0: /* Time and date */
            values.time_date.hour = gtk_spin_button_get_value(spin_hour);
            break;

        case 1: /* Time from now */
            values.time_from_now.hour = gtk_spin_button_get_value(spin_hour);
            break;
    }
}

void on_spin_minute(GtkSpinButton *spin, GtkScrollType *arg1, gpointer data)
{
    gint current_choice;

    if(gtk_spin_button_get_value(spin_minute) == 60) {
       gtk_spin_button_set_value(spin_minute, 0);
       gtk_spin_button_set_value(spin_hour, gtk_spin_button_get_value(spin_hour) + 1);
       on_spin_hour(spin_hour, NULL, NULL);
    } else if(gtk_spin_button_get_value(spin_minute) == -1) {
       gtk_spin_button_set_value(spin_minute, 59);
       gtk_spin_button_set_value(spin_hour, gtk_spin_button_get_value(spin_hour) - 1);
       on_spin_hour(spin_hour, NULL, NULL);
    }

    current_choice = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_time_type));

    switch(current_choice) {
        case 0: /* Time and date */
            values.time_date.min = gtk_spin_button_get_value(spin_minute);
            break;

        case 1: /* Time from now */
            values.time_from_now.min = gtk_spin_button_get_value(spin_minute);
            break;
    }
}

gboolean on_window_close(GtkWidget *widget, gpointer data)
{
   if(gui_systray_icon_is_showed())
      gtk_widget_hide(widget);
   else
      on_mnu_quit(widget, data);

   return TRUE;
}

void on_destroy(GtkWidget *widget, gpointer data)
{
    quit(QUIT_FAILURE); /* atexit called */
}

void on_btn_actual_time(GtkButton *widget, gpointer data)
{
    gui_set_actual_time(TRUE, TRUE);
}

/*
 * Start button
 */
void on_btn_start(GtkButton *button, gpointer data)
{
    static guint func_ref;
    GtkWidget *dialog;
    guint timeleft;
    gchar *path;

    time_t t = time(0);
    struct tm *tm_time = localtime(&t);

    /* Launch the action */
    if(values.action_scheduled == FALSE) {
        /* Test if the detection are ok */
        if(detect_error_dialog())
           return;


        /* Save sec */
        values.time_from_now.sec = tm_time->tm_sec;
        timeleft = time_left();

        /* Launch ! */
        if(timeleft > 30) {
           if(values.systray_icon != SYSTRAY_ICON_NEVER) {
               gui_systray_icon_show();
   
               /* Iconify with a notification ? */
               if(values.iconify && values.systray_icon != SYSTRAY_ICON_NEVER) { 
                   gtk_widget_hide(main_window);
                   gui_notification(_("I'm here !\n\nMove over me to see the countdown."), 
                                    NOTIFICATION_TYPE_FAST_INFO, NOTIFICATION_ICON_NONE);
               }
           }

           values.action_scheduled = TRUE;

           /* Change systray image icon */
           path = data_path("tray_icon2.png");
           gtk_image_set_from_file(GTK_IMAGE(tray_image), path);

           /* Time type widgets */
           gtk_widget_set_sensitive(GTK_WIDGET(calendar), 0);
           gtk_widget_set_sensitive(GTK_WIDGET(spin_hour), 0);
           gtk_widget_set_sensitive(GTK_WIDGET(spin_minute), 0);
           gtk_widget_set_sensitive(GTK_WIDGET(btn_actual_time), 0);
   
           /* Other */
           gtk_widget_set_sensitive(GTK_WIDGET(combo_action), 0);
           gtk_widget_set_sensitive(GTK_WIDGET(combo_time_type), 0);
           gtk_button_set_label(btn_start, _("_Stop"));    

           /* Timer : 1s*/
           func_ref = g_timeout_add(1000, timer_notify, NULL);
        }
        else
        {
           if(timeleft > 0 || values.time_choice == TIME_CHOICE_NOW) {
              if(confirm_dialog(values.current_action) == TRUE)
                 action_run(values.current_action);

           } else {
              dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_CLOSE,
                                     _("Chosen time already exceeded"));

              gtk_dialog_run(GTK_DIALOG(dialog));
              gtk_widget_destroy(dialog);
           }
        }
    }
    /* Stop the action */
    else {
        if(values.systray_icon != SYSTRAY_ICON_ALWAYS)
            gui_systray_icon_hide();

        if(values.iconify == TRUE && values.systray_icon != SYSTRAY_ICON_NEVER)
            gtk_widget_show(main_window);

        if(progress_confirm_open)
           progress_destroy();

        values.action_scheduled = FALSE;

        /* Change systray image icon */
        path = data_path("tray_icon.png");
        gtk_image_set_from_file(GTK_IMAGE(tray_image), path);


        /* Remove timer */
        g_source_remove(func_ref);

        /* Time Type widgets */
        on_combo_time_type(combo_time_type, NULL);

        /* Other */
        gtk_widget_set_sensitive(GTK_WIDGET(combo_action), 1);
        gtk_widget_set_sensitive(GTK_WIDGET(combo_time_type), 1);
        gtk_button_set_label(btn_start, _("_Start"));    

        /* Change tray tooltip */
        gtk_tooltips_set_tip(tray_tooltips, tray_eventbox, _("No action scheduled"), NULL);
    }
}

/* }}}1 */

/* Systray {{{1 */

void on_shutdown_now(GtkWidget *widget, gpointer data)
{
    if(confirm_dialog(values.action_shutdown) == TRUE) {
       gui_notification(_("Computer is going to shutdown now !"), NOTIFICATION_TYPE_IMPORTANT, NOTIFICATION_ICON_TIME);
       action_run(values.action_shutdown);
    }
}

void on_reboot_now(GtkWidget *widget, gpointer data)
{
    if(confirm_dialog(values.action_restart) == TRUE) {
      gui_notification(_("Computer is going to reboot now !"), NOTIFICATION_TYPE_IMPORTANT, NOTIFICATION_ICON_TIME);
      action_run(values.action_restart);
    }
}

void on_logout_now(GtkWidget *widget, gpointer data)
{
    if(confirm_dialog(values.action_logout) == TRUE) {
      gui_notification(_("Current session will be terminated !"), NOTIFICATION_TYPE_IMPORTANT, NOTIFICATION_ICON_TIME);
      action_run(values.action_logout);
    }
}

void on_tray_icon(GtkWidget *button, GdkEventButton *event, GtkWidget *window)
{
    switch(event->button) {
        case 1:
            if(!GTK_WIDGET_VISIBLE(window))
                gtk_window_present(GTK_WINDOW(window));
            else
                gtk_widget_hide(window);

            break;

        case 2:
            gtk_menu_popup(GTK_MENU(create_mnu_tray_action_now()), 
                           NULL, NULL, NULL, NULL, 
                           event->button, event->time);
            break;

        case 3:
            gtk_menu_popup(traymenu(), NULL, NULL, NULL, NULL, 
                           event->button, event->time);
            break;
    }
}

/* }}}1 */

/* Progress confirm {{{1 */

void on_progress_cancel(GtkWidget *widget, gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(data));
    on_btn_start(btn_start, NULL);
}

void on_progress_execute(GtkWidget *widget, gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(data));   
    action_run(values.current_action);
    on_btn_start(btn_start, NULL);
}

void on_progress_destroy(GtkWidget *widget, gpointer data)
{
    guint *func_ref;
    func_ref = data;

    g_source_remove(*func_ref);
    g_free(func_ref);

    progress_confirm_open = FALSE;
}

/* }}}1 */

/* vim:cindent:et:sw=4:ts=4:sts=4:tw=78:fenc=utf-8
 */
