//*****************************************************************************
//                                FileTasks.cpp                               *
//                               ---------------                              *
// Started     : 28/05/2005                                                   *
// Last Update : 19/03/2010                                                   *
// Copyright   : (C) 2005 by M.S.Waters                                       *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "main/FileTasks.hpp"
#include "main/FrmMain.hpp"

//*****************************************************************************
// Constructor.
//
// Argument List :
//   poFrmMain - A pointer to the parent frame

FileTasks::FileTasks( FrmMain * poFrmMain )
{
  // Set the pointer to the parent frame
  m_poFrmMain = poFrmMain;

  // Get the global configuration object
  m_poCfg = (wxConfig *) wxConfig::Get( );
}

//*****************************************************************************
// Destructor.

FileTasks::~FileTasks( )
{
}

//*****************************************************************************
// Execute the schematic import process.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bExecImport( void )
{
  bool       bRtn;
  TextCtrl * poTxtCtl;

  // Display the console page
  m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );

  // Convert the schematic file/s to a netlist file
  bRtn = m_oPrcGNetList.bExec( );

  // Print the results
  poTxtCtl = m_poFrmMain->m_oNbkTxtCtls.poGetPage( );
  m_oPrcGNetList.Print( *poTxtCtl );

  // Check for errors in the gnetlist output
  if( poTxtCtl->GetValue( ).Contains( wxT("ERROR") ) )     bRtn = FALSE;
  if( poTxtCtl->GetValue( ).Contains( wxT("Backtrace") ) ) bRtn = FALSE;

  // Delete the process log file
  m_oPrcGNetList.bDelLogFile( );

  return( bRtn );
}

//*****************************************************************************
// Initialize the Guile procedure name to be used by gnetlist when importing
// schematic files.

void  FileTasks::InitGuileProc( void )
{
  wxString  os1;

  // Set the Guile procedure name
  m_poCfg->SetPath( wxT("/gNetList") );
  os1 = m_poCfg->Read( wxT("GuileProc"), wxT("spice-sdb") );
  bSetGuileProc( os1 );
}

//*****************************************************************************
// Initialize the schematic file name/s.

void  FileTasks::InitSchemFiles( void )
{
  wxString  os1;

  // Get the schematic file name/s
  m_poCfg->SetPath( wxT("/Files") );
  os1 = m_poCfg->Read( wxT("Schematics"), wxT("") );
  if( os1.IsEmpty( ) )          return;

  // Set the schematic file name/s
  if( ! bSetSchemFiles( os1 ) ) return;

  // Return if a netlist file has also been specified
  os1 = m_poCfg->Read( wxT("NetList"), wxT("") );
  if( ! os1.IsEmpty( ) )        return;

  // Import the schematic file/s
  bImport( );
}

//*****************************************************************************
// Initialize the netlist file name.

void  FileTasks::InitNetLstFile( void )
{
  wxString  os1;

  // Return if a netlist file has already been loaded
  if( ! m_poFrmMain->m_oNetLst.bIsEmpty( ) ) return;

  // Get the netlist file name
  m_poCfg->SetPath( wxT("/Files") );
  os1 = m_poCfg->Read( wxT("NetList"), wxT("") );
  if( os1.IsEmpty( ) )                       return;

  // Set the netlist file name
  if( ! bSetNetLstFile( os1 ) )              return;

  // Open the netlist file
  bOpen( );

  // Does the netlist file contain simulator specific information?
  m_poCfg->SetPath( wxT("/Simulator") );
  if(      m_poFrmMain->m_oSimnNgs.m_oCmdPR.bIsValid( ) )
  {
    m_poCfg->Write( wxT("Engine"), CLP_NGSPICE );
    m_poFrmMain->m_oSimnGcp = m_poFrmMain->m_oSimnNgs;
  }
  else if( m_poFrmMain->m_oSimnGcp.m_oCmdPR.bIsValid( ) )
  {
    m_poCfg->Write( wxT("Engine"), CLP_GNUCAP  );
    m_poFrmMain->m_oSimnNgs = m_poFrmMain->m_oSimnGcp;
  }
}

//*****************************************************************************
// Display a error message dialog when schematic file/s can't be imported.

void  FileTasks::DlgErrSchems( void )
{
  wxString  os1, os2;
  size_t    szt1;

  for( szt1=0; szt1<m_oPrcGNetList.rofnaGetSchemFiles( ).GetCount( ); szt1++ )
    os2 << wxT("     ") << m_oPrcGNetList.rofnGetSchemFile( szt1 ).GetFullPath( )
        << wxT("\n");

  os1 << wxT("The schematic file/s couldn't be converted to a netlist file :\n")
      << wxT("\n")
      << os2
      << wxT("\n")
      << wxT("(This is usually because gnetlist encountered error/s while\n")
      << wxT("attempting to convert the schematic file/s to a netlist. Try\n")
      << wxT("examining the console output to determine where the problem\n")
      << wxT("occurred.)\n");

  m_poFrmMain->DlgErrMsg( wxT("Import Schematic/s Error"), os1 );
}

//*****************************************************************************
// Display a error message dialog when a netlist file can't be loaded.

void  FileTasks::DlgErrNetLst( void )
{
  wxString  os1;

  os1 << wxT("The netlist file couldn't be loaded :\n")
      << wxT("\n")
      << wxT("     ") << m_oPrcGNetList.rofnGetNetLstFile( ).GetFullPath( ) << wxT('\n')
      << wxT("\n")
      << wxT("(This is often because the file is in-complete, empty or\n")
      << wxT("doesn't exist.)\n");

  m_poFrmMain->DlgErrMsg( wxT("Load Netlist Error"), os1 );
}

//*****************************************************************************
// Delete temporary files which may have been generated by this application.
// Ie. given the circuit description file "<netlist>.ckt" delete files of the
//     following form :
//       <netlist>.ngspice.dc, <netlist>.ngspice.ac, ... etc.
//       <netlist>.gnucap.op,  <netlist>.gnucap.dc, .... etc.
//       gnetlist.log

bool  FileTasks::bDelTmpFiles( void )
{
  wxArrayString  osa1;
  wxFileName     ofn1;
  wxString       os1, os2;
  long           liTmpFileMgt;
  int            i1;

  // Determine the current temporary file management strategy
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("TmpFileMgt"), &liTmpFileMgt, (long) eTFMS_DELETE );

  // Return if nothing needs to be done
  if( liTmpFileMgt == (long) eTFMS_KEEP ) return( TRUE );

  // Get the path to the schematic or circuit description file
  ofn1 = m_oPrcGNetList.rofnGetNetLstFile( );
  ofn1.Normalize( );
  if( !ofn1.IsOk( ) || !ofn1.FileExists( ) ) return( FALSE );

  // Look for the gnetlist log file
  os1 = ofn1.GetPath( ) + wxT("/gnetlist.log");
  if( wxFileExists( os1 ) ) osa1.Add( os1 );

  os1 = ofn1.GetName( );
  i1 = os1.Find( wxT(".gspiceui") );
  if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
  ofn1.SetFullName( os1 );

  // Look for any GNU-Cap results file
  os1 = ofn1.GetFullPath( ) + wxT(".gnucap.??");
  for( os2=wxFindFirstFile( os1 ); !os2.IsEmpty( ); os2=wxFindNextFile( ) )
    osa1.Add( os2 );

  // Look for any Ng-Spice results files
  os1 = ofn1.GetFullPath( ) + wxT(".ngspice.??");
  for( os2=wxFindFirstFile( os1 ); !os2.IsEmpty( ); os2=wxFindNextFile( ) )
    osa1.Add( os2 );

  // If necessary prompt the user for permission to delete temporary files
  if( osa1.GetCount( ) > 0 )
  {
    if( m_poFrmMain!=NULL && m_poFrmMain->IsShown( ) )
    {
      // Prompt the user
      if( liTmpFileMgt == (long) eTFMS_PROMPT )
      {
        os1 = wxT("Delete Temporary Files");
        os2 = wxT("\nDelete the following temporary files :\n\n");
        for( i1=0; i1<(int)osa1.GetCount( ); i1++ )
          os2 << wxT("  ") << osa1.Item( i1 ) << wxT("   \n");
        i1 = wxMessageBox( os2, os1, wxYES_NO|wxICON_QUESTION, m_poFrmMain );
      }
      else i1 = wxYES;

      // Delete the temporary files
      if( i1 == wxYES )
        for( i1=0; i1<(int)osa1.GetCount( ); i1++ )
          wxRemoveFile( osa1.Item( i1 ) );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Do initialization tasks.

void  FileTasks::Initialize( void )
{
  InitGuileProc ( );
  InitSchemFiles( );
  InitNetLstFile( );
}

//*****************************************************************************
// Check that the gNetList binary can be found, if not display an error
// message.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bIsOkGNetList( void )
{
  wxString  os1;

  // Check that gnetlist exists and is accessible
  if( ! m_oPrcGNetList.bBinExists( ) )
  {
    if( m_poFrmMain != NULL )
    {
      os1 << wxT("\nCan't find ") << m_oPrcGNetList.rofnGetBinary( ).GetFullName( )
          << wxT(" which is required to import a schematic\nfile. ")
          << wxT("There is no path to it or it has not been installed.\n\n");
      m_poFrmMain->DlgErrMsg( wxT("Configuration Fault"), os1 );
    }
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the application main frame title.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetTitle( void )
{
  wxFileName  ofn1;
  wxString    os1;

  if( m_poFrmMain == NULL ) return( FALSE );

  // Create the title line
  os1 << wxT(" ") << APP_NAME << wxT(",  ") << APP_VERSION;
  if( ! rosGetNetLstFile( ).IsEmpty( ) )
  {
    ofn1 = rosGetNetLstFile( );
    if( ! ofn1.IsAbsolute( ) ) ofn1.MakeAbsolute( );
    if( ofn1.GetFullPath( ).StartsWith( ofn1.GetHomeDir( ) ) )
      ofn1.MakeRelativeTo( ofn1.GetHomeDir( ) );

    os1 << wxT("  -  ");
    if( ! ofn1.IsAbsolute( ) ) os1 << wxT("~/");
    os1 << ofn1.GetFullPath( );
  }

  // Set the main frames title line
  m_poFrmMain->wxFrame::SetTitle( os1 );

  return( TRUE );
}

//*****************************************************************************
// Set a Guile procedure name to be used for importing schematic files using
// gNetList.
//
// Argument List :
//   rosPName - The Guile procedure name
//              (Refer to gNetList documentation for list of procedure names)

bool  FileTasks::bSetGuileProc( const wxString & rosPName )
{
  if( ! m_oPrcGNetList.bSetGuileProc( rosPName ) ) return( FALSE );

  // Record the Guile procedure name currently selected
  m_poCfg->SetPath( wxT("/gNetList") );
  m_poCfg->Write( wxT("GuileProc"), rosGetGuileProc( ) );

  // Write any changes to the configuration file
  m_poCfg->Flush( );

  return( TRUE );
}

//*****************************************************************************
// Set the schematic file name/s.
//
// Argument List :
//   rosFNames - A string containing the full path and file name/s
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetSchemFiles( const wxString & rosFNames )
{
  // Attempt to set the schematic file name/s in the gNetList process object
  if( ! m_oPrcGNetList.bSetSchemFiles( rosFNames ) ) return( FALSE );

  // Set the schematic file name/s
  if( m_poFrmMain != NULL )
    m_poFrmMain->m_oNetLst.bSetSchemFiles( rosFNames );

  // Record the schematic file name/s in the configuration object
  m_poCfg->SetPath( wxT("/Files") );
  m_poCfg->Write( wxT("Schematics"), rosFNames );
  m_poCfg->Flush( );

  return( TRUE );
}

//*****************************************************************************
// Set the netlist file name.
//
// Argument List :
//   psFileName - A string containing the full path and file name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetNetLstFile( const wxString & rosFName )
{
  // Attempt to set the netlist file name in the gNetList process object
  if( ! m_oPrcGNetList.bSetNetLstFile( rosFName ) ) return( FALSE );

  // Configure stuff in the main frame
  if( m_poFrmMain != NULL )
  {
    m_poFrmMain->m_oNetLst.bSetLoadFile( rosFName );  // Set load file path
    m_poFrmMain->m_oNetLst.bSetSaveFile( rosFName );  // Set save file path
  }

  // Record the netlist file name in the configuration object
  m_poCfg->SetPath( wxT("/Files") );
  m_poCfg->Write( wxT("NetList"), rosGetNetLstFile( ) );
  m_poCfg->Flush( );

  return( TRUE );
}

//*****************************************************************************
// Get the currently selected Guile procedure.
//
// Return Values :
//   The currently selected Guile procedure

const wxString & FileTasks::rosGetGuileProc( void )
{
  return( m_oPrcGNetList.rosGetGuileProc( ) );
}

//*****************************************************************************
// Get an array containing the schematic file name/s.
//
// Return Values :
//   An array of schematic file names

const wxArrayString & FileTasks::rosaGetSchemFiles( void )
{
  static  wxArrayString  osa1;
  wxString  os1;
  size_t    szt1;

  osa1.Clear( );

  for( szt1=0; szt1<m_oPrcGNetList.rofnaGetSchemFiles( ).GetCount( ); szt1++ )
  {
    os1 = m_oPrcGNetList.rofnGetSchemFile( szt1 ).GetFullPath( );
    if( ! os1.IsEmpty( ) ) osa1.Add( os1 );
  }

  return( osa1 );
}

//*****************************************************************************
// Get the netlist file name.
//
// Return Values :
//   The netlist file name

const wxString & FileTasks::rosGetNetLstFile( void )
{
  static  wxString  osNetLstFile;

  osNetLstFile = m_oPrcGNetList.rofnGetNetLstFile( ).GetFullPath( );

  return( osNetLstFile );
}

//*****************************************************************************
// Display an open file dialog and set the netlist file name.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bDlgOpen( void )
{
  wxFileDialog * poDlgOpen;
  wxString       os1, os2;
  long           li1;

  // Can't display dialogue unless the application main frame has been created
  if( m_poFrmMain == NULL )                return( FALSE );

  // Create the different file filters
  os1 << wxT("All files (*)|*|")
      << wxT("Circuit files (*.ckt)|*.ckt|")
      << wxT("Circuit files (*.cir)|*.cir|")
      << wxT("Netlist files (*.net)|*.net");

  // Get the netlist file path from the global configuration object
  m_poCfg->SetPath( wxT("/Directories") );
  os2 = m_poCfg->Read( wxT("LastAccess"), wxGetHomeDir( ) );

  // Set the style bit pattern
#if wxCHECK_VERSION( 2,8,0 )
  li1 = wxFD_OPEN | wxFD_CHANGE_DIR | wxFD_FILE_MUST_EXIST;
#else
  li1 = wxOPEN | wxCHANGE_DIR | wxFILE_MUST_EXIST;
#endif

  // Create and configure the file open dialog
  poDlgOpen = new wxFileDialog( m_poFrmMain, wxT(""), wxT(""), wxT(""),
                                wxT(""), li1 );
  poDlgOpen->SetMessage( wxT("Open a Circuit Description File") );
  poDlgOpen->SetWildcard( os1 );
  poDlgOpen->SetFilterIndex( 1 );
  poDlgOpen->SetDirectory( os2 );

  // Display file open dialog
  if( poDlgOpen->ShowModal( ) != wxID_OK ) return( FALSE );

  // Set the netlist file name
  m_oPrcGNetList.bClear( );
  os1 = poDlgOpen->GetPath( );
  if( ! bSetNetLstFile( os1 ) )            return( FALSE );

  // Set the path last accessed in the global configuration object
  m_poCfg->SetPath( wxT("/Directories") );
  m_poCfg->Write( wxT("LastAccess"), poDlgOpen->GetDirectory( ) );
  m_poCfg->Flush( );

  return( TRUE );
}

//*****************************************************************************
// Display an import file/s dialog and set the schematic file name/s.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bDlgImport( void )
{
  wxFileDialog * poDlgImport;
  wxArrayString  osa1;
  wxString       os1, os2;
  long           li1;
  size_t         sz1;

  // Can't display dialogue unless the application main frame has been created
  if( m_poFrmMain == NULL )                  return( FALSE );

  // Create the different file filters
  os1 << wxT("gSchem files (*.sch)|*.sch|")
      << wxT("Protel II files (*.\?\?\?)|*.\?\?\?");

  // Get the netlist file path from the global configuration object
  m_poCfg->SetPath( wxT("/Directories") );
  os2 = m_poCfg->Read( wxT("LastAccess"), wxGetHomeDir( ) );

  // Set the style bit pattern
#if wxCHECK_VERSION( 2,8,0 )
  li1 = wxFD_OPEN | wxFD_CHANGE_DIR | wxFD_MULTIPLE | wxFD_FILE_MUST_EXIST;
#else
  li1 = wxOPEN | wxCHANGE_DIR | wxMULTIPLE | wxFILE_MUST_EXIST;
#endif

  // Create and configure the file import dialog
  poDlgImport = new wxFileDialog( m_poFrmMain, wxT(""), wxT(""), wxT(""),
                                  wxT(""), li1 );
  poDlgImport->SetMessage( wxT("Import a Schematic File/s") );
  poDlgImport->SetWildcard( os1 );
  poDlgImport->SetFilterIndex( 0 );
  poDlgImport->SetDirectory( os2 );

  // Display file import dialog
  if( poDlgImport->ShowModal( ) != wxID_OK ) return( FALSE );

  // Set the guile procedure and schematic file name/s
  m_oPrcGNetList.bClear( );
  bSetGuileProc( wxT("spice-sdb") );
  poDlgImport->GetPaths( osa1 );
  os1 = osa1.Item( 0 );
  for( sz1=1; sz1<osa1.GetCount( ); sz1++ )
    os1 << osa1.Item( sz1 ) << wxT(' ');
  if( ! bSetSchemFiles( os1 ) )              return( FALSE );

  // Set the path last accessed in the global configuration object
  m_poCfg->SetPath( wxT("/Directories") );
  m_poCfg->Write( wxT("LastAccess"), poDlgImport->GetDirectory( ) );
  m_poCfg->Flush( );

  return( TRUE );
}

//*****************************************************************************
// Open and load a circuit description (netlist) file into the FrmMain
// attribute simulation objects.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bOpen( void )
{
  bool  bRtn=TRUE;

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Clear the main frame object attributes
  m_poFrmMain->bClear( );

  // Attempt to load the circuit description file
  m_poFrmMain->m_oSimnNgs.bLoadFile( );
  m_poFrmMain->m_oSimnGcp.bLoadFile( );
  if( m_poFrmMain->m_oNetLst.bIsValid( ) )
    m_poFrmMain->SetStatusText( wxT(" Netlist file opened successfully"), 1 );
  else
  {
    DlgErrNetLst( );
    m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    bRtn = FALSE;
  }

  // Set the main frame title and the schematic file name/s if supplied
  bSetTitle( );
  bSetSchemFiles( m_poFrmMain->m_oNetLst.rosGetSchemFiles( ) );

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( bRtn );
}

//*****************************************************************************
// Import a schematic file by converting it to a netlist using gNetList and
// then loading it into the FrmMain simulation object.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bImport( void )
{
  bool  bRtn=TRUE;

  // Check that gNetList binary exists and is accessible
  if( ! bIsOkGNetList( ) ) return( FALSE );

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Clear the main frame object attributes
  m_poFrmMain->bClear( );

  // Convert the schematic file/s to a netlist file and load it
  if( bExecImport( ) )
  {
    bSetNetLstFile( m_oPrcGNetList.rofnGetNetLstFile( ).GetFullPath( ) );
    m_poFrmMain->m_oSimnNgs.bLoadFile( );
    m_poFrmMain->m_oSimnGcp.bLoadFile( );
    if( m_poFrmMain->m_oNetLst.bIsValid( ) )
      m_poFrmMain->SetStatusText( wxT(" Schematic file/s imported successfully"), 1 );
    else
    {
      DlgErrNetLst( );
      m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
      bRtn = FALSE;
    }
  }
  else
  {
    DlgErrSchems( );
    m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
    m_poFrmMain->m_oNbkTxtCtls.bSetPosn( -1 );
    bRtn = FALSE;
  }

  // Set the main frame title
  bSetTitle( );

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( bRtn );
}

//*****************************************************************************
// Reload a simulation object whether it be from a schematic or netlist file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bReload( void )
{
  bool  bRtn=TRUE;

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Re-initialize the text control notebook
  m_poFrmMain->m_oNbkTxtCtls.bInitialize( );

  // If schematic file/s have been specified perform an import operation
  if( ! m_oPrcGNetList.rofnaGetSchemFiles( ).IsEmpty( ) )
  {
    if( bImport( ) )
      m_poFrmMain->SetStatusText( wxT(" Schematic file/s re-imported successfully"), 1 );
    else
      bRtn = FALSE;
  }
  else if( m_oPrcGNetList.rofnGetNetLstFile( ).IsOk( ) )
  {
    if( bOpen( ) )
      m_poFrmMain->SetStatusText( wxT(" Netlist file reloaded successfully"), 1 );
    else
      bRtn = FALSE;
  }
  else
  {
    // There was no schematic or netlist to load, display an error message
    bRtn = FALSE;
    m_poFrmMain->DlgErrMsg( wxT("Reload Operation Error"),
                            wxT("No file is currently open.") );
  }

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( bRtn );
}

//*****************************************************************************
// Close the circuit description file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bClose( void )
{
  // Delete temporary files
  bDelTmpFiles( );

  // Clear file names
  m_oPrcGNetList.bClear( );
  bSetNetLstFile( wxT("") );
  bSetSchemFiles( wxT("") );

  return( TRUE );
}

//*****************************************************************************
// Do the necessary tasks before the application exits.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bExit( void )
{
  // Delete temporary file/s
  if( ! bDelTmpFiles( ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
