//*****************************************************************************
//                                NetList.cpp                                 *
//                               -------------                                *
// Started     : 01/09/2003                                                   *
// Last Update : 15/04/2010                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/NetList.hpp"

//*****************************************************************************
// Allocate storage for static data members.

wxFileName      NetList::m_ofnLoadFile;
wxFileName      NetList::m_ofnSaveFile;
ArrayFileName   NetList::m_ofnaSchemFiles;

wxArrayString   NetList::m_osaNetLst;
wxArrayString   NetList::m_osaTitle;
wxArrayString   NetList::m_osaIncludes;
ArrayComponent  NetList::m_oaCpnts;
wxArrayString   NetList::m_osaModels;
wxArrayString   NetList::m_osaSubCcts;

wxArrayString   NetList::m_osaNodeLbls;

bool            NetList::m_bIsValid = FALSE;

//*****************************************************************************
// Constructor.

NetList::NetList( void )
{
  m_bIsValid = FALSE;
}

//*****************************************************************************
// Destructor.

NetList::~NetList( )
{
}

//*****************************************************************************
// Extract the data that has been loaded into the string array this class is
// derived from.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtract( void )
{
  if( ! m_osaNetLst.IsEmpty( ) )
  {
    m_bIsValid = TRUE;

    if( ! bExtractTitle     ( ) ) m_bIsValid = FALSE;
    if( ! bExtractIncludes  ( ) ) m_bIsValid = FALSE;
    if( ! bExtractCpnts     ( ) ) m_bIsValid = FALSE;
    if( ! bExtractModels    ( ) ) m_bIsValid = FALSE;
    if( ! bExtractSubCcts   ( ) ) m_bIsValid = FALSE;
    if( ! bExtractNodeLbls  ( ) ) m_bIsValid = FALSE;
    if( ! bExtractSchemFiles( ) ) ;
  }
  else return( FALSE );

  return( m_bIsValid );
}

//*****************************************************************************
// Extract the title for the netlist.
// Traditionally the first line in the netlist is the title.
// An '*' at the start of a line seems to be used as a comment designator.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractTitle( void )
{
  // Check that the circuit isn't empty
  if( m_osaNetLst.IsEmpty( ) )          return( FALSE );

  // If the first line is empty assume there is no title
  wxString  os1 = m_osaNetLst.Item( 0 );
  if( os1.IsEmpty( ) )                  return( TRUE );

  // If the first line doesn't begin with an '*' then assume it is the title
  if( os1.GetChar( 0 ) != wxT('*') )
  {
    os1 = wxT("* ") + os1;
    m_osaTitle.Add( os1 );
    return( TRUE );
  }

  // If the netlist was created using gnetlist the first line will look
  // something like the following :
  //   gnetlist -v -g spice-sdb -o filter-lp-1.ckt filter-lp-1.sch
  if( os1.Contains( wxT("gnetlist") ) ) return( TRUE );

  // If the netlist was created by gSpiceUI the start of the file will look
  // something like the following :
  //   **************************************************************
  //   *  Electronic circuit simulation file generated by gSpiceUI  *
  //   *             Version 0.9.93 Alpha (28/12/2008)              *
  //   **************************************************************
  if(       m_osaNetLst.Item(0).find_first_not_of(wxT('*'))!=wxString::npos ) ;
  else if( !m_osaNetLst.Item(1).Contains( wxT("gSpiceUI") )                 ) ;
  else if( !m_osaNetLst.Item(2).Contains( wxT("Version") )                  ) ;
  else if(  m_osaNetLst.Item(3).find_first_not_of(wxT('*'))!=wxString::npos ) ;
  else                                  return( TRUE );

  // Accept all lines beginning with an '*' as the title
  for( size_t sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = m_osaNetLst.Item( sz1 );
    if( os1.GetChar( 0 ) != wxT('*') ) break;
    m_osaTitle.Add( os1 );
  }

  return( TRUE );
}

//*****************************************************************************
// Extract any include directives.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractIncludes( void )
{
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and an include directive
  if( m_osaNetLst.GetCount( ) < 4 ) return( TRUE );

  // Scan circuit description for include directives
  for( sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = m_osaNetLst.Item( sz1 );

    if( bIsSubCkt( os1 ) ) continue;
    if( os1.IsEmpty( )   ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".INCLUDE ") ) ) m_osaIncludes.Add( os1 );
  }

  return( TRUE );
}

//*****************************************************************************
// Extract all the component description lines from the circuit description.
// A component line is identified if the first character in the line is an
// alpha character and its length is greater than 8. The following examples
// shows a component description of minimum length (a resistor connected to
// nodes 1 and 0 of value 9 Ohm) :
//
//   R1 1 0 9
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractCpnts( void )
{
  wxString   os1, os2;
  size_t     sz1;
  Component  tCpnt;

  // Need at least a title line and 2 components
  if( m_osaNetLst.GetCount( ) < 3 ) return( FALSE );

  // Reset sub-circuit detection function
  os1 = wxT(".ENDS");
  bIsSubCkt( os1 );

  // Scan circuit description for components
  for( sz1=1; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    // Retrieve the next line from the netlist
    os1 = m_osaNetLst.Item( sz1 );

    // Ignore all component definitions inside a sub-circuit block
    if( bIsSubCkt( os1 ) )    continue;

    // Determine if this line is a valid component description
    tCpnt = os1;
    if( ! tCpnt.bIsValid( ) ) continue;

    // A valid component description was found
    m_oaCpnts.Add( tCpnt );
  }

  // Circuit description must have components
  if( m_oaCpnts.IsEmpty( ) )         return( FALSE );

  m_oaCpnts.Sort( &Component::iCompare );

  return( TRUE );
}

//*****************************************************************************
// Extract all the model description lines from the circuit description.
// A model description can consist of 1 or more lines and is identified if the
// line begins with ".MODEL ". Subsequent lines beginning with a '+' character
// are appended to the model description. The format is illustrated in the
// following example:
//
//   .MODEL CMOSN NMOS (LEVEL=2 LD=0.265073u TOX=418.0e-10
//   + NSUB=1.53142e+16 VTO=0.844345 KP=4.15964e-05 GAMMA=0.863074
//   + CJ=0.0003844 MJ=0.488400 CJSW=5.272e-10 MJSW=0.300200 PB=0.700000)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractModels( void )
{
  wxString  osModel;
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and a model description line
  if( m_osaNetLst.GetCount( ) < 4 ) return( TRUE );

  // Reset sub-circuit detection function
  os1 = wxT(".ENDS");
  bIsSubCkt( os1 );

  // Scan circuit description for models
  for( sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = m_osaNetLst.Item( sz1 );

    if( bIsSubCkt( os1 ) )                     continue;
    if( os1.IsEmpty( ) && osModel.IsEmpty( ) ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".MODEL ") ) )
    { // First line of model description found
      if( ! osModel.IsEmpty( ) ) m_osaModels.Add( osModel );
      osModel = os1;
    }
    else if( os1.GetChar( 0 ) == wxT('+') )
    { // Intermediate line of model description found
      if( osModel.Length( ) > 1 ) osModel << wxT('\n') << os1;
    }
    else if( ! osModel.IsEmpty( ) )
    { // Last line of model description found
      m_osaModels.Add( osModel );
      osModel.Empty( );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Extract any sub-circuit descriptions from the circuit description (netlist).
// The format of a sub-circuit description is illustrated in the following
// example :
//
//   .SUBCKT CCTNAME 1 5
//   R1 1 2 1K
//   R2 2 3 2K
//   R3 3 4 3K
//   R4 4 5 4K
//   .ENDS CCTNAME
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractSubCcts( void )
{
  wxString  osSubCct;
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and a sub-circuit description
  // containing 2 lines
  if( m_osaNetLst.GetCount( ) < 7 ) return( TRUE );

  // Scan circuit description for sub-circuits
  for( sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = m_osaNetLst.Item( sz1 );

    if( os1.IsEmpty( ) && osSubCct.IsEmpty( ) ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".SUBCKT ") ) )
    { // First line in sub-circuit description found
      osSubCct = os1;
    }
    else if( os2.StartsWith( wxT(".ENDS") ) )
    { // Last line in sub-circuit description found
      if( ! osSubCct.IsEmpty( ) )
      {
        osSubCct << wxT('\n') << os1;
        m_osaSubCcts.Add( osSubCct );
        osSubCct.Empty( );
      }
    }
    else if( ! osSubCct.IsEmpty( ) )
    { // Intermediate line in sub-circuit description found
      osSubCct << wxT('\n') << os1;
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Extract all nodes labels from the component line string array.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractNodeLbls( void )
{
  wxString  os1;
  size_t    sz1, sz2;
  int       i1;

  // Check that there are some components
  if( m_oaCpnts.IsEmpty( ) )     return( FALSE );

  for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
  {
    const Component & roCpnt = m_oaCpnts.Item( sz1 );
    if( roCpnt.m_osaNodes.IsEmpty( ) ) continue;

    for( sz2=0; sz2<roCpnt.m_osaNodes.GetCount( ); sz2++ )
    {
      os1 = roCpnt.m_osaNodes.Item( sz2 );
      if( m_osaNodeLbls.Index( os1 ) == wxNOT_FOUND ) m_osaNodeLbls.Add( os1 );
    }
  }

  if( m_osaNodeLbls.IsEmpty( ) ) return( FALSE );

  // Remove the earth/ground nodes if present
  if( ( i1=m_osaNodeLbls.Index( wxT("0") ) ) != wxNOT_FOUND )
    m_osaNodeLbls.RemoveAt( i1 );
  if( ( i1=m_osaNodeLbls.Index( wxT("GND") ) ) != wxNOT_FOUND )
    m_osaNodeLbls.RemoveAt( i1 );

  m_osaNodeLbls.Sort( &iStrCmp );

  return( TRUE );
}

//*****************************************************************************
// Extract the schematic file names from which the netlist was derived.
//
// When gSpiceUI imports a netlist from schematic file/s it places a reference
// to the schematic file/s near the top of the netlist. Eg. :
//   * Schematics : circuit1.sch circuit2.sch
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractSchemFiles( void )
{
  wxString  os1, os2;
  size_t    sz1;

  // Don't clear the current list of schematic file names
  // m_ofnaSchemFiles.Empty( );

  // Scan circuit description for schematic file name/s
  for( sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = m_osaNetLst.Item( sz1 );

    if( os1.IsEmpty( )   )             continue;
    if( os1.GetChar( 0 ) != wxT('*') ) continue;
    if( os1.StartsWith( wxT("* Schematic : "), &os2 ) )
    {
      if( ! bSetSchemFiles( os2 ) ) return( FALSE );
      break;
    }
  }

  return( TRUE );
}

//*****************************************************************************
// This function indicates whether a line in a sequence of line falls within a
// sub-circuit definition block.
//
// Argument List :
//   A line in a sequence of lines to be tested
//
// Return Values :
//   TRUE  - The line does    falls within a su-circuit definition block
//   FALSE - The line doesn't falls within a su-circuit definition block

bool  NetList::bIsSubCkt( wxString & roLine )
{
  static  bool  bIsSubCkt=FALSE;
  wxString  os1;

  os1 = roLine;
  os1.MakeUpper( );
  if( os1.StartsWith( wxT(".SUBCKT") ) ) bIsSubCkt = TRUE;
  if( os1.StartsWith( wxT(".ENDS") ) )   bIsSubCkt = FALSE;

  return( bIsSubCkt );
}

//*****************************************************************************
// Makes the object and its attributes empty and frees memory allocated to it.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bClear( void )
{
  m_osaNetLst  .Clear( );

  m_osaTitle   .Clear( );
  m_osaIncludes.Clear( );
  m_oaCpnts    .Clear( );
  m_osaModels  .Clear( );
  m_osaSubCcts .Clear( );

  m_osaNodeLbls.Clear( );

  m_bIsValid = FALSE;

  return( TRUE );
}

//*****************************************************************************
// Do the current object attributes constitute a valid netlist?
//
// Return Values:
//   TRUE  - Valid
//   FALSE - Not valid

bool  NetList::bValidate( void )
{
  m_bIsValid = TRUE;

  // Check that there is a title
  if( m_osaTitle.IsEmpty( ) )     { m_bIsValid = FALSE; return( FALSE ); }

  // Check that there are at least two components defined
  if( m_oaCpnts.GetCount( ) < 2 ) { m_bIsValid = FALSE; return( FALSE ); }

  return( TRUE );
}

//*****************************************************************************
// Load (or reload) a circuit description from file.
//
// Argument List :
//   rosFName - The name of the file to be loaded
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bLoadFile( const wxString & rosFName )
{
  wxString  os1;

  // Is this a load or reload?
  if( ! rosFName.IsEmpty( ) )
       { if( ! bSetLoadFile( rosFName )          )        return( FALSE ); }
  else { if( m_ofnLoadFile.GetFullPath( ).IsEmpty( ) )    return( FALSE ); }

  bClear( );  // Clear the object attributes

  // Open the file
  wxTextFile  oFileCct( m_ofnLoadFile.GetFullPath( ) );
  oFileCct.Open( );
  if( ! oFileCct.IsOpened( )       ) { oFileCct.Close( ); return( FALSE ); }
  if( oFileCct.GetLineCount( ) < 3 ) { oFileCct.Close( ); return( FALSE ); }

  // Load the file contents
  for( os1=oFileCct.GetFirstLine(); !oFileCct.Eof(); os1=oFileCct.GetNextLine() )
    m_osaNetLst.Add( os1 );

  // Need the following line in case the last line in the file is not empty
  // but has no line termination character
  if( os1.Length( ) > 0 ) m_osaNetLst.Add( os1 );

  oFileCct.Close( ); // Close the file

  // Attempt to extract the circuit description from file
  if( ! bExtract( ) )                                     return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Save (or resave) the circuit to file.
//
// Argument List :
//   rosFName - The name of the file to be saved
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSaveFile( const wxString & rosFName )
{
  wxString  os1;
  wxString  os2;
  size_t    sz1;

  // Is this a save or resave?
  if( ! rosFName.IsEmpty( ) )
       { if( ! bSetSaveFile( rosFName ) ) return( FALSE ); }
  else { if( ! m_ofnSaveFile.IsOk( )    ) return( FALSE ); }

  // Open the file
  wxTextFile  oFileCct( m_ofnSaveFile.GetFullPath( ) );
  if( oFileCct.Exists( ) )
       { if( ! oFileCct.Open( )   )       return( FALSE ); }
  else { if( ! oFileCct.Create( ) )       return( FALSE ); }

  // Clear the file if it contains lines
  for( sz1=oFileCct.GetLineCount( ); sz1>0; sz1-- )
    oFileCct.RemoveLine( 0 );

  // Save the circuit description title line/s
  os1 = wxT("**************************************************************");
  oFileCct.AddLine( os1 );
  os2 = wxT("*  Electronic circuit simulation file generated by gSpiceUI  *");
  oFileCct.AddLine( os2 );
  os2.Empty( );
  os2 << wxT("*             ") << APP_VERSION << wxT("              *");
  oFileCct.AddLine( os2 );
  oFileCct.AddLine( os1 );
  oFileCct.AddLine( wxT("") );

  // Save the schematic file name/s if specified
  if( ! m_ofnaSchemFiles.IsEmpty( ) )
  {
    os1 = wxT("* Schematic : ") + rosGetSchemFiles( );
    oFileCct.AddLine( os1 );
    oFileCct.AddLine( wxT("") );
  }

  // Save the title line/s
  if( ! m_osaTitle.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaTitle.GetCount( ); sz1++ )
      oFileCct.AddLine( m_osaTitle[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the include directives
  if( ! m_osaIncludes.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Include Directives") );
    for( sz1=0; sz1<m_osaIncludes.GetCount( ); sz1++ )
      oFileCct.AddLine( m_osaIncludes[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the component definition lines
  oFileCct.AddLine( wxT("* Component Definitions") );
  for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
    oFileCct.AddLine( m_oaCpnts[ sz1 ].rosGetString( ) );
  oFileCct.AddLine( wxT("") );

  // Save the sub-circuit definitions
  if( ! m_osaSubCcts.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Sub-Circuit Definitions") );
    for( sz1=0; sz1<m_osaSubCcts.GetCount( ); sz1++ )
      oFileCct.AddLine( m_osaSubCcts[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the model definition lines
  if( ! m_osaModels.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Model Definitions") );
    for( sz1=0; sz1<m_osaModels.GetCount( ); sz1++ )
      oFileCct.AddLine( m_osaModels[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the circuit description terminator
  oFileCct.AddLine( wxT(".END") );

  // Save the changes to disk
  oFileCct.Write( );
  oFileCct.Close( );

  return( TRUE );
}

//*****************************************************************************
// Set the load (netlist) file name.
//
// Argument List :
//   rosFName - The load file name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetLoadFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  ofn1 = rosFName;

  // Is the file name valid and does it exist?
  if( ! ofn1.IsOk( ) )       return( FALSE );
  if( ! ofn1.FileExists( ) ) return( FALSE );

  m_ofnLoadFile = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Set the save (netlist) file name.
//
// Argument List :
//   rosFName - The save file name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetSaveFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  ofn1 = rosFName;

  // Is the file name valid?
  if( ! ofn1.IsOk( ) ) return( FALSE );

  m_ofnSaveFile = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Set the schematic file name/s from which the netlist file was derived.
//
// Argument List :
//   rosaFName - The schematic file name/s
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetSchemFiles( const wxArrayString & rosaFNames )
{
  wxFileName  ofn1;
  size_t      sz1;

  m_ofnaSchemFiles.Empty( );

  for( sz1=0; sz1<rosaFNames.GetCount( ); sz1++ )
  {
    ofn1 = rosaFNames.Item( sz1 );

    // Is the file name valid and does it exist?
    if( ! ofn1.IsOk( ) )       continue;
    if( ! ofn1.FileExists( ) ) continue;

    m_ofnaSchemFiles.Add( ofn1 );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the schematic file name/s from which the netlist was derived.
//
// Argument List :
//   rosFName - The schematic file name/s
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetSchemFiles( const wxString & rosFNames )
{
  wxStringTokenizer  ostk1;
  wxArrayString      osa1;

  m_ofnaSchemFiles.Empty( );

  ostk1.SetString( rosFNames );
  while( ostk1.HasMoreTokens( ) )
    osa1.Add( ostk1.GetNextToken( ) );

  return( bSetSchemFiles( osa1 ) );
}

//*****************************************************************************
// Return a reference to a wxString object containing a list of the schematic
// file names.
//
// Argument List :
//   rosName - The component name (label)
//
// Return Values :
//   Success - The component reference in m_oaCpnts
//   Failure - An empty Component object

const wxString & NetList::rosGetSchemFiles( void )
{
  static  wxString  osSchemFiles;
  size_t  sz1;

  osSchemFiles.Empty( );

  for( sz1=0; sz1<m_ofnaSchemFiles.GetCount( ); sz1++ )
  {
    if( ! osSchemFiles.IsEmpty( ) )
      osSchemFiles << wxT(' ');
    osSchemFiles << m_ofnaSchemFiles.Item( sz1 ).GetFullPath( );
  }

  return( osSchemFiles );
}

//*****************************************************************************
// Get a reference to the Component object in m_oaCpnts with a specified name.
//
// Argument List :
//   rosName - The component name (label)
//
// Return Values :
//   Success - The component reference in m_oaCpnts
//   Failure - An empty Component object

const Component & NetList::roGetCpnt( const wxString & rosName )
{
  static  Component  oCpntEmpty;
  wxString  os1, os2;
  size_t    sz1;

  if( rosName  .IsEmpty( ) ) return( oCpntEmpty );
  if( m_oaCpnts.IsEmpty( ) ) return( oCpntEmpty );

  for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
    if( m_oaCpnts.Item( sz1 ).m_osName == rosName )
      return( m_oaCpnts.Item( sz1 ) );

  return( oCpntEmpty );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  NetList::Print( const wxString & rosPrefix )
{
  wxString  os1;
  size_t    sz1;

  os1 = m_ofnLoadFile.GetFullPath( );
  cout << rosPrefix.mb_str( ) << "m_ofnLoad        : "
       << os1.mb_str( ) << '\n';
  os1 = m_ofnSaveFile.GetFullPath( );
  cout << rosPrefix.mb_str( ) << "m_ofnSave        : "
       << os1.mb_str( ) << '\n';
  os1 = rosGetSchemFiles( );
  cout << rosPrefix.mb_str( ) << "m_osSchemFiles   : "
       << os1.mb_str( ) << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaNetLst[ ]   : ";
  if( ! m_osaNetLst.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaNetLst.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_osaNetLst.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaTitle[ ]    : ";
  if( ! m_osaTitle.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaTitle.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_osaTitle.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaIncludes[ ] : ";
  if( ! m_osaIncludes.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaIncludes.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_osaIncludes.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_oaCpnts[ ]     : ";
  if( ! m_oaCpnts.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_oaCpnts.Item( sz1 ).rosGetString( ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaModels[ ]   : ";
  if( ! m_osaModels.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaModels.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_osaModels.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaSubCcts[ ]  : ";
  if( ! m_osaSubCcts.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaSubCcts.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) cout << rosPrefix.mb_str( ) << "                   ";
      cout << m_osaSubCcts.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_bIsValid       : "
       << ( m_bIsValid ? "TRUE" : "FALSE") << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_NETLIST

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCpnt;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 3 )                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1.at( 0 ) == wxT('-') )
    {
      if( os1.at( 1 ) == wxT('h') ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
      else                          { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
    }
  }
  else                              { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Display the utility banner
  cout << "\n  Netlist Structure Test Utility"
       << "\n    Version 1.00 (08/05/2009)\n";

  // Create a NetList object
  NetList  tNetLst;

  cout << "\nLoad the netlist file : ";
  if( tNetLst.bLoadFile( os1 ) ) cout << "Success";
  else                           cout << "Failure";
  cout << "\n";

  cout << "\ntNetLst.Print( ) :\n";
  tNetLst.Print( );

  cout << "\n";

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS] [FILE]"
       << "\nOptions : -h   : Print usage (this message)"
       << "\n          FILE : A circuit description (NetList) file"
       << "\n\n";
}

#endif // TEST_NETLIST

//*****************************************************************************
