/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <gnome.h>
#include <gst/gst.h>

#include <gst/editor/editor.h>


/* class methods */
static void	gst_editor_item_class_init		(GstEditorItemClass *klass);
static void	gst_editor_item_init 			(GstEditorItem *item);
static void	gst_editor_item_set_property		(GObject *object, guint prop_id, 
                                                         const GValue *value, GParamSpec *pspec);
static void	gst_editor_item_get_property		(GObject *object, guint prop_id, 
                                                         GValue *value, GParamSpec *pspec);
static void	gst_editor_item_realize 		(GnomeCanvasItem *citem);
static gint	gst_editor_item_event			(GnomeCanvasItem *citem, GdkEvent *event);
static void	gst_editor_item_object_changed		(GstEditorItem *item, GstObject *object);
static void	gst_editor_item_resize_real	 	(GstEditorItem *item);
static void	gst_editor_item_repack_real 		(GstEditorItem *item);
static void	gst_editor_item_default_on_whats_this	(GstEditorItem *item);

/* callbacks on the parent item */
static void 	on_parent_item_position_changed 	(GstEditorItem *parent,
                                     			 GstEditorItem *item);

/* popup callbacks */
static void	on_whats_this				(GtkWidget *unused, GstEditorItem *item);


enum {
  ARG_0,
  ARG_WIDTH,
  ARG_HEIGHT,
  ARG_OBJECT,
};

enum {
  OBJECT_CHANGED,
  POSITION_CHANGED,
  LAST_SIGNAL
};

static GObjectClass *parent_class;
static guint gst_editor_item_signals[LAST_SIGNAL] = { 0 };
static GHashTable *editor_items = NULL;

static GnomeUIInfo menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("What's this?", "Give information on the currently selected item",
                          on_whats_this, "gtk-dialog-info"),
  GNOMEUIINFO_END
};

 
GType 
gst_editor_item_get_type (void) 
{
  static GType item_type = 0;

  if (!item_type) {
    static const GTypeInfo item_info = {
      sizeof (GstEditorItemClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_item_class_init,
      NULL,
      NULL,
      sizeof (GstEditorItem),
      0,
      (GInstanceInitFunc) gst_editor_item_init,
    };
      
    item_type = g_type_register_static (gnome_canvas_group_get_type (), "GstEditorItem", &item_info, 0);
  }
  return item_type;
}

static void 
gst_editor_item_class_init (GstEditorItemClass *klass) 
{
  GObjectClass *object_class;
  GnomeCanvasItemClass *citem_class;

  object_class = G_OBJECT_CLASS (klass);
  citem_class = GNOME_CANVAS_ITEM_CLASS (klass);

  parent_class = g_type_class_ref (gnome_canvas_group_get_type ());

  gst_editor_item_signals[OBJECT_CHANGED] =
    g_signal_new ("object-changed", G_OBJECT_CLASS_TYPE (klass), G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (GstEditorItemClass, object_changed), 
                  NULL, NULL, gst_editor_marshal_VOID__OBJECT, G_TYPE_NONE, 
                  1, GST_TYPE_OBJECT);
  gst_editor_item_signals[POSITION_CHANGED] =
    g_signal_new ("position-changed", G_OBJECT_CLASS_TYPE (klass), G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (GstEditorItemClass, position_changed), 
                  NULL, NULL, gst_editor_marshal_VOID__VOID, G_TYPE_NONE, 0);

  object_class->set_property = gst_editor_item_set_property;
  object_class->get_property = gst_editor_item_get_property;

  g_object_class_install_property (object_class, ARG_WIDTH,
       g_param_spec_double ("width", "width", "width",
			    0, G_MAXDOUBLE, 30, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, ARG_HEIGHT,
       g_param_spec_double ("height", "height", "height",
			    0, G_MAXDOUBLE, 10, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, ARG_OBJECT,
       g_param_spec_object ("object", "object", "object",
                             GST_TYPE_OBJECT, G_PARAM_READWRITE));

  citem_class->realize	= gst_editor_item_realize;
  citem_class->event	= gst_editor_item_event;
  klass->repack		= gst_editor_item_repack_real;
  klass->resize		= gst_editor_item_resize_real;
  klass->object_changed = gst_editor_item_object_changed;
  klass->whats_this	= gst_editor_item_default_on_whats_this;

  GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (klass, menu_items, 1);
}

static void 
gst_editor_item_init (GstEditorItem *item) 
{
  item->fill_color = 0xffffffff;
  item->outline_color = 0x333333ff;
  item->title_text = g_strdup ("rename me");
  item->textx = 1.0;
  item->texty = 1.0;
  item->height = 10;
  item->width = 30;
  item->textanchor = GTK_ANCHOR_NORTH_WEST;
}

static void 
gst_editor_item_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec) 
{
  GstEditorItem *item = GST_EDITOR_ITEM (object);

  switch (prop_id) {
    case ARG_WIDTH:
      item->width = g_value_get_double (value);
      break;
    case ARG_HEIGHT:
      item->height = g_value_get_double (value);
      break;
    case ARG_OBJECT:
      /* FIXME: unref the old object, if that's safe */
      g_signal_emit (object, gst_editor_item_signals[OBJECT_CHANGED], 0,
                     g_value_get_object (value), NULL);
      item->object = GST_OBJECT (g_value_get_object (value));
      if (item->title_text)
        g_free (item->title_text);
      item->title_text = g_strdup (item->object ? GST_OBJECT_NAME (item->object) : "");
      if (item->title)
        gnome_canvas_item_set (item->title, "text", item->title_text, NULL);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
  
  if (item->realized)
    gst_editor_item_resize (item);
}

static void 
gst_editor_item_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec) 
{
  GstEditorItem *item = GST_EDITOR_ITEM (object);

  switch (prop_id) {
    case ARG_WIDTH:
      g_value_set_double (value, item->width);
      break;
    case ARG_HEIGHT:
      g_value_set_double (value, item->height);
      break;
    case ARG_OBJECT:
      g_value_set_object (value, G_OBJECT (item->object));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void 
gst_editor_item_realize (GnomeCanvasItem *citem) 
{
  GstEditorItem *item = GST_EDITOR_ITEM (citem);

  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->realize)
    GNOME_CANVAS_ITEM_CLASS (parent_class)->realize (citem);

  item->border = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
                                        gnome_canvas_rect_get_type(),
                                        "width_units", 1.0,
                                        "fill_color_rgba", item->fill_color,
                                        "outline_color_rgba", item->outline_color,
                                        NULL);
  gnome_canvas_item_lower (item->border, 10);
  g_return_if_fail (item->border != NULL);
  GST_EDITOR_SET_OBJECT (item->border, item);

  item->title = gnome_canvas_item_new(GNOME_CANVAS_GROUP (citem),
                                      gnome_canvas_text_get_type(),
                                      "font", "Sans",
                                      "fill-color", "black",
                                      NULL);
  g_return_if_fail(item->title != NULL);
  gnome_canvas_item_set (item->title, "text", item->title_text, NULL);
  GST_EDITOR_SET_OBJECT(item->title,item);

  item->realized = TRUE;

  /* emission of position-changed on a parent item will propogate to all
     children */
  if (GST_IS_EDITOR_ITEM (citem->parent))
    g_signal_connect (citem->parent, "position-changed",
                      G_CALLBACK (on_parent_item_position_changed),
                      citem);

  if (G_OBJECT_TYPE (item) == GST_TYPE_EDITOR_ITEM)
    gst_editor_item_resize (item);
}

static void 
gst_editor_item_resize_real (GstEditorItem *item) 
{
  gdouble itemwidth, itemheight;

  /* it won't have a title prior to being realized */
  if (item->title) {
    g_object_get (G_OBJECT (item->title), "text-width", &itemwidth, NULL);
    item->t.w += itemwidth + 2.0;
    g_object_get (G_OBJECT (item->title), "text-height", &itemheight, NULL);
    item->t.h = MAX (item->t.h, itemheight + 2.0);
  }
  
  /* force the thing to grow if necessary */
  item->width = MAX (MAX (MAX (item->t.w, item->b.w),
                          item->l.w + item->c.w + item->r.w),
                     item->width);
  item->height = MAX (MAX (MAX (item->l.h, item->c.h), item->r.h) + item->t.h + item->b.h,
                      item->height);

  GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item))->repack (item);

//  g_signal_emit (G_OBJECT (item), gst_editor_item_signals[SIZE_CHANGED], 0, NULL);
}

static void 
gst_editor_item_repack_real (GstEditorItem *item) 
{
  gdouble x1,y1,x2,y2;

  if (!item->realized) return;

  x1 = 0;
  y1 = 0;
  x2 = x1 + item->width;
  y2 = y1 + item->height;

  /* resize the bordering box */
  gnome_canvas_item_set (item->border,
                         "x1", x1,
                         "y1", y1,
                         "x2", x2,
                         "y2", y2,
                         NULL);

  /* move the text to the right place */
  gnome_canvas_item_set (item->title,
                         "x", item->textx,
                         "y", item->texty,
                         "anchor", item->textanchor,
                         NULL);
}

static void
gst_editor_item_object_changed (GstEditorItem *item, GstObject *object)
{
  if (!editor_items)
    editor_items = g_hash_table_new (NULL, NULL);

  /* doesn't handle removals yet */
  g_hash_table_insert (editor_items, object, item);
}

static gint 
gst_editor_item_event (GnomeCanvasItem *citem,
                       GdkEvent *event)
{
  GstEditorItem *item;

  item = GST_EDITOR_ITEM (citem);

  switch(event->type) {
    case GDK_BUTTON_PRESS:
      if (event->button.button == 3) {
        GstEditorItemClass *itemclass = GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (citem));

        /* although we don't actually show a menu, stop the event propagation */
        if (itemclass->menu_items == 0)
          return TRUE;
        
        if (!itemclass->menu)
          itemclass->menu = gnome_popup_menu_new (itemclass->menu_items);
        
        gnome_popup_menu_do_popup (itemclass->menu, NULL, NULL, &event->button, item, NULL);
      }
      break;
    default:
      break;
  }

  /* we don't want the click falling through to the parent canvas item */
  return TRUE;
}

static void
gst_editor_item_default_on_whats_this (GstEditorItem *item)
{
  GtkWidget *dialog;

  dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE,
                                   "Items of type %s have not implemented a what's this page, sorry.",
                                   g_type_name (G_OBJECT_TYPE (item)));

  g_signal_connect_swapped (G_OBJECT (dialog), "response",
                            G_CALLBACK (gtk_widget_destroy),
                            G_OBJECT (dialog));

  gtk_widget_show (dialog);
}

/**********************************************************************
 * Callbacks on the parent editor item
 **********************************************************************/

static void
on_parent_item_position_changed (GstEditorItem *parent, GstEditorItem *item)
{
  g_return_if_fail (item != parent);
  
  g_signal_emit (G_OBJECT (item), gst_editor_item_signals[POSITION_CHANGED], 0, NULL);
}

/**********************************************************************
 * Popup menu callbacks
 **********************************************************************/

static void
on_whats_this (GtkWidget *unused, GstEditorItem *item)
{
  GstEditorItemClass *klass = GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item));
  
  g_return_if_fail (klass->whats_this != NULL);
  
  klass->whats_this (item);
}

/**********************************************************************
 * Public functions
 **********************************************************************/

void 
gst_editor_item_repack (GstEditorItem *item) 
{
  if (GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item))->repack)
    (GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item))->repack) (item);
}

void
gst_editor_item_resize (GstEditorItem *item)
{
  GstEditorItemBand empty = { 0.0, 0.0 };

  item->l = empty;
  item->r = empty;
  item->t = empty;
  item->b = empty;
  item->c = empty;

  if (GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item))->resize)
    (GST_EDITOR_ITEM_CLASS (G_OBJECT_GET_CLASS (item))->resize) (item);
}

GstEditorItem*
gst_editor_item_get (GstObject *object)
{
  GstEditorItem *ret;

  if (!editor_items)
    ret = NULL;
  else
    ret = (GstEditorItem*)g_hash_table_lookup (editor_items, object);

  return ret;
}

void
gst_editor_item_move (GstEditorItem *item, gdouble dx, gdouble dy)
{
  g_return_if_fail (GST_IS_EDITOR_ITEM (item));
  
  gnome_canvas_item_move ((GnomeCanvasItem*)item, dx, dy);
  
  g_signal_emit ((GObject*)item, gst_editor_item_signals[POSITION_CHANGED], 0, item);
}

