/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "caps-tree.h"


enum {
  NAME_COLUMN,
  INFO_COLUMN,
  NUM_COLUMNS
};

enum {
  PROP_0,
  PROP_ELEMENT_FACTORY,
  PROP_ELEMENT
};


static void	gst_element_browser_caps_tree_init	(GstElementBrowserCapsTree *ct);
static void	gst_element_browser_caps_tree_class_init(GstElementBrowserCapsTreeClass *klass);
static void 	gst_element_browser_caps_tree_set_property (GObject * object, guint prop_id, 
                                                            const GValue * value, GParamSpec * pspec);
static void 	gst_element_browser_caps_tree_get_property (GObject * object, guint prop_id,
                                                            GValue * value, GParamSpec * pspec);

static void	add_caps_to_tree 		(GstCaps *caps, GtkTreeStore *store,
                                                 GtkTreeIter *parent);
static gchar*	print_prop			(GstPropsEntry *prop);
static void	update_caps_tree		(GstElementBrowserCapsTree *ct);


static GtkScrolledWindowClass *parent_class = NULL;


GType
gst_element_browser_caps_tree_get_type (void)
{
  static GType element_browser_caps_tree_type = 0;
  
  if (!element_browser_caps_tree_type) {
    static const GTypeInfo element_browser_caps_tree_info = {
      sizeof(GstElementBrowserCapsTreeClass),
      NULL,
      NULL,
      (GClassInitFunc)gst_element_browser_caps_tree_class_init,
      NULL,
      NULL,
      sizeof(GstElementBrowserCapsTree),
      0,
      (GInstanceInitFunc)gst_element_browser_caps_tree_init,
    };
    element_browser_caps_tree_type =
      g_type_register_static (gtk_scrolled_window_get_type (), 
                              "GstElementBrowserCapsTree",
                              &element_browser_caps_tree_info, 0);
  }
  return element_browser_caps_tree_type;
}

static void
gst_element_browser_caps_tree_class_init (GstElementBrowserCapsTreeClass *klass)
{
  GObjectClass *gobject_class;
  
  gobject_class = G_OBJECT_CLASS (klass);
  
  parent_class = (GtkScrolledWindowClass*) g_type_class_ref (gtk_scrolled_window_get_type ());
  
  gobject_class->set_property = gst_element_browser_caps_tree_set_property;
  gobject_class->get_property = gst_element_browser_caps_tree_get_property;

  g_object_class_install_property (gobject_class, PROP_ELEMENT_FACTORY,
    g_param_spec_object ("element-factory", "Element factory", "Element factory",
                         gst_element_factory_get_type (), G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_ELEMENT,
    g_param_spec_object ("element", "Element", "Element",
                         gst_element_get_type (), G_PARAM_READWRITE));
}

static void gst_element_browser_caps_tree_init (GstElementBrowserCapsTree *ct)
{
  GtkTreeViewColumn *column;

  g_object_set (G_OBJECT (ct), "hadjustment", NULL, "vadjustment", NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (ct), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

  ct->store = gtk_tree_store_new (NUM_COLUMNS,
                                  G_TYPE_STRING,
                                  G_TYPE_STRING);
  ct->treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (ct->store));
  gtk_tree_view_set_model (GTK_TREE_VIEW (ct->treeview), GTK_TREE_MODEL (ct->store));
  column = gtk_tree_view_column_new_with_attributes ("Name",
                                                     gtk_cell_renderer_text_new (),
                                                     "text", NAME_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (ct->treeview), column);
  column = gtk_tree_view_column_new_with_attributes ("Info",
                                                     gtk_cell_renderer_text_new (),
                                                     "text", INFO_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (ct->treeview), column);
  
  gtk_widget_show (ct->treeview);
  gtk_container_add (GTK_CONTAINER (ct), ct->treeview);
  gtk_widget_show (GTK_WIDGET (ct));
}

static void
gst_element_browser_caps_tree_set_property (GObject* object, guint prop_id, 
                                            const GValue* value, GParamSpec* pspec)
{
  GstElementBrowserCapsTree *ct;
	    
  ct = GST_ELEMENT_BROWSER_CAPS_TREE (object);

  switch (prop_id) {
    case PROP_ELEMENT_FACTORY:
      ct->factory = (GstElementFactory*) g_value_get_object (value);
      break;
    case PROP_ELEMENT:
      ct->element = (GstElement*) g_value_get_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      return;
  }

  update_caps_tree (ct);
}

static void
gst_element_browser_caps_tree_get_property (GObject* object, guint prop_id, 
                                            GValue* value, GParamSpec* pspec)
{
  GstElementBrowserCapsTree *ct;
	    
  ct = GST_ELEMENT_BROWSER_CAPS_TREE (object);

  switch (prop_id) {
    case PROP_ELEMENT_FACTORY:
      g_value_set_object (value, (GObject*)ct->factory);
      break;
    case PROP_ELEMENT:
      g_value_set_object (value, (GObject*)ct->element);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

GtkWidget *gst_element_browser_caps_tree_new () 
{
  return GTK_WIDGET (g_object_new (gst_element_browser_caps_tree_get_type (), NULL));
}

static void
update_caps_tree (GstElementBrowserCapsTree *ct)
{
  const GList *pads, *templs;
  GtkTreeStore *store;
  GtkTreeIter iter;

  store = ct->store;

  gtk_tree_store_clear (store);

  if (ct->element) {
    pads = gst_element_get_pad_list (ct->element);
    while (pads) {
      GstPad *pad = (GstPad *)pads->data;
      GstCaps *caps = gst_pad_get_caps (pad);
      gchar *mime;
      
      if (caps) {
        GstType *type;
        type = gst_type_find_by_id (caps->id);
        mime = type->mime;
      }
      else {
        mime = "unknown/unknown";
      }
      
      gtk_tree_store_append (store, &iter, NULL);
      gtk_tree_store_set (store, &iter,
                          NAME_COLUMN, g_strdup (gst_pad_get_name (pad)),
                          INFO_COLUMN, mime,
                          -1);
      
      add_caps_to_tree (caps, store, &iter);
      
      pads = g_list_next (pads);
    }
  }
  
  if (ct->factory) {
    templs = ct->factory->padtemplates;
    while (templs) {
      GstPadTemplate *templ = (GstPadTemplate *)templs->data;
      GstCaps *caps = templ->caps;
      gchar *mime;
      
      if (caps) {
        GstType *type;
        type = gst_type_find_by_id (caps->id);
        mime = type->mime;
      }
      else {
        mime = "unknown/unknown";
      }
      
      gtk_tree_store_append (store, &iter, NULL);
      gtk_tree_store_set (store, &iter,
                          NAME_COLUMN, g_strdup (templ->name_template),
                          INFO_COLUMN, mime,
                          -1);
      
      add_caps_to_tree (caps, store, &iter);
      
      templs = g_list_next (templs);
    }
  }
}

static void
add_caps_to_tree (GstCaps *caps, GtkTreeStore *store,
                  GtkTreeIter *parent) 
{
  GtkTreeIter iter;
  GList *props;
  GstPropsEntry *prop;
  GstProps *properties;

  if (caps) {
    properties = gst_caps_get_props (caps);
    if (properties) {
      props = properties->properties;
      while (props) {
        prop = (GstPropsEntry*)(props->data);
        props = g_list_next(props);
        
        gtk_tree_store_append (store, &iter, parent);
        gtk_tree_store_set (store, &iter,
                            NAME_COLUMN, gst_props_entry_get_name (prop),
                            INFO_COLUMN, print_prop (prop),
                            -1);
      }
    }
  }
}

/* this should really be a GtkTreeViewDataFunc */
static gchar*
print_prop (GstPropsEntry *prop) 
{
  GstPropsType type;

  type = gst_props_entry_get_type (prop);

  switch (type) {
    case GST_PROPS_INT_TYPE:
    {
      gint val;
      gst_props_entry_get_int (prop, &val);
      return g_strdup_printf ("Integer: %d", val);
      break;
    }
    case GST_PROPS_INT_RANGE_TYPE:
    {
      gint min, max;
      gst_props_entry_get_int_range (prop, &min, &max);
      return g_strdup_printf ("Integer range: %d - %d", min, max);
      break;
    }
    case GST_PROPS_FLOAT_TYPE:
    {
      gfloat val;
      gst_props_entry_get_float (prop, &val);
      return g_strdup_printf ("Float: %f", val);
      break;
    }
    case GST_PROPS_FLOAT_RANGE_TYPE:
    {
      gfloat min, max;
      gst_props_entry_get_float_range (prop, &min, &max);
      return g_strdup_printf ("Float range: %f - %f", min, max);
      break;
    }
    case GST_PROPS_BOOLEAN_TYPE:
    {
      gboolean val;
      gst_props_entry_get_boolean (prop, &val);
      return g_strdup_printf ("Boolean: %s", val ? "TRUE" : "FALSE");
      break;
    }
    case GST_PROPS_STRING_TYPE:
    {
      const gchar *val;
      gst_props_entry_get_string (prop, &val);
      return g_strdup_printf ("String: %s", val);
      break;
    }
    case GST_PROPS_FOURCC_TYPE:
    {
      guint32 val;
      gst_props_entry_get_fourcc_int (prop, &val);
      return g_strdup_printf ("FourCC: '%c%c%c%c'",
             (gchar)( val        & 0xff), 
	     (gchar)((val >> 8)  & 0xff),
             (gchar)((val >> 16) & 0xff), 
	     (gchar)((val >> 24) & 0xff));
      break;
    }
    case GST_PROPS_LIST_TYPE:
    {
      const GList *list;
      gchar *ret;
      gint count = 0;

      gst_props_entry_get_list (prop, &list);
      ret = g_strdup_printf ("List: ");
      while (list) {
        GstPropsEntry *listentry;

        listentry = (GstPropsEntry*) (list->data);
        g_strconcat (ret, print_prop (listentry), count++ ? ", " : "", NULL);

        list = g_list_next (list);
      }
      return ret;
      break;
    }
    default:
      return g_strdup_printf("unknown props %d", type);
  }
}
