/* GStreamer Element
 * Copyright (C) 2008 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-bufferjoin
 *
 * <refsect2>
 * <para>
 * Joins consecutive buffers that have identical timestamps
 * (valid durations will be aggregated).
 *
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/base/gstadapter.h>

#define GST_TYPE_BUFFER_JOIN \
  gst_buffer_join_get_type ()
#define GST_BUFFER_JOIN(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_BUFFER_JOIN, GstBufferJoin))
#define GST_BUFFER_JOIN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_BUFFER_JOIN, GstBufferJoinClass))
#define GST_IS_BUFFER_JOIN(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_BUFFER_JOIN))
#define GST_IS_BUFFER_JOIN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_BUFFER_JOIN))

typedef struct _GstBufferJoin GstBufferJoin;
typedef struct _GstBufferJoinClass GstBufferJoinClass;

struct _GstBufferJoin
{
  GstElement parent;

  /* pads */
  GstPad *sinkpad, *srcpad;

  /* property */
  gboolean join_none;
  gboolean join_flags;

  GstBuffer *buffer;
};

struct _GstBufferJoinClass
{
  GstElementClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (buffer_join_debug);
#define GST_CAT_DEFAULT buffer_join_debug

static GstStaticPadTemplate sink_template = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate src_template = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

/* properties */
enum
{
  PROP_0,
  PROP_JOIN_NONE,
  PROP_JOIN_FLAGS
};

#define DEFAULT_PROP_JOIN_NONE   FALSE
#define DEFAULT_PROP_JOIN_FLAGS  FALSE

static void gst_buffer_join_finalize (GObject * object);
static GstFlowReturn gst_buffer_join_flush (GstBufferJoin * filter,
    gboolean send);
static GstCaps *gst_buffer_join_getcaps (GstPad * pad);
static gboolean gst_buffer_join_setcaps (GstPad * pad, GstCaps * caps);
static gboolean gst_buffer_join_event (GstPad * pad, GstEvent * event);
static GstFlowReturn gst_buffer_join_chain (GstPad * pad, GstBuffer * buf);
static GstStateChangeReturn gst_buffer_join_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_buffer_join_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_buffer_join_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstBufferJoin, gst_buffer_join, GstElement, GST_TYPE_ELEMENT);

static void
gst_buffer_join_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "Buffer Join",
      "Generic", "Joins consecutive buffers with identical timestamps",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_buffer_join_class_init (GstBufferJoinClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (buffer_join_debug, "bufferjoin", 0, "Buffer Join");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_buffer_join_finalize);

  gobject_class->set_property = gst_buffer_join_set_property;
  gobject_class->get_property = gst_buffer_join_get_property;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_JOIN_NONE,
      g_param_spec_boolean ("join-none", "Join None",
          "Join buffers with invalid time",
          DEFAULT_PROP_JOIN_NONE, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_JOIN_FLAGS,
      g_param_spec_boolean ("join-flags", "Join Flags",
          "Join buffers with different flags",
          DEFAULT_PROP_JOIN_NONE, G_PARAM_READWRITE));

  element_class->change_state =
      GST_DEBUG_FUNCPTR (gst_buffer_join_change_state);
}

static void
gst_buffer_join_init (GstBufferJoin * filter, GstBufferJoinClass * klass)
{
  filter->sinkpad = gst_pad_new_from_static_template (&sink_template, "sink");
  gst_pad_set_setcaps_function (filter->sinkpad,
      GST_DEBUG_FUNCPTR (gst_buffer_join_setcaps));
  gst_pad_set_event_function (filter->sinkpad,
      GST_DEBUG_FUNCPTR (gst_buffer_join_event));
  gst_pad_set_chain_function (filter->sinkpad,
      GST_DEBUG_FUNCPTR (gst_buffer_join_chain));
  gst_pad_set_getcaps_function (filter->sinkpad,
      GST_DEBUG_FUNCPTR (gst_buffer_join_getcaps));
  gst_element_add_pad (GST_ELEMENT (filter), filter->sinkpad);

  filter->srcpad = gst_pad_new_from_static_template (&src_template, "src");
  gst_pad_set_getcaps_function (filter->srcpad,
      GST_DEBUG_FUNCPTR (gst_buffer_join_getcaps));
  gst_element_add_pad (GST_ELEMENT (filter), filter->srcpad);

  filter->buffer = NULL;
  filter->join_none = DEFAULT_PROP_JOIN_NONE;
  filter->join_flags = DEFAULT_PROP_JOIN_FLAGS;
  gst_buffer_join_flush (filter, FALSE);
}


static void
gst_buffer_join_finalize (GObject * object)
{
  GstBufferJoin *filter = GST_BUFFER_JOIN (object);

  if (filter->buffer)
    gst_buffer_unref (filter->buffer);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


static GstCaps *
gst_buffer_join_getcaps (GstPad * pad)
{
  GstBufferJoin *filter;
  GstPad *otherpad;
  GstCaps *result;

  filter = GST_BUFFER_JOIN (GST_PAD_PARENT (pad));

  otherpad = (pad == filter->srcpad ? filter->sinkpad : filter->srcpad);
  result = gst_pad_peer_get_caps (otherpad);
  if (result == NULL)
    result = gst_caps_new_any ();

  return result;
}

static gboolean
gst_buffer_join_setcaps (GstPad * pad, GstCaps * caps)
{
  GstBufferJoin *filter = GST_BUFFER_JOIN (GST_PAD_PARENT (pad));

  /* may be a renegotiation; flush previous and accept new caps */
  gst_buffer_join_flush (filter, TRUE);

  return gst_pad_set_caps (filter->srcpad, caps);
}


static GstFlowReturn
gst_buffer_join_flush (GstBufferJoin * filter, gboolean send)
{
  GstFlowReturn ret = GST_FLOW_OK;

  if (filter->buffer) {
    if (send) {
      ret = gst_pad_push (filter->srcpad, filter->buffer);
      filter->buffer = NULL;
    } else {
      gst_buffer_unref (filter->buffer);
      filter->buffer = NULL;
    }
  }

  return ret;
}

static gboolean
gst_buffer_join_event (GstPad * pad, GstEvent * event)
{
  GstBufferJoin *filter;
  gboolean ret;

  filter = GST_BUFFER_JOIN (gst_pad_get_parent (pad));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_FLUSH_STOP:
      gst_buffer_join_flush (filter, FALSE);
      /* fall-through */
    default:
      ret = gst_pad_event_default (pad, event);
      break;
  }

  gst_object_unref (filter);
  return ret;
}

static GstFlowReturn
gst_buffer_join_chain (GstPad * pad, GstBuffer * buf)
{
  GstBufferJoin *filter;
  GstFlowReturn ret = GST_FLOW_OK;
  GstClockTime time;
  GstBufferFlag flags;

  filter = GST_BUFFER_JOIN (GST_PAD_PARENT (pad));

  time = GST_BUFFER_TIMESTAMP (buf);
  flags = GST_BUFFER_FLAGS (buf);
  if (filter->buffer &&
      ((time != GST_BUFFER_TIMESTAMP (filter->buffer)) ||
          (!filter->join_none && time == GST_CLOCK_TIME_NONE &&
              GST_BUFFER_TIMESTAMP (filter->buffer) == GST_CLOCK_TIME_NONE) ||
          (!filter->join_flags && flags != GST_BUFFER_FLAGS (filter->buffer))))
    ret = gst_buffer_join_flush (filter, TRUE);

  if (ret != GST_FLOW_OK)
    goto exit;

  if (filter->buffer) {
    GST_DEBUG_OBJECT (filter, "joining buffers at time %" GST_TIME_FORMAT,
        GST_TIME_ARGS (time));
    flags = GST_BUFFER_FLAGS (filter->buffer);
    filter->buffer = gst_buffer_join (filter->buffer, buf);
    GST_BUFFER_FLAGS (filter->buffer) = flags;
  } else
    filter->buffer = buf;

exit:
  return ret;
}


static void
gst_buffer_join_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstBufferJoin *filter;

  g_return_if_fail (GST_IS_BUFFER_JOIN (object));

  filter = GST_BUFFER_JOIN (object);

  switch (prop_id) {
    case PROP_JOIN_NONE:
      filter->join_none = g_value_get_boolean (value);
      break;
    case PROP_JOIN_FLAGS:
      filter->join_flags = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_buffer_join_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstBufferJoin *filter;

  g_return_if_fail (GST_IS_BUFFER_JOIN (object));

  filter = GST_BUFFER_JOIN (object);

  switch (prop_id) {
    case PROP_JOIN_NONE:
      g_value_set_boolean (value, filter->join_none);
      break;
    case PROP_JOIN_FLAGS:
      g_value_set_boolean (value, filter->join_flags);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_buffer_join_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;
  GstBufferJoin *filter = GST_BUFFER_JOIN (element);

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      gst_buffer_join_flush (filter, FALSE);
      break;
    default:
      break;
  }

done:
  return ret;
}
